/*
** ~ppr/src/libppr/options.c
** Copyright 1995, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This is a generic options parser.  An additional routine for reporting
** option parsing errors in filters is in the file foptions.c.
**
** This file was last modified 22 February 1996.
*/

#include "global_defines.h"
#include <string.h>
#include <ctype.h>

const char *options_string;		/* the string we are working on */
const char *options_error;		/* the error we found */
int options_error_context_index;	/* index to area of error */
int options_error_index;		/* precise point of error */

static const char *ptr = (char*)NULL;	/* point we are working at */
static int temp_error_index;
static start_skip;

/*
** Reset these routines, giving them a new set of options
** to parse.
*/
void options_start(const char *s)
    {
    ptr = options_string = s;		/* set ptr to start of options */
    options_error = "<NO ERROR>";	/* set error message to dummy value */
    options_error_context_index = 0;
    options_error_index = 0;		/* set error position indicator to string start */
    temp_error_index = 0;
    start_skip = 0;
    } /* end of options_start() */
    
/*
** Get the next name and value.  Return 1 if we are sucessfull, 
** zero if we are at the end of the string, and -1 if there is an 
** error.
**
** On each return, options_error points to the error message (if applicable), 
** options_string point to the string which was passed to options_start(),
** options_error_index points to the point at which the error was found
** and options_error_context_index points to the area of the error.
*/
int options_get_one(char *name, int maxnamelen, char *value, int maxvaluelen)
    {
    int namelen, valuelen;

    ptr += start_skip;
    options_error_index += start_skip;
    start_skip = 0;

    if(ptr == (char*)NULL)
    	{
    	options_error = "libppr: options.c: options_get_one(): options_start() was not called";
    	return -1;
    	}

    if( *ptr == (char)NULL )		/* If at end of options, */
    	return 0;			/* so indicate. */
    	
    options_error_context_index = temp_error_index;
    temp_error_index = options_error_index;

    namelen=strcspn(ptr,"= \t");	/* length til equals sign, space, or tab */

    if( namelen > (maxnamelen-1) )
    	{
    	options_error = "Option name is illegaly long";
    	return -1;
    	}

    if( namelen == 0 )
    	{
    	options_error = "Missing option name";
    	return -1;
    	}

    strncpy(name,ptr,namelen);		/* copy the name, */
    name[namelen]=(char)NULL;		/* and NULL terminate it */

    ptr += namelen;			/* move beyond name */
    options_error_index += namelen;

    if( *ptr != '=' )			/* if no equals sign, */
    	{				/* that is a problem */
	if( ptr[strspn(ptr," \t")] == '=' )
	    options_error="No spaces allowed before equals sign";
	else
	    options_error="Equals sign expected";

    	return -1;
    	}

    ptr++;				/* move beyond equals sign */
    options_error_index++;

    if( isspace(*ptr) )
    	{
    	options_error="No spaces allowed after equals sign";
    	return -1;
    	}

    valuelen=strcspn(ptr," \t");	/* length of value */

    if( valuelen > (maxvaluelen-1) )
    	{
    	options_error="Option value is illegaly long";
    	return -1;
    	}
    	
    if( valuelen==0 )
    	{
    	options_error="Empty value is not allowed";
    	return -1;
    	}

    strncpy(value, ptr, valuelen);	/* copy the value, */
    value[valuelen] = (char)NULL;	/* and NULL terminate it */

    start_skip = valuelen;				/* compute distance to start */
    start_skip += strspn(&ptr[start_skip]," \t");	/* of next name=value pair */

    return 1;
    } /* end of options_get_one() */
	
/* end of file */
