/*
** ~ppr/src/libppr/nodeid.c
** Copyright 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 20 December 1996.
*/

/*
** This module implements a set of functions which allow a PPR node name
** to be represented as a small integer.
*/

#include "global_defines.h"

/* #define DEBUG 1 */

/*
** This structure describes one entry in the node name table.
** If refcount is 0, the entry is not in use.
*/
struct NODEID
    {
    int refcount;
    char *name;
    } ;

/*
** This is the array of node names.
*/
static struct NODEID nodes[MAX_NODES];
static table_size = 0;

/*
** When a node name is fed to this function, it searches nodes[]
** for a matching entry.  If none is found, one is created with a 
** reference count of one.  If one is found, the reference count
** is incremented.  The index of the table entry, new or existing,
** is returned.
*/
int assign_nodeid(char *nodename)
    {
    int x;				/* roving index into nodename array */
    int empty_spot = -1;		/* first empty spot we see */

    for(x=0; x < table_size; x++)	/* Move thru the part of the table which */
    	{				/* has been used so far. */
	if( nodes[x].refcount == 0 )	/* If this is an unused node, */
	    {
	    if(empty_spot == -1)	/* if we haven't seen one yet, */
	    	empty_spot = x;		/* make a note of it; */
	    continue;			/* nothing else of interest in this entry. */
	    }

    	if(strcmp(nodes[x].name,nodename)==0)	/* If the names match, */
    	    {
    	    nodes[x].refcount++;		/* increment reference count, */
	    #ifdef DEBUG
	    debug("assign_nodeid(\"%s\"): refcount increased to %d", nodename, nodes[x].refcount);
	    #endif
    	    return x;				/* return the name */
    	    }
    	}

    if( empty_spot == -1 )			/* if no empty spot found, */
    	{					/* use space at end of table */
	empty_spot = table_size;

	if(++table_size > MAX_NODES)
	    fatal(lib_misc_fatal,"libppr: nodeid.c: nodename_to_nodeid(): table overflow");
    	}

    {
    int myalloc_save = myalloc_checkpoint_get();
    nodes[empty_spot].name = mystrdup(nodename);
    myalloc_checkpoint_put(myalloc_save);
    }
    nodes[empty_spot].refcount = 1;
    #ifdef DEBUG
    debug("assign_nodeid(\"%s\"): new node, refcount=1", nodename);
    #endif
    
    return empty_spot;
    } /* end of nodename_to_nodeid() */
    
/*
** The reference count of the indicated table entry
** is decremented.  If the decrement operation reduces
** the reference count to zero, the storage occupied
** by the name is freed.
*/
void free_nodeid(int nodeid)
    {
    if( nodeid < 0 || nodeid >= table_size )
    	fatal(lib_misc_fatal, "libppr: nodeid.c: free_nodeid(): %d is an invalid nodeid", nodeid);
    
    if( nodes[nodeid].refcount == 0 )
    	fatal(lib_misc_fatal, "libppr: nodeid.c: free_nodeid(): attempt to free invalid nodeid %d", nodeid);

    #ifdef DEBUG
    debug("free_nodeid(%d): refcount was %d", nodeid, nodes[nodeid].refcount);
    #endif

    if( --nodes[nodeid].refcount == 0 )		/* if decrementing the reference */
	{					/* count yields zero, */
	int myalloc_save = myalloc_checkpoint_get();
    	myfree(nodes[nodeid].name);
	myalloc_checkpoint_put(myalloc_save);

	if( (nodeid + 1) == table_size )	/* shrink the table if possible */
	    table_size--;
    	}
    } /* end of free_nodeid() */
    
/*
** This function returns the node name associated with
** a certain table entry.
*/
const char *nodeid_to_nodename(int nodeid)
    {
    #ifdef DEBUG
    debug("nodeid_to_nodename(%d)", nodeid);
    #endif

    if( nodeid < 0 || nodeid >= table_size )
    	fatal(lib_misc_fatal, "libppr: nodeid.c: nodeid_to_nodename(): %d is an invalid nodeid", nodeid);

    if( nodes[nodeid].refcount < 1 )
	fatal(lib_misc_fatal, "libppr: nodeid.c: nodeid_to_nodename(): node %d is already freed", nodeid);

    return nodes[nodeid].name;    
    } /* end of nodeid_to_nodename() */

/* end of file */
