/*
** ~ppr/src/libppr/myalloc.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file was last modified 19 November 1996.
*/

/*
** Memory allocator for PPR programs.
*/

#include "global_defines.h"
#include <stdlib.h>
#include <string.h>

/* #define DEBUG 1 */

static myalloc_blocks = 0;		/* number of blocks allocated */
static myalloc_blocks_saved;		/* number of blocks at last myalloc_checkpoint() */

/*
** PPR memory allocator.
*/
void *myalloc(size_t number, size_t size)
    {
    void *rval;

    #ifdef DEBUG
    debug("myalloc(number=%ld, size=%ld)", (long)number, (long)size);
    #endif

    myalloc_blocks++;

    if( (rval=malloc(size*number)) != (void*)NULL )
	{
        return rval;
        }
    else
        {
        fatal(lib_memory_fatal,"mymalloc(): out of memory");
        }
    } /* end of myalloc() */

/*
** PPR memory allocator, duplicate a string.
*/
char *mystrdup(const char *string)
    {
    char *rval;

    #ifdef DEBUG
    debug("mystrdup(\"%s\")", string);
    #endif

    myalloc_blocks++;

    if( (rval=(char*)malloc(strlen(string)+1)) != (char*)NULL )
        {
        strcpy(rval,string);
        return rval;
        }
    else
        {
        fatal(lib_memory_fatal,"mystrdup(): out of memory");
        }
    } /* end of mystrdup() */

/*
** PPR memory allocator, duplicate a part of a string.
*/
char *mystrndup(const char *string, size_t len)
    {
    char *rval;

    #ifdef DEBUG
    debug("mystrndup()");
    #endif

    myalloc_blocks++;

    if( ( rval = (char*)malloc(len+1) ) != (char*)NULL )
        {
        strncpy(rval,string,len);
        rval[len] = (char)NULL;
        return rval;
        }
    else
        {
        fatal(lib_memory_fatal,"mystrndup(): out of memory");
        }
    } /* end of mystrndup() */

/*
** PPR memory allocator, change the size of a block.
*/
void *myrealloc(void *ptr, size_t number, size_t size)
    {
    void *rval;
    
    #ifdef DEBUG
    debug("myrealloc()");
    #endif

    if( (rval = realloc(ptr, number*size)) != (void*)NULL )
	{
    	return rval;
    	}
    else
        {
        fatal(lib_memory_fatal,"myrealloc(): out of memory");
        }
    } /* end of myrealloc() */

/*
** PPR memory allocator, free a block.
*/
void myfree(void *ptr)
    {
    #ifdef DEBUG
    debug("myfree()");
    #endif

    myalloc_blocks--;
    free(ptr);
    } /* end of myfree() */

/*
** Save the value of myalloc_blocks.
*/
void myalloc_checkpoint(void)
    {
    myalloc_blocks_saved = myalloc_blocks;
    } /* end of myalloc_checkpoint() */
    
/*
** Routines to get and put the difference between myalloc_blocks
** and myalloc_blocks_saved.
**
** This is used by library routines and signal handler
** so they don't mess up the myalloc_checkpoint()/myalloc_assert()
** mechanisms of routines they are called from or routines
** they interupt.
*/
int myalloc_checkpoint_get(void) { return myalloc_blocks - myalloc_blocks_saved; }
void myalloc_checkpoint_put(int n) { myalloc_blocks_saved = myalloc_blocks - n; }

/*
** This is called to assert that myalloc_blocks differs
** from the figure saved by myalloc_checkpoint by a 
** certain amount.  This is called by a macro called
** myalloc_assert().  A possitive amount is increase, a negative
** amount is decrease.
*/
void _myalloc_assert(const char *file, int line, int assertion)
    {
    if( (myalloc_blocks_saved + assertion) != myalloc_blocks )
    	fatal(lib_memory_fatal, "myalloc_assert(): assertion failed at %s line %d (is %d, expected %d)",
    		file, line, myalloc_blocks, (myalloc_blocks_saved + assertion) );
    } /* end of _myalloc_assert() */

/* end of file */
