/*
** ~ppr/src/libppr/alert.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last modified 7 January 1997.
*/

/*
** This module contains routines for posting printer alerts.  It should be
** safe to call these routines from within a signal handler.  Notice that
** we build the alert line in a 256 byte buffer.  No alert line should even
** approach that length.  Previous versions of these functions did not have
** that limit because they used used the C I/O library, but that library
** may not be reentrant.
**
** Normally the alert is posted to the printer's alert log file, but if
** the printer name is "-" then the alert message is sent to stderr.
*/

#include "global_defines.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <time.h>

void valert(const char printer[], int dateflag, const char string[], va_list args)
    {
    int afile;
    char fname[MAX_PATH];
    time_t now;
    struct stat statbuf;

    if(printer == (char*)NULL)	/* if called as the result of a fatal */
    	return;			/* error before printer name determined, get out */

    /* We will be comparing the current time to the alert file modification time. */
    time(&now);

    /* The name of the alert log file for this printer. */
    sprintf(fname, ALERTDIR"/%s", printer);

    /* If the printer name is the special value "-", use stderr. */
    if(strcmp(printer, "-") == 0)
    	afile = 2;
    /* If can't stat() the alert log or it is older than one hour truncate it. */
    else if( stat(fname, &statbuf) || (difftime(now, statbuf.st_mtime) > 3600) ) 
	afile = open(fname, O_WRONLY | O_CREAT | O_TRUNC | O_APPEND, UNIX_644);
    /* If younger than one hour, append. */
    else
	afile = open(fname, O_WRONLY | O_CREAT | O_APPEND, UNIX_644);

    /* If we have a file we can use, */
    if(afile != -1)
        {
	char templine[256];

	templine[0] = (char)NULL;

        if(dateflag)
            {
            strcat(templine, "\n");		/* blank line */
            strcat(templine, ctime(&now));	/* print the time */
            }

	vsprintf(&templine[strlen(templine)], string, args);

	strcat(templine, "\n");
	write(afile, templine, strlen(templine));

	if(strcmp(printer, "-"))
	    close(afile);
        }       

    } /* end of valert() */

void alert(const char printer[], int dateflag, const char string[], ...)
    {
    va_list va;

    va_start(va,string);
    valert(printer, dateflag, string, va);
    va_end(va);
    }

/* end of file */
