/*
** ~ppr/src/interfaces/tcpip.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last revised 7 January 1997.
*/

/*
** Raw TCP/IP interface for communicating with HP Jetdirect cards or
** Extended Systems Pocket Print Servers.  This interface is either
** used directly to send PostScript, or is called by the gstcpip interface
** to send Ghostscript's output to the printer.
*/

#include "global_defines.h"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#ifndef NO_POLL
#include <stropts.h>
#endif
#include "interface.h"

/* Define this if you want debuging */
/* #define DEBUGGING 1 */

/* Set to TRUE by the SIGALRM handler to indicate a connect() timeout. */
int sigalrm_caught = FALSE;

/* Global name of printer for use when calling alert(). */
char *printer;			/* argv[1] */

/* The socket with the printer at the other end. */
int sockfd;

/* The number of unacknowledged control-D's. */
int eof_balance = 0;

/*
** Write lines to the debug file.
*/
#ifdef DEBUGGING
void debug(char *string, ... )
    {
    va_list va;
    FILE *file;

    va_start(va,string);
    if( (file=fopen(LOGDIR"/interface_tcpip","a")) != NULL )
        {
        fprintf(file,"DEBUG: (%ld) ",(long)getpid());
        vfprintf(file,string,va);
        fprintf(file,"\n");
        fclose(file);
        }
    va_end(va);
    } /* end of debug() */
#define DEBUG(a) debug a
#else
#define DEBUG(a)
#endif

/*
** Two routines used to exit with an error indication.  Normally these simply
** exit with the values EXIT_PRNERR and EXIT_PRNERR_NORETRY respectively,
** however if the undocumented option "patron=PID" is used, they will send
** SIGUSR1 or SIGUSR2 to the indicated process and exit with a normal code.
**
** This bizzar feature is used by the interface "gstcpip".
*/
long patron = -1;		/* PID of process to inform */

void exit_PRNERR(void)
    {
    DEBUG(("exit_PRNERR()"));

    if(patron != -1)			/* If we are begin run from Ghostscript, */
    	{
    	kill((pid_t)patron,SIGUSR1);	/* tell the shell script */
    	sleep(1);			/* allow interface script time to get signal and exit */
    	exit(EXIT_PRINTED);
    	}

    exit(EXIT_PRNERR);
    } /* end of exit_PRNERR() */
    
void exit_PRNERR_NORETRY(void)
    {
    DEBUG(("exit_PRNERR_NORETRY()"));

    if(patron != -1)
    	{
    	kill((pid_t)patron,SIGUSR2);
    	sleep(1);
    	exit(EXIT_PRINTED);
    	}
    exit(EXIT_PRNERR_NORETRY);
    } /* end of exit_PRNERR_NORETRY() */

/*
** Make the connection to the printer.
** Return the file descriptor.
*/
int make_connection(char *address, int connect_timeout, int sndbuf_size)
    {
    int sockfd;
    struct sockaddr_in printer_addr;
    char *ptr;
    struct hostent *hostinfo;
    int retval;

    /* Clear the printer address structure. */
    memset(&printer_addr, 0, sizeof(printer_addr));
    
    /* Parse the address into host and port. */
    if( strpbrk(address, " \t") != (char*)NULL )
    	{
    	alert(printer, TRUE, "Spaces and tabs not allowed in TCP/IP a printer address.");
    	alert(printer, FALSE, "\"%s\" does not conform to this requirement.", address);
    	exit_PRNERR_NORETRY();
    	}

    if( (ptr=strchr(address,':')) == (char*)NULL || ! isdigit(ptr[1]) )
    	{
    	alert(printer, TRUE, "TCP/IP printer address must be in form \"host:portnum\",");
    	alert(printer, FALSE, "\"%s\" does not conform to this requirement.", address);
	exit_PRNERR_NORETRY();
	}

    /* Put the port number in the structure. */
    printer_addr.sin_port = htons(atoi(ptr+1));

    /* Terminate the host part so that the port number will be ignored for now. */
    *ptr = (char)NULL;

    /*
    ** If convertion of a dotted address works, use it,
    ** otherwise, use gethostbyname().
    */
    if( (int)(printer_addr.sin_addr.s_addr=inet_addr(address)) != -1 )
	{		/* !!! cast to int shouldn't be, -1 should be INETADDR_NONE !!! */
	printer_addr.sin_family = AF_INET;
    	}
    else
    	{
	if( (hostinfo=gethostbyname(address)) == (struct hostent *)NULL )
	    {
	    alert(printer, TRUE, "TCP/IP interface can't get IP address for \"%s\".", address);
	    exit_PRNERR();	/* DNS failures can be temporary */
	    }

	printer_addr.sin_family = hostinfo->h_addrtype;
	memcpy(&printer_addr.sin_addr, hostinfo->h_addr_list[0], hostinfo->h_length);
    	}

    /*
    ** Now that inet_addr() and gethostbyname() have had a chance to 
    ** examine it, put the address back the way it was.
    */
    *ptr = ':';

    /* Create a socket. */
    if( (sockfd=socket(AF_INET, SOCK_STREAM, 0)) == -1 )
	{
	alert(printer, TRUE, "tcpip interface: socket() failed, errno=%d (%s)", errno, strerror(errno));
	exit_PRNERR();
	}		
	
    /*
    ** Connect the socket to the printer.
    ** Some systems, noteably Linux, fail to reliably implement
    ** connect() timeouts; that is why we use alarm() to 
    ** create our own timeout.
    */
    DEBUG(("calling connect()"));
    alarm(connect_timeout);
    retval = connect(sockfd, (struct sockaddr*) &printer_addr, sizeof(printer_addr));
    alarm(0);
    DEBUG(("connect() done"));

    /* If a timeout occured, */
    if(sigalrm_caught)
    	{
	alert(printer, TRUE, "Printer \"%s\" is not responding.", address);
	alert(printer, FALSE, "(Aborted after connect() blocked for %d seconds.)", connect_timeout);
    	exit_PRNERR();
    	}

    /* If connect() failed, */
    if(retval == -1)
	{
	DEBUG(("connect() failed, errno=%d (%s)", errno, strerror(errno)));

	if(errno == ETIMEDOUT)		/* I do not know why this does not occur. */
	    {				/* Because it doesn't, we use alarm(). */
	    alert(printer, TRUE, "Timeout while trying to connect to printer.");
	    alert(printer, FALSE, "(Connect() reported error ETIMEDOUT.)");
	    exit_PRNERR();
	    }

	if(errno == ECONNREFUSED)
	    {
	    alert(printer, TRUE, "Printer at \"%s\" has refused connection.", address);
	    exit_PRNERR();
	    }

	alert(printer, TRUE, "tcpip interface: connect() failed, errno=%d (%s)", errno, strerror(errno));
	exit_PRNERR();
	}
		
    /*
    ** Turn on the socket option which detects dead connexions.
    */
    {
    int true_variable = 1;
    if( setsockopt(sockfd, SOL_SOCKET, SO_KEEPALIVE, (void*)&true_variable, sizeof(true_variable)) == -1 )
    	{
    	alert(printer, TRUE, "tcpip interface: setsockopt() failed, errno=%d (%s)", errno, strerror(errno));
	exit_PRNERR_NORETRY();
    	}
    }

    /*
    ** If the user has supplied an explicit output
    ** buffer size setting, use it.
    */
    if(sndbuf_size != 0)
	{
	DEBUG(("setting SOL_SNDBUF to %d", sndbuf_size));

	if(setsockopt(sockfd, SOL_SOCKET, SO_SNDBUF, (void*)&sndbuf_size, sizeof(sndbuf_size) ) == -1)
	    {
	    alert(printer, TRUE, "tcpip interface: setsockopt() failed, errno=%d (%s)", errno, strerror(errno));
	    exit_PRNERR_NORETRY();
	    }    
	}

    return sockfd;
    } /* end of make_connection() */

/*
** Copy the file to the printer.
** Called only once.
*/
void copy_file(int sockfd, int write_size, int feedback)
    {
    int just_read, bytes_left, just_written;
    char *ptr;
    char *buffer;		/* Buffer for data transfer */
    int x;

    /*
    ** Allocate memory for the copy buffer.  We don't use
    **myalloc()  because we don't want to define fatal()
    ** for just this one trivial memory allocation.
    */
    if( (buffer = (char*)malloc( write_size * sizeof(char) )) == (char*)NULL )
	{
	alert(printer, TRUE, "tcpip interface: malloc() failed, errno=%d (%s)", errno, strerror(errno));
	exit_PRNERR();
	}
	
    do	{
	if( (just_read=read(0, buffer, write_size)) == -1 )
	    {
	    alert(printer, TRUE, "tcpip interface: error reading stdin, errno=%d (%s)", errno, strerror(errno));
	    exit_PRNERR();
	    }

	/*
	** Increment eof_balance for each control-D found in the block.
	** This code is skipt if there is not feedback from the printer. 
	*/
	if(feedback)
	    {
	    for(ptr=buffer,x=0; x < just_read; x++,ptr++)
		{
		if( *ptr == 4 )
		    eof_balance++;
		}
	    }

	/*
	** Write the whole buffer, making multiple calls 
	** to write() if necessary. 
	*/
	bytes_left = just_read;
	ptr = buffer;
	while(bytes_left)			
	    {
	    DEBUG(("write(sockfd,ptr,%d)",bytes_left));
	    if( (just_written=write(sockfd, ptr, bytes_left)) == -1 )
		{
		alert(printer, TRUE, "tcpip interface: error writing to socket, errno=%d (%s)", errno, strerror(errno));
		exit_PRNERR();
		}
	    DEBUG(("    %d",just_written));
	    ptr += just_written;
	    bytes_left -= just_written;				
	    }
	} while(just_read);

    /* Free the copy buffer. */
    free(buffer);
    } /* end of copy_file() */
	
/*
** This is the SIGPIPE handler which detects when
** we try to write to a socket which has become
** disconnected.
*/
void sigpipe_handler(int sig)
    {
    DEBUG(("SIGPIPE"));
    alert(printer, TRUE, "Connection to printer broken (SIGPIPE).");
    exit_PRNERR();
    } /* end of sigpipe_handler() */

/*
** This is the SIGALRM handler which detects if the alarm goes
** off before the remote system responds.  Notice that it does 
** nothing but set a flag.
*/
void sigalrm_handler(int sig)
    {
    DEBUG(("SIGALRM"));
    sigalrm_caught = TRUE;
    } /* end of sigalrm_handler() */

/*
** This is the SIGPOLL or SIGIO handler.  It handles data
** coming back from the printer.
*/
void sigpoll_handler(int sig)
    {
    char buffer[512];
    char cleanbuffer[512];
    int flags;
    int len;
    int countdown;
    char *ptr;
    int x;

    DEBUG(("SIGPOLL"));

    /* Get the current file flags for the socket connected to the printer. */
    if( (flags=fcntl(sockfd,F_GETFL)) == -1 )
    	{
    	alert(printer,TRUE,"tcpip interface: sigpoll_handler(): fcntl() failed, errno=%d (%s)",errno,strerror(errno));
    	exit_PRNERR_NORETRY();
    	}
    	
    /* 
    ** Set O_NONBLOCK.  We do this because we will want read() to return
    ** when there is not more data to read. 
    */
    if( fcntl(sockfd,F_SETFL,flags | O_NONBLOCK) == -1 )
    	{
    	alert(printer,TRUE,"tcpip interface: sigpoll_handler() fcntl(sockfd,F_SETFL,...) failed");
	alert(printer,FALSE,"while setting O_NONBLOCK, errno=%d (%s)",errno,strerror(errno));
	exit_PRNERR_NORETRY();
	}

    /* A loop to copy the data. */
    while( (len=read(sockfd,buffer,sizeof(buffer))) )
    	{
	if( len == -1 )		/* if error occured */
	    {
	    if(errno==EAGAIN)	/* This means no more data. */
	    	break;

	    alert(printer,TRUE,"tcpip interface: error reading from printer, errno=%d (%s)",errno,strerror(errno));
	    exit_PRNERR();
	    }    	

	/* 
	** Count the control-D's in the block.
	** While we do it, we copy anything which is not a
	** control-D into cleanbuffer[].  When we are done,
	** number of bytes in cleanbuffer[] will be stored in
	** "countdown". 
	*/
	for(x=0, ptr=buffer, countdown=0; x < len; x++, ptr++)
	    {
	    if(*ptr == 4)
	    	{
	    	eof_balance--;
	    	}
	    else
	    	{
	    	cleanbuffer[countdown++]=*ptr;
	    	}
	    }

	/* 
	** Write the whole block, using multiple calls to
	** write() if necessary.  We keep going until 
	** countdown, which was computed above in
	** reduced to zero.
	*/
	ptr = cleanbuffer;
	while( countdown > 0 )
	    {
	    if( (len=write(1,ptr,countdown)) == -1 )
	    	{
	    	alert(printer,TRUE,"tcpip interface: error writing to stdout, errno=%d (%s)",errno,strerror(errno));
	    	exit_PRNERR_NORETRY();
	    	}
	    countdown -= len;
	    ptr += len;
	    }
    	}
    
    /* Put the file flags back the way they where. */
    if( fcntl(sockfd,F_SETFL,flags) == -1 )
    	{
    	alert(printer,TRUE,"tcpip interface: sigpoll_handler() fcntl(sockfd,F_SETFL,...) failed");
	alert(printer,FALSE,"while restoring flags, errno=%d (%s)",errno,strerror(errno));
	exit_PRNERR_NORETRY();
	}
    
    DEBUG(("SIGPOLL done"));
    } /* end of sigpoll_handler() */

/* 
** Tie it all together.
*/
int main(int argc, char *argv[])
    {
    char *address;			/* argv[2] */
    char *options;			/* argv[3] */
    int feedback;			/* argv[7] */
    int sndbuf_size = 0;		/* size for SO_SNDBUF, 0 means don't set it */
    int write_size = 4096;		/* size of blocks to pass to write() when printing */
    int sleep_time = 0;			/* time to sleep() after printing */
    int eof_timeout = 0;		/* time to wait for control-d acknowledgement */
    int connect_timeout = 20;		/* connexion timeout in seconds */

    /* Make sure all the arguments are present. */
    if(argc < 8)
	{
	if(argc > 1)
	    alert(argv[1],TRUE,"tcpip interface: inappropriate number of parameters");
	else
	    fprintf(stderr,"tcpip interface: inappropriate number of parameters\n");
	exit_PRNERR_NORETRY();
	}
	
    /* 
    ** Name some of the arguments and 
    ** get the feedback value (TRUE or FALSE).
    */
    printer = argv[1];
    address = argv[2];
    feedback = atoi(argv[7]);

    /*
    ** If there is an environment variable declared, use it in stead of 
    ** the options field.  This is to get around a bug in Ghostscript
    ** which limits the length of the OutputFile string.
    **
    ** If the variable does not exist, we use argument 3.
    */
    if( (options = getenv("PPR_GS_INTERFACE_HACK_OPTIONS")) == (char*)NULL )
	options = argv[3];

    /* Parse the options string, searching for name=value pairs. */
    {
    char name[16];
    char value[16];
    int retval;

    options_start(options);
    while( (retval=options_get_one(name,sizeof(name),value,sizeof(value))) > 0 )
    	{
	/*
	** Undocumented "patron=PID" option.
	** Look for patron process id in options. 
	*/
	if( strcmp(name, "patron") == 0 )
	    {
	    patron = atoi(value);
	    }
	/*
	** The delay after closing connection, before exiting.
	*/
	else if( strcmp(name, "sleep") == 0 )
	    {
	    sleep_time = atoi(value);		    
	    }
	/*
	** Maximum number of seconds to wait for control-D's.
	** The form "eoftimeout" has been `undocumented'.
	*/
	else if(strcmp(name, "eoftimeout") == 0 || strcmp(name, "eof_timeout") == 0)
	    {
	    eof_timeout = atoi(value);
	    }
	/*
	** Size for TCP/IP send buffer
	*/
	else if( strcmp(name,"sndbuf_size") == 0 )
	    {
	    sndbuf_size = atoi(value);
	    }
	/*
	** Size of writes() to printer
	*/
	else if( strcmp(name,"write_size") == 0 )
	    {
	    write_size = atoi(value);
	    }
	/*
	** Connect timeout
	*/
	else if( strcmp(name,"connect_timeout") == 0 )
	    {
	    connect_timeout = atoi(value);
	    }
	/*
	** Catch anything else.
	*/
	else
	    {
	    options_error = "Unrecognized keyword in printer options";
	    retval = -1;
	    break;
	    }	
	  
	} /* end of while() loop */

    if(retval == -1)
    	{
    	alert(printer,TRUE,"Option parsing error:  %s", options_error);
    	alert(printer,FALSE,"%s",options);
    	alert(printer,FALSE,"%*s^ right here", options_error_index, "");
    	exit(EXIT_PRNERR_NORETRY);
    	}
    }

    /* Describe the options in the debuging output. */
    DEBUG(("printer=\"%s\", address=\"%s\", feedback=%d",printer,address,feedback));
    DEBUG(("sleep_time=%d, eof_timeout=%d, patron=%d",sleep_time,eof_timeout,patron));
    DEBUG(("connect_timeout=%d, sndbuf_size=%d, write_size=%d",connect_timeout,sndbuf_size,write_size));

    /* 
    ** Install SIGPIPE handler to detect
    ** broken connections. 
    */
    signal(SIGPIPE,sigpipe_handler);	

    /* Install SIGALRM handler for connect() timeouts. */
    signal(SIGALRM,sigalrm_handler);

    /* Connect to the printer */
    sockfd = make_connection(address, connect_timeout, sndbuf_size);

    /* 
    ** If we are accepting feedback, set up a handler for it.
    ** We will have to do this differently depending on
    ** whether this is a BSD system with SIGIO or a System V
    ** system with SIGPIPE. 
    */
    if(feedback)
	{
	/* Fill in a signal handler description structure. */
	struct sigaction sig;
        sig.sa_handler = sigpoll_handler;
        sigemptyset(&sig.sa_mask);
        sigaddset(&sig.sa_mask,SIGALRM); /* block alarm signals */
        sig.sa_flags = SA_RESTART;
                        
	/*
	** BSD way 
	*/
	#ifdef NO_POLL
	/* Install the signal handler. */
        sigaction(SIGIO, &sig, NULL);

	/* 
	** Say we want the SIGIO signals for the file
	** descriptor sent to this process. 
	*/
	if(fcntl(sockfd,F_SETOWN,getpid()) < 0)
	    {
	    alert(printer,TRUE,"tcpip interface: fcntl() failed on socket, error=%d (%s)",errno,strerror(errno));
	    exit_PRNERR();
	    }

	/* Turn on SIGIO for the pipe file descriptor. */
	if(fcntl(sockfd,F_SETFL,FASYNC) < 0)
	    {
	    alert(printer,TRUE,"tcpip interface: fcntl() failed on socket, errno=%d (%s)",errno,strerror(errno));
	    exit_PRNERR();
	    }

	/*
	** System V way
	*/
	#else
	/* Install the signal handler. */
        sigaction(SIGPOLL, &sig, NULL);

	if( ioctl(sockfd,I_SETSIG, S_RDNORM ) )
	    {
	    alert(printer,TRUE,"tcpip interface: ioctl() failed on socket, errno=%d (%s)",errno,strerror(errno));
	    exit_PRNERR();
	    }
	#endif
        }

    /* Copy the file from STDIN to the printer. */
    copy_file(sockfd, write_size, feedback);

    /*
    ** Wait for all the control-D's to be acknowledged.
    ** If there is no feedback from the printer, eof_balance
    ** will have always been zero.
    **
    ** If eof_timeout is greater than zero, it is the maximum
    ** number of seconds we should wait.
    **
    ** eof_timeout is set with the option "eoftimeout=SECONDS".
    */
    {
    int time_waiting = 0;
    while(eof_balance > 0)
	{
	DEBUG(("sleeping while waiting for eoj acknowledgement"));

	sleep(1);

	if( (eof_timeout > 0) && (++time_waiting > eof_timeout) )
	    break;
	}
    }

    /* Close the connection */
    close(sockfd);	
	
    /*
    ** Pocket print servers have been known to reject a new
    ** connection for a few seconds after closing the previous one.
    ** If more than one job is in the queue at one time, this can result
    ** in every other print attempt producing a fault.  This 
    ** problem is minor and can go unnoticed, but we will have
    ** an option to sleep for a specified number of seconds 
    ** after closing the connection.
    **
    ** sleep_time is set with the option "sleep=SECONDS".
    */
    if(sleep_time > 0)
	{
	DEBUG(("Sleeping for %d seconds for printer recovery",sleep_time));
	sleep(sleep_time);
	}

    DEBUG(("Done"));

    /* We can assume that it was printed. */
    return EXIT_PRINTED;
    } /* end of main() */
	
/* end of file */
