/*
** ~ppr/src/interfaces/serial.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** PPR interface to drive a serial printer.
**
** This file was last modified 1 May 1996.
*/

#include "global_defines.h"
#include "interface.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/wait.h>
#include <ctype.h>
#include <termios.h>
#include <string.h>
#include <sys/ioctl.h>		/* for Linux */

/*
** Uncomment this to include debugging code. Since the debugging
** output appears on stdout, the debugging code must not be in
** when this interface is actually being used with PPR.
*/
/* #define DEBUG 1 */
/* #define DEBUG_CHILD 1 */

/* The size of the read and write buffers in bytes. */
#define BUFFER_SIZE 5120

/* These global variables point to the arguments. */
char *printer_name;
char *printer_address;
char *printer_options;
char *printer_jobname;
int   printer_jobbreak;
int   printer_feedback;

/*
** This global tells whether we have to wait for our control-D
** end-of-file signals to be acknowledged.
*/
int wait_for_acknowledgement = TRUE;

/* This variable counts the number of unacknowledged control-D's. */
int eof_balance = 0;

/* 
** This variable describes the modem lines which we must
** check for before considering the printer to be on line.
*/
int online_modem = 0;

/*
** Handler for SIGHUP.
*/
void sighup_handler(int sig)
    {
    #ifdef DEBUG
    printf("SIGHUP received\n");
    #endif

    alert(printer_name,TRUE,"Hangup detected on printer port (SIGHUP).");
    exit(EXIT_PRNERR);
    } 

/*
** Break handler.
*/
void sigint_handler(int sig)
    {
    #ifdef DEBUG
    printf("SIGINT received\n");
    #endif

    alert(printer_name,TRUE,"Break received on printer port (SIGINT).");
    exit(EXIT_PRNERR);
    }

/*
** SIGALRM  handler, we get this if open() takes too long.
**
** We exit with the code EXIT_ENGAGED which causes PPRD to set
** the printer's status to "Otherwise engaged or off line".
** PPRD will try again in 60 seconds.
**
** This is not considered an error, so we do not call alert().
*/
void sigalrm_handler(int sig)
    {
    #ifdef DEBUG
    printf("SIGALRM received\n");
    #endif
    
    exit(EXIT_ENGAGED);
    }

/*
** This routine catches the signals from the child half which
** tell the parent that a Control-D has been received.
**
** This two process operation is crude, but I don't have a 
** directly connected printer so I didn't want to make it too
** complicated as I can't test it very well.
*/
void sigusr2_handler(int sig)
    {
    #ifdef DEBUG
    printf("SIGUSR2 received\n");
    #endif
    
    eof_balance--;
    }

/*
** This is a handler for abnormal child termination.
** It is removed just before normal child termination is expected.
*/
void sigchild_handler(int sig)
    {
    int wstat;
    
    #ifdef DEBUG
    printf("SIGCHLD received\n");
    #endif

    while( wait(&wstat) == -1 )	/* Keep going until it works. */
    	{
    	if(errno!=EINTR)	/* If any error other than interupted system call, */
    	    {			/* then, send an additional error message and get out. */
    	    alert(printer_name,FALSE,"serial interface: wait() failed, %s",strerror(errno));
    	    exit(EXIT_PRNERR_NORETRY);
    	    }
    	}

    if(WCOREDUMP(wstat))	/* Check for core dump. */
    	{
    	alert(printer_name,TRUE,"Child half of serial interface program dumped core.");
	exit(EXIT_PRNERR_NORETRY);
	}
	
    if(WIFSIGNALED(wstat))	/* If child killed, */
    	{			/* die as though self caught signal. */
    	exit(EXIT_SIGNAL);
    	}

    if( ! WIFEXITED(wstat) )
    	{
    	alert(printer_name,TRUE,"serial interface: incomprehensible value returned by wait()"); 
	exit(EXIT_PRNERR_NORETRY);
	}

    exit(WEXITSTATUS(wstat));    /* pass on the child's exit status */
    } /* end of sigchild_handler() */

/*
** This is the meat.
*/
int main(int argc, char *argv[])
    {
    pid_t child_pid, parent_pid;	/* process id of each process */
    int wstat;				/* Wait status, used to wait for child */
    int port;				/* file handle of the printer port */
    char buffer[BUFFER_SIZE];		/* read and write buffer */
    char *ptr;				/* pointer into the buffer */
    int len,writelen;			/* length of stuff to read and write */
    int x;				/* General use in loops */
    sigset_t signal_set;		/* storage for set containing SIGUSR2 */
    struct termios settings;		/* printer port settings */    
    struct stat statbuf;		/* buffer for stat on the port */

    if(argc < 5)
        {
	fprintf(stderr,"serial: insufficient parameters\n");
        exit(EXIT_PRNERR_NORETRY);
	}
		
    /* Name the arguments. */
    printer_name=argv[1];
    printer_address=argv[2];
    printer_options=argv[3];
    printer_jobname=argv[4];
    printer_jobbreak=atoi(argv[5]);
    printer_feedback=atoi(argv[6]);

    /* Check for unusable job break methods. */
    if( printer_jobbreak == JOBBREAK_SIGNAL || printer_jobbreak == JOBBREAK_SIGNAL_PJL )
    	{
    	alert(printer_name,TRUE,"Job break methods \"signal\" and \"signal/pjl\" are not");
    	alert(printer_name,FALSE,"not compatible with the interface \"serial\".");
    	exit(EXIT_PRNERR_NORETRY);
    	}

    /* Save the parent's process id for use by the child half. */
    parent_pid=getpid();

    /* Set the SIGHUP handler in case the printer goes off line. */
    signal(SIGHUP,sighup_handler);

    /* Set the SIGINT handler in case the printer sends a break. */
    signal(SIGINT,sigint_handler);

    /* Set up a SIGUSR2 handler for signals from child half. */
    signal(SIGUSR2,sigusr2_handler);

    /* Set up a SIGCHLD handler in case the child has to exit. */
    signal(SIGCHLD,sigchild_handler);

    /* 
    ** Set up a SIGALRM handler because we use alarm() to
    ** detect when an open() of the printer port is taking
    ** too long.
    */
    signal(SIGALRM,sigalrm_handler);

    /*
    ** Make sure the address we were given is a tty.
    ** If stat() failes, we will ignore the error 
    ** for now, we will let open() catch it.
    */
    if( stat(printer_address,&statbuf) == 0 )
    	{
	if( ! (statbuf.st_mode & S_IFCHR) )
	    {
	    alert(printer_name,TRUE,"The file \"%s\" is not a tty.");
	    exit(EXIT_PRNERR_NORETRY);
	    }
	}	    

    /*
    ** Open the port.  We use alarm() to give this operation a 
    ** timeout since printers which are off line have been known
    ** to cause open() to block indefinitely.
    */
    #ifdef DEBUG
    printf("Opening printer port\n");
    #endif
    alarm(5);		/* give it 5 seconds */
    port=open(printer_address,O_RDWR);
    alarm(0);		/* cancel alarm */

    if(port == -1)	/* If error, */
    	{
	switch(errno)
	    {
	    case EACCES:
	    	alert(printer_name,TRUE,"Access to port \"%s\" denied.",printer_address);
		exit(EXIT_PRNERR_NORETRY);
	    case EIO:
	    	alert(printer_name,TRUE,"Hangup or error while opening \"%s\".",printer_address);
		exit(EXIT_PRNERR);		
	    case ENFILE:
	    	alert(printer_name,TRUE,"System open file table is full.");
	    	exit(EXIT_STARVED);
	    case ENOENT:	/* file not found */
	    case ENOTDIR:	/* path not found */
	    	alert(printer_name,TRUE,"The port \"%s\" does not exist.",printer_address);
	    	exit(EXIT_PRNERR_NORETRY);
	    case ENXIO:
	    	alert(printer_name,TRUE,"The device file \"%s\" exists, but the device doesn't.",printer_address);
		exit(EXIT_PRNERR_NORETRY);
	    case ENOSR:
	    	alert(printer_name,TRUE,"System is out of streams.");
	    	exit(EXIT_STARVED);
	    default:
	    	alert(printer_name,TRUE,"serial interface: open(\"%s\",O_RDWR) failed, %s",printer_address,strerror(errno));
		exit(EXIT_PRNERR_NORETRY);
	    }	    
    	}

    /*
    ** Set the printer port's forground process group id.  I believe
    ** the reason for this is so that we will get the signals that
    ** indicate that nasty things have happened to the connection
    ** to the printer.
    */
    if( tcsetpgrp(port,parent_pid) == -1 )
	{
	alert(printer_name,TRUE,"serial interface: tcsetpgrp() failed");
	exit(EXIT_PRNERR);
	}

    /*
    ** Wait for output to drain and then get the current port settings.
    ** Getting the current settings is said to be important because the 
    ** termios structure might have some implementations specific bits 
    ** that we should not mess with.
    **
    ** (See POSIX Programmers Guide, Donald Lewine, page 161.)
    */
    tcdrain(port);
    tcgetattr(port,&settings);

    /*
    ** Establish some default port settings.
    ** I wonder if we are messing with some of those implementation
    ** specific bits.
    */
    settings.c_iflag=BRKINT | IGNPAR | IXON | IXOFF;
    settings.c_cflag=B9600 | CS8 | CREAD | HUPCL;
    settings.c_oflag=0;
    settings.c_lflag=0;
    settings.c_cc[VMIN]=1;	/* Don't return with zero characters */
    settings.c_cc[VTIME]=1;	/* Wait 0.1 second for next character */

    /*
    ** Use the options to set the baud rate and such.
    ** This is a parser and interpreter.
    **
    ** The options string is in the form:
    ** speed=9600 parity=none bits=8 xonxoff=yes wait=yes
    */
    {
    char name[16];
    char value[16];
    int retval;

    options_start(printer_options);
    while( (retval=options_get_one(name,sizeof(name),value,sizeof(value))) > 0 )
    	{
	/* Intepret the keyword. */
	if( strcmp(name, "speed")==0 )
	    {
	    if( cfsetispeed(&settings,atoi(value)) == -1 )
		{
		options_error = "Invalid baud rate";
		retval = -1;
		break;
	    	}
	    }
	else if( strcmp(name, "xonxoff")==0 )
	    {
	    int answer;
	    if( (answer=torf(value)) == ANSWER_UNKNOWN )
	    	{
		options_error = "Illegal \"xonxoff\" value";
		retval = -1;
		break;
		}	    	    
	    if(answer)		/* on */
	    	settings.c_iflag |= IXON | IXOFF;
	    else			/* off */
	    	settings.c_iflag &= ~(IXON | IXOFF);
	    }  	
	else if( strcmp(name,"wait")==0 )
	    {
	    if( (wait_for_acknowledgement=torf(value)) == ANSWER_UNKNOWN )
		{
		options_error = "Illegal argument to \"wait\" option in printer options";
		retval = -1;
		break;
		}
	    }
	else if( strcmp(name, "parity")==0 )
	    {
	    if( strcmp(value, "none")==0 )
	    	{
	    	settings.c_cflag &= ~PARENB;		/* clear parity enable */
	    	}
	    else if(strcmp(value, "even")==0)
	    	{
		settings.c_cflag &= ~PARODD;		/* clear odd parity */
	    	settings.c_cflag |= PARENB;		/* enable parity */
	    	}
	    else if(strcmp(value, "odd")==0)
	    	{
		settings.c_cflag |= PARODD;		/* set odd parity */
	    	settings.c_cflag |= PARENB;		/* enable parity */
	    	}
	    }
	else if( strcmp(name, "bits")==0 )
	    {
	    int bits;
	    if( (bits=atoi(value)) != 7 && bits != 8 )
	    	{
		options_error = "Argument to \"bits\" option is not 7 or 8 in printer options.";
		retval = -1;
		break;
	    	}
	    settings.c_cflag &= ~CSIZE;	/* clear old setting */
	    if(bits==7)
		settings.c_cflag |= CS7;
	    else
		settings.c_cflag |= CS8;
	    }
	else if( strcmp(name,"online") == 0 )
	    {
	    if( icmp(value, "CTS") == 0 )
		online_modem = TIOCM_CTS;
	    else if( icmp(value, "DSR") == 0 )
		online_modem = TIOCM_DSR;
	    else if( icmp(value, "CD") == 0 )
		online_modem = TIOCM_CAR;
	    else if( icmp(value, "none") == 0 )
	    	online_modem = 0;
	    else
		{
		options_error = "The only valid online= options are \"CTS\", \"DSR\", and \"CD\"";
		retval = -1;
		break;
		}
	    }
	else
	    {
	    options_error = "Unrecognized keyword in printer options";
	    retval = -1;
	    break;
	    }

	} /* end of while() */
	  
    if(retval == -1)
    	{
    	alert(printer_name, TRUE, "Option parsing error:  %s", options_error);
    	alert(printer_name, FALSE, "%s",printer_options);
    	alert(printer_name, FALSE, "%*s^ right here", options_error_index, "");
    	exit(EXIT_PRNERR_NORETRY);
    	}
    }
    
    /* Write the new port settings. */
    if( tcsetattr(port,TCSANOW,&settings) == -1 )
    	{
    	alert(printer_name,TRUE,"serial interface: tcsetattr() failed, errno=%d (%s)",errno,strerror(errno));
    	exit(EXIT_PRNERR);
    	}

    /*
    ** Make sure the necessary modem control lines are on to
    ** indicate that the printer is on line.
    */
    if( online_modem != 0 )
	{
	int x;
    
	if( ioctl(port, TIOCMGET, &x) == -1 )
	    {
	    alert(printer_name,TRUE,"serial interface: ioctls(port,TIOCMGET,&x) failed, errno=%d (%s)",errno,strerror(errno));    	
	    exit(EXIT_PRNERR);    	
    	    }

	if( (x & online_modem) == 0 )
	    {
	    exit(EXIT_ENGAGED);	    
	    }
	}

    /* Fork() into sending and receiving processes. */
    if( (child_pid=fork()) ==  -1 )
    	{
    	alert(printer_name,TRUE,"serial interface: fork() failed");
    	exit(EXIT_STARVED);
    	}
    	
    /*
    ** Child half, copy stuff from printer, sending SIGUSR2 to
    ** parent every time we receive a Control-D. 
    **
    ** If it does not detect an error, the child never exits of
    ** its own volition, it just keeps running until the parent
    ** half kills it.
    */
    if(child_pid==0)
    	{
	int newlen;
	int c;
	
	#ifdef DEBUG_CHILD
	printf("I am the child\n");
	#endif

	while(1)	/* plan to run until parent kills us */
    	    {
	    while( (len=read(port,buffer,BUFFER_SIZE)) == -1 )
	    	{
		if(errno!=EINTR)
		    {
		    alert(printer_name,TRUE,"serial interface: read() from printer failed, errno=%d",errno);
		    exit(EXIT_PRNERR);
		    }
	    	}

	    #ifdef DEBUG_CHILD
	    printf("child read %d bytes: \"%.*s\"\n",len,len,buffer);
	    if(len)
	    	{
	    	for(x=0; x<len; x++)
	    	    printf("%d ",buffer[x]);
	    	printf("\n");
	    	}
	    #endif

	    /* 
	    ** Check for hangup.
	    ** See termio(7), "Modem Disconnect"
	    */
	    if(len==0)
	    	{
	    	alert(printer_name,TRUE,"Hangup detected on printer port (EOF).");
	    	exit(EXIT_PRNERR);
	    	}

	    /* 
	    ** Write buffer to stdout, except for control-D's
	    ** for which we send SIGUSR2 to parent.
	    */
	    newlen=0;
	    ptr=buffer;
	    for(x=0; x <= len; x++)	/* This is right! */
	    	{
		c=ptr[newlen];
	
		if(c==4 || x==len)	
		    {	    		/* if control-D or end */ 
		    #ifdef DEBUG_CHILD
		    printf("writing %d bytes to stdout\n",newlen);
		    #endif
		    while(newlen)
		    	{
		    	while( (writelen=write(1,ptr,newlen)) == -1 )
		    	    {
			    if(errno!=EINTR)
			    	{
			    	alert(printer_name,TRUE,"serial interface: write() to pprdrv failed, errno=%d",errno);
		    	    	exit(EXIT_PRNERR);
		    	    	}
		    	    }
		    	ptr+=writelen;
		    	newlen-=writelen;
		    	}
		    }
		else
		    {
		    newlen++;
		    }

		if(c==4)		/* if it was a control-D, */
		    {			/* Tell parent, */
		    #ifdef DEBUG_CHILD
		    printf("sending SIGUSR2 to parent\n");
		    #endif

		    kill(parent_pid,SIGUSR2);
		    ptr++;		/* and skip past the control-D */
		    }
		}
	    } /* end of child loop, loop never ends */
	} /* end of if child */
    
    /*
    ** Parent half, copy stuff to printer, incrementing a counter for
    ** every Control-D we send and decrementing it every time we
    ** receive SIGUSR2. 
    */
    while(1)
    	{
	/* Read a block from stdin. */
    	while( (len=read(0,buffer,BUFFER_SIZE)) == -1 )
    	    {
    	    if(errno!=EINTR)
    	    	{
		if(errno==EIO)
		    alert(printer_name,TRUE,"serial interface: physical IO error on printer port");
		else 		    
    	    	    alert(printer_name,TRUE,"serial interface: error reading from pprdrv, errno=%d",errno);

	    	exit(EXIT_PRNERR);
	    	}
    	    }

	#ifdef DEBUG
	printf("read %d bytes from pprdrv\n",len);
	#endif
	
	if(len==0)		/* if no more data, we are done */
	    break;

	/* Increment eof_balance for each control-D found in the block. */
	for(ptr=buffer,x=0; x < len; x++,ptr++)
	    {
	    if( *ptr == 4 )
	    	eof_balance++;
	    }

	/* Write the block to the printer port. */
	ptr=buffer;
	while(len)
	    {
	    while( (writelen=write(port,ptr,len)) == -1 )
	    	{
	    	if(errno!=EINTR)
	    	    {
		    if(errno==ENXIO)
			alert(printer_name,TRUE,"Hangup on printer port (errno=ENXIO).");
		    else
	    	    	alert(printer_name,TRUE,"serial interface: write() to printer port failed, errno=%d.",errno);

	    	    exit(EXIT_PRNERR);
	    	    }
	    	}
	    #ifdef DEBUG
	    printf("wrote %d bytes to printer port\n",writelen);
	    #endif
	    
	    len-=writelen;
	    ptr+=writelen;
	    }
    	}    

    /* 
    ** If we can expect to receive control-D's from the printer and
    ** the options did not contain "wait=no" and the control-D 
    ** disparity counter is positive, wait for enough SIGUSR2's 
    ** to reduce it to zero.  (The child half sends up a SIGUSR2 
    ** whenever it receives a control-D).
    */
    if( printer_feedback && wait_for_acknowledgement )
    	{
    	sigemptyset(&signal_set);		/* Make a mask to block SIGUSR2 */
    	sigaddset(&signal_set,SIGUSR2);
    	while( sigprocmask(SIG_BLOCK,&signal_set,(sigset_t*)NULL), eof_balance > 0 )
    	    {
	    #ifdef DEBUG
	    printf("Waiting for acknowledgement, eof_balance=%d\n",eof_balance);
	    #endif

	    sigpause(SIGUSR2);    	
    	    }
    	}
    
    /* Kill our child and wait for it to die. */
    sleep(1);
    #ifdef DEBUG
    printf("Killing child\n");
    #endif
    signal(SIGCHLD,SIG_IGN);		/* Take out old SIGCHLD handler */
    kill(child_pid,SIGTERM);    	/* Send the signal */
    wait(&wstat);			/* Reap the child */
    
    /* We are done. */
    close(port);			/* explicitly close printer port */
    exit(EXIT_PRINTED);			/* exit, telling pprdrv we suceeded */
    } /* end of main() */

/* end of file */
