/*
** ~ppr/src/interfaces/lpr.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell and Damian Ivereigh.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file was last modified 13 December 1996.
*/

/*
** This interface sends the print job to another computer
** using the Berkeley LPR protocol.
*/

#include "global_defines.h"
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/utsname.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include "interface.h"

/* The file for the the IDs sent to the remote system. */
#define LPR_NEXTID VAR_SPOOL_PPR"/lpr_nextid"

/* Timeout for connect(). */
#define CONNECT_TIMEOUT 20

/* Global name of printer for use when calling alert(). */
char *printer;

/* The maximum length of a control file "P" line argument: */
#define MAX_P 31

/*
** Make the connection to the printer.
** Return the file descriptor.
*/
int make_connection(char *address)
    {
    int sockfd;				/* handle of socket we will connect to LPR server */
    struct sockaddr_in printer_addr;	/* internet address of LPR server */
    struct hostent *hostinfo;		/* return value of gethostbyname() */
    int retval;

    /* Check for any obvious syntax error in the address */
    if( strpbrk(address, " \t") != (char*)NULL )
    	{
    	alert(printer, TRUE, "Spaces and tabs not allowed in printer LPR address.");
    	exit(EXIT_PRNERR_NORETRY);
    	}

    /* Clear the printer internet address structure. */
    memset(&printer_addr, 0, sizeof(printer_addr));

    /*
    ** If the address specifies the number of a port to use, use that,
    ** otherwise, find out which port we should connect 
    ** to by looking in the /etc/services database.  It
    ** we don't find it, we will use the value 515.
    */
    {
    char *ptr;
    
    if( (ptr=strchr(address, ':')) != (char*)NULL )
	{
	if( (printer_addr.sin_port=atoi(ptr+1)) == 0 )
	    {
	    alert(printer, TRUE, "Bad port specification in printer address.");
	    exit(EXIT_PRNERR_NORETRY);
	    }

	*ptr = (char)NULL;	/* truncate */
	}
    else			/* If address doesn't specify the port, */
    	{
        struct servent *sp;
	if ((sp=getservbyname("printer", "tcp"))!=NULL)
	    {
	    printer_addr.sin_port = sp->s_port;
	    }
	else
	    {
	    /* Use the default */
	    printer_addr.sin_port = htons(515);
	    }
	}
    }	
    
    /* 
    ** If converstion of a dotted address works, use it,
    ** otherwise, use gethostbyname() to find the address. 
    */
    if( (int)(printer_addr.sin_addr.s_addr=inet_addr(address)) != -1 )
	{			/* cast to int is bad, -1 should be INETADDR_NONE */
	printer_addr.sin_family = AF_INET;
    	}
    else
    	{
	if( (hostinfo=gethostbyname(address)) == (struct hostent *)NULL )
	    {
	    alert(printer, TRUE, "Can't get IP address for \"%s\".", address);
	    /* exit(EXIT_PRNERR_NORETRY); */
	    exit(EXIT_PRNERR);	/* hard to decide */
	    }

	printer_addr.sin_family = hostinfo->h_addrtype;
	memcpy(&printer_addr.sin_addr, hostinfo->h_addr_list[0], hostinfo->h_length);
    	}

    /*
     * Open a TCP/IP socket which we will use to connect
     * to the LPD server.
     */
    if( (sockfd=socket(AF_INET, SOCK_STREAM, 0)) == -1 )
	{
	alert(printer, TRUE, "lpr: socket() failed, errno=%d (%s)", errno, strerror(errno));
	exit(EXIT_PRNERR);
	}		

    /*
    ** If we are root, get a priviledged port number
    ** assigned to our socket.
    */
    if( geteuid() == 0 )
	{
	u_short lport = IPPORT_RESERVED - 1;	/* first reserved port */
	struct sockaddr_in sin;
	
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);

	while(TRUE)
	    {
	    sin.sin_port = htons(lport);
	    if( bind(sockfd, (struct sockaddr *)&sin, sizeof(sin)) != -1 )
	        break;

	    if(errno == EADDRINUSE)	/* If there error is that this port used, */
		{
		if( --lport == IPPORT_RESERVED/2 )
		    {
		    alert(printer, TRUE, "lpr interface: no free reserved ports");
		    exit(EXIT_PRNERR);
		    }
		continue;
		}

	    alert(printer, TRUE, "lpr interface: bind() failed, port=%d, errno=%d (%s)", (int)lport, errno, strerror(errno));	
	    exit(EXIT_PRNERR);
	    }
	}

    /*
    ** Connect the socket to the printer.
    ** Since some systems, such as Linux, fail to implement
    ** connect() timeouts, we will do it with alarm().
    */
    alarm(CONNECT_TIMEOUT);
    retval = connect(sockfd, (struct sockaddr*) &printer_addr, sizeof(printer_addr));
    alarm(0);

    if (retval==-1)
	{
	if(errno==ETIMEDOUT)
	    {
	    alert(printer, TRUE, "Timeout while trying to connect to remote system \"%s\".", address);
	    exit(EXIT_PRNERR);
	    }
	if(errno==ECONNREFUSED)
	    {
	    alert(printer, TRUE, "Remote system \"%s\" has refused the connection.", address);
	    if( geteuid() != 0)
		{
		alert(printer, FALSE, "(Probably because the lpr interface is not suid root");
		alert(printer, FALSE, "and therefore could not use a priviledged port.)");
		}
	    exit(EXIT_PRNERR);
	    }
	if(errno==EADDRINUSE)
	    {
	    alert(printer,TRUE, "lpr interface: connect() set errno to EADDRINUSE!");
	    alert(printer,FALSE, "(This indicates a bug in PPR!)");
	    exit(EXIT_PRNERR_NORETRY);
	    }
	alert(printer, TRUE, "lpr interface: connect() failed, errno=%d (%s)", errno, strerror(errno));
	exit(EXIT_PRNERR);
	}

    /* Turn on the socket option which detects dead connections. */
    {
    int truevalue = 1;
    if( setsockopt(sockfd,SOL_SOCKET,SO_KEEPALIVE,(void*)&truevalue,sizeof(truevalue)) == -1 )
    	{
    	alert(printer,TRUE,"lpr interface: setsockopt() failed, errno=%d (%s)",errno,strerror(errno));
	exit(EXIT_PRNERR_NORETRY);
    	}
    }

    return sockfd;
    } /* end of make_connection() */

/*
** Return the next id number.  The string "file" is the full path and
** name of the file to use.  The file is locked, the last id is read,
** and incremented making the new id, and the new id is written back 
** to the file, the file is closed, and the new id is returned.  
*/
int get_next_id(char *file)
    {
    int idfilefd;                   /* file descriptor of id file */
    FILE *idfile;
    int tid;                        /* for holding id */

    idfilefd=open(file,O_RDWR,S_IRUSR | S_IWUSR); /* open the id file */

    if(idfilefd != -1)		    /* if open worked, */
        {          
	if(lock_exclusive(idfilefd,TRUE))
	    {
	    alert(printer,TRUE,"lpr interface: can't lock \"%s\"",file);
	    exit(EXIT_PRNERR_NORETRY);
	    }
	    
        idfile=fdopen(idfilefd,"r+");	/* read and update */

        fscanf(idfile,"%d",&tid);	/* get the last value used */
        tid++;				/* add one to it */
        if( (tid<0) || (tid>999) )	/* if id unreasonable or too large */
            tid=1;                  	/* force it to one */
        rewind(idfile);             	/* move to start, for write */
        }
    else                            	/* does not exist */
        {
        tid=0;                      	/* start id's at 0 */
        idfile=fopen(file,"w");     	/* create a new file */
        if(idfile==(FILE*)NULL)
	    {
            alert(printer,TRUE,"lpr interface: can't create new id file \"%s\"",file);
            exit(EXIT_PRNERR_NORETRY);
            }
        }

    fprintf(idfile,"%d\n",tid);     /* write the new id */
    fclose(idfile);                 /* and close the id file */
    return tid;
    } /* end of get_next_id() */

/*
** Copy stdin to the named file, return the open file
** and set the length.
*/
int file_stdin(char *tempfile, int *length)
    {
    char *copybuf;			/* buffer for copy operation */
    int copyfd;				/* file descriptor of temp file */
    int rbytes, wbytes;			/* bytes read and written */

    /* Set length initially to zero. */
    *length=0;

    /* Copy stdin to a temporary file */
    if( (copybuf=(char*)malloc((size_t)16384)) == (void*)NULL)
	{
	alert(printer,TRUE,"lpr interface: malloc() failed, errno=%d",errno);
	exit(EXIT_PRNERR);
	}

    if( (copyfd=open(tempfile,O_RDWR | O_CREAT,UNIX_755)) == -1 )
    	{
    	alert(printer,TRUE,"lpr interface: open() failed for transfer file, errno=%d.",errno);
    	exit(EXIT_PRNERR);
    	}
    
    unlink(tempfile);

    /* copy until end of file */
    while( (rbytes=read(0,copybuf,4096)) )
    	{
    	if(rbytes==-1)
	    {
	    alert(printer,TRUE,"lpr interface: internal interface error: read() failed, errno=%d.",errno);
	    exit(EXIT_PRNERR);
	    }

    	if( (wbytes=write(copyfd,copybuf,rbytes)) != rbytes)
    	    {
    	    if(wbytes==-1)
    	    	alert(printer,TRUE,"lpr interface: error writing transfer file: write() failed, errno=%d",errno);
    	    else
    	    	alert(printer,TRUE,"lpr interface: disk full error writing transfer file.");

	    exit(EXIT_PRNERR);
    	    }

        *length += rbytes;		/* add to total */
	}
    	
    free(copybuf);			/* free the buffer */

    lseek(copyfd,(off_t)0,SEEK_SET);	/* return to start of file */

    return copyfd;
    } /* end of file_stdin() */

/*
** Send a command to the remote system.
*/
void write_cmd(int fd, char *text, int length)
    {
    int result;
    
    if( (result=write(fd,text,length)) == -1 )
    	{
    	alert(printer,TRUE,"lpr interface: write() to remote system failed, errno=%d",errno);
    	exit(EXIT_PRNERR);
    	}

    if( result != length )
    	{
    	alert(printer,TRUE,"lpr interface: write() to remote system did not complete");
    	exit(EXIT_PRNERR);
    	}
    	
    } /* end of write_cmd() */
    
/*
** Look for a result code and return TRUE if it is favourable. 
*/
int ok(int fd)
    {
    char result;
    int retval;
    
    if( (retval=read(fd,&result,1)) == -1 )
    	{
    	alert(printer,TRUE,"lpr interface: read from remote system failed, errno=%d",errno);
    	exit(EXIT_PRNERR);
    	}

    if( retval != 1 )
    	{
    	alert(printer,TRUE,"lpr interface: %d bytes read from remote system when 1 expected",retval);
    	exit(EXIT_PRNERR);
    	}    

    if(result == 0)
    	return TRUE;
    else
        return FALSE;        
    } /* end of ok() */

/*
** Copy the data to the printer.
** Called only once.
*/
void send_data_file(int source, int sockfd)
    {
    int just_read, bytes_left, just_written;
    char *ptr;
    char buffer[4096];		/* Buffer for data transfer */

    do	{
	if( (just_read=read(source,buffer,sizeof(buffer))) == -1 )
	    {
	    alert(printer,TRUE,"lpr interface: error reading temp file, errno=%d",errno);
	    exit(EXIT_PRNERR);
	    }
	bytes_left=just_read;
	ptr=buffer;
	while(bytes_left)			
	    {
	    if( (just_written=write(sockfd,ptr,bytes_left)) == -1 )
		{
	 	alert(printer,TRUE,"lpr interface: error writing to socket, errno=%d",errno);
		exit(EXIT_PRNERR);
		}
	    ptr += just_written;
	    bytes_left -= just_written;				
	    }
	} while(just_read);
    } /* end of send_data_file() */
	
/*
** This is the SIGPIPE handler which detects when
** we try to write to a socket which has become
** disconnected.
*/
void sigpipe_handler(int sig)
    {
    alert(printer,TRUE,"Connection to printer broken (SIGPIPE)");
    exit(EXIT_PRNERR);
    } /* end of sigpipe_handler() */

/*
** This is the SIGALRM handler which detects if the alarm goes
** off before the remote system responds.
*/
void sigalrm_handler(int sig)
    {
    alert(printer,TRUE,"Remote LPR/LPD system is not responding.");
    exit(EXIT_PRNERR);
    } /* end of sigalrm_handler() */

/*
** Main function.
*/
int main(int argc, char *argv[])
    {
    /* Command line options.  The printer name is global. */
    char *address;
    char *options;
    char *jobname;
    char *forline;

    /* The person, extracted from the forline. */
    char *person;
    int person_len;

    /* The address broken up into its components. */
    char *rprinter;			/* name of printer */
    char *rhost;			/* host printer is on */

    int banner = FALSE;			/* Normally will discarage banner on remote system */

    char cf_name[MAX_PATH];		/* name to give control file on remote system */
    char control_file[1024];		/* The actual control file contents. */
    char df_name[MAX_PATH];		/* name to give data file on remote system */
    int df_length;			/* length of data file (we must tell recipient) */

    int sockfd;				/* open connection to LPD */
    struct utsname sysinfo;		/* for uname() call */
    int rqueueid;			/* remote queue id */
    char command[256];			/* staging area for command to LPD */
    char temp_file[MAX_PATH];		/* file to store stdin in */
    int tempfd;				/* file descriptor for temp_file[] */
    char typecode = 'f';		/* Code to use in cf file */
    
    int len;
    char *ptr, *ptr2;

    /* Check to see we have at least the required parameters. */
    if(argc < 9)
	{
	if(argc > 1)
	    alert(printer, TRUE, "lpr interface: insufficient parameters");
	else
	    fprintf(stderr, "lpr interface: insufficient parameters");
	exit(EXIT_PRNERR_NORETRY);
	}

    /* Name some of the parameters. */
    printer = argv[1];
    address = argv[2];
    options = argv[3];
    jobname = argv[4];
    forline = argv[8];

    /* seperate the host and printer parts of the address */
    len = strcspn(address, "@");
    address[len] = (char)NULL;
    rprinter = address;
    rhost = &address[len+1];
    
    /* Check to be sure we got both a printer and a host. */
    if( (strlen(rprinter)==0) || (strlen(rhost)==0) )
	{
    	alert(printer, TRUE, "Printer address is invalid, syntax is \"printer@host\".");
    	exit(EXIT_PRNERR_NORETRY);
    	}

    /* Settle on a P line for the control file: */
    person = forline;
    if(person[0] == (char)NULL) person = "unknown";
    person_len = strcspn(person, "@");
    if(person_len > MAX_P) person_len = 31;

    /* Parse the options string, searching for name=value pairs. */
    len = 0;
    ptr2 = ptr = options;		/* setting ptr2 keeps GCC happy */
    while(*ptr)
    	{
	ptr += strspn(ptr," \t");	/* Eat up leading space */
	
	if(len==0)			/* If no keyword started, */
	    ptr2=ptr;			/* mark start of this one. */

	len += strcspn(ptr,"= \t");	/* Count size of keyword */
	ptr = &ptr2[len];		/* move ptr to end of keyword */
	ptr += strspn(ptr," \t");	/* eat up any space before equals */

	if(*ptr=='=')			/* if end of keyword */
	    {
	    ptr += strspn(ptr,"= \t");	/* eat equals and space after */

	    /* Make sure there is an argument left. */
	    if(*ptr == (char)NULL)
	    	{
	    	alert(printer,TRUE,"Missing argument in printer options: %s",ptr2);
	    	exit(EXIT_PRNERR_NORETRY);
	    	}

	    /* Intepret the keyword. */
	    if(len == 6 && strncmp(ptr2, "banner", 6) == 0)
	    	{
		if( (banner=torf(ptr)) == ANSWER_UNKNOWN )
		    {
		    alert(printer,TRUE,"Option \"banner=\" must be true or false");
		    exit(EXIT_PRNERR_NORETRY);
		    }
	    	}
	    else if(len==12 && strncmp(ptr2,"lpr_typecode",12)==0)
		{
		if (*ptr=='o' || *ptr=='f')
		    typecode=*ptr;
		else
		    {
		    alert(printer, TRUE, "Option \"lpr_typecode=\" must be 'o' or 'f'");
		    exit(EXIT_PRNERR_NORETRY);
		    }
		}
	    else
	    	{
		alert(printer, TRUE, "Unrecognized keyword in printer options: %*s", len, ptr2);
		exit(EXIT_PRNERR_NORETRY);
	    	}	
	  
	    /* Eat up the argument */
	    ptr += strcspn(ptr," \t");	    	
	    len=0;			/* clear for next keyword */
	    }
	else				/* Illegal character */
	    {
	    alert(printer, TRUE, "Syntax error in printer options:");
	    alert(printer, FALSE, "%s", options);
	    alert(printer, FALSE, "%*s^right here",
	    	(int)(ptr-options), "");
	    exit(EXIT_PRNERR_NORETRY);
	    }
    	}

    /* make a temporary file name */
    sprintf(temp_file,"%s/%ld",TEMPDIR,(long)getpid());

    /* copy stdin to a temporary file */
    tempfd = file_stdin(temp_file, &df_length);

    /* Get the name of this system */
    if( uname(&sysinfo) == -1 )
    	{
    	alert(printer, TRUE, "lpr interface: uname() failed, errno=%d (%s)", errno, strerror(errno));
    	exit(EXIT_PRNERR_NORETRY);
    	}	

    /* Generate a queue id. */
    rqueueid=get_next_id(LPR_NEXTID);

    /* Generate the data file and queue file names */
    sprintf(df_name, "dfA%03d%s", rqueueid, sysinfo.nodename);
    sprintf(cf_name, "cfA%03d%s", rqueueid, sysinfo.nodename);

    /* Build the `control file' */
    sprintf(control_file,	"H%s\n"
    				"P%.*s\n"
    				"%c%s\n"
    				"U%s\n"
    				"N%s\n"
    				"%s",
    		sysinfo.nodename,		/* host */
		person_len, person,		/* person from argument 8 */
		typecode,			/* lpr typecode */
    		df_name,			/* file to print */
    		df_name,			/* file to unlink */
    		jobname,			/* name of job, to keep System V LP happy */
    		banner ? "Lsomebody\n" : "");	/* Should the receiving system print a banner? */

    /*
    ** Install SIGPIPE handler to detect
    ** broken connections.
    */
    signal(SIGPIPE,sigpipe_handler);	

    /* Install SIGALRM handler for connect() timeouts. */
    signal(SIGALRM,sigalrm_handler);

    /* Connect to the remote system. */
    sockfd = make_connection(rhost);

    /* Say we want to send a job. */
    sprintf(command, "\002%s\n", rprinter);
    write_cmd(sockfd, command, strlen(command));
    
    /* Check if the response if favorable. */
    if( ! ok(sockfd) )
    	{
    	alert(printer, TRUE, "Remote LPR/LPD system \"%s\" refuses to accept job for \"%s\".", rhost, rprinter);    
	exit(EXIT_PRNERR);
	}    
    
    /* Try to send the data file */
    sprintf(command, "\003%d %s\n", df_length, df_name);
    write_cmd(sockfd, command, strlen(command));

    /* Check if response is favourable. */
    if( ! ok(sockfd) )
    	{
    	alert(printer, TRUE, "Remote LPR/LPD system \"%s\" does not have room for data file.", address);
    	exit(EXIT_PRNERR);
    	}

    /* Send the data file */
    send_data_file(tempfd, sockfd);
    
    /* Send the zero byte */
    command[0] = (char)0;
    write_cmd(sockfd, command, 1);
    
    /* Check if response if favourable. */
    if( ! ok(sockfd) )
    	{
    	alert(printer,TRUE,"Remote LPR/LPD system \"%s\" denies correct receipt of data file.",address);
    	exit(EXIT_PRNERR);
    	}
    	    
    /* Try to send the control file */
    sprintf(command,"\002%d %s\n",(int)strlen(control_file),cf_name);
    write_cmd(sockfd,command,strlen(command));

    /* Check if response is favourable. */
    if( ! ok(sockfd) )
    	{
    	alert(printer,TRUE,"Remote LPR/LPD system does not have room for control file.");
    	exit(EXIT_PRNERR);
    	}

    /* Send the control file. */
    write_cmd(sockfd,control_file,strlen(control_file));
    
    /* Send the zero byte */
    command[0]=(char)0;
    write_cmd(sockfd,command,1);
    
    /* Check if response if favourable. */
    if( ! ok(sockfd) )
    	{
    	alert(printer,TRUE,"Remote LPR/LPD system \"%s\" denies correct receipt of control file.",address);
    	exit(EXIT_PRNERR);
    	}
    	    
    /* Close the files we still have open. */
    close(sockfd);
    close(tempfd);    	
    
    /* Tell pprdrv that we did our job correctly. */
    exit(EXIT_PRINTED);    
    } /* end of main() */
    
/* end of file */
