/*
** ~ppr/src/interfaces/atalk_cap.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 7 January 1997.
*/

/*
** AppleTalk interface for the Columbia AppleTalk Program.
*/

/* #define DEBUG 1 */

/*
** Includes for CAP.  You must copy abpap.h from the CAP source
** directory "lib/cap" to "/usr/include/netat".
**
** The last file, "cap_proto.h" is part of PPR.  It provides some
** prototypes the CAP include files do not.
**
** The order of these includes is important. 
*/
#include <sys/time.h>
#include <netat/appletalk.h>
#include <netat/abpap.h>
#include "cap_proto.h"		/* prototypes CAP does not provide */

/*
** Now for some more normal includes. 
*/
#include <global_defines.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include "interface.h"

/* misc globals */
char *printer_name;
#define LocalQuantum 1
#define MaxRemoteQuantum 8
int sigusr1_caught=FALSE;
int sigalrm_caught=FALSE;
sigset_t sigset_sigusr1;
	
/* a forward reference */
int receive(int cno, int flag);

/* Retry variables.  There is as yet no code to consult these values. */
int open_retries = 10;			/* number of times to retry pap_open() */
int lookup_retries = 8;			/* number of times to retry NBP lookup */
int lookup_interval = 1;		/* interval between retries */
int is_laserwriter = TRUE;		/* Is this a LaserWriter compatible PostScript printer? */
long patron = -1;			/* PID of gsatalk */
int status_update_interval;

/* 
** write a debugging line to the log file
*/
#ifdef DEBUG
void debug(const char string[], ... )
    {
    va_list va;
    FILE *file;

    va_start(va,string);
    if( (file=fopen(LOGDIR"/interface_atalk","a")) != NULL )
        {
        fprintf(file,"DEBUG: (%ld) ", (long)getpid());
        vfprintf(file,string,va);
        fprintf(file,"\n");
        fclose(file);
        }
    va_end(va);
    } /* end of debug() */
#endif

/*
** Two routines used to exit with an error indication.  Normally these simply
** exit with the values EXIT_PRNERR and EXIT_PRNERR_NORETRY respectively,
** however if the undocumented option "patron=PID" is used, they will send
** SIGUSR1 or SIGUSR2 to the indicated process and exit with a normal code.
**
** This bizzar feature is used by the interface "gsatalk".
** This usage of SIGUSR1 has nothing to do with its usage when
** the atalk interface is operating in the normal manner.
*/
void exit_PRNERR(void)
    {
    #ifdef DEBUG
    debug("exit_PRNERR()");
    #endif

    if(patron != -1)			/* If we are begin run from Ghostscript, */
    	{
    	kill((pid_t)patron,SIGUSR1);	/* tell the shell script */
    	sleep(1);			/* allow interface script time to get signal and exit */
    	exit(EXIT_PRINTED);
    	}

    exit(EXIT_PRNERR);
    } /* end of exit_PRNERR() */
    
void exit_PRNERR_NORETRY(void)
    {
    #ifdef DEBUG
    debug("exit_PRNERR_NORETRY()");
    #endif

    if(patron != -1)
    	{
    	kill((pid_t)patron, SIGUSR2);
    	sleep(1);
    	exit(EXIT_PRINTED);
    	}
    exit(EXIT_PRNERR_NORETRY);
    } /* end of exit_PRNERR_NORETRY() */

/*
** Handler for SIGUSR1.  We will receive SIGUSR1 from pprdrv when pprdrv
** wants us to send a EOJ indication over the AppleTalk connexion.
*/
void sigusr1_handler(int signum)
    {
    sigusr1_caught=TRUE;
    fcntl(0,F_SETFL,O_NONBLOCK);
    } /* end of sigusr1_handler() */

/*
** Handler for SIGALRM.
*/
void sigalrm_handler(int signum)
    {
    sigalrm_caught=TRUE;
    } /* end of sigalrm_handler() */

/*
** Open a connection to the printer.
** Return -1 if the name is not found.
**
** This routine is called from open_printer().
*/
int basic_open_printer(const char *address, int *wlen)
    {
    PAPStatusRec status;
    int ocomp;
    int cno;
    int ret;
    PAPSOCKET *ps;

    #ifdef DEBUG
    debug("basic_open_printer(\"%s\")",address);
    #endif

    if( (ret=PAPOpen(&cno,address,LocalQuantum,&status,&ocomp)) != 0 )
	{
	/*
	** CAP returns this dummy status value if NBP
	** fails to find the printer name on the network.
	**
	** The CAP source file abpapc.c, function PAPOpen()
	** contains the sprintf() format string "%Can't find %s".
	** The "%C" is not valid and the result is undefined.
	** Therefor, we look for the intended output (in case
	** someone fixes the bug) and two different observed
	** results. 
	*/
	if( strncmp(&status.StatusStr[1], "%Can't find ", 11) == 0
		|| strncmp(&status.StatusStr[1], "Can't find ", 11) == 0
		|| strncmp(&status.StatusStr[1], "an't find ", 11) == 0 )
	    {
	    #ifdef DEBUG
	    debug("Can't find it");
	    #endif	
	    return -1;			/* let open_printer() deal with it */
	    }
	/*
	** We don't know how to deal with other errors.
	*/
	else
	    {
	    alert(printer_name, TRUE, "atalk interface: PAPOpen() returned %d, %s", ret, &status.StatusStr[1]);
	    exit_PRNERR();
	    }
	}

    /* 
    ** Get pointer to PAP socket internal data. 
    ** We will use this to determine how long we
    ** have been waiting for a connection and to
    ** determine the remote flow quantum.
    */
    ps = cnotopapskt(cno);

    /* Wait for PAPOpen() to complete. */
    do	{
	#ifdef DEBUG
	debug("Waiting");
	#endif

	abSleep(4,TRUE);		/* Let ATalk work for up to 1 second */

	if( ps->po.papO.wtime > 10 )	/* If we have been waiting too long */
	    {				/* (that is, more than 10 seconds), */
	    #ifdef DEBUG
	    debug("basic_open_printer() timeout");
	    debug("(%.*s)", (int)status.StatusStr[0], &status.StatusStr[1]);
	    #endif

	    /* Write the status string in printer response format. */
	    printf("%%%%[ %.*s ]%%%%\n", (int)status.StatusStr[0], &status.StatusStr[1]);

	    /* Try to cancel the PAPOpen(). */
	    PAPClose(cno);

	    /* 
	    ** Tell pprdrv that the printer is otherwise
	    ** engaged or off line.
	    */
	    exit(EXIT_ENGAGED);
	    }
	
	} while(ocomp > 0);
	
    if(ocomp < 0)	/* if ocomp indicates an error */
	{
	alert(printer_name,TRUE,"atalk interface: PAPOpen() completion code: %d",ocomp);
	exit_PRNERR();
	}

    /*
    ** If the status contains the string "status: PrinterError:", print it 
    ** even if the open was sucessful.
    */
    if(status.StatusStr[0] > 21 && strncmp("status: PrinterError:", &status.StatusStr[1], 21) == 0)
	printf("%%%%[ %.*s ]%%%%\n", (int)status.StatusStr[0], &status.StatusStr[1]);

    /* 
    ** Determine how many bytes at most we 
    ** may send to the remote end. 
    */
    *wlen = ps->rflowq > MaxRemoteQuantum ? MaxRemoteQuantum * 512 : ps->rflowq * 512;

    #ifdef DEBUG
    debug("basic_open_printer() returning cno=%d",cno);
    #endif
    
    return cno;
    } /* end of basic_open_printer() */

/*
** This routine is called from open_printer().
*/
void hide_printer(int cno, const char type[], int typelen)
    {
    char buf[512];
    int ret;
    int wcomp;
    
    #ifdef DEBUG
    debug("hide_printer(%d,%s,%d",cno,type,typelen);
    #endif

    sprintf(buf,"%%!PS-Adobe-3.0 ExitServer\n"
    	"0 serverdict begin exitserver\n"
    	"statusdict begin\n"
    	"currentdict /appletalktype known{/appletalktype}{/product}ifelse\n"
    	"(%.*s) def\n"
    	"end\n%%%%EOF\n",typelen,type);

    if( (ret=PAPWrite(cno,buf,strlen(buf),TRUE,&wcomp)) < 0)
    	{
    	alert(printer_name,TRUE,"atalk interface: hide_printer(): PAPWrite() failed while changing AppleTalk type, ret=%d",ret);
    	exit_PRNERR();
    	}

    do	{			/* Wait for the response, */
	abSleep(4,TRUE);	/* throwing it away. */
	} while(! receive(cno, FALSE));    	
    
    #ifdef DEBUG
    debug("hide_printer() done");
    #endif
    } /* end of hide_printer() */

/*
** This routine opens a printer, hiding it and re-opening 
** it if necessary.
*/
int open_printer(const char address[], int *wlen)
    {
    int cno;			/* PAP connexion number */
    char unrenamed[100];	/* space for building printer name */
    int namelen;
    #ifdef GNUC_HAPPY
    int typeoffset = 0;
    int typelen = 0;
    int zoneoffset = 0;
    #else
    int typeoffset, typelen, zoneoffset;
    #endif
    int renamed = FALSE;	/* set to TRUE if we rename the printer */
    
    #ifdef DEBUG
    debug("open_printer(\"%s\")",address);
    #endif

    /* Split up the name now as a syntax check. */
    if( ! ( (namelen=strcspn(address, ":"))
    		&& (namelen <= 32)
		&& (address[namelen] == ':')
		&& (typelen=strcspn(&address[(typeoffset=(namelen+1))], "@"))
		&& (typelen <= 32)
		&& (address[(zoneoffset=(namelen+2+typelen))-1] == '@')
		&& (strlen(&address[zoneoffset]) <= 32)
		) )
	{
	alert(printer_name, TRUE, "Syntax error in printer address.");
	exit_PRNERR_NORETRY();
	}

    while(TRUE)		/* really, just a goto */
    	{
    	/* First, try with the name as is. */
    	if( (cno=basic_open_printer(address,wlen)) >= 0 )
    	    return cno;
    	
    	/* 
    	** If we were looking for a printer of type "LaserWriter", there is
    	** nothing more we can do.  Also, if we have renamed a printer
    	** and it has not re-appeared yet, there is no point in trying
    	** to rename it again. 
    	*/
    	if( (strncmp(&address[typeoffset],"LaserWriter",typelen)==0) || renamed || !is_laserwriter )
    	    {
    	    alert(printer_name,TRUE,"Printer \"%s\" not found.",address);
    	    exit_PRNERR();
    	    }

    	/* 
    	** Previously we were looking for a hidden printer.  We didn't 
    	** find it, so now try with the type changed to "LaserWriter". 
    	*/
    	sprintf(unrenamed,"%.*s:LaserWriter@%s",namelen,address,&address[zoneoffset]);
    	if( (cno=basic_open_printer(unrenamed,wlen)) < 0 )
    	    {
    	    alert(printer_name,TRUE,"Printer \"%s\" not found,",address);
    	    alert(printer_name,FALSE,"nor is \"%s\".",unrenamed);
    	    exit_PRNERR();
    	    }			
    
        /* now, send the code to the printer to hide it. */
        hide_printer(cno,&address[typeoffset],typelen);
        renamed=TRUE;
    
        /* Close the connection, we can do nothing more with it. */
        PAPClose(cno);

	/* Give the printer time to recover. */
	sleep(10);
	} /* end of while(TRUE) */
    
    } /* end of open_printer() */

/*
** Receive any data the printer has for us.
**
** Return TRUE if an end of job indicator is received.
**
** If the flag is TRUE, the data is copied to stdout, 
** otherwise it is discarded.
*/
int receive(int cno, int flag)
    {
    static int rcomp=0;		/* read complete flag, must be static */
    static int rlen=0;		/* bytes read, must be static */
    static int eoj=0;
    static char rbuf[LocalQuantum * 512];
    int paperr;
	
    #ifdef DEBUG
    debug("receive(%d,%d)",cno,flag);
    #endif

    if(rcomp > 0)			/* If PAPread() not done, return. */
	{
	#ifdef DEBUG
	debug("PAPRead() not done");
	#endif
	return FALSE;
	}

    if(rcomp < 0)			/* If there was an error, */
	{				/* then make it a printer alert. */
	alert(printer_name,TRUE,"atalk interface: PAPRead() completion error: %d",rcomp);
	exit_PRNERR();
	}
	
    #ifdef DEBUG
    if(rlen)
	{
	debug("%d bytes received:",rlen);
	debug("%.*s",rlen,rbuf);
	}
    #endif

    if(rlen && flag)			/* If we have something, */
	{				/* and we are not hiding a printer, */
	write(1,rbuf,rlen);		/* send it to stdout. */
	}
	
    if(eoj)				/* If the read that just completed */
	{				/* detected EOJ, return TRUE. */
	#ifdef DEBUG
	debug("EOJ from printer");
	#endif
	rlen=0;
	eoj=0;
    	return TRUE;
    	}
		
    /* Start a new PAPRead(). */
    #ifdef DEBUG
    debug("Starting new PAPRead()");
    #endif
    if( (paperr=PAPRead(cno,rbuf,&rlen,&eoj,&rcomp)) < 0 )
    	{
    	alert(printer_name,TRUE,"atalk interface: PAPRead() returned %d",paperr);
    	exit_PRNERR();
    	}

    #ifdef DEBUG
    debug("receive() done");
    #endif

    return FALSE;			/* not server EOF */
    } /* end of receive() */
	
/*
** Copy the job from stdin to the printer.
** Return TRUE if no more jobs.
*/
int copy_job(int cno, int wlen)
    {
    int wcomp=0;		/* PAPWrite() completion flag */
    char tobuf[wlen];		/* Buffer to read blocks form pipe and write them to pritner */
    int tolen=0;		/* number of bytes to write to printer (initial zero is for GNU-C) */
    int paperr;			/* return value of PAPWrite() */
    int server_eoj=FALSE;

    #ifdef DEBUG
    debug("copy_job(%d,%d)",cno,wlen);
    #endif

    do	{
	if( ! server_eoj )	/* We must always receive first */
	    server_eoj = receive(cno,TRUE);

	if(wcomp <= 0)		/* if PAPWrite() done or error */
	    {
	    if(wcomp < 0)	/* if error */
	    	{
	    	alert(printer_name,TRUE,"atalk interface: copy_job(): PAPWrite() completion error: %d",wcomp);
	    	exit_PRNERR();
	    	}

	    /* PAPWrite() is done, read next block from the pipe on stdin. */
	    sigprocmask(SIG_UNBLOCK,&sigset_sigusr1,(sigset_t*)NULL);
	    sigalrm_caught=FALSE;
	    alarm(1);
	    while( (tolen=read(0,tobuf,wlen)) == -1 )
	    	{
		if(errno==EAGAIN)	/* Posix response for no data. */
		    {
		    tolen=0;
		    break;
		    }

	    	if(errno!=EINTR)	/* if not interupted system call, */
	    	    {
	    	    alert(printer_name,TRUE,"Pipe read error, errno=%d (%s)",errno,strerror(errno));
	    	    exit_PRNERR();
	    	    }

		if(sigalrm_caught)	/* if read() timed out */
		    {
		    sigalrm_caught=FALSE;
		    tolen=-1;		/* set special flag value */
		    break;
		    }
	    	}
	    alarm(0);
	    sigprocmask(SIG_BLOCK,&sigset_sigusr1,(sigset_t*)NULL);

	    #ifdef DEBUG
	    debug("%d bytes from pipe",tolen);
	    #endif

	    /* If there really is a block, dispatch it to the printer. */
	    if( tolen!=-1 && (paperr=PAPWrite(cno,tobuf,tolen,tolen==0?TRUE:FALSE,&wcomp)) < 0 )
	    	{
	    	alert(printer_name,TRUE,"atalk interface: copy_job(): PAPWrite() returned %d",paperr);
	    	exit_PRNERR();
	    	}

	    } /* end of if write done or error completing PAPwrite() */

	abSleep(4,TRUE);	/* give atalk time to work */

	#ifdef DEBUG
	debug("wcomp=%d",wcomp);
	#endif
	} while(tolen || wcomp);
	
    #ifdef DEBUG
    debug("end of job sent, waiting for response");
    #endif

    /*
    ** If we are sending to a LaserWriter compatible printer, wait for
    ** the rest of the response.  (That is, wait until the server
    ** (that is, the printer) sends eoj.)
    */
    if(is_laserwriter)
	{
	do  {
	    abSleep(4,TRUE);
	    } while( ! server_eoj && ! (server_eoj = receive(cno,TRUE)) );
	}
	
    #ifdef DEBUG
    debug("copy_job() done");
    #endif

    return sigusr1_caught;	/* return TRUE if more jobs coming */
    } /* end of copy_job() */

/*
** Main (Program entry point)
*/
int main(int argc, char *argv[])
    {
    char *printer_address;	/* pointer to "name:type@zone" */
    char *printer_options;	/* interface options passed to us by pprdrv */
    int printer_jobbreak;
    int printer_cno;		/* connection number */
    int wlen;
    
    /*
    ** If we do not have the bare minimum of parameters, say so.
    ** Use "-" for the printer name when calling alert() so that
    ** alert will print the message on stderr.
    */
    if(argc < 3)
        {
        alert("-", TRUE, "atalk interface: insufficient parameters");
        exit_PRNERR_NORETRY();
	}
		
    /* Save the printer name and address in named variables. */
    printer_name = argv[1];
    if( (printer_address = getenv("PPR_GS_INTERFACE_HACK_ADDRESS")) == (char*)NULL )
	printer_address = argv[2];
    if( (printer_options = getenv("PPR_GS_INTERFACE_HACK_OPTIONS")) == (char*)NULL )
	printer_options = argc >= 4 ? argv[3] : "";
    printer_jobbreak = argc >= 7 ? atoi(argv[6]) : JOBBREAK_NONE;

    /* Check the length of the printer address. */
    if(strlen(printer_address) > (32+1+32+1+32))
	{
	alert(printer_name, TRUE, "Printer address \"%s\" is too long\n", printer_address);
	exit_PRNERR_NORETRY();
        }
            	
    /* Parse the options string, searching for name=value pairs. */
    {
    char name[25];
    char value[16];
    int retval;

    options_start(printer_options);
    while( (retval=options_get_one(name, sizeof(name), value, sizeof(value))) > 0 )
    	{
	if(strcmp(name, "patron") == 0)
	    {
	    patron = atoi(value);
	    }
	else if(strcmp(name, "is_laserwriter") == 0)
	    {
	    if( (is_laserwriter=torf(value)) == ANSWER_UNKNOWN )
	    	{
	    	options_error = "is_laserwriter must be set to \"true\" or \"false\"";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "open_retries") == 0)
	    {
	    open_retries = atoi(value);
	    if(open_retries < -1 || open_retries == 0)
	    	{
		options_error = "Value must be a positive integer or -1 (infinity)";
		retval = -1;
		break;
	    	}	    
	    }
	else if(strcmp(name, "lookup_retries") == 0)
	    {
	    lookup_retries = atoi(value);
	    if(lookup_retries < 1)
	    	{
	    	options_error = "Value must be a positive integer";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "lookup_interval") == 0)
	    {
	    lookup_interval = atoi(value);
	    if(lookup_interval < 1)
	    	{
	    	options_error = "Value must be a positive integer";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "status_update_interval") == 0)
	    {
	    if( (status_update_interval = atoi(value)) < 0 )
	    	{
	    	options_error = "Value must be a positive integer or zero";
	    	retval = -1;
	    	break;
	    	}
	    }
	else
	    {
	    options_error = "Unrecognized keyword in printer options";
	    retval = -1;
	    break;
	    }
    	}

    if(retval == -1)
    	{
    	alert(printer_name,TRUE,"Option parsing error:  %s", options_error);
    	alert(printer_name,FALSE,"%s",printer_options);
    	alert(printer_name,FALSE,"%*s^ right here", options_error_index, "");
    	exit_PRNERR_NORETRY();
    	}
    }

    /* 
    ** Define a signal set which contains SIGUSR1
    ** we will use this for blocking and unblocking it. 
    */
    sigemptyset(&sigset_sigusr1);
    sigaddset(&sigset_sigusr1,SIGUSR1);
	
    /* 
    ** Install the USR1 signal handler, block SIGUSR1,
    ** and let pprdrv know we are ready to receive
    ** SIGUSR1 end of job notifications.
    **
    ** (If the jobbreak method is "none" we will not do this.
    ** This condition exists so that ppad may invoke the 
    ** interface to make queries during setup without fear
    ** of being hit by a signal.  As of version 1.30, this feature
    ** of ppad is not implemented yet.)
    */
    signal(SIGUSR1,sigusr1_handler);
    sigprocmask(SIG_BLOCK,&sigset_sigusr1,(sigset_t*)NULL);
    if( printer_jobbreak != JOBBREAK_NONE )
	kill(getppid(),SIGUSR1);

    /*
    ** Install the handler for SIGALRM which we will be
    ** using to cause read() to timeout in copy_job() so
    ** we can call receive().
    */
    signal(SIGALRM,sigalrm_handler);

    abInit(FALSE);	/* initialize AppleTalk (no display) */
    nbpInit();		/* initialize Name Binding Protocol */
    PAPInit();		/* initialize Printer Access Protocol */
	
    /* open the connection to the printer */
    printer_cno=open_printer(printer_address,&wlen);
	
    /* copy the data */
    while( copy_job(printer_cno,wlen) ) /* continue printing jobs */
	{				/* while we get SIGUSR1 job seperators. */
	fcntl(0,F_SETFL,0);		/* Clear the non-blocking flag. */
	sigusr1_caught=FALSE;		/* Clear EOJ indicator. */
	kill(getppid(),SIGUSR1);	/* Tell pprdrv that we understood EOJ. */
	}
	
    /* close the connection to the printer */
    PAPClose(printer_cno);

    return EXIT_PRINTED;
    } /* end of main() */
	
/* end of file */
