/*
** ~ppr/src/interfaces/atalk_ali.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 2 April 1997.
*/

/*
** Apple Printer Access Protocol (PAP) interface.
**
** This version works with the AT&T/Apple AppleTalk Library Interface 
** or the ALI compatibility library which David Chappell wrote for Netatalk.
**
** The AppleTalk Library Interface was jointly defined by AT&T and Apple 
** Computer.  It is described in "NetBIOS and Transport Interface
** Programmer's Reference, NCR publication number D2-0678-A, November 1991.
*/

/*
** If DEBUG is defined then this interface will write 
** debugging information to /var/spool/ppr/logs/interface_atalk.
*/
/* #define DEBUG 1 */

/*
** This enables a hack which causes copy_job() to finish if
** status_update_interval is non-zero and it notices that
** the printer status has been idle for approximately the
** indicated number of seconds after the job is done.
*/
/* #define IDLE_HACK_TIMEOUT 180 */

#include "global_defines.h"
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include "interface.h"

/* Includes for ALI */
#include <at/appletalk.h>
#include <at/nbp.h>
#include <at/pap.h>

/* Prototypes which are missing from the AT&T ALI include files: */
#include "pap_proto.h"

/* Values for address_status parameter of open_printer() */
#define ADDRESS_STATUS_UNKNOWN 0
#define ADDRESS_STATUS_CONFIRM 1
#define ADDRESS_STATUS_RECENT 2

#define MAX_ADDRESS_CACHE 200		/* max size of address cache file */
char *printer_name;			/* name to use with alert() */
#define LocalQuantum 1          	/* The number of 512 byte buffers we provide */
#define MaxWriteQuantum 8		/* Maximum size of output buffer in 512 byte blocks */
int sigusr1_caught = FALSE;		/* set TRUE by the SIGUSR1 signal handler */
sigset_t sigset_sigusr1;		/* a signal set containing SIGUSR1 */
struct timeval *select_timeout_to_use;	/* select() timeout to use */
struct timeval timeval_zero = {0,0};	/* zero seconds */

/* Forward reference. */
int receive(int fd, int flag);

/* Option variables: */
int open_retries = 10;			/* number of times to retry pap_open() */
int lookup_retries = 8;			/* number of times to retry NBP lookup */
int lookup_interval = 1;		/* interval between retries */
int is_laserwriter = TRUE;		/* Is this a LaserWriter compatible PostScript printer? */
long patron = -1;			/* PID of gsatalk */
int status_update_interval = 0;		/* ask for status this often */
int address_cache = TRUE;   	    	/* Should we use the address cache? */

#ifdef DEBUG
#define DODEBUG(a) debug a
#else
#define DODEBUG(a)
#endif

/* 
** Write a debugging line to the log file.
*/
#ifdef DEBUG
void debug(const char string[], ... )
    {
    va_list va;
    FILE *file;

    va_start(va,string);
    if( (file=fopen(LOGDIR"/interface_atalk","a")) != NULL )
        {
        fprintf(file,"DEBUG: (%ld) ",(long)getpid());
        vfprintf(file,string,va);
        fputc('\n',file);
        fclose(file);
        }
    va_end(va);
    } /* end of debug() */
#endif

/*
** Two routines used to exit with an error indication.  Normally these simply
** exit with the values EXIT_PRNERR and EXIT_PRNERR_NORETRY respectively,
** however if the undocumented option "patron=PID" is used, they will send
** SIGUSR1 or SIGUSR2 to the indicated process and exit with a normal code.
**
** This bizzar feature is used by the interface "gsatalk".
** This usage of SIGUSR1 has nothing to do with its usage when
** the atalk interface is operating in the normal manner.
*/
void exit_PRNERR(void)
    {
    DODEBUG(("exit_PRNERR()"));

    if(patron != -1)			/* If we are begin run from Ghostscript, */
    	{
    	kill((pid_t)patron,SIGUSR1);	/* tell the shell script */
    	sleep(1);			/* allow interface script time to get signal and exit */
    	exit(EXIT_PRINTED);
    	}

    exit(EXIT_PRNERR);
    } /* end of exit_PRNERR() */
    
void exit_PRNERR_NORETRY(void)
    {
    DODEBUG(("exit_PRNERR_NORETRY()"));

    if(patron != -1)
    	{
    	kill((pid_t)patron,SIGUSR2);
    	sleep(1);
    	exit(EXIT_PRINTED);
    	}
    exit(EXIT_PRNERR_NORETRY);
    } /* end of exit_PRNERR_NORETRY() */

/*
** Catch user signal 1.
** This signal indicates that we should consider a zero byte
** read() from the pipe on stdin to be an indication of end of job.
*/
void sigusr1_handler(int signum)
    {
    DODEBUG(("SIGUSR1 caught"));
    sigusr1_caught = TRUE;			/* set indicator */
    select_timeout_to_use = &timeval_zero;	/* sabatoge any select() which may slip thru in copy_job() */
    } /* end of sigusr1_handler */

/*
** Handler for signals which are likely to terminate this interface.
** We must catch these signals so we can call exit().  The NATALI
** AppleTalk library uses atexit() to install termination handlers.
*/
void term_handler(int signum)
    {
    DODEBUG(("Killed by signal \"%s\".", strsignal(signum)));
    exit(EXIT_SIGNAL);        
    } /* end of term_handler() */

/*
** Call this to explain what went wrong if the connection breaks.
** Basically, this function is used to explain a failed pap_read()
** or a failed pap_write().
*/
void fatal_pap_error(int pap_error, int sys_error)
    {
    DODEBUG(("fatal pap error, pap_error=%d (%s), sys_error=%d (%s)", pap_error, pap_strerror(pap_error), sys_error, strerror(sys_error)));

    switch(pap_error)
        {
        case PAPSYSERR:
            alert(printer_name, TRUE, "Unix system error, sys_error=%d (%s).", sys_error, strerror(sys_error) );
            break;
        case PAPHANGUP:
            alert(printer_name, TRUE, "Unexpected hangup.  (Printer turned off while printing?)");
	    break;
        case PAPTIMEOUT:
            alert(printer_name, TRUE, "Timeout error.  (Network communication problem.)");
            break;
        default:
            alert(printer_name, TRUE, "atalk interface: PAP error, pap_error=%d (%s)", pap_error, pap_strerror(pap_error)); 
            break;
        }
    exit_PRNERR();
    } /* end of fatal_pap_error() */

/*
** Rename the printer in order to hide it.
**
** This routine must be called after the printer has been
** opened under its original type of "LaserWriter".
**
** This routine is called by open_printer() below.
*/
void hide_printer(int fd, const char newtype[])
    {
    char writebuf[256];

    DODEBUG(("hiding printer, newtype=\"%s\"",newtype));

    /*
    ** Code to change the printer type.  This code seems to 
    ** temporarily change the type of all LaserWriter compatible
    ** printers.  Notice that it does not get the exitserver code
    ** or exitserver password from the PPD file.  This could
    ** be considered a deficiency, but it is not a serious one.
    */
    strcpy(writebuf,	"%!PS-Adobe-3.0 ExitServer\n"
			"0 serverdict begin exitserver\n"
			"statusdict begin\n"
			"currentdict /appletalktype known{/appletalktype}{/product}ifelse\n(");
    strcat(writebuf,	newtype);
    strcat(writebuf,	") def\n"
			"end\n"
			"%%EOF\n");

    /* Write the block, appending an end of job indication. */
    if( pap_write(fd,writebuf,strlen(writebuf),1,PAP_WAIT) == -1 )
        fatal_pap_error(pap_errno, errno);

    /* Throw away the output until EOJ. */
    while( ! receive(fd,FALSE)) ;
    } /* end of hide_printer() */

/*
** Return a handle which refers to the printer after filling in addr.
** If it fails, return -1.
*/
int basic_open_printer(at_entity_t *entity, at_nbptuple_t *addr, int *wlen, int do_lookup)
    {
    at_retry_t retries;
    u_short quantum;
    int fd;			/* file discriptor of open printer connexion */
    int ret;			/* return value for various functions */
    u_char more;
    unsigned char status[256];	/* status string from pap_open() */

    DODEBUG(("basic_open_printer(entity={ {%d, \"%.*s\"}, {%d, \"%.*s\"}, {%d, \"%.*s\"} }, addr={%d:%d:%d}, wlen=?, do_lookup=%d)",
    	entity->object.len, entity->object.len, entity->object.str,
    	entity->type.len, entity->type.len, entity->type.str,
    	entity->zone.len, entity->zone.len, entity->zone.str,
    	(int)addr->enu_addr.net, (int)addr->enu_addr.node, (int)addr->enu_addr.socket, do_lookup));

    /*
    ** Set the retry parameters to the values specified
    ** by the interface options.
    */
    retries.retries = lookup_retries;
    retries.interval = lookup_interval;

    /* Resolve the printer name to an address. */
    if( do_lookup && (ret=nbp_lookup(entity, addr, 1, &retries, &more)) != 1 )
        {				/* if failed to return 1 entry */
        if(ret == -1)			/* if lookup error */
            {				/* (i.e. worse than `not found') */
	    if(nbp_errno==NBPBADNAME)
		{
	    	alert(printer_name, TRUE, "Invalid printer address length.");
		exit_PRNERR_NORETRY();
	    	}
            alert(printer_name,TRUE,"Printer name lookup failed, nbp_errno=%d (%s), errno=%d (%s)",
            	nbp_errno, nbp_strerror(nbp_errno), errno, strerror(errno) ); 
	    exit_PRNERR();
            }
        if( ret > 1 )			/* if multiple printers */
	    {
            alert(printer_name, TRUE, "%d printers answer to \"%s\".", ret, printer_name);
	    exit_PRNERR();
            }

	DODEBUG(("basic_open_printer(): printer not found"));

	return -1;
        } /* end of if name lookup fails */

    /* Open a circuit to the printer. */
    quantum = LocalQuantum;		/* tell how many 512 byte buffers we will privide */     
    if( (fd=pap_open(addr, &quantum, status, (short)open_retries)) == -1 )
        {
        if(pap_errno==PAPBUSY)		/* If error is because printer */
            {				/* is busy, */
	    DODEBUG(("basic_open_printer(): printer is busy"));

	    /* 
	    ** Print the status on stdout in the form in which the
	    ** printer itself would send it.  This is so that
	    ** we can tell what it is busy doing.
	    */
	    printf("%%%%[ %.*s ]%%%%\n", (int)status[0], &status[1]);

	    exit(EXIT_ENGAGED);		/* Use special exit code. */
            }
        else				/* if other error, */
            {				/* other errors, just print message. */
	    if(pap_errno == PAPSYSERR)
		{
		alert(printer_name, TRUE, "atalk interface: pap_open() failed, pap_errno=%d (%s), errno=%d (%s)",
			pap_errno, pap_strerror(pap_errno), errno, strerror(errno) );
		}
	    else
		{
		alert(printer_name, TRUE, "atalk interface: pap_open() failed, pap_errno=%d (%s)",
			pap_errno, pap_strerror(pap_errno));
		}
	    exit_PRNERR();
            }
	}

    /*
    ** If the status contains the string "status: PrinterError:", print it 
    ** even if the open was sucessful.
    */
    if(status[0] > 21 && strncmp("status: PrinterError:", &status[1], 21) == 0)
	printf("%%%%[ %.*s ]%%%%\n", (int)status[0], &status[1]);

    /*
    ** Limit remote flow quantum to the max we can do
    ** and compute how many bytes we can write at a time.
    */
    if(quantum > MaxWriteQuantum)
	quantum = MaxWriteQuantum;
    *wlen = quantum * 512;

    DODEBUG(("circuit open, status is: {%d, \"%.*s\"}", (int)status[0], (int)status[0], &status[1] ));

    return fd;
    } /* end of basic_open_printer() */

/*
** Return a handle which refers to the printer.
*/
int open_printer(const char atalk_name[], at_nbptuple_t *addr, int *wlen, int address_status)
    {
    at_entity_t entity;     /* broken up name */
    int fd;                 /* file discriptor of open printer */
    char newtype[33];       /* type to rename printer to */
    int renamed = FALSE;
    int do_lookup = TRUE;

    DODEBUG(("open_printer(\"%s\", wlen)", atalk_name));

    /* Parse the printer name, type, and zone into seperate data items. */
    if( nbp_parse_entity(&entity, atalk_name) )
        {
        alert(printer_name, TRUE, "Syntax error in printer address.");
	exit_PRNERR_NORETRY();
        }

    if(address_status != ADDRESS_STATUS_UNKNOWN)
    	{
    	do_lookup = FALSE;

	if(address_status == ADDRESS_STATUS_CONFIRM)
	    {
	    if(nbp_confirm(&entity, &(addr->enu_addr), (at_retry_t*)NULL) == -1)
	    	do_lookup = TRUE;
	    }
	}

    /*
    ** We will pass at most twice thru three times thru this loop.
    **
    ** Pass 1: Try to open the printer with the name as given.  Stop
    **         here if the entity type is "LaserWriter" or the interface
    **         option is_laserwriter is FALSE.
    ** Pass 2: Try to open as type "LaserWriter" and rename
    **         if sucessful
    ** Pass 3: Try again to open with name as given
    */
    while(TRUE)
	{
	/*
	** First, try the name as it is.
	** (Second time thru the loop, try it as it originally was.)
	*/
	if( (fd = basic_open_printer(&entity, addr, wlen, do_lookup)) >= 0 )
	    return fd;

	/*
	** If we were already looking for an entity of type "LaserWriter" 
	** or we have already tried to hide it or we aren't sending
	** to a laserwriter at all, then give up now.
        */
        if( (entity.type.len==11 && strncmp(entity.type.str,"LaserWriter",11)==0) || renamed || !is_laserwriter )
	    {
            alert(printer_name,TRUE,"Printer \"%s\" not found.", atalk_name);
	    exit_PRNERR();
            }

        /* Save the type we looked for in "newtype". */
        strncpy(newtype, entity.type.str, entity.type.len);
        newtype[(int)entity.type.len] = (char)NULL;	/* (int) is for GNU-C */

        /* Change the type to "LaserWriter". */
        strcpy(entity.type.str, "LaserWriter");
        entity.type.len = 11;

        /* Try again with the type of "LaserWriter". */
	if( (fd = basic_open_printer(&entity, addr, wlen, TRUE)) == -1 )
	    {
            alert(printer_name,TRUE,"Printer \"%s\" not found,", atalk_name);
	    alert(printer_name,FALSE,"nor is \"%.*s:LaserWriter@%.*s\".",
		(int)entity.object.len,entity.object.str,
		(int)entity.zone.len,entity.zone.str);
	    exit_PRNERR();
            }

	/* Now, hide the printer. */
	hide_printer(fd, newtype);
        if(pap_close(fd) == -1)		/* close connexion */
	    {
            alert(printer_name,TRUE,"atalk interface: pap_close(): failed, pap_errno=%d (%s), errno=%d (%s)",
            	pap_errno, pap_strerror(pap_errno), errno, strerror(errno) );
	    exit_PRNERR();
            }

	/* Say we renamed it and get ready for the next try. */
	renamed = TRUE;
	entity.type.len = strlen(newtype);
	strncpy(entity.type.str, newtype, entity.type.len);
	do_lookup = TRUE;
        } /* end of while(TRUE) */

    } /* end of open_printer() */

/*
** Receive data from the printer.
**
** Unlike the receive() function in atalk_cap.c, this function will
** block if there is no data to read.
**
** If the printer sends us EOJ, we will return TRUE.
**
** If the flag is TRUE, the data is copied to stdout, 
** otherwise it is discarded.
*/
int receive(int papfd, int flag)
    {
    u_char eoj;				/* used with pap_read() */
    int len;				/* number of bytes read */
    char readbuf[LocalQuantum*512];	/* buffer for messages from the printer */

    DODEBUG(("receive(papfd=%d, flag=%s)", papfd, flag ? "TRUE" : "FALSE"));

    if( (len=pap_read(papfd,readbuf,(LocalQuantum*512),&eoj)) == -1 )
        fatal_pap_error(pap_errno, errno);

    DODEBUG(("%d bytes read from printer, eoj=%d", len, eoj));

    /* 
    ** Copy the printer message to stdout and flush to
    ** make sure pprdrv gets it right away.
    */
    if(flag)
	{
	fwrite(readbuf,sizeof(char),len,stdout);     
	fflush(stdout);
	}
	
    DODEBUG(("%.*s", len, readbuf));

    DODEBUG(("receive() done"));

    return (eoj ? TRUE : FALSE);
    } /* end of receive() */

/*
** Copy one job.
**
** Return TRUE if we detect that pprdrv has closed its
** end of the pipe.
**
** Notice that the select() code differs slightly depending on 
** whether _NATALI_PAP is defined.  It will by defined if
** David Chappell's AppleTalk Library Interface compatiblity 
** library for Netatalk is being used in stead of the AT&T
** implementation.
*/
int copy_job(int papfd, at_nbptuple_t *addr, int wlen)
    {
    int stdin_closed = FALSE;		/* return value: TRUE if end of file on pipe on stdin */
    int lookret;			/* return value of pap_look() */
    int tosend = -1;			/* bytes left to send, -1 means buffer empty, -2 means EOJ sent */
    fd_set rfds;			/* set of file descriptors for select() */
    int server_eoj = FALSE;		/* TRUE if a pap_read() detected EOJ */
    char writebuf[MaxWriteQuantum*512];	/* data to be sent to the printer */
    int previous_sigusr1_caught;
    int selret;				/* return value from select() */
    int writes_blocked = FALSE;
    struct timeval idle_delay;		/* used for select() while waiting for something to do */
    int status_update_countdown = 1;
    #ifdef IDLE_HACK_TIMEOUT
    int idle_hack_count = 0;
    #endif

    DODEBUG(("copy_job(papfd=%d, wlen=%d)", papfd, wlen));

    sigusr1_caught = FALSE;
    
    /*
    ** Read bytes from the pipe and write them to the printer
    ** reading data from the printer whenever necessary.
    **
    ** This will continue until we get an end of job indication
    ** from the printer in acknowledgement of the one that 
    ** we will send.
    **
    ** If the EOJ block has been sent, tosend will equal -2.
    ** If an EOJ block has been received from the printer,
    ** server_eoj will be true.
    **
    ** So, if we are sending to a LaserWriter, this loop continues
    ** until we receive an EOJ indication from the LaserWriter in
    ** acknowledgement of our EOJ indicator.
    **
    ** If we are not sending to a LaserWriter, this loop continues
    ** only until we have sent an EOJ indication.
    */
    while( ! server_eoj && (is_laserwriter || tosend != -2) )
	{
	/*
	** If the send buffer is empty, see if there is data on stdin.
	** We know that the send buffer is empty when tosend equals -1.
	** 
	** If there is no data available now, read() will return -1, setting
	** errno to EAGAIN.  If the other end has closed the pipe, read()
	** will return 0.
	*/
	if(tosend == -1)
	    {
	    DODEBUG(("copy_job(): reading stdin"));

	    if( (tosend=read(0, writebuf, wlen)) == -1 )
            	{
	    	if(errno == EAGAIN)		/* If no data to read, */
		    {
		    if(sigusr1_caught)		/* If at a job break, */
		    	tosend = 0;		/* arrange to send EOJ block, */
		    else			/* otherwise, */
		    	tosend = -1;		/* the send buffer is still empty. */
		    }
		else				/* If other read() error, */
		    {				/* interface has failed. */
		    alert(printer_name, TRUE, "atalk interface: copy_job(): read error, errno=%d (%s)", errno, strerror(errno));
		    exit_PRNERR();
		    }
		}
	    else if(tosend == 0)		/* If end of file, */
	    	{
		DODEBUG(("copy_job(): pipe closed"));
	    	stdin_closed = TRUE;
	    	}
	    DODEBUG(("copy_job(): %d bytes read", tosend));
	    }

	/*
	** Use pap_look() to see what is happening on the PAP connexion.
	*/
	if( (lookret=pap_look(papfd)) == -1 )
	    {
	    alert(printer_name, TRUE, "atalk interface: copy_job(): pap_look() failed, pap_errno=%d (%s), errno=%d (%s)", pap_errno, pap_strerror(errno), errno, strerror(errno) );
	    exit_PRNERR();
	    }
	DODEBUG(("copy_job(): pap_look() returned %d (%s)", lookret, pap_look_string(lookret) ));

	/*
	** If data has been received from the printer,
	** read it now and go back to the top of the loop.
	**
	** Reading as soon as the printer wants us to is
	** the trick to preventing a deadlock.
	*/
	if(lookret == PAP_DATA_RECVD)
	    {
	    DODEBUG(("copy_job(): calling receive()"));
	    server_eoj = receive(papfd, TRUE);
	    continue;
	    }
	    
	/*
	** If pap_look() indicates that a previously blocked write 
	** can now be made or that it is all right to try to write,
	** though it may block and we have something to send, try now.
	*/
	if( (lookret==PAP_WRITE_ENABLED || (!writes_blocked && lookret==0)) && tosend >= 0 )
	    {
	    DODEBUG(("copy_job(): sending %d bytes to printer", tosend));

	    if( pap_write(papfd, writebuf, tosend, tosend>0?0:1, PAP_NOWAIT) == -1 )
		{
		if( pap_errno == PAPBLOCKED )
		    {
		    writes_blocked = TRUE;
		    DODEBUG(("copy_job(): writes blocked"));
		    }
		else	/* a real error */
		    {
		    fatal_pap_error(pap_errno, errno);
		    }
		}
	    else	/* pap_write() was entirely sucessful, */
	    	{
		writes_blocked = FALSE;

	    	if( tosend != 0 )	/* If not the EOJ block, */
		    tosend = -1;	/* prompt to load another one. */
	    	else			/* If it was the EOJ block, use a */
	    	    tosend = -2;	/* value to say "don't read any more from stdin". */
		}
            }

	/*
	** Has pap_look() returned something which indicates an error?
	** (Note: we will have handled PAP_DATA_RECVD above.)
	*/
	if(lookret != 0 && lookret != PAP_WRITE_ENABLED)
	    {
	    if(lookret == PAP_DISCONNECT)
		fatal_pap_error(PAPHANGUP, 0);
	    alert(printer_name, TRUE, "atalk interface: copy_job(): pap_look() returned %d (%s)", lookret, pap_look_string(lookret));
	    exit_PRNERR();
	    }

	/*
	** If we get here we can assume we are blocked on some front.
	**
	** We can't just keep calling pap_look() because that would eat
	** up scads of CPU time.  We will use select() to block right here
	** until there is some likelyhood that pap_look() will return
	** something different from what it returned last time.
	**
	** More specifically, we use select() to wait for something to
	** happen on the PAP connection, data to be received on stdin,
	** or SIGUSR1 to be received.
	*/
	previous_sigusr1_caught = sigusr1_caught;
	selret = 0;
	while(selret == 0 && (!sigusr1_caught || previous_sigusr1_caught))
	    {
	    FD_ZERO(&rfds);				/* empty set of read file descriptors */
	    if(tosend == -1) FD_SET(0, &rfds);		/* If we want data to send, tell select() to return if we get it. */
	    FD_SET(papfd, &rfds);			/* We want to know if anything happens on PAP connexion */

	    #define SELECT_TIME 5
	    idle_delay.tv_sec = SELECT_TIME;		/* kept short for NATALI */
	    idle_delay.tv_usec = 0;
	    select_timeout_to_use = &idle_delay;

	    /*
	    ** Unblock SIGUSR1 and call select().  If SIGUSR1 is received 
	    ** the moment we unblock it, it will change select_timeout_to_use
	    ** so that select() will return immediately.  Otherwise, select()
	    ** will return when it is interupted by SIGUSR1 or when the timeout 
	    ** expires.
	    */
	    DODEBUG(("copy_job(): calling select()"));
	    sigprocmask(SIG_UNBLOCK, &sigset_sigusr1, (sigset_t*)NULL);
	    selret = select(papfd+1, &rfds, (fd_set*)NULL, (fd_set*)NULL, select_timeout_to_use);
	    sigprocmask(SIG_BLOCK, &sigset_sigusr1, (sigset_t*)NULL);

	    if(selret == -1)
		{
		if(errno != EINTR)
		    {
	    	    alert(printer_name, TRUE, "atalk interface: copy_job(): select() failed, errno=%d (%s)", errno, strerror(errno) );
	    	    exit_PRNERR();
	    	    }
		DODEBUG(("copy_job(): select() interupted"));
		}

	    DODEBUG(("copy_job(): select() returned %d", selret));

	    /*
	    ** We are still inside the inner loop which waits 
	    ** at the bottom of the outer loop for something
	    ** to happen.
	    **
	    ** If nothing has happened on the PAP endpoint and
	    ** SIGUSR1 has not just been caught,
	    */
	    if(selret == 0 && (!sigusr1_caught || previous_sigusr1_caught))
		{
		/*
		** If we are using NATALI rather than the AT&T STREAMS 
		** based AppleTalk, we must call a PAP function every
		** so often, otherwise important protocol things such as
		** tickle packets and timeouts won't work.
		*/
		#ifdef _NATALI_PAP
		if(pap_look(papfd) != 0)
		    break;
		#endif

		/*
		** Since we don't have anything better to do, we might as well 
		** inquire into the present status of the printer.
		*/
		DODEBUG(("status_update_countdown = %d\n", status_update_countdown));
		if(status_update_interval > 0 && (status_update_countdown-=SELECT_TIME) <= 0)
		    {
		    static int first_update_sent = FALSE;
		    unsigned char status[256];

		    if(pap_status(addr, status) == -1)
			{
			DODEBUG(("copy_job(): pap_status() failed, pap_errno=%d", pap_errno));		    
			}
		    else
			{
			if(!first_update_sent)
			    {
			    fputs("%%[ PPR status updates follow ]%%\n", stdout);
			    first_update_sent = TRUE;
			    }
			printf("%%%%[ %.*s ]%%%%\n", status[0], &status[1]);
			fflush(stdout);

			/* This is a hack for queue hangs: */
			#ifdef IDLE_HACK_TIMEOUT
			if(tosend == -2 && strncmp(&status[1], "status: idle", 12) == 0)
			    {
			    if( (idle_hack_count++ * status_update_interval) > IDLE_HACK_TIMEOUT )
				{
				server_eoj = 1;	/* so outer loop will stop */
				break;		/* break out of inner loop */
				}
			    }
			#endif
			}
		    status_update_countdown = status_update_interval;
		    }
		} /* end of if nothing happened */
	    } /* end of idle loop */
	    
        } /* until EOJ from printer (at least for LaserWriters) */

    DODEBUG(("copy_job(): returning %s", stdin_closed ? "TRUE" : "FALSE"));
    return stdin_closed;
    } /* end of copy_job() */
              
/*
** main function
*/
int main(int argc, char *argv[])
    {
    char *printer_appletalk_name;			/* argument: string name of printer */
    char *printer_options;				/* argument: interface options */
    int printer_jobbreak;				/* argument: jobbreak method */
    int printer_fd;					/* open circuit to the printer */
    int wlen;						/* max bytes to write to printer */
    at_nbptuple_t printer_address;			/* AppleTalk address of the printer */
    int address_status = ADDRESS_STATUS_UNKNOWN;	/* Do we need to look up the AppleTalk address? */

    DODEBUG(("atalk interface starting, %s", datestamp()));

    /* 
    ** Make sure we have sufficient parameters.  Notice that we call
    ** alert with the printer name set to "-" since we don't know it
    ** yet.  This will result in the error message being sent to stderr.
    */
    if(argc < 3)
	{
        alert("-", TRUE, "atalk interface: insufficient parameters");
        exit_PRNERR_NORETRY();
        }

    /*
    ** Save the printer name, address, options, and the jobbreak method
    ** in named variables.
    **
    ** In the case of the printer's AppleTalk name and the interface
    ** options, we will use the values in certain environment variables
    ** in preference to the ones in the argument list.  We accept them
    ** from environment variables because Ghostscript has trouble with
    ** long OutputFile strings.  Ghostscript is used by the gsatalk
    ** interface.
    */
    printer_name = argv[1];
    if( (printer_appletalk_name = getenv("PPR_GS_INTERFACE_HACK_ADDRESS")) == (char*)NULL )
	printer_appletalk_name = argv[2];
    if( (printer_options = getenv("PPR_GS_INTERFACE_HACK_OPTIONS")) == (char*)NULL )
	printer_options = argc >= 4 ? argv[3] : "";
    printer_jobbreak = argc >= 7 ? atoi(argv[6]) : JOBBREAK_NONE;

    /* Make sure all address components are no longer than the legal length. */
    if(strlen(printer_appletalk_name) > (32+1+32+1+32))
	{
    	alert(printer_name, TRUE, "Printer address \"%s\" is too long.", printer_appletalk_name);
	exit_PRNERR_NORETRY();
    	}

    /* Parse the options string, searching for name=value pairs. */
    {
    char name[25];
    char value[16];
    int retval;

    options_start(printer_options);
    while( (retval=options_get_one(name,sizeof(name),value,sizeof(value))) > 0 )
    	{
	if(strcmp(name,"patron") == 0)
	    {
	    patron = atoi(value);
	    }
	else if(strcmp(name,"is_laserwriter") == 0)
	    {
	    if( (is_laserwriter=torf(value)) == ANSWER_UNKNOWN )
	    	{
	    	options_error = "is_laserwriter must be set to \"true\" or \"false\"";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "open_retries") == 0)
	    {
	    open_retries = atoi(value);
	    if(open_retries < -1 || open_retries == 0)
	    	{
		options_error = "Value must be a positive integer or -1 (infinity)";
		retval = -1;
		break;
	    	}	    
	    }
	else if(strcmp(name, "lookup_retries") == 0)
	    {
	    lookup_retries = atoi(value);
	    if(lookup_retries < 1)
	    	{
	    	options_error = "Value must be a positive integer";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "lookup_interval") == 0)
	    {
	    lookup_interval = atoi(value);
	    if(lookup_interval < 1)
	    	{
	    	options_error = "Value must be a positive integer";
	    	retval = -1;
	    	break;
	    	}
	    }
	else if(strcmp(name, "status_update_interval") == 0)
	    {
	    if( (status_update_interval = atoi(value)) < 0 )
		{
		options_error = "Value must be a positive integer or zero";
		retval = -1;
		break;
		}
	    }
	else if(strcmp(name, "address_cache") == 0)
	    {
	    if( (address_cache = torf(value)) == ANSWER_UNKNOWN )
	    	{
	    	options_error = "address_cache must be set to \"true\" or \"false\"";
	    	retval = -1;
	    	break;
	    	}	    
	    }
	else
	    {
	    options_error = "Unrecognized keyword in printer options";
	    retval = -1;
	    break;
	    }
    	}

    if(retval == -1)
    	{
    	alert(printer_name,TRUE,"Option parsing error:  %s", options_error);
    	alert(printer_name,FALSE,"%s", printer_options);
    	alert(printer_name,FALSE,"%*s^ right here", options_error_index, "");
    	exit_PRNERR_NORETRY();
    	}
    }

    /* Create a signal set for blocking SIGUSR1. */
    sigemptyset(&sigset_sigusr1);
    sigaddset(&sigset_sigusr1, SIGUSR1);

    /* Make sure stdin is in non-blocking mode. */
    fcntl(0, F_SETFL, O_NONBLOCK);

    /*
    ** Install a signal handler for the end of job signal
    ** and tell our parent (pprdrv) that it is in place
    ** by sending it SIGUSR1.
    **
    ** (If the jobbreak method is "none" we will not do this.
    ** This condition exists so that ppad may invoke the 
    ** interface to make queries during setup without fear
    ** of being hit by a signal.  As of version 1.30, ppad does
    ** not yet use this feature.)
    */
    signal(SIGUSR1, sigusr1_handler);
    sigprocmask(SIG_BLOCK,&sigset_sigusr1,(sigset_t*)NULL);
    if( printer_jobbreak != JOBBREAK_NONE )
	kill(getppid(),SIGUSR1);   

    /*
    ** Install a handler for any signal which is likely to be used
    ** to terminate this interface in order to cancel the job
    ** or halt the printer.
    */
    signal(SIGTERM, term_handler);	/* cancel or halt printer */
    signal(SIGHUP, term_handler);	/* parent (pprdrv) died */
    signal(SIGPIPE, term_handler);	/* conceivable as signal of parent's death */
    signal(SIGINT, term_handler);	/* control-C (during testing) */

    /*
    ** If the address_cache= option is true and the
    ** printer name is not "-" (which indicates that 
    ** this interface is being used to contact a printer
    ** which does not have a queue yet), then
    ** read cached printer address information.
    */
    if(address_cache && strcmp(printer_name, "-"))
	{
	int fd;
	char fname[MAX_PATH];
	
	sprintf(fname, ADDRESS_CACHE"/%s", printer_name);
	if( (fd=open(fname, O_RDONLY)) == -1 )
	    {
	    if(errno != ENOENT)
		alert(printer_name, TRUE, "atalk interface: can't open \"%s\" for read, errno=%d (%s)", fname, errno, strerror(errno));
	    }
	else		/* file exists and is open */
	    {
	    char temp[MAX_ADDRESS_CACHE+1];
	    int len;

	    len = read(fd, temp, MAX_ADDRESS_CACHE);

	    if(len == -1)
	    	alert(printer_name, TRUE, "atalk interface: read() of address cache file failed, errno=%d (%s)", errno, strerror(errno));
	    else if(len < 10 || len > MAX_ADDRESS_CACHE)	/* really paranoid! */
	    	alert(printer_name, TRUE, "atalk interface: address cache file is corrupt, length=%d", len);
	    else
	      {
	      /* NULL terminate the data from the file */
	      temp[len] = (char)NULL;

	      if(strncmp("atalk\n", temp, 6) == 0)	/* if address is for this interface, */
	    	{
		char *p = temp + 6;
		
	    	if(strcspn(p, "\n") == (len=strlen(printer_appletalk_name)) && strncmp(p, printer_appletalk_name, len) == 0)
	    	    {					/* and the printer address has not changed, */
		    int net, node, socket;

		    p+=len;
		    p++;
		    
		    /* Read the AppleTalk address */
		    if(sscanf(p, "%d %d %d", &net, &node, &socket) != 3)
			{
			alert(printer_name, TRUE, "atalk interface: sscanf() failed on cached address");
			}
		    else
		    	{
			struct stat statbuf;

			address_status = ADDRESS_STATUS_CONFIRM;

			/* See if the cached address is very recent */
			if(fstat(fd, &statbuf) == -1)
			    {
			    alert(printer_name, TRUE, "atalk interface: can't fstat address cache file, errno=%d (%s)", errno, strerror(errno));
			    }
			else
			    {
			    time_t time_now;
			    time(&time_now);
			    if((time_now - statbuf.st_mtime) < 20)
			    	address_status = ADDRESS_STATUS_RECENT;
			    }

			printer_address.enu_addr.net = net;
			printer_address.enu_addr.node = node;
			printer_address.enu_addr.socket = socket;
		    	}
	    	    }
	    	}
	      } /* end of if file is of plausible length */

	    /* close the cache file */
	    close(fd);

	    /* remove file since we will create a new one if we print sucessfully */
	    unlink(fname);
	    } /* end of if file open suceeded */
	}

    /*
    ** Open a PAP connexion to the printer.
    */
    printer_fd = open_printer(printer_appletalk_name, &printer_address, &wlen, address_status);

    /*
    ** The main loop.
    ** Call copy_job() until it detects that stdin has been closed.
    */
    while( ! copy_job(printer_fd, &printer_address, wlen) )
        {
        sigusr1_caught = FALSE;		/* Clear flag which indicates EOJ signal from pprdrv. */
	kill(getppid(), SIGUSR1);	/* Respond to pprdrv's SIGUSR1. */
        }

    /*
    ** Close the PAP ciruit to the printer.
    */
    DODEBUG(("closing PAP connection"));
    if(pap_close(printer_fd) == -1)
	{
        alert(printer_name, TRUE, "atalk interface: pap_close(): failed, pap_errno=%d (%s), errno=%d (%s)",
        	pap_errno, pap_strerror(pap_errno), errno, strerror(errno) );
	exit_PRNERR();
        }

    /*
    ** Cache the printer address:
    */
    if(address_cache && strcmp(printer_name, "-"))
	{
	int fd;
	char fname[MAX_PATH];
	
	sprintf(fname, ADDRESS_CACHE"/%s", printer_name);
	if( (fd=open(fname, O_WRONLY | O_CREAT | O_TRUNC, UNIX_644)) == -1 )
	    {
	    alert(printer_name, TRUE, "atalk interface: can't open \"%s\" for write, errno=%d (%s)", fname, errno, strerror(errno));
	    }
	else
	    {
	    char temp[200];
	    int towrite, written;
	    sprintf(temp, "atalk\n%s\n%d %d %d\n", printer_appletalk_name, 
	    	(int)printer_address.enu_addr.net,
	    	(int)printer_address.enu_addr.node,
	    	(int)printer_address.enu_addr.socket);
	    if( (written=write(fd, temp, towrite=strlen(temp))) == -1 )
	    	alert(printer_name, TRUE, "atalk interface: write() to address cache file failed, errno=%d (%s)", errno, strerror(errno));
	    if(written != towrite)
		alert(printer_name, TRUE, "atalk interface: towrite=%d, written=%d", towrite, written);
	    close(fd);
	    }
	}
	
    return EXIT_PRINTED;
    } /* end of main() */

/* end of file */
