/*
** ~ppr/src/indexttf/indexttf.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file was last revised 31 November 1996.
*/

/*
** The purpose of this program is to make a GDBM database of the 
** TrueType fonts available to PPR.  The PostScript names are the keys,
** the file names are the data.
**
** This program is invoked in this manner:
**
** /usr/ppr/bin/indexttf /usr/ppr/TTFonts/somefont.ttf /otherdir/otherfont.ttf
**
** The indicated files are added to the database.  All file names must have
** absolute paths.  You may use shell wildcards such as "*.ttf" to specify
** many files at at time.  The only way to delete files from the database
** is to delete the database file "/etc/ppr/ttfonts" and run indexttf to
** put back the ones you want.
**
** If this program seems a little odd, if some of the functions seem
** like overkill, it is because it is a stripped down version of
** pprdrv_tt2.c.
*/

#include "global_defines.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>

/* In case myalloc() and friends fail. */
const int lib_memory_fatal = 1;

#ifdef TT_DBM_gdbm
#include <gdbm.h>
/* needed for gdbm 1.5 */
extern gdbm_error gdbm_errno;
#endif

/* Types used in TrueType font files. */
#define BYTE unsigned char
#define USHORT unsigned short int
#define ULONG unsigned int

/* This structure tells what we have found out about */
/* the current font. */
struct TTFONT
    {
    char *filename;			/* Name of TT file */
    FILE *file;				/* the open TT file */

    unsigned int numTables;		/* number of tables present */
    BYTE *offset_table; 		/* Offset table in memory */
    } ;

/*
** Handle fatal errors.
** Print a message and exit.
*/
void fatal(int exitval, const char message[], ... )
    {
    va_list va;

    printf("\n");		/* close line and flush */

    va_start(va,message);
    fprintf(stderr,"Fatal: ");
    vfprintf(stderr,message,va);
    fprintf(stderr,"\n");
    va_end(va);
    exit(exitval);
    } /* end of fatal() */

/*---------------------------------------
** Endian conversion routines.
** These routines take a BYTE pointer
** and return a value formed by reading
** bytes starting at that point. 
**
** These routines read the big-endian
** values which are used in TrueType
** font files.
---------------------------------------*/

/*
** Get an Unsigned 32 bit number.
*/
ULONG getULONG(BYTE *p)
    {
    int x;
    ULONG val=0;    

    for(x=0; x<4; x++)
	{
	val *= 0x100;
	val += p[x];	
	}
    
    return val;
    } /* end of ftohULONG() */

/*
** Get an unsigned 16 bit number.
*/
USHORT getUSHORT(BYTE *p)
    {
    int x;
    USHORT val=0;    

    for(x=0; x<2; x++)
	{
	val *= 0x100;
	val += p[x];	
	}
    
    return val;
    } /* end of getUSHORT() */

/*-----------------------------------------------------------------------
** Load a TrueType font table into memory and return a pointer to it.
** The font's "file" and "offset_table" fields must be set before this
** routine is called.
**
** This first argument is a TrueType font structure, the second 
** argument is the name of the table to retrieve.  A table name
** is always 4 characters, though the last characters may be 
** padding spaces.
-----------------------------------------------------------------------*/
BYTE *GetTable(struct TTFONT *font, char *name)
    {
    BYTE *ptr;
    unsigned int x;
    
    /* We must search the table directory. */
    ptr = font->offset_table + 12;
    x=0;
    while(TRUE)
    	{
	if( strncmp((char*)ptr,name,4) == 0 )
	    {
	    ULONG offset,length;
	    BYTE *table;

	    offset = getULONG( ptr + 8 );
	    length = getULONG( ptr + 12 );	    
	    table = (BYTE*)myalloc( sizeof(BYTE), length );

	    if( fseek( font->file, (long)offset, SEEK_SET ) )
	    	fatal(1,"TrueType font \"%s\" may be corrupt (reason 3)",font->filename);

	    if( fread(table,sizeof(BYTE),length,font->file) != (sizeof(BYTE) * length))
		fatal(1,"TrueType font \"%s\" may be corrupt (reason 4)",font->filename);
		
	    return table;
	    }

    	x++;
    	ptr += 16;
    	if(x == font->numTables)
	    fatal(1,"TrueType font \"%s\" is missing table '%s'",font->filename,name);
    	}

    } /* end of GetTable() */

/*--------------------------------------------------------------------
** Get the PostScript name for this font from the 'name' table.
--------------------------------------------------------------------*/
char *get_postscript_name(struct TTFONT *font)
    {
    BYTE *table_ptr,*ptr2;
    int numrecords;			/* Number of strings in this table */
    BYTE *strings;			/* pointer to start of string storage */
    int x;
    int platform,encoding;		/* Current platform id, encoding id, */
    int language,nameid;		/* language id, name id, */
    int offset,length;			/* offset and length of string. */
    char *postname=(char*)NULL;
    
    table_ptr = GetTable(font,"name");
    numrecords = getUSHORT( table_ptr + 2 );
    strings = table_ptr + getUSHORT( table_ptr + 4 );
    
    ptr2 = table_ptr + 6;
    for(x=0; x < numrecords; x++,ptr2+=12)
    	{
	platform = getUSHORT(ptr2);
	encoding = getUSHORT(ptr2+2);
	language = getUSHORT(ptr2+4);
	nameid = getUSHORT(ptr2+6);
	length = getUSHORT(ptr2+8);
	offset = getUSHORT(ptr2+10);

	/* PostScript name */
	if( platform == 1 && nameid == 6 )
	    {
	    postname = (char*)myalloc(sizeof(char),length+1);
	    strncpy(postname,(char*)(strings+offset),length);
	    postname[length] = (char)NULL;
	    break;
	    }
    	}

    myfree(table_ptr);
    
    return postname;
    } /* end of Read_name() */

/*------------------------------------------------------------------
** Open a Truetype font file and get its PostScript name.
------------------------------------------------------------------*/
char *do_ttfont(char *filename)
    {
    struct TTFONT font;
    char *psname;
    
    /* Save the file name for error messages. */
    font.filename=filename;

    /* Open the font file */
    if( (font.file = fopen(filename,"r")) == (FILE*)NULL )
    	fatal(1,"Failed to open TrueType font \"%s\"",filename);

    /* Allocate space for the unvarying part of the offset table. */
    font.offset_table = (BYTE*)myalloc( 12, sizeof(BYTE) );
    
    /* Read the first part of the offset table. */
    if( fread( font.offset_table, sizeof(BYTE), 12, font.file ) != 12 )
    	fatal(1,"TrueType font \"%s\" may be corrupt (reason 1)",filename);
    
    /* Determine how many directory entries there are. */
    font.numTables = getUSHORT( font.offset_table + 4 );
    
    /* Expand the memory block to hold the whole thing. */
    font.offset_table = (BYTE*)myrealloc( font.offset_table, (12 + font.numTables * 16), sizeof(BYTE) );
    
    /* Read the rest of the table directory. */
    if( fread( font.offset_table + 12, sizeof(BYTE), (font.numTables*16), font.file ) != (font.numTables*16) )
    	fatal(1,"TrueType font \"%s\" may be corrupt (reason 2)",filename);
    
    /* Load information from the "name" table. */
    psname=get_postscript_name(&font);

    /* We are done with the TrueType font file. */
    fclose(font.file);

    /* Free the memory occupied by tables. */
    myfree(font.offset_table);

    return psname;
    } /* end of do_ttfont() */

/*
** This is the error handler routine
** which we tell gdbm_open() to use.
**
** All it does is squirel away the error message string. 
*/
#ifdef TT_DBM_gdbm
#ifdef __cplusplus
extern "C" void derr(...);
void derr(...) { }
#else
static void derr(char *message) { }
#endif
#endif

/*
** Main
*/
int main(int argc, char *argv[])
    {
    int x;
    char *postname;
    int errcode;

    #ifdef TT_DBM_gdbm
    GDBM_FILE dbfile;
    datum key;
    datum data;

    if( (dbfile=gdbm_open(TTDBNAME,0,GDBM_WRCREAT,S_IRUSR|S_IWUSR,derr)) == (GDBM_FILE)NULL )
	fatal(1,"Error opening database, errno=%d (%s), gdbm_errno=%d.",errno,strerror(errno),gdbm_errno);
    #endif

    for(x=1; x < argc; x++)
    	{
	printf("%s",argv[x]);

	if( argv[x][0] != '/' )
	    fatal(1,"All file names must have absolute paths");

    	if( (postname=do_ttfont(argv[x])) == (char*)NULL )
    	    fatal(1,"TrueType font file \"%s\" has no PostScript name",argv[x]);

	printf(", \"%s\"", postname);

	#ifdef TT_DBM_gdbm
	key.dptr = postname;
	key.dsize = strlen(postname);

	data.dptr = (char*)argv[x];
	data.dsize = strlen(argv[x]);

  	errcode=gdbm_store(dbfile,key,data,GDBM_INSERT);
	if(errcode == 0)
	    printf(", added");
  	else if(errcode == 1)
            printf(", already exists");
        else
            fatal(1,"errno=%d (%s), gdbm_errno=%d",errno,strerror(errno),gdbm_errno);
	#endif
    
	myfree(postname);
	printf("\n");
    	}

    #ifdef TT_DBM_gdbm
    gdbm_close(dbfile);
    #else
    fputs("No database code, this program did nothing.\n", stderr);
    #endif

    return 0;    
    } /* end of main() */

/* end of file */
