/*
** ~ppr/src/dotmatrix/postscript.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file last modified 7 June 1996.
*/

#include <stdio.h>
#include "filter_dotmatrix.h"

/*
** These two variables are used for minimizing the amount
** of PostScript code generated by movements.  "ls" is the
** magnitude of the last vertical movement.  "lm" is the 
** position achieved by the last absolute horizontal positioning
** command generated.
*/
static int ls;
static int lm;

/*
** The PostScript names of the four fonts
** which will be at our disposal.  These 
** names are not fixed because sometimes it
** will be necessary to use the IBM version
** of Courier while at other times the Adobe
** version will do.  (The IBM version has more
** characters though the extra ones are not 
** encoded in the standard encoding.)
*/
static char *font_normal;
static char *font_bold;
static char *font_oblique;
static char *font_boldoblique;

/*
** The name of the PostScript encoding to use.  If 
** "StandardEncoding" is ok, this variable will
** be set to a NULL.
*/
static char *encoding_name;

/*
** This subroutine is called at the top of the document.
*/
void top_of_document(void)
    {
    /*
    ** For each of the four fonts, if code page 437 encoding was
    ** selected, choose the IBM version of Courier only if pass1
    ** revealed that non-ASCII characters are required from that
    ** font.
    */
    if(encoding==ENCODING_CP437 && uses_nonascii_normal)
    	font_normal="IBMCourier";
    else
    	font_normal="Courier";

    if(encoding==ENCODING_CP437 && uses_nonascii_bold)
    	font_bold="IBMCourier-Bold";
    else
    	font_bold="Courier-Bold";

    if(encoding==ENCODING_CP437 && uses_nonascii_oblique)
    	font_oblique="IBMCourier-Italic";
    else
    	font_oblique="Courier-Oblique";

    if(encoding==ENCODING_CP437 && uses_nonascii_boldoblique)
    	font_boldoblique="IBMCourier-BoldItalic";
    else
    	font_boldoblique="Courier-BoldOblique";

    /*
    ** Select the encoding we will use.  Don't use a special
    ** encoding unless it will actually be useful.
    */
    if(encoding!=ENCODING_STANDARD && (uses_nonascii_normal || uses_nonascii_bold
		|| uses_nonascii_oblique || uses_nonascii_boldoblique) )
	{
	if(encoding==ENCODING_ISOLATIN1)
	    encoding_name="ISOLatin1Encoding";
	else if(encoding==ENCODING_CP437)
	    encoding_name="CP437Encoding";
	}
    else
    	{
    	encoding_name=(char*)NULL;
    	}
	
    puts("%!PS-Adobe-3.0");
    puts("%%Creator: PPR dotmatrix printer emulator");
    puts("%%Pages: (atend)");
    puts("%%DocumentData: Clean7Bit");
    puts("%%DocumentNeededResources: procset "DOTMATRIX);

    /* If we need graphics routines, say so. */
    if(uses_graphics)
    	{
    	if(level2)
    	    fputs("%%+ procset "DOTMATRIXG2"\n",stdout);
    	else
    	    fputs("%%+ procset "DOTMATRIXG1"\n",stdout);
    	}

    /*
    ** If we will have to re-encode the font, mention the
    ** resources needed to do it.
    */
    if(encoding_name != (char*)NULL)
	{
	puts("%%+ procset "REENCODE);
	printf("%%%%+ encoding %s\n",encoding_name);
	}

    /* Emmit comments for those fonts we will actually use. */
    if(uses_normal)
	printf("%%%%+ font %s\n",font_normal);
    if(uses_bold)
	printf("%%%%+ font %s\n",font_bold);
    if(uses_oblique)
	printf("%%%%+ font %s\n",font_oblique);
    if(uses_boldoblique)
	printf("%%%%+ font %s\n",font_boldoblique);

    /* If we will use proportional spacing, mention those proceedure sets. */
    if(uses_proportional1 || uses_proportional2 || uses_proportional3 || uses_proportional4)
    	puts("%%+ procset "NEWMETRICS);
    if(uses_proportional1)
    	puts("%%+ procset "METRICSEPSON1);
    if(uses_proportional2)
    	puts("%%+ procset "METRICSEPSON2);
    if(uses_proportional3)
    	puts("%%+ procset "METRICSEPSON3);
    if(uses_proportional4)
    	puts("%%+ procset "METRICSEPSON4);

    /* If colour required, name that proceedure set. */
    if(uses_colour)
    	puts("%%+ procset "COLOUR);

    /* Name the document's requirements. */
    if(uses_colour)
    	fputs("%%Requirements: color\n",stdout);
    	
    /*
    ** If level 2 PostScript required, say so.  Level 2 features
    ** are only required if we have generated compressed graphics.
    */
    printf("%%%%LanguageLevel: %d\n",(level2 && uses_graphics) ? 2 : 1);

    puts("%%EndComments");
    puts("");

    /*
    ** In the prolog, we insert proceedure sets and encodings.
    ** Begin by downloading the proceedure set with dot matrix
    ** printer emulation proceedures.
    */
    puts("%%BeginProlog");
    puts("%%IncludeResource: procset "DOTMATRIX);

    /* If we will be printing graphics, emmit the routines here. */
    if(uses_graphics)
    	{
    	if(level2)
    	    fputs("%%IncludeResource: procset "DOTMATRIXG2"\n",stdout);
    	else
    	    fputs("%%IncludeResource: procset "DOTMATRIXG1"\n",stdout);
    	}

    /*
    ** If a character set other than Standard was selected and non-ASCII
    ** characters are used in at least one of the four fonts, then
    ** download the required encoding and a proceedure to re-encode
    ** a font.
    */
    if(encoding_name != (char*)NULL )
	{
	puts("%%IncludeResource: procset "REENCODE);
	printf("%%%%IncludeResource: encoding %s\n",encoding_name);
	}
	
    /*
    ** If we are using proportional spacing, download 
    ** the proportional font metrics procedure sets.
    */
    if(uses_proportional1 || uses_proportional2 || uses_proportional3 || uses_proportional4)
    	puts("%%IncludeResource: procset "NEWMETRICS);
    if(uses_proportional1)
    	puts("%%IncludeResource: procset "METRICSEPSON1);
    if(uses_proportional2)
    	puts("%%IncludeResource: procset "METRICSEPSON2);
    if(uses_proportional3)
    	puts("%%IncludeResource: procset "METRICSEPSON3);
    if(uses_proportional4)
    	puts("%%IncludeResource: procset "METRICSEPSON4);

    /* If colour needed, download that proceedure set. */
    if(uses_colour)
    	puts("%%IncludeResource: procset "COLOUR);

    fputs("%%EndProlog\n\n",stdout);

    /*
    ** In the setup section, we can execute things.
    ** We begin by putting our dictionary on the top
    ** of the stack.
    */
    puts("%%BeginSetup");
    puts("pprdotmatrix begin");
    puts("pprdotmatrix_init");

    /*
    ** If we will use Pinwriter 6 units, change some of
    ** the scaling variables.
    */
    if(emulation & EMULATION_24PIN_UNITS)
	{
	puts("/xfactor 360 72 div def");	/* convert horizontal units to PostScript units */
    	puts("/yfactor 360 72 div def");	/* same for vertical units */
	puts("/bunit 60 def");			/* printer's basic unit */
    	}

    /*
    ** If we are emulating a narrow carriage printer,
    ** we must arrange to have things shifted 0.25
    ** inch to the right.
    */
    if( emulation & EMULATION_8IN_LINE )
	puts("/xshift 0.25 inch def");

    /*
    ** If an x or y shift has been specified, emmit it.
    */
    if( xshift )
	printf("/xshift xshift %d add def\n", xshift);
    if( yshift )
	printf("/yshift yshift %d add def\n", yshift);

    /* Download all the fonts we need. */
    if(uses_normal)
	printf("%%%%IncludeResource: font %s\n",font_normal);
    if(uses_bold)
	printf("%%%%IncludeResource: font %s\n",font_bold);
    if(uses_oblique)
	printf("%%%%IncludeResource: font %s\n",font_oblique);
    if(uses_boldoblique)
	printf("%%%%IncludeResource: font %s\n",font_boldoblique);

    /* For each font, if we need non-ASCII characters, re-encode it. */
    if(encoding_name != (char*)NULL)
	{
	if(uses_nonascii_normal)
	    printf("/%s /Courier %s ReEncode\n",font_normal,encoding_name);

	if(uses_nonascii_bold)
	    printf("/%s /Courier-Bold %s ReEncode\n",font_bold,encoding_name);

	if(uses_nonascii_oblique)
	    printf("/%s /Courier-Oblique %s ReEncode\n",font_oblique,encoding_name);

	if(uses_nonascii_boldoblique)
	    printf("/%s /Courier-BoldOblique %s ReEncode\n",font_boldoblique,encoding_name);
	}

    /* Find and scale the fonts we will use. */
    if(uses_normal)
	fputs("/f /Courier findfont 12 scalefont def\n",stdout);
    if(uses_bold)
	fputs("/fb /Courier-Bold findfont 12 scalefont def\n",stdout);
    if(uses_oblique)
	fputs("/fo /Courier-Oblique findfont 12 scalefont def\n",stdout);
    if(uses_boldoblique)
	fputs("/fbo /Courier-BoldOblique findfont 12 scalefont def\n",stdout);

    /* find and scale the proportional fonts we will use. */
    if(uses_proportional1)
	{
	fputs("/Courier /PSCourier MetricsEpson_Courier NewMetrics\n",stdout);
	fputs("/pf /PSCourier findfont 12 scalefont def\n",stdout);
	}    	
    if(uses_proportional2)
	{
	fputs("/Courier-Bold /PSCourier-Bold MetricsEpson_Courier-Bold NewMetrics\n",stdout);
	fputs("/pfb /PSCourier-Bold findfont 12 scalefont def\n",stdout);
	}    	
    if(uses_proportional3)
	{
	fputs("/Courier-Oblique /PSCourier-Oblique MetricsEpson_Courier-Oblique NewMetrics\n",stdout);
	fputs("/pfo /PSCourier-Oblique findfont 12 scalefont def\n",stdout);
	}    	
    if(uses_proportional4)
	{
	fputs("/Courier-BoldOblique /PSCourier-BoldOblique MetricsEpson_Courier-BoldOblique NewMetrics\n",stdout);
	fputs("/pfbo /PSCourier-BoldOblique findfont 12 scalefont def\n",stdout);
	}    	

    /*
    ** If there are any normal characters, it is ok to select
    ** normal characters as the default.
    */
    if(uses_normal)
	fputs("f 1 sf\n",stdout);

    fputs("%%EndSetup\n\n",stdout);
    } /* end of top_of_document() */

/*
** This subroutine is called at the bottom of the document.
*/
void bottom_of_document(void)
    {
    puts("%%Trailer");
    puts("end % pprdotmatrix");
    printf("%%%%Pages: %d\n",current_page);
    puts("%%EOF");
    } /* end of bottom_of_document() */

/*
** This subroutine is called at the top of each page.
*/
void top_of_page(void)
    {
    /* increment page number */
    current_page++;

    /* current point has been lost */
    postscript_xpos=postscript_ypos=-1;
    
    /* Colour setting has been lost. */
    postscript_print_colour=COLOUR_BLACK;

    /* emmit appropriate postscript code to start a page */
    printf("%%%%Page: %d %d\n",current_page,current_page);
    puts("%%BeginPageSetup");
    puts("bp");
    puts("%%EndPageSetup");

    /* Rest the line output code */
    buffer_top_of_page_reset();
    
    /* Reset the line spacing compression variables */
    lm = ls = 0;
    } /* end of top_of_page() */

/*
** This subroutine is called at the bottom of each page.
*/
void bottom_of_page(void)
    {
    puts("%%PageTrailer");
    puts("ep");
    puts("");
    } /* end of bottom_of_page() */

/*
** Issue PostScript code to aquire the position named in
** xpos and ypos.
*/
void achieve_position(void)
    {
    #ifdef DEBUG_COMMENTS
    if( xpos != postscript_xpos || ypos != postscript_ypos )
	printf("%% xpos=%d, ypos=%d, lm=%d, ls=%d, postscript_xpos=%.1f, postscript_ypos=%d\n",
		xpos, ypos, lm, ls, postscript_xpos, postscript_ypos);
    #endif

    /* If both x and y must change, */
    if( xpos != postscript_xpos && ypos != postscript_ypos )
    	{
	if(xpos==lm && ypos==(postscript_ypos-ls))
	    {
	    fputs("n\n",stdout);
	    }
	else
	    {
    	    printf("%d %d mxy\n", xpos, ypos);
    	    lm = xpos;
    	    ls = postscript_ypos - ypos;
    	    }
    	}

    /* If only y must change, */
    else if( ypos != postscript_ypos )
	{
	printf("%d my\n", ypos);
	ls = postscript_ypos - ypos;
	}

    /* If only x must change, */
    else if( xpos != postscript_xpos )
    	{
	int movement = xpos - postscript_xpos;
	
	if(movement > 0 && movement < HORIZONTAL_UNITS)
	    printf("%d m\n", movement);
	else
    	    printf("%d mx\n", xpos);
    	}

    /* We believe at one of the above did the job. */
    postscript_xpos = xpos;
    postscript_ypos = ypos;
    } /* end of achieve_position() */
    
/* end of file */
