#!/usr/bin/perl
#
# ~ppr/src/commentators/audio.perl
# Copyright 1995, 1996, 1997, Trinity College Computing Center.
# Written by David Chappell.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies and that both that
# copyright notice and this permission notice appear in supporting
# documentation.  This software is provided "as is" without express or
# implied warranty.
#
# Last modified 31 January 1997.
#

#
# This is intended as an example commentator.  It is not however the
# simplest possible commentator.  A commentator is invoked like this:
#
# commentators/audio <address> <options> <printer> <code> <cooked_message> <raw1> <raw2>
#
# The <address> is a parameter we use to tell where to send the message
# for this commentator addresses may be in the form "dnsname:portnum", such 
# as "chappell.pc.trincoll.edu:15009", in which case the file is copied into
# a Samba share area and a TCP/IP connexion is opened to a MS-Windows
# computer running pprpopup and it is instructed to play the sound in
# the share area.  If the address is in the form "/dev/somedev", then 
# it is sent to the indicated local device.
#
# To extract the message strings from this file use:
# sed -n -e "s/^.*\'(.*\)'.*$/\1/p" audio.perl
#

# These will be filled in when this script is installed:
$HOMEDIR = "?";
$VAR_SPOOL_PPR = "?";

unshift(@INC, "$HOMEDIR/lib");
require 'pprpopup.pl';
require 'play_local.pl';
require 'speach.pl';

# If this is non-zero, debugging messages will be printed on stdout.  Any
# message written on stdout or stderr will find their way into the pprdrv
# log file.
$DEBUG = 0;

# All messages are catagorized by importance, with 1 being the most important
# and 10 being the least important.  Message with an importance less than or 
# equal to the desired detail level indicated by this variable are spoken, 
# others are discarded.
my $DETAIL_LEVEL = 10;

# Should we play silly sounds?
my $SILLY_SOUNDS = 0;

#
# The type of commentary (the possible values of our $CODE argument. 
# The official definitions of these values are in ~ppr/src/include/pprdrv.h.
#
$COM_PRINTER_ERRORS = 1;		# %%[ PrinterError: xxxxx ]%%
$COM_PRINTER_STATUS = 2;		# %%[ status: xxxxx ] %%%
$COM_IMPATIENCE = 4;			# printer bogging down
$COM_EXIT = 8;				# why did it exit?
$COM_MUNDANE_EXIT = 16;			# printed and incapable
$COM_PRINTER_STATUS_UPDATES = 32;	# status messages during printing

#===================================================================
# Various classes of silly sounds:
#===================================================================

# Job printed:
$SILLY_PRINTED = [10, "*trumpet1", 10, "*trumpet2", 2, "*flyby"];

# Printer faults:
$SILLY_NO_RETRY = [100, "*siren"];
$SILLY_FAULT = [10, "*ominousgong", 10, "*mystery", 20, "*sonar", 10, "*clank"];

# Printing interupted:
$SILLY_CANCELED = [5, "*biglaser", 5, "*blatblatblat",
	5, "*eraser", 5, "*machinegun", 25, "*raygun", 5, "*riffleshots"];

# No progress:
$SILLY_STALLED = [15, "*mystery", 40, "*ominousgong"];
$SILLY_WASNTSTALLED = [25, "*flyby"];
$SILLY_NOLONGERSTALLED = [25, "*carstart"];
$SILLY_NOLONGERSTALLED2 = [25, "*tinklemusic"];

# Off line or otherwise engaged:
$SILLY_BLOCKED = [15, "*bwabwa"];

$SILLY_ATTENTION = [25, "*policewhistle.au", 25, "*alarm", 25, "*foghorn",
	25, "*computer"];
$SILLY_JAM = [100, "*crash"];
$SILLY_JOBERROR = [25, "*glass", 25, "*bounce"];

$SILLY_INCAPABLE = [100, "*bounce"];

#==============================================================
# The events and their sound files, grouped by $CODE value.
#
# Each array has 4 elements:
# 0	play if the selected detail level is this or higher
# 1	silly sound to play at start
# 2	phrase for body of message
# 3	silly sound to play at end
#==============================================================

#
# $CODE == $COM_PRINTER_ERRORS
# (Printer sent "%%[ PrinterError: xxxxx ]%%".)
#
# Also $COM_PRINTER_STATUS and $COM_PRINTER_STATUS_UPDATES
# for those not defined in next section.
#
$events_errors{"cover open"}		= [ 5, $SILLY_ATTENTION, 'has its cover open' ];
$events_errors{"resetting printer"}	= [ 5, undef, 'is resetting' ];
$events_errors{"warming up"}		= [ 5, undef, 'is warming up' ];
$events_errors{"no paper tray"}		= [ 5, $SILLY_ATTENTION, 'needs its paper tray back' ];
$events_errors{"paper jam"}		= [ 1, $SILLY_JAM, 'has a paper jam' ];
$events_errors{"out of paper"}		= [ 1, $SILLY_ATTENTION, 'is out of paper' ];
$events_errors{"manual feed"}		= [ 2, $SILLY_ATTENTION, 'requires manual feeding' ];
$events_errors{"miscellaneous error"}	= [ 1, undef, 'reports an error which it describes as miscellaneous' ];
$events_errors{"fatal error"}		= [ 1, undef, 'has suffered a fatal error' ];
$events_errors{"silly message"}		= [ 6, undef, 'has sent a silly message' ];
$events_errors{"toner is low"}		= [ 5, undef, 'needs more toner' ];

#
# $CODE == $COM_PRINTER_STATUS and $CODE == $COM_PRINTER_STATUS_UPDATES
# (Printer sent "%%[ status: xxxxx ] %%".)
#
$events_status{"busy"}			= [ 6, undef, 'is otherwise engaged' ];
$events_status{"idle"}			= [ 6, undef, 'is idle' ];
$events_status{"off line"}		= [ 1, undef, 'is off line' ];

#
# $CODE == $COM_PRINTER_STATUS_UPDATES
#
$events_status_updates{"busy"}		= [ 6, undef, 'is busy printing' ];

# $CODE == $COM_IMPATIENCE
# If the printer seems too slow we will receive
# one of the messages below.
$events_impatience{"may be stalled"}		= [ 6, $SILLY_STALLED, 'may be stalled' ];
$events_impatience{"probably stalled"}		= [ 3, $SILLY_STALLED, 'is probably stalled' ];
$events_impatience{"wasn't stalled"}		= [ 6, $SILLY_WASNTSTALLED, 'is not stalled' ];
$events_impatience{"no longer stalled"}		= [ 2, $SILLY_NOLONGERSTALLED, 'is no longer stalled', $SILLY_NOLONGERSTALLED2 ];

# $CODE == $COM_EXIT or $CODE == $COM_MUNDANE_EXIT
# Describe the circumstances under which pprdrv exited.
$events_exit{"has printed a job"}			= [ 9, $SILLY_PRINTED, 'has printed a document' ];
$events_exit{"printer error"}				= [ 1, $SILLY_FAULT, 'has failed to print due to a communications error' ];
$events_exit{"printer error, no retry"}			= [ 1, $SILLY_NO_RETRY, 'has something badly wrong with it' ];
$events_exit{"interface core dump"}			= [ 1, undef, 'has a defective interface program' ];
$events_exit{"job error"}				= [ 5, $SILLY_JOBERR, 'has detected an error in the current job' ];
$events_exit{"postscript error"}			= [ 8, $SILLY_JOBERR, 'has detected a postscript error in the current job' ];
$events_exit{"interface program killed"}		= [ 1, undef, 'has stopt printing because somebody killed the interface program' ];
$events_exit{"printing halted"}				= [ 5, $SILLY_CANCELED, 'has stopt printing the current job as ordered' ];
$events_exit{"otherwise engaged or off-line"}		= [ 8, undef, 'is otherwise engaged or off line', $SILLY_BLOCKED ];
$events_exit{"starved for system resources"}		= [ 1, undef, 'is not printing due to an insufficiency of system resources', $SILLY_BLOCKED ];
$events_exit{"incapable of printing this job"}		= [10, $SILLY_INCAPABLE, 'is not capable of printing the current job' ];

#==============================================================
# Main
#==============================================================

# Assign names to the command line parameters.
my($ADDRESS, $OPTIONS, $PRINTER, $CODE, $COOKED, $RAW1, $RAW2) = @ARGV;

# Parse the options parameter which is a series of name=value pairs.
my $option;
foreach $option (split(/[\s]+/, $OPTIONS))
  {
  my($name, $value) = split(/=/, $option);
  if( $name eq "level" )
    { $DETAIL_LEVEL=$value; }
  elsif( $name eq "voice" )
    { speach_voice($value); }
  elsif( $name eq "silly_sounds" )
    {
    if($value =~ /^[1ty]/i)
	{ $SILLY_SOUNDS = 1 }
    else
	{ $SILLY_SOUNDS = 0 }
    }
  else
    { die "Unrecognized option: $name\n"; }
  }

# Debugging code:
if($DEBUG)
  {
  $| = 1;
  print "\$ADDRESS=\"$ADDRESS\", \$OPTIONS=\"$OPTIONS\", \$PRINTER=$PRINTER, \$CODE=$CODE,\n";
  print "\$COOKED=\"$COOKED\", \$RAW1=\"$RAW1\", \$RAW2=\"$RAW2\"\n";
  print "\$DETAIL_LEVEL=$DETAIL_LEVEL\n";
  }

# Start with an almost empty playlist:
my @playlist = ('the printer');

# And an empty introductory sound
my $intro_sound = undef;
my $closing_sound = undef;

#
# Convert the printer name to a sound file name.  If there
# is no sound file for this printer, build it out of the 
# sound files for the characters of its name.
#
if( -r "$SOUNDS_DIR/$PRINTER.au" )
    {
    push(@playlist, "$PRINTER");
    }
else
    {
    spellout(\@playlist, $PRINTER);
    }

#
# If the event was a printer error, append the sound file
# for that error.
#
# Notice that if there is a cooked version of the message
# we use that, otherwise we use the raw version.  We will probably 
# only get a match on the raw one when there is no cooked one if 
# we have in the table a raw message which is not translated in
# lw_errors.conf.
#
if($CODE == $COM_PRINTER_ERRORS)
    {
    my $message = $COOKED ne "" ? $COOKED : $RAW1;

    if( defined( $EVENT_SOUND = $events_errors{$message} ) )
	{
	$this_level = $EVENT_SOUND->[0];
	push(@playlist, $EVENT_SOUND->[2]);
	$intro_sound = $EVENT_SOUND->[1];
	$closing_sound = $EVENT_SOUND->[3];
	}
    else
	{
	$this_level = 1;
	push(@playlist, 'has reported an unrecognized error');
	}
    }

#
# If it is a status message, look it up in the list of status messages,
# if it is not found there, look it up in the list of printer errors.
# (Many printer status messages are really errors such as "cover open".)
#
# Again notice that we prefer the cooked version but will use the raw
# message.
#
elsif($CODE == $COM_PRINTER_STATUS)
    {
    my $message = $COOKED ne "" ? $COOKED : $RAW1;

    # Status can sometimes be an error message:
    if( defined($EVENT_SOUND = $events_status{$message}) || defined($EVENT_SOUND = $events_errors{$message}) )
	{
	$this_level = $EVENT_SOUND->[0];
	push(@playlist, $EVENT_SOUND->[2]);
	$intro_sound = $EVENT_SOUND->[1];
	$closing_sound = $EVENT_SOUND->[3];
	}
    else
	{
	$this_level = 1;
	push(@playlist, 'has reported an unrecognized status');
	}
    }

elsif($CODE == $COM_PRINTER_STATUS_UPDATES)
    {
    my $message = $COOKED ne "" ? $COOKED : $RAW1;

    unshift(@playlist, 'status update', '(pause)');

    # Status can sometimes be an error message:
    if( defined($EVENT_SOUND = $events_status_updates{$message})  || defined($EVENT_SOUND = $events_status{$message}) || defined($EVENT_SOUND = $events_errors{$message}) )
	{
	$this_level = $EVENT_SOUND->[0];
	push(@playlist, $EVENT_SOUND->[2]);
	$intro_sound = $EVENT_SOUND->[1];
	$closing_sound = $EVENT_SOUND->[3];
	}
    else
	{
	$this_level = 1;
	push(@playlist, 'has reported an unrecognized status');
	}
    }

#
# If the problem is that the printer is stalled, build
# a message.  Some of the messages are ready-recorded,
# others we must build from components.
#
elsif($CODE == $COM_IMPATIENCE)
    {
    if( defined( $EVENT_SOUND = $events_impatience{$COOKED} ) )
	{
	$this_level = $EVENT_SOUND->[0];
	push(@playlist, $EVENT_SOUND->[2]);
	$intro_sound = $EVENT_SOUND->[1];
	$closing_sound = $EVENT_SOUND->[3];
	}
    elsif( $COOKED =~ /^stalled for ([0-9]+) minutes$/ )
	{
	$this_level = 2;
	push(@playlist, 'has been stalled for');
	speak_time_interval(\@playlist, ($1 * 60));
	$intro_sound = $SILLY_STALLED;
	}
    else
	{
	$this_level = 1;
	push(@playlist, 'is the subject of an unrecognized stall message');
	}
    }

#
# If it is a pprdrv exit message,
#
elsif($CODE == $COM_EXIT || $CODE == $COM_MUNDANE_EXIT)
    {
    if( defined( $EVENT_SOUND = $events_exit{$COOKED} ) )
	{
	$this_level = $EVENT_SOUND->[0];
	push(@playlist, $EVENT_SOUND->[2]);
	$intro_sound = $EVENT_SOUND->[1];
	$closing_sound = $EVENT_SOUND->[3];
	}
    else
	{
	$this_level = 1;
	push(@playlist, 'is the subject of an invalid exit message');
	}
    }

#
# If none of the above,
#
else
    {
    $this_level = 1;
    push(@playlist, 'is named in an invalid event report');
    }

# If an introductory sound was chosen, push it onto the front
# of the play list.
if($SILLY_SOUNDS)
    {
    if(defined($intro_sound))
	{ unshift(@playlist, $intro_sound); }
    if(defined($closing_sound))
	{ push(@playlist, $closing_sound); }
    }

# Print the message for debugging purposes:
if($DEBUG) { print join(' ', @playlist), "\n"; }

#
# Decide if the requested level of detail requires us to play
# this announcement.  If it does, play the selected sound files.
#
if( $this_level <= $DETAIL_LEVEL )
  { play_many_au($ADDRESS, @playlist); }
else
  { if($DEBUG) { print "Not important enough to play.\n"; } }

# We are done.
if($DEBUG) { print "Done\n\n"; }
exit(0);
