/* Driver for NEC P6 mono and color
 */

#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include "pnmread.h"

extern int optind;
extern char *optarg;

/* This mask defines the order how the pixels will be copied into the output
   buffer */
static char bit_value[8] = {
  128,
  64,
  32,
  16,
  8,
  4,
  2,
  1
};

typedef struct {
  int x_dpi;
  int y_dpi;
  int pins;
  int dbl;
  int real360;
  int bytes;
  int maxwidth;
  int formfeed;
  int linefeed;
  int htabs;
  int interlace;
  int printeven;
  int printodd;
  int reset;
  int y_margin;
  int x_margin;
  int init;
  char odd_blue[3];
  char even_blue[3];
  char odd_yellow[3];
  char even_yellow[3];
  char odd_red[3];
  char even_red[3];
  char odd_black[3];
  char even_black[3];
} NecP6_Opts;

#ifdef __STDC__
static void necp6_decode_args(int argc, char *argv[], NecP6_Opts *necp6_opts)
#else
static void necp6_decode_args(argc, argv, necp6_opts)
int argc;
char *argv[];
NecP6_Opts *necp6_opts;
#endif

{
  int opt;
  int tmp;
  int xtmp[3];

  /* print out options */
  if(!argc) {
    fprintf(stderr, "    -o 360x360dpi [default] (Max. width: 4896)\n");
    fprintf(stderr, "    -o 180x180dpi           (Max. width: 2446)\n");
    fprintf(stderr, "    -o real360x360dpi       (Max. width: 4896)\n");
    fprintf(stderr, "    -o formfeed -o linefeed -o noreset -o noinit\n");
    fprintf(stderr, "    -o x_margin=[-]pixel  -o y_margin=[-]pixel\n");
    fprintf(stderr, "    -o htabs=int (1/180'')\n");
    fprintf(stderr, "    [-o interlace_360 | -o interlace_180]\n");
    fprintf(stderr, "    [-o print_odd | -o print_even]\n");
    fprintf(stderr, "    -o odd=hex  (default: 333)\n");
    fprintf(stderr, "    -o even=hex (default: ccc)\n");

    if(strcmp(argv[0], "necp6_mono")) {
      fprintf(stderr, "    -o odd_cyan=(hex)    -o even_cyan=(hex)\n");
      fprintf(stderr, "    -o odd_magenta=(hex) -o even_magenta=(hex)\n");
      fprintf(stderr, "    -o odd_yellow=(hex)  -o odd_yellow=(hex)\n");
      fprintf(stderr, "    -o even_black=(hex)  -o even_black=(hex)\n");
    }

    return;
  }

  /* default values for options */
  necp6_opts->x_dpi          = 360;
  necp6_opts->y_dpi          = 360;
  necp6_opts->x_margin       = 0;
  necp6_opts->y_margin       = 0;
  necp6_opts->pins           = 24;
  necp6_opts->dbl            = 1;
  necp6_opts->real360        = 0;
  necp6_opts->bytes          = 3;
  necp6_opts->maxwidth       = 4896;
  necp6_opts->formfeed       = 0;
  necp6_opts->linefeed       = 0;
  necp6_opts->interlace      = 0;
  necp6_opts->htabs          = 0;
  necp6_opts->reset          = 1;
  necp6_opts->init           = 1;
  necp6_opts->printodd       = 1;
  necp6_opts->printeven      = 1;
  necp6_opts->odd_blue[0]    = 0x3;
  necp6_opts->even_blue[0]   = 0xC;
  necp6_opts->odd_red[0]     = 0x3;
  necp6_opts->even_red[0]    = 0xC;
  necp6_opts->odd_yellow[0]  = 0x3;
  necp6_opts->even_yellow[0] = 0xC;
  necp6_opts->odd_black[0]   = 0x3;
  necp6_opts->even_black[0]  = 0xC;
  necp6_opts->odd_blue[1]    = 0x3;
  necp6_opts->even_blue[1]   = 0xC;
  necp6_opts->odd_red[1]     = 0x3;
  necp6_opts->even_red[1]    = 0xC;
  necp6_opts->odd_yellow[1]  = 0x3;
  necp6_opts->even_yellow[1] = 0xC;
  necp6_opts->odd_black[1]   = 0x3;
  necp6_opts->even_black[1]  = 0xC;
  necp6_opts->odd_blue[2]    = 0x3;
  necp6_opts->even_blue[2]   = 0xC;
  necp6_opts->odd_red[2]     = 0x3;
  necp6_opts->even_red[2]    = 0xC;
  necp6_opts->odd_yellow[2]  = 0x3;
  necp6_opts->even_yellow[2] = 0xC;
  necp6_opts->odd_black[2]   = 0x3;
  necp6_opts->even_black[2]  = 0xC;

  optind = 0;

  /* reading arguments */
  while((opt = getopt(argc, argv, "d:o:")) != -1)
    switch(opt) {
    case 'o':
      if(!strcmp(optarg, "360x360dpi")) {
	necp6_opts->x_dpi    = 360;
	necp6_opts->y_dpi    = 360;
	necp6_opts->y_dpi    = 24;
	necp6_opts->dbl      = 1;
	necp6_opts->maxwidth = 4896;
      } else

      if(!strcmp(optarg, "180x180dpi")) {
	necp6_opts->x_dpi    = 180;
	necp6_opts->y_dpi    = 180;
	necp6_opts->pins     = 24;
	necp6_opts->dbl      = 0;
	necp6_opts->maxwidth = 2448;
      } else

      if(!strcmp(optarg, "real360x360dpi")) {
	necp6_opts->x_dpi    = 360;
	necp6_opts->y_dpi    = 360;
	necp6_opts->y_dpi    = 24;
	necp6_opts->dbl      = 1;
	necp6_opts->maxwidth = 4896;
	necp6_opts->real360  = 1;
      } else

      if(!strcmp(optarg, "formfeed")) {
	necp6_opts->formfeed = 1;
      } else

      if(!strcmp(optarg, "linefeed")) {
	necp6_opts->linefeed = 1;
      } else

      if(!strcmp(optarg, "noreset")) {
	necp6_opts->reset = 0;
      } else

      if(!strcmp(optarg, "noinit")) {
	necp6_opts->init = 0;
      } else

      if(!strcmp(optarg, "htabs")) {
	necp6_opts->htabs = 10;
      } else

      if(!strncmp(optarg, "htabs=", 6)) {
	sscanf(optarg + 6, "%d", &necp6_opts->htabs);
      } else

      if(!strncmp(optarg, "x_margin=", 9)) {
	sscanf(optarg + 9, "%d", &necp6_opts->x_margin);
      } else

      if(!strncmp(optarg, "y_margin=", 9)) {
	sscanf(optarg + 9, "%d", &necp6_opts->y_margin);
      } else

      if(!strcmp(optarg, "interlace_360")) {
	necp6_opts->interlace = 1;
      } else

      if(!strcmp(optarg, "interlace_180")) {
	necp6_opts->interlace = 2;
      } else

      if(!strcmp(optarg, "print_even")) {
	necp6_opts->printeven = 1;
	necp6_opts->printodd  = 0;
      } else

      if(!strcmp(optarg, "print_odd")) {
	necp6_opts->printodd  = 1;
	necp6_opts->printeven = 0;
      } else

      if(!strncmp(optarg, "odd=", 4)) {
	sscanf(optarg + 4, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->odd_blue[0]   = xtmp[0];
	necp6_opts->odd_red[0]    = xtmp[0];
	necp6_opts->odd_yellow[0] = xtmp[0];
	necp6_opts->odd_black[0]  = xtmp[0];
	necp6_opts->odd_blue[1]   = xtmp[1];
	necp6_opts->odd_red[1]    = xtmp[1];
	necp6_opts->odd_yellow[1] = xtmp[1];
	necp6_opts->odd_black[1]  = xtmp[1];
	necp6_opts->odd_blue[2]   = xtmp[2];
	necp6_opts->odd_red[2]    = xtmp[2];
	necp6_opts->odd_yellow[2] = xtmp[2];
	necp6_opts->odd_black[2]  = xtmp[2];
      } else

      if(!strncmp(optarg, "even=", 5)) {
	sscanf(optarg + 5, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->even_blue[0]   = xtmp[0];
	necp6_opts->even_red[0]    = xtmp[0];
	necp6_opts->even_yellow[0] = xtmp[0];
	necp6_opts->even_black[0]  = xtmp[0];
	necp6_opts->even_blue[1]   = xtmp[1];
	necp6_opts->even_red[1]    = xtmp[1];
	necp6_opts->even_yellow[1] = xtmp[1];
	necp6_opts->even_black[1]  = xtmp[1];
	necp6_opts->even_blue[2]   = xtmp[2];
	necp6_opts->even_red[2]    = xtmp[2];
	necp6_opts->even_yellow[2] = xtmp[2];
	necp6_opts->even_black[2]  = xtmp[2];
      } else

      if(!strncmp(optarg, "odd_cyan=", 9)) {
	sscanf(optarg + 9, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->odd_blue[0] = xtmp[0];
	necp6_opts->odd_blue[1] = xtmp[1];
	necp6_opts->odd_blue[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "even_cyan=", 10)) {
	sscanf(optarg + 10, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->even_blue[0] = xtmp[0];
	necp6_opts->even_blue[1] = xtmp[1];
	necp6_opts->even_blue[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "odd_magenta=", 12)) {
	sscanf(optarg + 12, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->odd_red[0] = xtmp[0];
	necp6_opts->odd_red[1] = xtmp[1];
	necp6_opts->odd_red[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "even_magenta=", 13)) {
	sscanf(optarg + 13, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->even_red[0] = xtmp[0];
	necp6_opts->even_red[1] = xtmp[1];
	necp6_opts->even_red[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "odd_yellow=", 11)) {
	sscanf(optarg + 11, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->odd_yellow[0] = xtmp[0];
	necp6_opts->odd_yellow[1] = xtmp[1];
	necp6_opts->odd_yellow[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "even_yellow=", 12)) {
	sscanf(optarg + 12, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->even_yellow[0] = xtmp[0];
	necp6_opts->even_yellow[1] = xtmp[1];
	necp6_opts->even_yellow[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "odd_black=", 10)) {
	sscanf(optarg + 10, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->odd_black[0] = xtmp[0];
	necp6_opts->odd_black[1] = xtmp[1];
	necp6_opts->odd_black[2] = xtmp[2];
      } else

      if(!strncmp(optarg, "even_black=", 11)) {
	sscanf(optarg + 11, "%1x%1x%1x", xtmp+0, xtmp+1, xtmp+2);
	necp6_opts->even_black[0] = xtmp[0];
	necp6_opts->even_black[1] = xtmp[1];
	necp6_opts->even_black[2] = xtmp[2];

      } else {
	fprintf(stderr, "%s: Warning: unknown option %s\n", argv[0], optarg);
      }

      break;
    }
}

/* printer specific defines */
#define YELLOW 4
#define RED    1
#define BLUE   2
#define BLACK  0

#define NECP6_LINEFEED_FULL(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(24, (fd))

#define NECP6_LINEFEED_HIGH(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(23, (fd));\
NECP6_LF((fd))

#define NECP6_LINEFEED_180(fd,i)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc((i), (fd))

#define NECP6_LINEFEED_HIGH_24(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(12, (fd))

#define NECP6_LINEFEED_HIGH_23(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(11, (fd));\
NECP6_LF((fd))

#define NECP6_LINEFEED_HIGH_22(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(11, (fd))

#define NECP6_LINEFEED_HIGH_25(fd)\
fputc('\033', (fd));\
fputc('J', (fd));\
fputc(12, (fd));\
NECP6_LF((fd))

#define NECP6_INIT_1(fd)\
fputc(033,(fd));\
fputc('@',(fd))

#define NECP6_INIT_2(fd)\
fputc(033,(fd));\
fputc('P',(fd));\
fputc(033,(fd));\
fputc('l',(fd));\
fputc(000,(fd));\
fputc('\r',(fd));\
fputc(034,(fd));\
fputc(063,(fd));\
fputc(001,(fd))

#define NECP6_FORMFEED(fd)\
fputc(12,(fd))

#define NECP6_EXIT(fd)\
fputc(033,(fd));\
fputc('@',(fd))

#define NECP6_COLOR(fd, color)\
fputc(033,(fd));\
fputc('r',(fd));\
fputc(color,(fd))

#define NECP6_HTAB(fd, cnt)\
fputc(033, (fd));\
fputc('\\', (fd));\
fputc((cnt) & 0xff, (fd));\
fputc((cnt) >> 8, (fd))

#define NECP6_PRINT_360_PREFIX(fd, cnt)\
fputc(034, (fd));\
fputc('Z', (fd));\
fputc((cnt) & 0xff, (fd));\
fputc((cnt) >> 8, (fd))

#define NECP6_PRINT_180_PREFIX(fd, cnt)\
fputc(033, (fd));\
fputc('*', (fd));\
fputc(39, (fd));\
fputc((cnt) & 0xff, (fd));\
fputc((cnt) >> 8, (fd))

#define NECP6_CR(fd)\
fputc(13, (fd))

#define NECP6_LF(fd)\
fputc(10, (fd))

/* an temporary array */
static char *necp6_null = NULL;

/* checks wether printer buffer is empty */
#ifdef __STDC__
static int not_empty(char *b, int len)
#else
static int not_empty(b, len)
char *b;
int len;
#endif
{
  while(len--) {
    if(b[len])
      return(len + 1);
  }

  return(0);
}

/* prints one buffer in 360 dpi with */
#ifdef __STDC__
static void necp6_print_stripe(FILE *fd, char *buffer, int len,
			       NecP6_Opts *opts)
#else
static void necp6_print_stripe(fd, buffer, len, opts)
FILE *fd;
char *buffer;
int len;
NecP6_Opts *opts;
#endif
{
  int cnt = len / opts->bytes;
  int minleer;
  int minleer1;
  int i = 0;
  int leer = 0;
  int leerbegin;
  int len1;
  int skip = (opts->x_dpi / 180) * opts->bytes;

  /* If htabs then optimizing */
  if(opts->htabs == 0) {

    NECP6_PRINT_360_PREFIX(fd, cnt);
    fwrite(buffer, 1, len, fd);

  } else {

    /* minimal amount of blank space which will be skipped with tabs */
    minleer = opts->htabs * skip;

    while(i < len) {
      leer = i;

      do {
	/* searching next empty area */
	for(leerbegin = leer; buffer[leerbegin] && leerbegin < len;
	    leerbegin++);

	minleer1 = leerbegin + minleer;

	/* checking the length of the empty area */
	for(leer = leerbegin; !buffer[leer] && leer < len; leer++);

      /* until the empty area is large enough */
      } while(leer < minleer1 && leer < len);

      /* end of the buffer ? */
      if(leer == len) {
	leerbegin = len;
      }

      /* is there a printable area before the empty area -> print it */
      if(i < leerbegin) {
	len1 = leerbegin - i;

	len1 = len1 % opts->bytes
	  ? len1 + opts->bytes - (len1 % opts->bytes) : len1;

	leerbegin = i + len1;

	cnt = len1 / opts->bytes;

	NECP6_PRINT_360_PREFIX(fd, cnt);
	fwrite(buffer + i, 1, len1, fd);
      }

      /* skip the emtpy area with tab */
      len1 = (leer - leerbegin) / skip;
      NECP6_HTAB(fd, len1);

      i = leerbegin + len1 * skip;
    }
  }
}

/* stores the last used color */
static int necp6_last_color = 0;

/* prints one buffer with 360 dpi */
#ifdef __STDC__
static void necp6_print_stripe_360(FILE *fd, char *buffer, int offset, int len,
				   int color, NecP6_Opts *opts)
#else
static void necp6_print_stripe_360(fd, buffer, offset, len, color, opts)
FILE *fd;
char *buffer;
int offset;
int len;
int color;
NecP6_Opts *opts;
#endif
{
  int len1;
  char *tmp_buf;
  int i;
  char *c;
  int doublebytes;
  int skip;

  if(buffer) {

    /* if there is a negative margin -> skipping the columns at the beginning */
    if(opts->x_margin < 0) {
      if((skip = opts->x_margin * opts->bytes * (-1)) > len)
	return;
      offset += skip;
      len -= skip;

    /* positive margin -> tab to the right position */
    } else if(opts->x_margin > 0) {
      skip = opts->x_margin / 2;
      NECP6_HTAB(fd, skip);

      if(skip % 2) {
	NECP6_PRINT_360_PREFIX(fd, 1);
	fwrite(necp6_null, 1, opts->bytes, fd);
      }
    }

    /* print only if buffer not empty */
    if(len1 = not_empty(buffer + offset, len)) {

      /* select the right color */
      if(color != necp6_last_color) {
	NECP6_COLOR(fd, color);
	necp6_last_color = color;
      }

      len1 = len1 % opts->bytes
	? len1 + opts->bytes - (len1 % opts->bytes) : len1;

      /* real 360 dpi -> buffer will be splitted into two buffer. One buffer
       * the even columns, one buffer the odd columns
       */
      if(opts->real360) {

	tmp_buf = malloc(len1);
	c = buffer + offset;
	doublebytes = opts->bytes * 2;

	/* even columns */
	for(i = 0; i < len1; i++) {
	  if((i % doublebytes) < opts->bytes) {
	    tmp_buf[i] = 0;
	  } else {
	    tmp_buf[i] = c[i];
	  }
	}

	necp6_print_stripe(fd, tmp_buf, len1, opts);

	/* odd columns */
	for(i = 0; i < len1; i++) {
	  if((i % doublebytes) < opts->bytes) {
	    tmp_buf[i] = c[i];
	  } else {
	    tmp_buf[i] = 0;
	  }
	}

	necp6_print_stripe(fd, tmp_buf, len1, opts);

	free(tmp_buf);

      } else {

	/* print without optimazion */
	necp6_print_stripe(fd, buffer + offset, len1, opts);
      }
    }

    NECP6_CR(fd);
  }
}

/* prints the buffer in 180 dpi */
#ifdef __STDC__
static void necp6_print_stripe_180(FILE *fd, char *buffer, int len,
			       int color, NecP6_Opts *opts)
#else
static void necp6_print_stripe_180(fd, buffer, len, color, opts)
FILE *fd;
char *buffer;
int len;
int color;
NecP6_Opts *opts;
#endif
{
  int len1;
  int cnt;
  int skip;

  if(buffer) {

    /* if negative margin -> skip the columns at the beginning */
    if(opts->x_margin < 0) {
      if((skip = opts->x_margin * opts->bytes * (-1)) > len)
	return;
      buffer += skip;
      len -= skip;

      /* positive margin -> tab to the right position */
    } else if(opts->x_margin > 0) {
      NECP6_HTAB(fd, opts->x_margin);
    }

    if(len1 = not_empty(buffer, len)) {
      /* select the color */
      if(color != necp6_last_color) {
	NECP6_COLOR(fd, color);
	necp6_last_color = color;
      }

      len1 = len1 % opts->bytes
	? len1 + opts->bytes - (len1 % opts->bytes) : len1;

      cnt = len1 / opts->bytes;

      /* print the buffer */
      NECP6_PRINT_180_PREFIX(fd, cnt);
      fwrite(buffer, 1, len1, fd);
    }

    NECP6_CR(fd);
  }
}

/* copy the upper 3 halfbytes and sets lower region to zero */
#ifdef __STDC__
static void bitcopy_u_u(char *dest, char *src, int len)
#else
static void bitcopy_u_u(dest, src, len)
char *dest;
char *src;
int len;
#endif
{
  int i;

  if(dest) {

    src += len;

    for(i = 0; i < len; i++) {
      switch(i%3) {
      case 0:
	*(dest++) = *(src++);
	break;
      case 1:
	*(dest++) = *(src++) & 0xF0;
	break;
      case 2:
	*(dest++) = '\0';
	src++;
	break;
      }
    }
  }
}

/* copy the lower region to the upper region, settung lower region to zero */
#ifdef __STDC__
static void bitcopy_l_u(char *dest, char *src, int len)
#else
static void bitcopy_l_u(dest, src, len)
char *dest;
char *src;
int len;
#endif
{
  int i;

  if(dest) {

    src += len;

    for(i = 0, src++; i < len; i++) {
      switch(i%3) {
      case 0:
	*dest = (*src & 0xF) << 4;
	src++;
	*dest |= (*src & 0xF0) >> 4;
	dest++;
	break;
      case 1:
	*(dest++) = (*(src++) & 0xF) << 4;
	break;
      case 2:
	*(dest++) = '\0';
	src++;
	break;
      }
    }
  }
}

/* copy the upper region to the lower region */
#ifdef __STDC__
static void bitcopy_u_l(char *dest, char *src, int len)
#else
static void bitcopy_u_l(dest, src, len)
char *dest;
char *src;
int len;
#endif
{
  int i;

  if(dest) {

    src += len;

    for(i = 0, dest++; i < len; i++) {
      switch(i%3) {
      case 0:
	*(dest++) |= (*src & 0xF0) >> 4;
	*dest      = (*(src++) & 0xF) << 4;
	break;
      case 1:
	*(dest++) |= (*(src++) & 0xF0) >> 4;
	break;
      case 2:
	dest++;	src++;
	break;
      }
    }
  }
}

/* copy upper region to upper region. setting lower region to zero
 * using mask
 */
#ifdef __STDC__
static void bitcopy_u_u_mask(char *dest, char *src, int len, char mask[3])
#else
static void bitcopy_u_u_mask(dest, src, len, mask)
char *dest;
char *src;
int len;
char mask[3];
#endif
{
  int i;
  char m1, m2;

  m1 = (mask[0] << 4) | mask[1];
  m2 = mask[2] << 4;

  if(dest) {
    for(i = 0; i < len; i++) {
      switch(i%3) {
      case 0:
	*(dest++) = *(src++) & m1;
	break;
      case 1:
	*(dest++) = *(src++) & 0xF0 & m2;
	break;
      case 2:
	*(dest++) = '\0';
	src++;
	break;
      }
    }
  }
}

/* copy lower region to lower region using mask
 */
#ifdef __STDC__
static void bitcopy_l_l_mask(char *dest, char *src, int len, char mask[3])
#else
static void bitcopy_l_l_mask(dest, src, len, mask)
char *dest;
char *src;
int len;
char mask[3];
#endif
{
  int i;
  char m1, m2;

  m1 = mask[0];
  m2 = (mask[1] << 4) | mask[2];

  if(dest) {
    for(i = 0; i < len; i++) {
      switch(i%3) {
      case 0:
	dest++;
	src++;
	break;
      case 1:
	*(dest++) |= *(src++) & 0xF & m1;
	break;
      case 2:
	*(dest++) = *(src++) & m2;
	break;
      }
    }
  }
}

/* copy lower region to upper region. setting lower region to zero
 * using mask
 */
#ifdef __STDC__
static void bitcopy_l_u_mask(char *dest, char *src, int len, char mask[3])
#else
static void bitcopy_l_u_mask(dest, src, len, mask)
char *dest;
char *src;
int len;
char mask[3];
#endif
{
  int i;
  char m1, m2;

  m1 = mask[0];
  m2 = (mask[1] << 4) | mask[2];

  if(dest) {
    for(i = 0, src++; i < len; i++) {
      switch(i%3) {
      case 0:
	*dest = (*src & 0xF & m1) << 4;
	src++;
	*dest |= (*src & 0xF0 & m2) >> 4;
	dest++;
	break;
      case 1:
	*(dest++) = (*(src++) & 0xF & m2) << 4;
	break;
      case 2:
	*(dest++) = '\0';
	src++;
	break;
      }
    }
  }
}

/* copy upper region to lower region using mask
 */
#ifdef __STDC__
static void bitcopy_u_l_mask(char *dest, char *src, int len, char mask[3])
#else
static void bitcopy_u_l_mask(dest, src, len, mask)
char *dest;
char *src;
int len;
char mask[3];
#endif
{
  int i;
  char m1, m2;

  m1 = (mask[0] << 4) | mask[1];
  m2 = mask[2] << 4;

  if(dest) {
    for(i = 0, dest++; i < len; i++) {
      switch(i%3) {
      case 0:
	*(dest++) |= (*src & 0xF0 & m1) >> 4;
	*dest      = (*(src++) & 0xF & m1) << 4;
	break;
      case 1:
	*(dest++) |= (*(src++) & 0xF0 & m2) >> 4;
	break;
      case 2:
	dest++;	src++;
	break;
      }
    }
  }
}

/* stores the interlaced buffers */
static char *buffer_yellow_old  = NULL;
static char *buffer_red_old     = NULL;
static char *buffer_blue_old    = NULL;
static char *buffer_black_old   = NULL;

static char *buffer_yellow_act  = NULL;
static char *buffer_red_act     = NULL;
static char *buffer_blue_act    = NULL;
static char *buffer_black_act   = NULL;

static int sav_len;

/* prints all buffers in 360 or 180 dpi */
#ifdef __STDC__
static void necp6_print_buffer(FILE *fd, char *buffer_yellow,
			       char *buffer_red, char *buffer_blue,
			       char *buffer_black, int len,
			       NecP6_Opts *opts)
#else
static void necp6_print_buffer(fd, buffer_yellow, buffer_red, buffer_blue,
			       buffer_black, len, opts)
FILE *fd;
char *buffer_yellow;
char *buffer_red;
char *buffer_blue;
char *buffer_black;
int len;
NecP6_Opts *opts;
#endif
{
  int c;
  int len2;
static int first = 0;

  if(len) {
    if(opts->x_dpi == 360) {
      switch(opts->interlace) {

      case 2: /* interlace_180 */
	len2 = len * 2;

	/* First pass ? */
	if(!buffer_black_old) {

	  sav_len = len;

	  if(buffer_yellow) {
	    buffer_yellow_old = malloc(len2);
	    buffer_red_old    = malloc(len2);
	    buffer_blue_old   = malloc(len2);
	    buffer_yellow_act = malloc(len2);
	    buffer_red_act    = malloc(len2);
	    buffer_blue_act   = malloc(len2);
	  }

	  buffer_black_old  = malloc(len2);
	  buffer_black_act  = malloc(len2);

	  bitcopy_u_u_mask(buffer_yellow_act, buffer_yellow, len2,
			   opts->odd_yellow);
	  bitcopy_u_u_mask(buffer_red_act,    buffer_red,    len2,
			   opts->odd_red);
	  bitcopy_u_u_mask(buffer_blue_act,   buffer_blue,   len2,
			   opts->odd_blue);
	  bitcopy_u_u_mask(buffer_black_act,  buffer_black,  len2,
			   opts->odd_black);

	  bitcopy_l_l_mask(buffer_yellow_act, buffer_yellow, len2,
		          opts->even_yellow);
	  bitcopy_l_l_mask(buffer_red_act,    buffer_red,    len2,
			   opts->even_red);
	  bitcopy_l_l_mask(buffer_blue_act,   buffer_blue,   len2,
			   opts->even_blue);
	  bitcopy_l_l_mask(buffer_black_act,  buffer_black,  len2,
			   opts->even_black);

	  bitcopy_u_u_mask(buffer_yellow_old, buffer_yellow, len2,
			   opts->even_yellow);
	  bitcopy_u_u_mask(buffer_red_old,    buffer_red,    len2,
			   opts->even_red);
	  bitcopy_u_u_mask(buffer_blue_old,   buffer_blue,   len2,
			   opts->even_blue);
	  bitcopy_u_u_mask(buffer_black_old,  buffer_black,  len2,
			   opts->even_black);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_act, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_act,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_act,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_act,  0, len, BLACK,  opts);
	  }

	  if(opts->printodd) {
	    necp6_print_stripe_360(fd,buffer_yellow_old, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,  0, len, BLACK,  opts);
	  }

	  NECP6_LF(fd);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_act,len,len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_act,  len, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_act, len, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_act,len, len, BLACK,  opts);
	  }

	  if(opts->printodd) {
	    necp6_print_stripe_360(fd,buffer_yellow_old,len,len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,  len, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old, len, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,len, len, BLACK,  opts);
	  }

	  bitcopy_l_u_mask(buffer_yellow_old, buffer_yellow, len2,
			   opts->odd_yellow);
	  bitcopy_l_u_mask(buffer_red_old,    buffer_red,    len2,
			   opts->odd_red);
	  bitcopy_l_u_mask(buffer_blue_old,   buffer_blue,   len2,
			   opts->odd_blue);
	  bitcopy_l_u_mask(buffer_black_old,  buffer_black,  len2,
			   opts->odd_black);

	  NECP6_LINEFEED_HIGH_23(fd);

	} else { /* Not the first pass */

	  bitcopy_u_l_mask(buffer_yellow_old, buffer_yellow, len2,
			   opts->even_yellow);
	  bitcopy_u_l_mask(buffer_red_old,    buffer_red,    len2,
			   opts->even_red);
	  bitcopy_u_l_mask(buffer_blue_old,   buffer_blue,   len2,
			   opts->even_blue);
	  bitcopy_u_l_mask(buffer_black_old,  buffer_black,  len2,
			   opts->even_black);

	  if(opts->printodd) {
	    necp6_print_stripe_360(fd,buffer_yellow_old, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,  0, len, BLACK,  opts);
	  }

	  NECP6_LF(fd);

	  if(opts->printodd) {
	    necp6_print_stripe_360(fd,buffer_yellow_old,len,len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,  len, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old, len, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,len, len, BLACK,  opts);
	  }

	  NECP6_LINEFEED_HIGH_23(fd);

	  bitcopy_u_u_mask(buffer_yellow_act, buffer_yellow, len2,
			   opts->odd_yellow);
	  bitcopy_u_u_mask(buffer_red_act,    buffer_red,    len2,
			   opts->odd_red);
	  bitcopy_u_u_mask(buffer_blue_act,   buffer_blue,   len2,
			   opts->odd_blue);
	  bitcopy_u_u_mask(buffer_black_act,  buffer_black,  len2,
			   opts->odd_black);
	  bitcopy_l_l_mask(buffer_yellow_act, buffer_yellow, len2,
			   opts->even_yellow);
	  bitcopy_l_l_mask(buffer_red_act,    buffer_red,    len2,
			   opts->even_red);
	  bitcopy_l_l_mask(buffer_blue_act,   buffer_blue,   len2,
			   opts->even_blue);
	  bitcopy_l_l_mask(buffer_black_act,  buffer_black,  len2,
			   opts->even_black);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_act, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_act,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_act,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_act,  0, len, BLACK,  opts);
	  }

	  NECP6_LF(fd);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_act,len,len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_act,   len,len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_act,  len,len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_act, len,len, BLACK,  opts);
	  }

	  NECP6_LINEFEED_HIGH_23(fd);

	  bitcopy_l_u_mask(buffer_yellow_old, buffer_yellow, len2,
			   opts->odd_yellow);
	  bitcopy_l_u_mask(buffer_red_old,    buffer_red,    len2,
			   opts->odd_red);
	  bitcopy_l_u_mask(buffer_blue_old,   buffer_blue,   len2,
			   opts->odd_blue);
	  bitcopy_l_u_mask(buffer_black_old,  buffer_black,  len2,
			   opts->odd_black);
	}
	break;

      case 1: /* interlace_360 */

	/* FIrst pass ? */
	if(!buffer_black_old) {
	  if(opts->printodd) {
	    necp6_print_stripe_360(fd, buffer_yellow, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd, buffer_red,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd, buffer_blue,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd, buffer_black,  0, len, BLACK,  opts);
	  }

	  NECP6_LF(fd);

	  if(buffer_yellow) {
	    buffer_yellow_old = malloc(len);
	    buffer_red_old    = malloc(len);
	    buffer_blue_old   = malloc(len);
	  }
	  buffer_black_old  = malloc(len);

	  sav_len = len;

	  bitcopy_u_u(buffer_yellow_old, buffer_yellow, len);
	  bitcopy_u_u(buffer_red_old,    buffer_red,    len);
	  bitcopy_u_u(buffer_blue_old,   buffer_blue,   len);
	  bitcopy_u_u(buffer_black_old,  buffer_black,  len);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_old, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,  0, len, BLACK,  opts);
	  }

	  bitcopy_l_u(buffer_yellow_old, buffer_yellow, len);
	  bitcopy_l_u(buffer_red_old,    buffer_red,    len);
	  bitcopy_l_u(buffer_blue_old,   buffer_blue,   len);
	  bitcopy_l_u(buffer_black_old,  buffer_black,  len);

	  NECP6_LINEFEED_HIGH_24(fd);

	} else { /* not the first pass */

	  bitcopy_u_l(buffer_yellow_old, buffer_yellow, len);
	  bitcopy_u_l(buffer_red_old,    buffer_red,    len);
	  bitcopy_u_l(buffer_blue_old,   buffer_blue,   len);
	  bitcopy_u_l(buffer_black_old,  buffer_black,  len);

	  if(opts->printeven) {
	    necp6_print_stripe_360(fd,buffer_yellow_old, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd,buffer_red_old,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd,buffer_blue_old,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd,buffer_black_old,  0, len, BLACK,  opts);
	  }

	  NECP6_LINEFEED_HIGH_23(fd);

	  bitcopy_l_u(buffer_yellow_old, buffer_yellow, len);
	  bitcopy_l_u(buffer_red_old,    buffer_red,    len);
	  bitcopy_l_u(buffer_blue_old,   buffer_blue,   len);
	  bitcopy_l_u(buffer_black_old,  buffer_black,  len);

	  if(opts->printodd) {
	    necp6_print_stripe_360(fd, buffer_yellow, 0, len, YELLOW, opts);
	    necp6_print_stripe_360(fd, buffer_red,    0, len, RED,    opts);
	    necp6_print_stripe_360(fd, buffer_blue,   0, len, BLUE,   opts);
	    necp6_print_stripe_360(fd, buffer_black,  0, len, BLACK,  opts);
	  }

	  NECP6_LINEFEED_HIGH_25(fd);
	}
	break;

      default: /* No interlacing */

	if(first == 1) {
	  first = 2;
	} else {
	  NECP6_LINEFEED_HIGH(fd);
	}

	necp6_print_stripe_360(fd, buffer_yellow, 0, len, YELLOW, opts);
	necp6_print_stripe_360(fd, buffer_red,    0, len, RED,    opts);
	necp6_print_stripe_360(fd, buffer_blue,   0, len, BLUE,   opts);
	necp6_print_stripe_360(fd, buffer_black,  0, len, BLACK,  opts);

	NECP6_LF(fd);

	necp6_print_stripe_360(fd, buffer_yellow, len, len, YELLOW, opts);
	necp6_print_stripe_360(fd, buffer_red,    len, len, RED,    opts);
	necp6_print_stripe_360(fd, buffer_blue,   len, len, BLUE,   opts);
	necp6_print_stripe_360(fd, buffer_black,  len, len, BLACK,  opts);

	break;
      }
    } else { /* 180 dpi */

      if(first == 1) {
	first = 2;
      } else {
	NECP6_LINEFEED_FULL(fd);
      }

      necp6_print_stripe_180(fd, buffer_yellow, len, YELLOW, opts);
      necp6_print_stripe_180(fd, buffer_red   , len, RED,    opts);
      necp6_print_stripe_180(fd, buffer_blue  , len, BLUE,   opts);
      necp6_print_stripe_180(fd, buffer_black , len, BLACK,  opts);
    }
  } else { /* before the first pass or after tha last pass */

    if(necp6_null) { /* after the last pass */
      switch(opts->interlace) {
      case 1: /* interlace_360 */
	if(opts->printeven) {
	  necp6_print_stripe_360(fd,buffer_yellow_old,0,sav_len, YELLOW, opts);
	  necp6_print_stripe_360(fd,buffer_red_old,   0,sav_len, RED,    opts);
	  necp6_print_stripe_360(fd,buffer_blue_old,  0,sav_len, BLUE,   opts);
	  necp6_print_stripe_360(fd,buffer_black_old, 0,sav_len, BLACK,  opts);
	}

	if(buffer_yellow_old) {
	  free(buffer_yellow_old);
	  free(buffer_red_old);
	  free(buffer_blue_old);
	}

	free(buffer_black_old);

	buffer_yellow_old = NULL;
	buffer_red_old    = NULL;
	buffer_blue_old   = NULL;
	buffer_black_old  = NULL;
	break;

      case 2: /* interlace_180 */
	if(opts->printodd) {
	  necp6_print_stripe_360(fd,buffer_yellow_old,0,sav_len, YELLOW, opts);
	  necp6_print_stripe_360(fd,buffer_red_old,   0,sav_len, RED,    opts);
	  necp6_print_stripe_360(fd,buffer_blue_old,  0,sav_len, BLUE,   opts);
	  necp6_print_stripe_360(fd,buffer_black_old, 0,sav_len, BLACK,  opts);
	}

	NECP6_LF(fd);

	if(opts->printodd) {
	  necp6_print_stripe_360(fd,buffer_yellow_old,len,sav_len,YELLOW,opts);
	  necp6_print_stripe_360(fd,buffer_red_old,   len,sav_len,RED,   opts);
	  necp6_print_stripe_360(fd,buffer_blue_old,  len,sav_len,BLUE,  opts);
	  necp6_print_stripe_360(fd,buffer_black_old, len,sav_len,BLACK, opts);
	}

	if(buffer_yellow_old) {
	  free(buffer_yellow_old);
	  free(buffer_red_old);
	  free(buffer_blue_old);
	  free(buffer_yellow_act);
	  free(buffer_red_act);
	  free(buffer_blue_act);
	}

	free(buffer_black_old);
	free(buffer_black_act);

	buffer_yellow_old = NULL;
	buffer_red_old    = NULL;
	buffer_blue_old   = NULL;
	buffer_black_old  = NULL;
	buffer_yellow_act = NULL;
	buffer_red_act    = NULL;
	buffer_blue_act   = NULL;
	buffer_black_act  = NULL;

	break;
      }

      free(necp6_null);
      necp6_null = NULL;

    } else { /* before first pass */
      necp6_null = malloc(opts->bytes);
      memset(necp6_null, 0, opts->bytes);
      first = 1;
    }
  }
}

/* prints monochrome graphics */
#ifdef __STDC__
int necp6_mono(FILE *in, FILE *out, int argc, char *argv[])
#else
int necp6_mono(in, out, argc, argv)
FILE *in;
FILE *out;
int argc;
char *argv[];
#endif
{
  NecP6_Opts necp6_opts;
  Pnm_Info *pnm_info;
  char *out_buffer;
  long line, col;
  int pixel;
  int outlen;
  int byte;
  int out_index;
  int linlen;
  int dblplusone;
  int value;
  char *outp;
  int tmp1;
  int eightpdbl;
  int i;

  /* printing usage */
  if(!in) {
    fprintf(stderr, "\n  Options for driver '%s'\n", argv[0]);
    necp6_decode_args(0, argv, NULL);
    return(1);
  }

  /* reading header from pnm file */
  if(!(pnm_info = pnm_get_info(in))) {
    fprintf(stderr, "%s: Can't decode PNM file\n", argv[0]);
    return(1);
  }

  switch(pnm_info->type) {
  case PGM:
    fprintf(stderr, "%s: Greyscales ignored\n", argv[0]);
    break;

  case PPM:
    fprintf(stderr, "%s: Colors ignored\n", argv[0]);
    break;

  case PBM:
    break;

  default:
    fprintf(stderr, "%s: Unknown PNM type\n", argv[0]);
    return(1);
    break;
  }

  /* decoding driver specific args */
  necp6_decode_args(argc, argv, &necp6_opts);

  if(necp6_opts.y_margin > pnm_info->height)
    return(0);

  if(pnm_info->width + necp6_opts.x_margin > necp6_opts.maxwidth) {
    fprintf(stderr, "%s:  picture with %d larger than %d (Margin: %d)\n",
	    argv[0], pnm_info->width, necp6_opts.maxwidth, necp6_opts.x_margin);
    return(1);
  }

  /* some useful constants */
  dblplusone = necp6_opts.dbl + 1;
  eightpdbl  = dblplusone * 8;

  linlen     = necp6_opts.bytes * pnm_info->width;
  outlen     = linlen * (necp6_opts.dbl + 1);

  /* allocating output buffer */
  if(!(out_buffer = malloc(outlen))) {
    fprintf(stderr, "%s: malloc error\n", argv[0]);
    return(1);
  }

  /* init printer */
  if(necp6_opts.init) {
    NECP6_INIT_1(out);
  }

  NECP6_INIT_2(out);

  /* if margin negative -> skipping the lines at the beginning */
  if(necp6_opts.y_margin < 0) {
    for(line = necp6_opts.y_margin; line < 0; line++) {
      for(col = 0; col < pnm_info->width; col++) {
	(*pnm_info->get_pixel_mono)(&pixel, in);
      }
      (*pnm_info->get_pixel_mono)(NULL, in);
    }
    pnm_info->height += necp6_opts.y_margin;

  /* positive margin -> tab to the right position */
  } else if(necp6_opts.y_margin > 0) {
    if(necp6_opts.x_dpi == 360) {
      line = necp6_opts.y_margin / 2;
      if(necp6_opts.y_margin % 2) {
	NECP6_LF(out);
      }
      tmp1 = line / 127;
      for(i = 0; i < tmp1; i++) {
	NECP6_LINEFEED_180(out,127);
      }
      if(line % 127) {
	NECP6_LINEFEED_180(out,line % 127);
      }
    } else {
      tmp1 = necp6_opts.y_margin / 127;
      for(i = 0; i < tmp1; i++) {
	NECP6_LINEFEED_180(out,127);
      }
      if(necp6_opts.y_margin % 127) {
	NECP6_LINEFEED_180(out,necp6_opts.y_margin % 127);
      }
    }
  }

  /* initialization */
  necp6_print_buffer(out, NULL, NULL, NULL, NULL, 0, &necp6_opts);

  /* processing line by line */
  for(line = 0; line < pnm_info->height; line++) {

    /* if buffer full -> printing line */
    if((tmp1 = line % (necp6_opts.pins * dblplusone)) == 0) {

      if(line) {
	necp6_print_buffer(out, NULL, NULL, NULL, out_buffer, linlen,
			   &necp6_opts);
	}

      /* clearing output buffer */
      memset(out_buffer, 0, outlen);
    }

    /* some useful constants */
    value = bit_value[(tmp1 / dblplusone) % 8];
    outp  = out_buffer + (tmp1 / eightpdbl);

    /* 360 dpi will be printed interlaced. The interlaced line is after the
     * first line
     */
    if(necp6_opts.dbl && (line % 2) == 1)
      outp += linlen;

    /* processing all pixels of one scan line */
    for(col = 0; col < pnm_info->width; col++, outp += necp6_opts.bytes) {

      /* reading pixel value */
      if((*pnm_info->get_pixel_mono)(&pixel, in)) {
	fprintf(stderr, "%s: Can't read pixels from PNM file\n", argv[0]);
	free(out_buffer);
	NECP6_EXIT(out);
	return(1);
      }

      /* setting pixel in output buffer */
      if(pixel)
	*outp |= value;
    }

    /* end of scan line */
    (*pnm_info->get_pixel_mono)(NULL, in);
  }

  /* print last buffer */
  necp6_print_buffer(out, NULL, NULL, NULL, out_buffer, linlen, &necp6_opts);

  necp6_print_buffer(out, NULL, NULL, NULL, NULL, 0, &necp6_opts);

  if(necp6_opts.dbl == 0) {
    tmp1 *= 2;
  }

  switch(necp6_opts.interlace) {
  case 1:
    tmp1 -= 24;
    break;
  case 2:
    tmp1 -= 25;
    break;
  }

  if(tmp1 < 0) {
    tmp1 = 0;
  }

  /* Move paper to end of the picture */
  while(tmp1--) {
    NECP6_LF(out);
  }

  if(necp6_opts.linefeed) {
    NECP6_LINEFEED_FULL(out);
  }

  if(necp6_opts.formfeed) {
    NECP6_FORMFEED(out);
  }

  /* reset printer */
  if(necp6_opts.reset) {
    NECP6_EXIT(out);
  }

  /* freeing memory */
  free(out_buffer);

  return(0);
}

/* printing color graphic */
#ifdef __STDC__
int necp6_color(FILE *in, FILE *out, int argc, char *argv[])
#else
int necp6_color(in, out, argc, argv)
FILE *in;
FILE *out;
int argc;
char *argv[];
#endif
{
  NecP6_Opts necp6_opts;
  Pnm_Info *pnm_info;
  char *out_buffer_red, *out_buffer_yellow, *out_buffer_blue,
       *out_buffer_black;
  long line, col;
  int pixel_yellow, pixel_red, pixel_blue, pixel_black;
  int outlen;
  int byte;
  int out_index;
  int linlen;
  int dblplusone;
  int value;
  char *outp_red, *outp_yellow, *outp_blue, *outp_black;
  int tmp1;
  int eightpdbl;
  int i;

  /* printing usage */
  if(!in) {
    fprintf(stderr, "\n  Options for driver '%s'\n", argv[0]);
    necp6_decode_args(0, argv, NULL);
    return(1);
  }

  /* reading header from pnm file */
  if(!(pnm_info = pnm_get_info(in))) {
    fprintf(stderr, "%s: Can't decode PNM file\n", argv[0]);
    return(1);
  }

  /* decoding driver specific args */
  necp6_decode_args(argc, argv, &necp6_opts);

  if(necp6_opts.y_margin > pnm_info->height)
    return(0);

  if(pnm_info->width + necp6_opts.x_margin > necp6_opts.maxwidth) {
    fprintf(stderr, "%s: picture with %d larger than %d (Margin: %d)\n",
	    argv[0], pnm_info->width, necp6_opts.maxwidth, necp6_opts.x_margin);
    return(1);
  }

  /* some useful constants */
  dblplusone = necp6_opts.dbl + 1;
  eightpdbl  = dblplusone * 8;

  linlen     = necp6_opts.bytes * pnm_info->width;
  outlen     = linlen * (necp6_opts.dbl + 1);

  /* allocating output buffer */
  if(!(out_buffer_red = malloc(outlen))) {
    fprintf(stderr, "%s: malloc error\n", argv[0]);
    return(1);
  }
  if(!(out_buffer_yellow = malloc(outlen))) {
    fprintf(stderr, "%s: malloc error\n", argv[0]);
    free(out_buffer_red);
    return(1);
  }
  if(!(out_buffer_blue = malloc(outlen))) {
    fprintf(stderr, "%s: malloc error\n", argv[0]);
    free(out_buffer_red);
    free(out_buffer_yellow);
    return(1);
  }
  if(!(out_buffer_black = malloc(outlen))) {
    fprintf(stderr, "%s: malloc error\n", argv[0]);
    free(out_buffer_red);
    free(out_buffer_yellow);
    free(out_buffer_black);
    return(1);
  }

  /* init printer */
  if(necp6_opts.init) {
    NECP6_INIT_1(out);
  }

  NECP6_INIT_2(out);

  /* negative margin -> skipping lines at the beginning */
  if(necp6_opts.y_margin < 0) {
    for(line = necp6_opts.y_margin; line < 0; line++) {
      for(col = 0; col < pnm_info->width; col++) {
	(*pnm_info->get_pixel_color)(&pixel_yellow, &pixel_red,
				     &pixel_blue, &pixel_black, in);
      }
      (*pnm_info->get_pixel_color)(NULL, NULL, NULL, NULL, in);
    }
    pnm_info->height += necp6_opts.y_margin;

  /* positive margin -> tab to the right position */
  } else if(necp6_opts.y_margin > 0) {
    if(necp6_opts.x_dpi == 360) {
      line = necp6_opts.y_margin / 2;
      if(necp6_opts.y_margin % 2) {
	NECP6_LF(out);
      }
      tmp1 = line / 127;
      for(i = 0; i < tmp1; i++) {
	NECP6_LINEFEED_180(out,127);
      }
      if(line % 127) {
	NECP6_LINEFEED_180(out,line % 127);
      }
    } else {
      tmp1 = necp6_opts.y_margin / 127;
      for(i = 0; i < tmp1; i++) {
	NECP6_LINEFEED_180(out,127);
      }
      if(necp6_opts.y_margin % 127) {
	NECP6_LINEFEED_180(out,necp6_opts.y_margin % 127);
      }
    }
  }

  /* initialization */
  necp6_print_buffer(out, NULL, NULL, NULL, NULL, 0, &necp6_opts);

  /* processing line by line */
  for(line = 0; line < pnm_info->height; line++) {

    /* if buffer full -> printing line */
    if((tmp1 = line % (necp6_opts.pins * dblplusone)) == 0) {

      if(line) {
	necp6_print_buffer(out, out_buffer_yellow, out_buffer_red,
			   out_buffer_blue, out_buffer_black, linlen,
			   &necp6_opts);
	}

      /* clearing output buffer */
      memset(out_buffer_yellow, 0, outlen);
      memset(out_buffer_red,    0, outlen);
      memset(out_buffer_blue,   0, outlen);
      memset(out_buffer_black,  0, outlen);
    }

    /* some useful constants */
    value = bit_value[(tmp1 / dblplusone) % 8];
    outp_yellow  = out_buffer_yellow + (tmp1 / eightpdbl);
    outp_red     = out_buffer_red    + (tmp1 / eightpdbl);
    outp_blue    = out_buffer_blue   + (tmp1 / eightpdbl);
    outp_black   = out_buffer_black  + (tmp1 / eightpdbl);

    /* 360 dpi will be printed interlaced. The interlaced line is after the
     * first line
     */
    if(necp6_opts.dbl && (line % 2) == 1) {
      outp_yellow += linlen;
      outp_red    += linlen;
      outp_blue   += linlen;
      outp_black  += linlen;
    }

    /* processing all pixels of one scan line */
    for(col = 0; col < pnm_info->width; col++,
	outp_yellow += necp6_opts.bytes,
	outp_red    += necp6_opts.bytes,
	outp_blue   += necp6_opts.bytes,
	outp_black  += necp6_opts.bytes) {

      /* reading pixel value */
      if((*pnm_info->get_pixel_color)(&pixel_yellow, &pixel_red,
				      &pixel_blue, &pixel_black, in)) {
	fprintf(stderr, "%s: Can't read pixels from PNM file\n", argv[0]);
	free(out_buffer_red);
	free(out_buffer_yellow);
	free(out_buffer_blue);
	free(out_buffer_black);
	NECP6_EXIT(out);
	return(1);
      }

      /* setting pixel in output buffer */
      if(pixel_yellow)
	*outp_yellow |= value;
      if(pixel_red)
	*outp_red    |= value;
      if(pixel_blue)
	*outp_blue   |= value;
      if(pixel_black)
	*outp_black  |= value;
    }

    /* end of scan line */
    (*pnm_info->get_pixel_color)(NULL, NULL, NULL, NULL, in);
  }

  /* print last buffer */
  necp6_print_buffer(out, out_buffer_yellow, out_buffer_red,
		     out_buffer_blue, out_buffer_black, linlen,
		     &necp6_opts);

  /* deinitialization */
  necp6_print_buffer(out, NULL, NULL, NULL, NULL, 0, &necp6_opts);

  if(necp6_opts.dbl == 0) {
    tmp1 *= 2;
  }

  switch(necp6_opts.interlace) {
  case 1:
    tmp1 -= 24;
    break;
  case 2:
    tmp1 -= 25;
    break;
  }

  if(tmp1 < 0) {
    tmp1 = 0;
  }

  /* Move paper to end of the picture */
  while(tmp1--) {
    NECP6_LF(out);
  }

  if(necp6_opts.linefeed) {
    NECP6_LINEFEED_FULL(out);
  }

  if(necp6_opts.formfeed) {
    NECP6_FORMFEED(out);
  }

  /* reset printer */
  if(necp6_opts.reset) {
    NECP6_EXIT(out);
  }

  /* freeing memory */
  free(out_buffer_red);
  free(out_buffer_yellow);
  free(out_buffer_blue);
  free(out_buffer_black);

  return(0);
}
