indexing
	
	description: "A printer variable that can be set to various options via a textual command"
	author: "Glenn Maughan <glennm@insect.sd.monash.edu.au>"
	status: "See notice at end of class"
	date: "1996/01/18 10:02:43"
	revision: "1.1.1.1"
	
class PRINTER_SET_VARIABLE
	
inherit
	
	DISPLAYABLE
	
	CHARACTER_ROUTINES
		export
			{NONE} all
		end
	
creation
	
	make
	
feature {ANY} -- Initialisation
	
	make is
			-- Build variable lists
		do
			!! options.make
			options.compare_objects
		end -- make
	
feature {ANY} -- Status report
	
	syntax: STRING
			-- The printer variable syntax (as understood by the relevant printer).
	
	options: LINKED_LIST [SET_OPTION]
			-- The valid values that the variable can be set to.
	
	default_option: SET_OPTION
			-- The default value that the variable will be set to.
	
	is_valid_option (option: STRING): BOOLEAN is
			-- Is `option' a valid option value for this variable?
		require
			option_exists: option /= Void
		do
			from
				options.start
			until
				options.exhausted or Result
			loop
				Result := options.item.is_match_for (option)
				options.forth
			end
		end -- is_valid_option
	
	is_user_selected: BOOLEAN
			-- Has the user selected an option for this setting?  If
			-- there is no user_selected_option then the default option
			-- is used.
	
	user_selected_option: STRING
			-- The user selected option
	
	compose_command_string: STRING is
			-- Build a string representing the variable set command.  Use
			-- the default value if no user_selected_option.
		require
			user_selected: is_user_selected
		do
			Result := clone (syntax)
			Result.append (" ")
			-- check for a user selected option, otherwise use the default
			if user_selected_option /= Void then
				Result.append (user_selected_option)
			else
				Result.append (default_option.value)
			end
		end -- compose_command_string
	
feature {ANY} -- Status setting
	
	set_user_selected is
			-- Set this option to user selected
		do
			is_user_selected := True
		end -- set_user_selected
	
	set_user_selected_option (selected_option: STRING) is
			-- Set the user selected option to `selected_option'.
		require
			valid_option: is_valid_option (selected_option)
		do	       
			user_selected_option := selected_option
			set_user_selected
		ensure
			is_selected: is_user_selected
		end -- set_user_selected_option
	
	set_syntax (new_syntax: STRING) is
			-- Set the syntax to `new_syntax'.  Will clone the syntax.
		require
			valid_syntax: new_syntax /= Void
		do
			syntax := clone (new_syntax)
		end -- set_syntax
	
	set_default_option (new_default: SET_OPTION) is
			-- Set the default_option variable value to `new_default'.
		require
			valid_default: new_default /= Void
		do
			default_option := new_default
		end -- set_default_option
	
feature {DISPLAYABLE} -- Display
	
	display_indented (file: IO_MEDIUM; level: INTEGER) is
			-- Output textual representation indented `level' spaces on `file'.
		do
			display_indent (file, level)
			file.putstring ("syntax = `")
			file.putstring (printable_string (syntax))
			file.putstring ("' options = ")
			display_options (file, level)
			file.putstring (" default = ")
			default_option.display (file)
			file.putstring ("%N")
		end -- display_indented 
	
feature {ANY} -- Display
	
	switch_string: STRING is
			-- String representation of command line syntax for this variable.
		local
			range_option: SET_OPTION_RANGE
			number_options: INTEGER
			pad: STRING
		do
			Result := "["
			-- add all of the options
			from
				options.start
				number_options := 1
			until
				options.exhausted
			loop
				
				-- check whether we have a normal or range option
				--| this should be done with dynamic binding!
				range_option ?= options.item
				if range_option /= Void then
					-- there is only one option.  Display it and the
					-- default afterwards.
					Result.append (options.item.out)
					Result.append (" {")
					Result.append (default_option.value)
					Result.append ("}")
				else
					-- check whether the option is default
					if options.item.is_match_for (default_option.value) then
						Result.append ("{")
					end
					Result.append (options.item.out)
					-- check whether the option is default
					if options.item.is_match_for (default_option.value) then
						Result.append ("}")
					end
					-- check whether this is the last option
					if not options.islast then
						Result.append (" | ")
					end 
					-- have we built a full line of options?
					if not options.islast and (number_options \\ Max_options_per_line) = 0 then
						-- go to next line with an indent
						Result.append ("%N")
						!! pad.make (Indent_step * 2)
						pad.fill_blank
						Result.append (pad)
					end
				end
				options.forth
				number_options := number_options + 1
			end
			Result.append ("]")
		end -- switch_string
	
feature {NONE} -- Constants
	
	Max_options_per_line: INTEGER is 6
			-- Maximum number of options to display per line
	
feature {NONE} -- Display
	
	display_options (file: IO_MEDIUM; level: INTEGER) is
		require
			valid_file: file /= Void and then (file.exists and file.is_open_write)
			valid_indent: level >= 0
		do
			from
				options.start
			until
				options.exhausted
			loop
				options.item.display (file)
				if not options.islast then
					file.putstring (", ")
				end
				options.forth
			end
		end -- display_options
	
end -- class PRINTER_SET_VARIABLE

--| hpmodeset - Print file encapsulation utility.
--| Copyright (C) 1995 Glenn Maughan <glennm@insect.sd.monash.edu.au>
--|
--| This program is free software; you can redistribute it and/or modify
--| it under the terms of the GNU General Public License as published by
--| the Free Software Foundation; either version 2 of the License, or
--| (at your option) any later version.
--|
--| This program is distributed in the hope that it will be useful,
--| but WITHOUT ANY WARRANTY; without even the implied warranty of
--| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--| GNU General Public License for more details.
--|
--| You should have received a copy of the GNU General Public License
--| along with this program; if not, write to the Free Software
--| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
