indexing
	
	description: "Configuration settings for a printer"
	author: "Glenn Maughan <glennm@insect.sd.monash.edu.au>"
	status: "See notice at end of class"
	date: "1996/01/18 10:02:43"
	revision: "1.1.1.1"
	
class PRINTER_CONFIGURATION
	
inherit
	
	DISPLAYABLE
	
	CHARACTER_ROUTINES
		export
			{NONE} all
		end
	
	STORABLE
	
	PRINTER_SET_CONSTANTS
		export
			{NONE} all
		end
	
creation
	
	make, make_retrieve
	
feature {ANY} -- Initialisation
	
	make is
			-- Build the tables for the configuration
		do
			!! globals.make (10)
			!! languages.make (3)
			!! variables.make (40)
		end -- make
	
	make_retrieve is
			-- Build an empty configuration ready for retrieving from
			-- persistent store.
		do
		end -- make_retrieve
	
feature {ANY} -- Status report
	
	globals: HASH_TABLE [STRING, STRING]
			-- The global printer setting variables.  Indexed on the setting
			-- name with the setting value stored in the table.
	
	languages: HASH_TABLE [PRINTER_LANGUAGE, STRING]
			-- The printer languages supported in this configuration.  Indexed
			-- on the language name and stores the printer language information object.
	
	variables: HASH_TABLE [PRINTER_SET_VARIABLE, STRING]
			-- The variables that may be changed in this configuration.
			-- Indexed on the variable name and stores the variable information object.
	
	maximum_read_for_magic: INTEGER is
			-- The maximum number of bytes to read to find all magic numbers
			-- for each printer language.
		require
			valid_languages: languages /= Void
			non_empty_languages: not languages.empty
		local
			index: INTEGER
			keys: ARRAY [STRING]
		do
			Result := 0
			-- iterate over all languages and determine the maximum read
			-- number by adding the offset and length of magic
			from
				keys := languages.current_keys
				index := keys.lower
			variant 
				keys.upper - index
			until
				index > keys.upper
			loop
				Result := max (Result, languages.item (keys.item (index)).magic_offset 
					       + languages.item (keys.item (index)).magic_string.count)
				-- do next language
				index := index + 1
			end
		end -- maximum_read_for_magic
	
	language_syntax_for_magic (magic_buffer: STRING): STRING is
			-- The language set syntax for the printer code found in `magic_buffer'.
		require
			valid_buffer: magic_buffer /= Void
			correct_buffer_length: magic_buffer.count = maximum_read_for_magic
			valid_language: is_valid_language (magic_buffer)
		local
			keys: ARRAY [STRING]
			index: INTEGER    
			found: BOOLEAN
		do 
			-- check each language magic numbers against the magic_buffer to
			-- determine whether it exists.
			from
				keys := languages.current_keys
				index := keys.lower
			variant
				keys.upper - index
			until
				index > keys.upper or found
			loop
				found := (magic_buffer.
					  substring_index (languages.item (keys.item (index)).magic_string, 
							   languages.item (keys.item (index)).magic_offset) > 0)
				if found then
					Result := languages.item (keys.item (index)).command
				end
				-- check next language
				index := index + 1
			end 
		end -- language_syntax_for_magic	 
	
	language_for_magic (magic_buffer: STRING): STRING is
			-- The language name for the printer code found in `magic_buffer'.
		require
			valid_buffer: magic_buffer /= Void
			correct_buffer_length: magic_buffer.count = maximum_read_for_magic
			valid_language: is_valid_language (magic_buffer)
		local
			keys: ARRAY [STRING]
			index: INTEGER    
			found: BOOLEAN
		do 
			-- check each language magic numbers against the magic_buffer to
			-- determine whether it exists.
			from
				keys := languages.current_keys
				index := keys.lower
			variant
				keys.upper - index
			until
				index > keys.upper or found
			loop
				found := (magic_buffer.
					  substring_index (languages.item (keys.item (index)).magic_string, 
							   languages.item (keys.item (index)).magic_offset) > 0)
				if found then
					Result := keys.item (index)
				end
				-- check next language
				index := index + 1
			end 
		end -- language_for_magic	 
	
	is_valid_language (magic_buffer: STRING): BOOLEAN is
			-- Does the printer code in `magic_buffer' constitute a valid
			-- printer language?
		require
			valid_buffer: magic_buffer /= Void
			correct_buffer_length: magic_buffer.count = maximum_read_for_magic
		local
			keys: ARRAY [STRING]
			index: INTEGER
		do
			-- check each language magic numbers against the magic_buffer to
			-- determine whether it exists.
			from
				keys := languages.current_keys
				index := keys.lower
			variant
				keys.upper - index
			until
				index > keys.upper or Result
			loop
				Result := (magic_buffer.
					   substring_index (languages.item (keys.item (index)).magic_string, 
							    languages.item (keys.item (index)).magic_offset) > 0)
				-- check next language
				index := index + 1
			end 
		end -- is_valid_language
	
feature {ANY} -- Validation
	
	is_valid: BOOLEAN is
			-- Is this configuration valid?
		do
			-- Must have at least one language, at least one variable and
			-- all globals. 
			-- assume everything was ok
			Result := True
			-- check if anything is not ok
			if languages.count = 0 then 
				Result := False
				validation_error_string := "Error in configuration file: No languages specified."
			elseif variables.count = 0 then
				Result := False
				validation_error_string := "Error in configuration file: No variables specified."
			elseif not all_globals_exist then
				Result := False
				validation_error_string := "Error in configuration file: Global definition missing."
			end
		end -- is_valid
	
	validation_error_string: STRING
			-- Validation error message.
	
feature {DISPLAYABLE} -- Display
	
	display_indented (file: IO_MEDIUM; level: INTEGER) is
			-- Output textual representation indented `level' spaces on `file'.
		do
			display_indent (file, level)
			file.putstring ("Printer Configuration%N")
			display_indent (file, level)
			file.putstring ("Globals:%N")
			display_globals (file, level + Indent_step)
			display_indent (file, level)
			file.putstring ("Languages:%N")
			display_languages (file, level + Indent_step)
			display_indent (file, level)
			file.putstring ("Variables:%N")
			display_variables (file, level + Indent_step)
		end -- display_indented
	
feature {NONE} -- Display
	
	display_globals (file: IO_MEDIUM; level: INTEGER) is
		require
			valid_file: file /= Void and then (file.exists and file.is_open_write)
			valid_indent: level >= 0
		local
			sorted_globals: SORTED_TWO_WAY_LIST [STRING]
		do
			-- get the keys out of the globals table and
			-- sort them
			!! sorted_globals.make
			sorted_globals.fill (globals.current_keys)
			from
				sorted_globals.start
			until
				sorted_globals.exhausted
			loop
				-- display each global variable and value
				display_indent (file, level)
				file.putstring (sorted_globals.item)
				file.putstring (" = `")
				file.putstring (printable_string (globals.
								  item (sorted_globals.item)))
				file.putstring ("'%N")
				sorted_globals.forth
			end
		end -- display_globals
	
	display_languages (file: IO_MEDIUM; level: INTEGER) is
		require
			valid_file: file /= Void and then (file.exists and file.is_open_write)
			valid_indent: level >= 0
		local
			sorted_languages: SORTED_TWO_WAY_LIST [STRING]
		do
			-- get the keys out of the languages table and
			-- sort them
			!! sorted_languages.make
			sorted_languages.fill (languages.current_keys)
			from
				sorted_languages.start
			until
				sorted_languages.exhausted
			loop
				-- display each global variable and value
				display_indent (file, level)
				file.putstring (sorted_languages.item)
				file.putstring (": ")
				languages.item (sorted_languages.item).display (file)
				sorted_languages.forth
			end
		end -- display_languages
	
	display_variables (file: IO_MEDIUM; level: INTEGER) is
		require
			valid_file: file /= Void and then (file.exists and file.is_open_write)
			valid_indent: level >= 0
		local
			sorted_languages: SORTED_TWO_WAY_LIST [STRING]
		do
			-- get thy keys out of the variables table and
			-- sort them
			!! sorted_languages.make
			sorted_languages.fill (variables.current_keys)
			from
				sorted_languages.start
			until
				sorted_languages.exhausted
			loop
				-- display each global variable and value
				display_indent (file, level)
				file.putstring (sorted_languages.item)
				file.putstring (": ")
				variables.item (sorted_languages.item).display (file)
				sorted_languages.forth
			end
		end -- display_variables
	
feature {NONE} -- Validation
	
	all_globals_exist: BOOLEAN is
			-- Do entries for every global option exist in the global table?
		do
			Result := globals.has (White_space) 
				and globals.has (Job_header) 
				and globals.has (Job_trailer) 
				and globals.has (Reset_command)
		end -- all_globals_exist
	
	max (a, b: INTEGER): INTEGER is
			-- The maximum of a or b
		do
			if a > b then
				Result := a
			else
				Result := b
			end 
		end -- max
	
end -- class PRINTER_CONFIGURATION

--| hpmodeset - Print file encapsulation utility.
--| Copyright (C) 1995 Glenn Maughan <glennm@insect.sd.monash.edu.au>
--|
--| This program is free software; you can redistribute it and/or modify
--| it under the terms of the GNU General Public License as published by
--| the Free Software Foundation; either version 2 of the License, or
--| (at your option) any later version.
--|
--| This program is distributed in the hope that it will be useful,
--| but WITHOUT ANY WARRANTY; without even the implied warranty of
--| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--| GNU General Public License for more details.
--|
--| You should have received a copy of the GNU General Public License
--| along with this program; if not, write to the Free Software
--| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
