indexing

	description: "HP command line mode setting application"
	author: "Glenn Maughan <glennm@insect.sd.monash.edu.au>"
	status: "See notice at end of class"
	date: "1996/01/18 10:02:42"
	revision: "1.1.1.1"
   
class HP_SET_MODE
	
inherit
	
	EXTENDED_ARGUMENTS
		export
			{NONE} all
		end
	
	PRINTER_SET_CONSTANTS
		export
			{NONE} all
		end
	
creation
	
	make
	
feature {NONE} -- Initialisation
	
	make is
			-- Run the set mode command line interface
		
		do
			io.error.putstring (Copyright_message)    
			-- create default printer settings
			!! file_list.make
			-- parse the command line setting all values
			parse_config_command_line 
			if valid_arguments then
				-- read configuration settings
				read_configuration
				-- check that the configuration worked ok
				if not config_generator.config_error then
					-- get the configuration
					configuration := config_generator.configuration
					-- parse the general options
					parse_general_command_line
					if valid_arguments then
						if show_usage then
							display_usage
						elseif show_config then
							display_config
						elseif show_license then
							display_license
						else
							-- parse the printer options 
							parse_printer_command_line
							if valid_arguments then
								-- put the pjl commands or the
								-- printer files
								if pjl_only then
									put_pjl_commands
								else
									put_printer_files  
								end
							else
								io.error.putstring ("Try hpmodeset -help%N")
							end	
						end
					else
						io.error.putstring ("Try hpmodeset -help%N")
					end
				else
					io.error.putstring (config_generator.config_error_string)
				end
			end
		rescue
			io.error.putstring ("Error: hpmodeset terminated abnormally.%N")
			io.error.putstring ("Please record the problem and send to: glennm@insect.sd.monash.edu.au.%N")
		end -- make
	
feature {ANY} -- Settings
	
	config_generator: HP_CONFIGURATION
	
	configuration: PRINTER_CONFIGURATION
	
	file_list: LINKED_LIST [PLAIN_TEXT_FILE]
			-- List of file names for printing
	
feature {NONE} -- Command line arguments
	
	environment: EXECUTION_ENVIRONMENT
	
	Hp_mode_set_home: STRING is "HPMODESETHOME"
			-- The environment variable for the home directory
	
	Analyser_name: STRING is "config-analyser.store"
			-- Name of the lexical analyser for config texts.
	
	Config_name: STRING is "config"
			-- Name of the config text file
	
	user_config_name: STRING
			-- Name of the config text file as given by the user in a
			-- -configfile option.  If Void Config_name is used.
	
	read_configuration is
			-- Read or rebuild the configuration structures
		local
			path, config_file, config_store_file, analyser_file: STRING 
			config_parser: HP_CONFIG
		do
			!! environment
			!! config_generator
				!! config_parser.make
			-- get the utility home path. First check for the home
			-- environment variable, otherwise use the current directory.
			path := environment.get (Hp_mode_set_home)
			if path /= Void then
				config_file := clone (path)
				config_file.append ("/")
				if user_config_name /= Void then
					config_file.append (user_config_name)
				else
					config_file.append (Config_name)
				end
				analyser_file := clone (path)
				analyser_file.append ("/")
				analyser_file.append (Analyser_name)
				config_store_file := clone (path)
				config_store_file.append ("/")
				if user_config_name /= Void then
					config_store_file.append (user_config_name)
				else
					config_store_file.append (Config_name)
				end
				config_store_file.append (".store")
			else
				if user_config_name /= Void then
					config_file := clone (user_config_name)
				else
					config_file := clone (Config_name)
				end
				analyser_file := Analyser_name
				if user_config_name /= Void then
					config_store_file := clone (user_config_name)
				else
					config_store_file := clone (Config_name)
				end
				config_store_file.append (".store")
			end
			config_generator.set_parser (config_parser, analyser_file)
			config_generator.set_configuration_file (config_file, config_store_file)
			config_generator.read
		end -- read_configuration
	
	put_printer_files is
			-- Output each printer file to stdout separated by enter
			-- language command depending on the file type
		do
			from
				number_files_output := 0
				file_list.start
			until
				file_list.exhausted
			loop  
				-- if the file isnt open then open it.  stdin will always
				-- be open
				if file_list.item.is_closed then
					file_list.item.open_read
				end
				-- put job header, language and file 
				put_job_language (file_list.item) --| calls put_job_header and put_job_file
				file_list.forth
			end
			-- put the trailer only if a file was output
			if number_files_output > 0 then
				put_job_trailer
			end
		end -- put_printer_files
	
	number_files_output: INTEGER 
			-- The number of files output to stdout.
	
	put_pjl_commands is
			-- Output the PJL header only
		do
			-- send the header and trailer
			put_job_header
			put_job_trailer
			-- notify the user
			io.error.putstring ("Output PJL commands%N")
		end -- put_pjl_commands
	
	put_job_header is
			-- Send the job header and option commands to stdout
		require
			configuration_set: configuration /= Void
		local
			index: INTEGER
			keys: ARRAY [STRING]
			globals: HASH_TABLE [STRING, STRING]
		do
			-- get each component of the job header from the globals and
			-- print on stdout
			globals := configuration.globals
			io.putstring (globals.item (Job_header))
			io.putstring (globals.item (End_of_command))
			-- put all the variable options
			-- iterate over all variables printing the
			-- compose_command_string if the variable is user_selected
			from
				keys := configuration.variables.current_keys
				index := keys.lower
			variant
				keys.upper - index
			until
				index > keys.upper
			loop
				-- check whether the variables is user_selected
				if configuration.variables.item (keys.item (index)).is_user_selected then
					-- output the compose command string
					io.putstring (configuration.variables.item (keys.item (index)).compose_command_string)
					-- end the command
					io.putstring (globals.item (End_of_command))
				end
				-- do next variable
				index := index + 1
			end
		end -- put_job_header
	
	put_job_language (input_file: PLAIN_TEXT_FILE) is
			-- Put the language selection command to stdout
		require
			configuration_set: configuration /= Void
			valid_file: input_file /= Void
			open_input_file: input_file.is_open_read
		local
			magic_buffer: STRING
			pad: STRING
		do
			-- check that the file is readable.  
			if input_file.file_readable then
				-- read a buffer of data from the file.
				input_file.read_stream (configuration.maximum_read_for_magic)
				magic_buffer := clone (input_file.last_string)
				-- check length of buffer.  If too short pad with spaces.
				if magic_buffer.count < configuration.maximum_read_for_magic then
					!! pad.make (configuration.maximum_read_for_magic - magic_buffer.count)
					pad.fill_blank
					magic_buffer.append (pad)
				end
				-- check whether the printer language is known
				if configuration.is_valid_language (magic_buffer) then 
					-- put the header
					put_job_header
					-- put the language command
					io.putstring (configuration.language_syntax_for_magic (magic_buffer))
				        io.putstring (configuration.globals.item (End_of_command))
					-- notify the user of file language
					io.error.putstring ("`")
					io.error.putstring (input_file.name)
					io.error.putstring ("' (")
					io.error.putstring (configuration.language_for_magic (magic_buffer))
					io.error.putstring (")")
					-- put the file and read buffer
					put_job_file (magic_buffer, input_file)
					-- put the language leave command
					io.putstring (configuration.languages.item (configuration.language_for_magic (magic_buffer)).
						      leave_command)
				        -- increment the number_files counter
				        number_files_output := number_files_output + 1
				else
					-- if we are allowed to output straight text then
					-- output it otherwise notify user of ignored file
					if ignore_text then
						io.error.putstring ("`")
						io.error.putstring (input_file.name)
						io.error.putstring ("' contains an unknown printer language, ignored.%N")
					else
						-- put the header
						put_job_header
						-- notify the user of file language (text)
						io.error.putstring ("`")
						io.error.putstring (input_file.name)
						io.error.putstring ("' (text)")
						-- put the file and read buffer
						put_job_file (magic_buffer, input_file)
						-- increment the number_files counter
						number_files_output := number_files_output + 1
					end
				end
			else 
				io.error.putstring ("`")
				io.error.putstring (input_file.name)
				io.error.putstring ("' is not readable (or empty), ignored.%N")
			end
		end -- put_job_language
	
	Read_buffer_size: INTEGER is 4096
	
	put_job_file (magic_buffer: STRING; file: PLAIN_TEXT_FILE) is
			-- Write the contents of `buffer' and `file' to stdout.
		require
			configuration_set: configuration /= Void
			valid_file: valid_file (file) and then file.is_open_read
		local
			stdin_count: INTEGER
		do
			-- put the start of the file
			io.putstring (magic_buffer)
			-- read and write the rest of the file
			-- check whether is it stdin, if so read until end of file
			if file = io.input then
				-- read a stream and write it
				from 
					stdin_count := magic_buffer.count
					file.read_stream (Read_buffer_size)
				until
					file.end_of_file
				loop
					stdin_count := stdin_count + file.last_string.count
					io.putstring (file.last_string)
					file.read_stream (Read_buffer_size)
				end
				-- put the last stream read
				stdin_count := stdin_count + file.last_string.count
				io.putstring (file.last_string) 
				io.error.putstring (" ")
				io.error.putint (stdin_count)
				io.error.putstring (" bytes%N")
			else
				-- treat as a normal file
				file.read_stream (file.count)
				io.putstring (file.last_string)
				io.error.putstring (" ")
				io.error.putint (magic_buffer.count + file.count)
				io.error.putstring (" bytes%N")
				file.close
			end
		end -- put_job_file
	
	put_job_trailer is
			-- Display the job trailer on stdout
		require
			configuration_set: configuration /= Void 
		local
			globals: HASH_TABLE [STRING, STRING]
		do
			globals := configuration.globals
			io.putstring (globals.item (Job_trailer))
			io.putstring (globals.item (End_of_command))
			io.putstring (globals.item (Reset_command))
			io.putstring (globals.item (End_of_command))
		end -- put_job_trailer
	
	Copyright_message: STRING is "hpmodeset version 2.1, (C) Copyright 1995/1996 Glenn Maughan%N"
	
	display_usage is
			-- Output usage message
		require
			configuration_exists: configuration /= Void
		local
			sorted_variables: SORTED_TWO_WAY_LIST [STRING]
		do
			io.error.putstring ("%NSynopsis:%N%N")
			io.error.putstring ("    hpmodeset [-help | -config | -license]%N")
			io.error.putstring ("    hpmodeset [<config switches>] [<option switches>] [<printer switches>] file ...%N")
			io.error.putstring ("%NGeneral switches:%N%N")	
			io.error.putstring ("    -help               display this help message and current printer switches%N")
			io.error.putstring ("    -showconfig         display the printer configuration information%N")
			io.error.putstring ("    -license            display the copyright and license information%N")
			io.error.putstring ("%NConfiguration switches:%N%N")
			io.error.putstring ("    -configfile=<file>  use <file> as the configuration file and store%N")
			io.error.putstring ("                        in <file>.store%N")
			io.error.putstring ("%NOption switches:%N%N")
			io.error.putstring ("    -notext             do not allow unknown print file languages to be output%N")
			io.error.putstring ("    -pjlonly            only output the PJL commands, ignore any files listed%N")
			io.error.putstring ("%NPrinter switches: (default values shown in braces)%N%N")
			-- display each available switch, the default value and valid options.
			from
				-- get the variable keys and sort them
				!! sorted_variables.make
				sorted_variables.fill (configuration.variables.current_keys)
				sorted_variables.start
			until
				sorted_variables.exhausted
			loop
				io.error.putstring ("    -")
				io.error.putstring (sorted_variables.item)
				io.error.putstring ("=")
				io.error.putstring (configuration.variables.item (sorted_variables.item).switch_string)
				io.error.new_line
				sorted_variables.forth
			end
		end -- display_usage
	
	display_config is
			-- Output configuration settings
		require
			configuration_exists: configuration /= Void
		do
			configuration.display (io.error)
		end -- display_config
	
	display_license is
			-- Output license message
		do
			io.error.putstring ("This program is free software; you can redistribute it and/or modify%N")
			io.error.putstring ("if under the terms of the GNU General Public License as published by%N")
			io.error.putstring ("the Free Software Foundation; either version 2 of the License, or%N")
			io.error.putstring ("(at your option) any later version.%N%N")
			io.error.putstring ("This program is distributed in the hope that it will be useful,%N")
			io.error.putstring ("but WITHOUT ANY WARRANTY; without even the implied warranty of%N")
			io.error.putstring ("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the%N")
			io.error.putstring ("GNU General Public License for more details.%N%N")
			io.error.putstring ("You should have received a copy of the GNU General Public License%N")
			io.error.putstring ("along with this program; if not, write to the Free Software%N")
			io.error.putstring ("Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.%N")
		end -- display_license
			
	Config_option: STRING is "showconfig"
	Configfile_option: STRING is "configfile="
	Help_option: STRING is "help"
	License_option: STRING is "license"
	Notext_option: STRING is "notext"
	Pjlonly_option: STRING is "pjlonly"
			-- command line options
	
	valid_arguments: BOOLEAN
			-- Are the command line arguments valid?
	
	show_usage: BOOLEAN
			-- Should the usage message be displayed?
	
	show_config: BOOLEAN
			-- Should the configuration settings be displayed?
	
	show_license: BOOLEAN
			-- Should the license information be displayed?
	
	ignore_text: BOOLEAN
			-- Should unknown print file types be ignored?
	
	pjl_only: BOOLEAN
			-- Should the PJL header be output only and all files ignored?
	
	last_argument_position: INTEGER
			-- The maximum position of the arguments parsed.  Search for
			-- file names after this point.
	
	parse_config_command_line is
			-- Get the command line arguments for the config options
		local
			next_argument: INTEGER
		do 
			set_option_sign ('-')
			valid_arguments := True
			-- parse the config options
			next_argument := has_word_option (Configfile_option) 
			if next_argument > 0 then
				user_config_name := coalesced_option_word_value (Configfile_option)
				last_argument_position := max (last_argument_position, next_argument)
			end	    
		end -- parse_config_command_line
	
	parse_general_command_line is
			-- Get the command line arguments for the general options
		local
			next_argument: INTEGER
		do 
			set_option_sign ('-')
			valid_arguments := True
			-- parse the general options
			if has_word_option (Help_option) > 0 then
				show_usage := True
			end
			if has_word_option (Config_option) > 0 then
				show_config := True
			end
			if has_word_option (License_option) > 0 then
				show_license := True
			end
			next_argument := has_word_option (Notext_option) 
			if next_argument > 0 then
				ignore_text := True
				last_argument_position := max (last_argument_position, next_argument)
			end
			next_argument := has_word_option (Pjlonly_option)
			if next_argument > 0 then
				pjl_only := True
				last_argument_position := max (last_argument_position, next_argument)
			end
		end -- parse_general_command_line
	
	parse_printer_command_line is
			-- Get the command line arguments for the printer options
		local
			next_argument: INTEGER
			keys: ARRAY [STRING]
			index: INTEGER
			new_option: STRING
			search_option: STRING
		do 
			valid_arguments := True
			-- parse for each argument in the configuration variables
			from
				keys := configuration.variables.current_keys
				index := keys.lower
			until
				index > keys.upper or not valid_arguments
			loop
				-- build the search option.  switch with = appended
				search_option := clone (keys.item (index))
				search_option.append ("=")
				-- search for the switch
				next_argument := has_word_option (search_option) 
				-- check whether it existed
				if next_argument > 0 then 
					debug ("options")
						io.error.putstring ("Option ")
						io.error.putstring (keys.item (index))
						io.error.putstring (" selected.%N")
					end
					last_argument_position := max (last_argument_position, next_argument)
					-- set the variable to user selected
					configuration.variables.item (keys.item (index)).set_user_selected
					-- check whether it has a separate word option
					new_option := coalesced_option_word_value (search_option)
					if new_option /= Void and then not new_option.empty then
						-- check that the option is valid
						if configuration.variables.item (keys.item (index)).is_valid_option (new_option) then
							debug ("options")
								io.error.putstring ("      value: ")
								io.error.putstring (new_option)
								io.error.new_line
							end
							configuration.variables.item (keys.
								   	      item (index)).set_user_selected_option (new_option)
						else
							io.error.putstring ("Invalid option `")
							io.error.putstring (new_option)
							io.error.putstring ("' for variable ")
							io.error.putstring (keys.item (index))
							io.error.new_line
							valid_arguments := False
						end
					end
				end
				-- do the next variable
				index := index + 1
			end	    
			-- get the list of files appearing after the last command
			-- line option.
			if valid_arguments then
				get_file_list
			end
		end -- parse_printer_command_line
	
	get_file_list is
			-- Collect the files at the end of the argument list.  There
			-- must be at least one.
		local
			next_file: STRING
			next_position: INTEGER
			file: PLAIN_TEXT_FILE
		do
			from
				-- move past the last argument
				next_position := last_argument_position + 1
			until
				next_position > argument_count
			loop
				-- check for stdin marker '-'
				if argument (next_position).is_equal ("-") then
					file_list.extend (io.input)
				else
					-- check the validity of the filename
					!! file.make (argument (next_position))
					if valid_file (file) then
						file_list.extend (file)
					else
						valid_arguments := False
						io.error.putstring ("Error: Invalid file: ")
						io.error.putstring (argument (next_position))
						io.error.putstring (", please check that the file exists and can be read.%N")
					end
				end
				next_position := next_position + 1
			end
			-- check that at least one file was given unless pjl_only is set
			if not pjl_only and then file_list.empty then
				valid_arguments := False
				io.error.putstring ("Error: Print file not specified, you must specify at least one print file.%N")
			end
		end -- get_file_list
	
	valid_file (file: PLAIN_TEXT_FILE): BOOLEAN is
			-- Does `file' exist and is it readable?
		require
			valid_file: file /= Void
		do
			-- check if it is stdin, if so then True
			if file = io.input then
				Result := True
			else
				Result := file.exists and file.is_readable
			end
		end -- valid_file
	
	max (a, b: INTEGER): INTEGER is
			-- The maximum of a or b
		do
			if a > b then
				Result := a
			else
				Result := b
			end 
		end -- max
	
end -- class HP_SET_MODE

--| hpmodeset - Print file encapsulation utility.
--| Copyright (C) 1995 Glenn Maughan <glennm@insect.sd.monash.edu.au>
--|
--| This program is free software; you can redistribute it and/or modify
--| it under the terms of the GNU General Public License as published by
--| the Free Software Foundation; either version 2 of the License, or
--| (at your option) any later version.
--|
--| This program is distributed in the hope that it will be useful,
--| but WITHOUT ANY WARRANTY; without even the implied warranty of
--| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--| GNU General Public License for more details.
--|
--| You should have received a copy of the GNU General Public License
--| along with this program; if not, write to the Free Software
--| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
