indexing

	description: "Configuration for the hpmodeset utility"
	author: "Glenn Maughan <glennm@insect.sd.monash.edu.au>"
	status: "See notice at end of class"
	date: "1996/01/18 10:02:42"
	revision: "1.1.1.1"

class HP_CONFIGURATION
   
inherit
   
   HP_CONFIG_LEXICAL
      export 
	 {NONE} all
      redefine
	 store_analyzer,
	 retrieve_analyzer
      end
   
   SEMANTIC_INFORMATION
      export
	 {NONE} all
      undefine
	 copy, consistent, is_equal, setup
      end
   
feature {ANY} -- Initialisation
   
   make is
	 -- Build a new empty configuration for the hp mode set tool
      do
      end -- make
   
feature {ANY} -- Configuration 
   
   configuration_file_name: STRING
	 -- Name of the file to read for configuration
   
   analyzer_file_name: STRING
	 -- Name of the file that does or will contain the lexical analyzer
	 -- for this configuration language.
   
   configuration_store_file_name: STRING
	 -- Name of the file to read the configuration structure from.
   
   root_line: CONSTRUCT
	 -- Top level construct in the configuration language.
   
   configuration: PRINTER_CONFIGURATION
	 -- The configuration either read from file or parsed from the
	 -- config file.
   
feature {ANY} -- Modification
   
   config_error: BOOLEAN
	 -- Did an error occur while reading configuration?
   
   config_error_string: STRING
	 -- String message representing the configuration error.
   
   set_configuration_file (new_config_file, new_store_file: STRING) is
	 -- Set the name of the configuration file to `new_file' and
	 -- the configuration store file to `new_store_file'.
      require
	 valid_file: new_config_file /= Void
	 valid_store_file: new_store_file /= Void
      do
	 configuration_file_name := new_config_file
	 configuration_store_file_name := new_store_file
      end -- set_configuration
   
   set_parser (root_construct: like root_line; analyzer_file: STRING) is
	 -- Set the parser root line to `root_construct' and set the name
	 -- of the lexical analyzer file name to `analyzer_file_name'
      require
	 valid_construct: root_construct /= Void 
	 valid_file_name: analyzer_file /= Void
      do
	 root_line := root_construct
	 analyzer_file_name := analyzer_file
      ensure
	 root_line_set: root_line = root_construct
	 analyzer_file_set: analyzer_file_name.is_equal (analyzer_file)
      end -- set_parser
   
   read is
	 -- Read the configuration from the configuration file
      require
	 root_line_set: root_line /= Void
	 config_files_set: configuration_file_name /= Void and configuration_store_file_name /= Void
	 analyser_files_set: analyzer_file_name /= Void
      local
	 file: PLAIN_TEXT_FILE
	 tried: BOOLEAN
      do   
	 -- make sure the analyzer is ready
	 setup_analyzer
	 -- check the input file
	 if is_valid_file (configuration_file_name) then
	    -- check whether the configuration needs to be rebuilt or read
	    if is_newer_store_file (configuration_file_name, configuration_store_file_name) or tried then
	       io.error.putstring ("Reading configuration... ")
	       -- parse the root line
	       root_line.document.set_input_file (configuration_file_name)
	       root_line.document.wipe_out
	       root_line.document.get_token
	       root_line.document.go_i_th (1)
	       root_line.parse;
	       debug ("parse_tree")
		  !! file.make ("tree.out")
		  file.create_read_write
		  root_line.display (file)
		  file.close
	       end 
	       -- if the parse worked then apply the semantics
	       if root_line.parsed then
		  root_line.semantics
		  debug ("check")
		     io.putstring ("Semantic information is ")
		     if info.check_clear then
			io.putstring ("clear%N")
		     else
			io.putstring ("NOT clear%N")
		     end
		  end
		  -- get the new configuration and store locally
		  configuration := info.configuration
		  -- check the validity of the configuration
		  if configuration.is_valid then  
		     io.error.putstring ("Done.%N")
		     -- store the new configuration
		     configuration.store_by_name (configuration_store_file_name)
		  else
		     config_error := True
		     config_error_string := configuration.validation_error_string
		  end
	       else
		  -- display the last error stored
		  config_error := True 
		  config_error_string := root_line.document.error_message
	       end
	    else
	       -- retrieve the configuration structure
	       !! configuration.make_retrieve
	       configuration ?= configuration.retrieve_by_name (configuration_store_file_name)
	       --| may cause exception on read error therefore retry
	       --| and build
	    end
	 else
	    config_error := True
	    config_error_string := "Invalid configuration file: "
	    config_error_string.append (configuration_file_name)
	    config_error_string.append ("%N")
	 end
      rescue
	 -- the retrieve must have failed 
	 if not tried then
	    tried := True
	    retry
	 end
      end -- read
   
feature {NONE} -- Implementation
   
   is_valid_file (file_name: STRING): BOOLEAN is
	 -- Does the file named `file_name' exist?
      require
	 valid_file_name: file_name /= Void
      local
	 file: RAW_FILE
      do
	 !! file.make (file_name)
	 Result := file.exists
      end -- is_valid_file
   
   is_newer_store_file (first_file_name, second_file_name: STRING): BOOLEAN is
	 -- Is the file in `first_file_name' newer than the file in
	 -- `second_file_name'?  Returns true if the second file does
	 -- not exist.
      local
	 file1, file2: RAW_FILE
      do
	 !! file1.make (first_file_name)
	 !! file2.make (second_file_name)
	 if not file2.exists then
	    Result := True
	 else
	    Result := (file1.exists and file2.exists) and then (file1.date > file2.date)
	 end
      end -- is_newer_store_file
   
   setup_analyzer is
	 -- Build the lexical analyzer, either use the existing
	 -- analyzer, retrieve one from disk or create a new one.
      require
	 valid_root_line: root_line /= Void
      local
	 analyzer_file: RAW_FILE;
	 tried: BOOLEAN
      do
	 !! analyzer_file.make (Analyzer_file_name);
	 -- use existing analyzer if it has been retrieved, or
	 -- retrieve the analyzer from file if the file exists, or
	 -- build a new one.
	 if analyzer /= Void then
	    root_line.document.set_lexical (analyzer)
	 elseif analyzer_file.exists and not tried then
	    --| may cause a failure, if it does then retry and build
	    --| the analyzer.
	    retrieve_analyzer (Analyzer_file_name);
	    root_line.document.set_lexical (analyzer); 
	 else
	    build (root_line.document);
	    -- store the built analyzer 
	    analyzer := root_line.document.analyzer
	    store_analyzer (Analyzer_file_name) 
	 end;
      rescue
	 -- the retrieve must have failed 
	 if not tried then
	    tried := True
	    retry
	 end
      end -- setup_analyzer
    
   store_analyzer (file_name: STRING) is
	 -- Store `analyzer' in file named `file_name'.
      require else
	 initialized: initialized
      local
	 store_file: RAW_FILE
	 directory: DIRECTORY
      do
	 if analyzer = Void then
	    !! analyzer.make
	 end;
	 !! store_file.make_open_write (file_name)
	 analyzer.independent_store (store_file)
	 store_file.close;
      end; -- store_analyzer
   
   retrieve_analyzer (file_name: STRING) is
	 --  Retrieve `analyzer' from file named `file_name'.
      local
	 retrieved_file: RAW_FILE
      do
	 if analyzer = Void then
	    !! analyzer.make
	 end;
	 !! retrieved_file.make_open_read (file_name);
	 analyzer ?= analyzer.retrieved (retrieved_file);
	 retrieved_file.close;
      end; -- store_analyzer
     
end -- class HP_CONFIGURATION

--| hpmodeset - Print file encapsulation utility.
--| Copyright (C) 1995 Glenn Maughan <glennm@insect.sd.monash.edu.au>
--|
--| This program is free software; you can redistribute it and/or modify
--| it under the terms of the GNU General Public License as published by
--| the Free Software Foundation; either version 2 of the License, or
--| (at your option) any later version.
--|
--| This program is distributed in the hope that it will be useful,
--| but WITHOUT ANY WARRANTY; without even the implied warranty of
--| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--| GNU General Public License for more details.
--|
--| You should have received a copy of the GNU General Public License
--| along with this program; if not, write to the Free Software
--| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
