/*********************************************************************************/
/*                                                                               */
/*  FenForm - Document Printer                                                   */
/*  Vn 2.0                                                                       */
/*                                                                               */
/*  Project Leader - Mike Eggleston                                              */
/*                                                                               */
/*  Copyright 2005 - 2008 Fenland Software Ltd                                   */
/*  Date: 28/08/2008                                                             */
/*                                                                               */
/*  Postscript Module - fenform_ps.c                                             */
/*                                                                               */
/*  Postscript Specific Functions                                                */
/*                                                                               */
/*********************************************************************************/
/*                                                                               */
/*  Licence: GPL Vn2                                                             */
/*                                                                               */
/*  This program is free software; you can redistribute it and/or modify         */
/*  it under the terms of the GNU General Public License as published by         */
/*  the Free Software Foundation; either version 2 of the License, or            */
/*  (at your option) any later version.                                          */
/*                                                                               */
/*  This program is distributed in the hope that it will be useful,              */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of               */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                */
/*  GNU General Public License for more details.                                 */
/*                                                                               */
/*  You should have received a copy of the GNU General Public License            */
/*  along with this program; if not, write to the Free Software                  */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    */
/*                                                                               */
/*                                                                               */
/*                                                                               */
/*********************************************************************************/


#include "fenform.h"


/******************************/
/*  Functions                 */
/******************************/

int FENFORM_PS_StartDocument( FF_Handle *Hndl )
{

    int n,
        tabpos = 0;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Creating File Header\n" );

    Hndl->output_ps = PS_new( );
    PS_open_fp( Hndl->output_ps, Hndl->ofile );

    if( FENFORM_PS_LoadForm( Hndl ) == 0 ) {

     PS_set_info( Hndl->output_ps, "Title", "FenForm Output" );
     PS_set_info( Hndl->output_ps, "Author", Hndl->user_name );
     PS_set_info( Hndl->output_ps, "Creator", PROGNAME );

     if( Hndl->frm.orientation == 'L' )
      PS_set_info( Hndl->output_ps, "Orientation", "Landscape" );
     else
      PS_set_info( Hndl->output_ps, "Orientation", "Portrait" );

     /* Set default tabs. */
     if( Hndl->verbose )
      syslog( LOG_INFO, "Setting Default Horizontal Tabs\n" );

     for( n = 0; n < MAX_H_TABS; n++ )
     {
      tabpos += DFT_HTAB;
      Hndl->frm.htabs[ n ] = tabpos * Hndl->frm.curr_char_width;
     }

     Hndl->cur_line = Hndl->frm.length - Hndl->frm.line_space;
     Hndl->start_col = Hndl->frm.l_margin;

     return( FENFORM_PS_LoadFonts( Hndl ) );

    } else {

     return( -1 );

    }
}


void FENFORM_PS_EndDocument( FF_Handle *Hndl )
{

    if( Hndl->verbose )
     syslog( LOG_INFO, "Writing PS File End\n" );

    PS_close( Hndl->output_ps );
    PS_delete( Hndl->output_ps );

}


int FENFORM_PS_ReadFormHeader( FF_Handle *Hndl, int form_file_dscptr )
{

    /***********************************************************/
    /* Header Layout                                           */
    /*                                                         */
    /* 9 Bytes Const %FENFORM%                                 */
    /* 2 Bytes - Header Size (Bytes)                           */
    /* 1 Byte - Major version                                  */
    /* 2 Bytes - Minor version                                 */
    /* NULL terminated string - User Name                      */
    /* NULL terminated string - Form Name                      */
    /* NULL terminated string - Creation Date                  */
    /* NULL terminated string - Creation Program               */
    /* NULL terminated string - Copyright Notice               */
    /* 2 Bytes - Form Length                                   */
    /* 2 Bytes - Form Width                                    */
    /* 2 Bytes - Left Margin                                   */
    /* 2 Bytes - Top Margin                                    */
    /* 1 Byte - LPI                                            */
    /* 1 Byte - CPI                                            */
    /*                                                         */
    /***********************************************************/

    FF_HeaderData Hdr;

    char *Buffer,
         *s;

    int  temp_int;

    /* Clear the header structure */
    memset( &Hdr, 0, sizeof( FF_HeaderData ) );

    /* Read the header data */
    Hdr.file_id = malloc( 10 * sizeof( char ) );
    read( form_file_dscptr, Hdr.file_id, 9 );
    Hdr.file_id[ 9 ] = '\0';

    if( strncmp ( Hdr.file_id, "%FENFORM%", 9 ) ) {
     syslog( LOG_INFO, "Invalid form file\n" ); 
     return( -1 );
    }

    read( form_file_dscptr, &Hdr.header_size, 2 );

    /* No character string in the header can be longer than */
    /* the entire header so we can allocate a single buffer */
    /* large enough to hold the entire header.              */

    Buffer = malloc( Hdr.header_size * sizeof( char ) );
    if( ! Buffer ) {
     syslog( LOG_INFO, "Unable to allocate header buffer.\n" ); 
     return( -1 );
    }
    memset( Buffer, 0, Hdr.header_size * sizeof( char ) );

    read( form_file_dscptr, &Hdr.major_version, 1 );
    if( Hdr.major_version > FORM_VERSION ) {
     syslog( LOG_INFO, "Unrecognized form version number\n" ); 
     return( -1 );
    }
    read( form_file_dscptr, &Hdr.minor_version, 2 );

    s = Buffer;
    do {
     read( form_file_dscptr, s, 1 );
    } while( *s++ );
    Hdr.user_name = malloc( ( strlen( Buffer ) + 1 ) * sizeof( char ) );
    if( Hdr.user_name )
     strcpy( Hdr.user_name, Buffer );

    s = Buffer;
    do {
     read( form_file_dscptr, s, 1 );
    } while( *( s++ ) );
    Hdr.form_name = malloc( ( strlen( Buffer ) + 1 ) * sizeof( char ) );
    if( Hdr.form_name )
     strcpy( Hdr.form_name, Buffer );

    s = Buffer;
    do {
     read( form_file_dscptr, s, 1 );
    } while( *( s++ ) );
    Hdr.creation_date = malloc( ( strlen( Buffer ) + 1 ) * sizeof( char ) );
    if( Hdr.creation_date )
     strcpy( Hdr.creation_date, Buffer );

    s = Buffer;
    do {
     read( form_file_dscptr, s, 1 );
    } while( *( s++ ) );
    Hdr.creation_pgm = malloc( ( strlen( Buffer ) + 1 ) * sizeof( char ) );
    if( Hdr.creation_pgm )
     strcpy( Hdr.creation_pgm, Buffer );

    s = Buffer;
    do {
     read( form_file_dscptr, s, 1 );
    } while( *( s++ ) );
    Hdr.copyright_notice = malloc( ( strlen( Buffer ) + 1 ) * sizeof( char ) );
    if( Hdr.copyright_notice )
     strcpy( Hdr.copyright_notice, Buffer );

    /* Free the buffer memory */
    free( Buffer );

    read( form_file_dscptr, &Hdr.form_length, 2 );
    read( form_file_dscptr, &Hdr.form_width, 2 );

    read( form_file_dscptr, &Hdr.left_margin, 2 );
    read( form_file_dscptr, &Hdr.top_margin, 2 );

    read( form_file_dscptr, &Hdr.lpi, 1 );
    read( form_file_dscptr, &Hdr.cpi, 1 );

    if( Hdr.lpi > 0 )
     Hndl->frm.line_space = POINTS_PER_INCH / (float)Hdr.lpi;

    if( Hdr.cpi > 0 ) {
     Hndl->frm.base_char_width = POINTS_PER_INCH / (float)Hdr.cpi;
     Hndl->frm.curr_char_width = Hndl->frm.base_char_width;
    }

    Hndl->frm.length = (float)Hdr.form_length;
    Hndl->frm.width = (float)Hdr.form_width;

    if( Hdr.form_width > Hdr.form_length )
     Hndl->frm.orientation = 'L';
    else
     Hndl->frm.orientation = 'P';

    Hndl->frm.l_margin = (float)Hdr.left_margin;
    Hndl->frm.r_margin = (float)Hdr.form_width;

    Hndl->frm.length -= (float)Hdr.top_margin;

    if( lseek( form_file_dscptr, Hdr.header_size, SEEK_SET ) == Hdr.header_size )
     return( 0 );
    else {
     syslog( LOG_INFO, "Form File error.  Unable to position to start of form.\n" ); 
     return( -1 );
    }
}


int FENFORM_PS_LoadForm( FF_Handle *Hndl )
{

    int form_file_dscptr,
        image_byte,
        error_flag = 0;

    if( ! Hndl->curr_form )
     return( error_flag );

    if( Hndl->verbose )
     syslog( LOG_INFO, "Loading Form\n" );

    if( ( form_file_dscptr = open( Hndl->curr_form, O_RDONLY ) ) > 0 ) {

     error_flag = FENFORM_PS_ReadFormHeader( Hndl, form_file_dscptr );
     if( error_flag == 0 ) {

      if( Hndl->verbose )
       syslog( LOG_INFO, "Loading Form Image - %s\n", Hndl->curr_form );

      fprintf( Hndl->ofile, "%%!PS-Adobe-3.0\n");
      fprintf( Hndl->ofile, "/ImageData\ncurrentfile\n<< /Filter /SubFileDecode\n   /DecodeParms << /EODCount 0 /EODString (*EOD*) >>\n>> /ReusableStreamDecode filter\n" );

      if( Hndl->frm.orientation == 'L' ) {
       fprintf( Hndl->ofile, "90 rotate\n");
       fprintf( Hndl->ofile, "0 -%f translate\n", Hndl->frm.length );
      }

      while( read( form_file_dscptr, &image_byte, 1 ) ) {
       fputc( image_byte, Hndl->ofile );
      }

      close( form_file_dscptr );

      fprintf( Hndl->ofile, "*EOD*\ndef\n\n/IDForm\n<< /FormType 1\n");

      if( Hndl->frm.orientation == 'L' )
       fprintf( Hndl->ofile, "   /BBox [0 0 %f %f]\n", Hndl->frm.length, Hndl->frm.width );
      else
       fprintf( Hndl->ofile, "   /BBox [0 0 %f %f]\n", Hndl->frm.width, Hndl->frm.length );

      fprintf( Hndl->ofile, "   /Matrix [ 1 0 0 1 0 0]\n   /PaintProc\n" );
      fprintf( Hndl->ofile, "   { pop\n       /ostate save def\n         /showpage {} def\n         /setpagedevice /pop load def\n" );
      fprintf( Hndl->ofile, "         ImageData 0 setfileposition ImageData cvx exec\n       ostate restore\n   } bind\n>> def\n" );

     }

    } else {

     syslog( LOG_INFO, "Unable to Load Form Image - %s\n", Hndl->curr_form );
     error_flag = -1;

    }

    return( error_flag );

}


char *FENFORM_PS_FindFontFile( FF_Handle *Hndl, char *font_name )
{

    FILE *fontmap_fp;

    char buffer[ LINE_LEN + 1 ],
         *fontmap,
         *font = NULL,
         *font_file = NULL;

    static char *font_path = NULL;

    fontmap_fp = fopen( Hndl->cfg.font_map, "r" );

    if( ! fontmap_fp ) {
     syslog( LOG_INFO, "Font map (%s) not found\n", Hndl->cfg.font_map );
     return( NULL );

    } else {

     font = realloc( font, (size_t) ( strlen( font_name ) + 1 ) * sizeof( char ) );
     if( ! font ) {
      syslog( LOG_INFO, "Unable to allocate memory for font\n" );
      return( NULL );
     }

     strcpy( font, font_name );

     do {
      fseek( fontmap_fp, 0L, SEEK_SET );
      while( fgets( buffer, LINE_LEN, fontmap_fp ) ) {
       if( *buffer == '/' ) {
        font_file = FENFORM_CFG_ExtractParameter( buffer + 1 );
        if( ( font_file )
         && ( strcmp( buffer + 1, font ) == 0 ) ) {
         if( *font_file == '(' ) {
          font_file++;

          /* Remove the closing parenthesis */
          *strchr( font_file, ')' ) = '\0';

          /* Remove the file "extension" */
          *strchr( font_file, '.' ) = '\0';

          break;

         } else {

          if( *font_file == '/' ) {
           *(font_file + strcspn( font_file, " \t;\n" ) ) = '\0'; 
           font = realloc( font, (size_t) strlen( font_file ) * sizeof( char ) );
           if( ! font ) {
            syslog( LOG_INFO, "Unable to allocate memory for font\n" );
            return( NULL );
           }
           strcpy( font, font_file + 1 );
           break;
          }
         }
        }
       }
      }
     } while( *font_file == '/' );
    }

    if( font_file && *font_file != '/' && *font_file != '(' ) {
     font_path = malloc( (size_t)( strlen( Hndl->cfg.font_map ) + 1 ) * sizeof( char ) );
     if( ! font_path ) {
      syslog( LOG_INFO, "Unable to allocate memory for font_path\n" );
      return( NULL );
     }

     strcpy( font_path, Hndl->cfg.font_map );
     *( strrchr( font_path, '/' ) + 1 ) = '\0';
     font_path = realloc( font_path, (size_t)( strlen( font_path ) + strlen( font_file) + 1 ) * sizeof( char ) );
     if( ! font_path ) {
      syslog( LOG_INFO, "Unable to allocate memory for font_path\n" );
      return( NULL );
     }

     font_path = strcat( font_path, font_file );

     return( font_path );

    } else {

     return( NULL );
    }

}


int FENFORM_PS_LoadFonts( FF_Handle *Hndl )
{

    char *font_file = NULL;
    const char *font_encoding = NULL;
    int ret = 0;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Loading PS Fonts\n" );

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.std_font );
    if( font_file )
     Hndl->cfg.std_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.std_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_font );
    if( font_file )
     Hndl->cfg.bold_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.bold_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_font );
    if( font_file )
     Hndl->cfg.dblstrike_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.dblstrike_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.italic_font );
    if( font_file )
     Hndl->cfg.italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.italic_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_italic_font );
    if( font_file )
     Hndl->cfg.bold_italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.bold_italic_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_italic_font );
    if( font_file )
     Hndl->cfg.dblstrike_italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.dblstrike_italic_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.nlq_font );
    if( font_file )
     Hndl->cfg.nlq_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.nlq_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_nlq_font );
    if( font_file )
     Hndl->cfg.bold_nlq_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.bold_nlq_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_nlq_font );
    if( font_file )
     Hndl->cfg.dblstrike_nlq_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.dblstrike_nlq_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.nlq_italic_font );
    if( font_file )
     Hndl->cfg.nlq_italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 0 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.nlq_italic_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_nlq_italic_font );
    if( font_file )
     Hndl->cfg.bold_nlq_italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.bold_nlq_italic_font );
     ret = -1;
    }

    font_file = FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_nlq_italic_font );
    if( font_file )
     Hndl->cfg.dblstrike_nlq_italic_font_id = PS_findfont( Hndl->output_ps, font_file, font_encoding, 1 );
    else {
     syslog( LOG_INFO, "Font File %s for %s font not loaded\n", font_file, Hndl->cfg.dblstrike_nlq_italic_font );
     ret = -1;
    }

    return( ret );

}


void FENFORM_PS_StartPage( FF_Handle *Hndl )
{

    Hndl->h_tab = 0;
    Hndl->v_tab = 0;
    Hndl->cur_line = Hndl->frm.length - Hndl->frm.line_space;
    Hndl->start_col = Hndl->frm.l_margin;
    Hndl->cur_col = Hndl->start_col;
    Hndl->text_mode = FF_OFF;
    Hndl->font_set = FF_OFF;
    Hndl->new_page = FF_OFF;

    if( Hndl->frm.orientation == 'L' )
     PS_begin_page( Hndl->output_ps, Hndl->frm.length, Hndl->frm.width );
    else
     PS_begin_page( Hndl->output_ps, Hndl->frm.width, Hndl->frm.length );

    if( Hndl->curr_form ) {
     fprintf( Hndl->ofile, "IDForm execform\nrestore\nsave\n" );

     if( Hndl->frm.orientation == 'L' ) {
      PS_rotate( Hndl->output_ps, 90 );
      PS_translate( Hndl->output_ps, 0, 0 - Hndl->frm.length );
     }
    }

}


void FENFORM_PS_PrintText( FF_Handle *Hndl )
{

    if( Hndl->font_flags & FF_UNDERLINE_FLAG )
     PS_set_parameter( Hndl->output_ps, "underline", "true" );
    else
     PS_set_parameter( Hndl->output_ps, "underline", "false" );

    if( Hndl->font_flags & FF_OVERSCORE_FLAG )
     PS_set_parameter( Hndl->output_ps, "overline", "true" );
    else
     PS_set_parameter( Hndl->output_ps, "overline", "false" );

    if( Hndl->ri_end > Hndl->ri_start )
     FENFORM_PrintReverseImage( Hndl );
    else
    {
     if( Hndl->shadow_end > Hndl->shadow_start )
      FENFORM_PrintShadow( Hndl );
     else
      PS_show_xy( Hndl->output_ps, Hndl->line_of_text, Hndl->start_col, Hndl->cur_line + Hndl->line_offset );
    }

}


void FENFORM_PS_SetFont( FF_Handle *Hndl )
{

    int font_id;

    if( Hndl->verbose )
    {
     syslog( LOG_INFO, "Setting New Font\n" );
     syslog( LOG_INFO, "  Font = %s  Size = %f", Hndl->curr_font, Hndl->frm.curr_char_width );
    }

    PS_setfont( Hndl->output_ps, Hndl->curr_font_id, 120 * Hndl->frm.curr_char_width / POINTS_PER_INCH );

    Hndl->font_set = FF_ON;

}


void FENFORM_PS_PrintShadow( FF_Handle *Hndl )
{

    float line,
          line_width;

    line_width = Hndl->frm.line_space * 0.8;
  
    line = ( Hndl->cur_line + line_width / 2 ) - ( line_width / _SUBSCRIPT_OFFSET );

    PS_setlinewidth( Hndl->output_ps, line_width );
    PS_setgray( Hndl->output_ps, SHADOW_DEPTH );
    PS_moveto( Hndl->output_ps, Hndl->shadow_start, line );
    PS_lineto( Hndl->output_ps, Hndl->shadow_end, line );
    PS_stroke( Hndl->output_ps );

    FENFORM_SetColour( Hndl );
    PS_setlinewidth( Hndl->output_ps, Hndl->cfg.line_width );
    PS_show_xy( Hndl->output_ps, Hndl->line_of_text, Hndl->start_col, Hndl->cur_line + Hndl->line_offset );

    Hndl->shadow_start = Hndl->cur_col;
    Hndl->shadow_end = 0;

}



void FENFORM_PS_PrintReverseImage( FF_Handle *Hndl )
{

    float line,
          line_width;

    line_width = Hndl->frm.line_space * 0.8;
  
    line = ( Hndl->cur_line + line_width / 2 ) - ( line_width / _SUBSCRIPT_OFFSET );

    PS_setlinewidth( Hndl->output_ps, line_width );
    PS_moveto( Hndl->output_ps, Hndl->ri_start, line );
    PS_lineto( Hndl->output_ps, Hndl->ri_end, line );
    PS_stroke( Hndl->output_ps );
    PS_setgray( Hndl->output_ps, 1 );

    PS_show_xy( Hndl->output_ps, Hndl->line_of_text, Hndl->start_col, Hndl->cur_line + Hndl->line_offset );

    FENFORM_SetColour( Hndl );
    PS_setlinewidth( Hndl->output_ps, Hndl->cfg.line_width );

    Hndl->ri_start = Hndl->cur_col;
    Hndl->ri_end = 0;

}


void FENFORM_PS_NewPage( FF_Handle *Hndl )
{

    /****************************/
    /* If "one line multi-size" */
    /* was selected, cancel it. */
    /****************************/

    if( Hndl->ms_one_line )
    {
     /* Cancel Multi Size */
     FENFORM_MultiSize( Hndl, 0 );
     Hndl->ms_one_line = FF_OFF;
    }

    FENFORM_PrintText( Hndl );
    PS_end_page( Hndl->output_ps );

    Hndl->new_page = FF_ON;

}


/*********************************************************************************/
/*                                                                               */
/*                             Postscript Module                                 */
/*                                                                               */
/*                      End of Source File - fenform_ps.c                        */
/*                                                                               */
/*********************************************************************************/
