/*********************************************************************************/
/*                                                                               */
/*  FenForm - Document Printer                                                   */
/*  Vn 2.0                                                                       */
/*                                                                               */
/*  Project Leader - Mike Eggleston                                              */
/*                                                                               */
/*  Copyright 2005 - 2008 Fenland Software Ltd                                   */
/*  Date: 28/08/2008                                                             */
/*                                                                               */
/*  Barcodes Module - fenform_bc.c                                               */
/*                                                                               */
/*  Barcode specific functions                                                   */
/*                                                                               */
/*********************************************************************************/
/*                                                                               */
/*  Licence: GPL Vn2                                                             */
/*                                                                               */
/*  This program is free software; you can redistribute it and/or modify         */
/*  it under the terms of the GNU General Public License as published by         */
/*  the Free Software Foundation; either version 2 of the License, or            */
/*  (at your option) any later version.                                          */
/*                                                                               */
/*  This program is distributed in the hope that it will be useful,              */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of               */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                */
/*  GNU General Public License for more details.                                 */
/*                                                                               */
/*  You should have received a copy of the GNU General Public License            */
/*  along with this program; if not, write to the Free Software                  */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    */
/*                                                                               */
/*                                                                               */
/*                                                                               */
/*********************************************************************************/

#include "fenform.h"

#define _RM4SCC_PITCH	24
#define _POSTNET_PITCH	22
#define _PLANET_PITCH	22


/* These are the representations of the digits 0-9 */
static char *postnet_chars[] = {
  "11000", "00011", "00101", "00110", "01001",
  "01010", "01100", "10001", "10010", "10100" };


/* These are the representations of the digits 0-9 */
static char *planet_chars[] = {
  "00111", "11100", "11010", "11001", "10110",
  "10101", "10011", "01110", "01101", "01011" };


static char rm4scc_alpha[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";


static char *rm4scc_chars[] = {
  "3300", "3210", "3201", "2310", "3201",
  "2211", "3120", "3030", "3021", "2130",
  "2121", "2031", "3102", "3012", "3003",
  "2112", "2103", "2013", "1320", "1230",
  "1221", "0330", "0321", "0231", "1302",
  "1212", "1203", "0312", "0303", "0213",
  "1122", "1032", "1023", "0132", "0123",
  "0033" };



/******************************/
/*  Functions                 */
/******************************/

float FENFORM_BC_PrintPostnetCharacter( FF_Handle *Hndl, char *code )
{
    char x;

    float start_h,
          height,
          pitch,
          width,
          code_length;

    int n;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Starting function 'print_postnet_char'" );

    fprintf( Hndl->ofile, "\n%% Printing Postnet/Planet barcode\n" );

    pitch = (float) ( ( POINTS_PER_INCH * 100 ) / _POSTNET_PITCH );
    n = (int) pitch;
    pitch = (float) n / 100.0;

    width = (float) ( ( POINTS_PER_INCH * 100 ) / ( _POSTNET_PITCH * 4 ) );
    n = (int) width;
    width = (float) n / 100.0;

    /* Check for illegal character */
    if( *code == '*' ) {

     code_length = ( strlen( code ) - 1 ) * pitch;
     height = POINTS_PER_INCH / 15.0;

     switch( Hndl->ofile_fmt ) {

      case PS:  fprintf( Hndl->ofile, "%.2f setlinewidth\nnewpath\n", height );
                fprintf( Hndl->ofile, "%f %f a\n", Hndl->cur_col, Hndl->cur_line );
                fprintf( Hndl->ofile, "%f %f l\n",  Hndl->cur_col + code_length, Hndl->cur_line );
                fprintf( Hndl->ofile, "closepath\nstroke\n" );
                break;
     }

     code_length += ( 3 * pitch );

    } else {

     fprintf( Hndl->ofile, "[\n%%  height  xpos   ypos  width\n" );

     start_h = Hndl->cur_col;

     while( x = *code++ ) {
      switch (x) {

       case '0': height = POINTS_PER_INCH / 20.0;
                  break;

       case '1': height = POINTS_PER_INCH / 10.0;
                  break;
      }

      fprintf( Hndl->ofile, "   [%5.2f %6.2f %6.2f %5.2f]\n", height, start_h, Hndl->cur_line, width );
      start_h += pitch;

     }

     fprintf( Hndl->ofile, "]       { {} forall setlinewidth moveto 0 exch rlineto stroke} bind forall\n" );
     code_length = start_h - Hndl->cur_col + 2 * pitch;
    }

    fprintf( Hndl->ofile, "%% End Postnet/Planet barcode\n\n" );

    return( code_length );

}


float FENFORM_BC_Print_RM4SCC_Character( FF_Handle *Hndl, char *code )
{
    char x;

    float start_h,
          start_v,
          height,
          centre,
          pitch,
          width,
          code_length;

    int n;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Starting function 'FENFORM_Print_RM4SCC_Character'" );

    fprintf( Hndl->ofile, "%% Printing RM4SCC barcode\n" );

    fprintf( Hndl->ofile, "[\n%%  height  xpos   ypos  width\n" );

    centre = Hndl->cur_line + 6.9;

    pitch = (float) ( (POINTS_PER_INCH * 100) / _RM4SCC_PITCH );
    n = (int) pitch;
    pitch = (float) n / 100.0;

    width = (float) ( (POINTS_PER_INCH * 100) / (_RM4SCC_PITCH * 4) );
    n = (int) width;
    width = (float) n / 100.0;

    start_h = Hndl->cur_col + 2 * pitch;

    while(x = *code++) {
     switch (x) {

      case '0': start_v = centre - 6.85;
                height = 13.7;
                break;

      case '1': start_v = centre - 1.70;
                height = 8.55;
                break;

      case '2': start_v = centre - 6.85;
                height = 8.55;
                break;

      case '3': start_v = centre - 1.70;
                height = 3.40;
                break;
     }

     fprintf( Hndl->ofile, "   [%5.2f %6.2f %6.2f %5.2f]\n", height, start_h, start_v, width );
     start_h += pitch;
    }

    fprintf( Hndl->ofile, "]       { {} forall setlinewidth moveto 0 exch rlineto stroke} bind forall\n" );
    fprintf( Hndl->ofile, "%% End RM4SCC barcode\n\n" );

    code_length = start_h - Hndl->cur_col + 2 * pitch;

    return( code_length );

}


char *FENFORM_BC_EncodePostnet( FF_Handle *Hndl, char *str)
{
    char x,
         *s,
         *code_string;

    int n,
        check_digit = 0;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Starting function 'encode_postnet'" );

    /* Allocate space for the code string                */
    /* Allow an extra 5 chars for the check digit plus   */
    /* 3 chars for the start, stop, and NULL terminator  */

    code_string = realloc( NULL, (5 * strlen(str) + 8) * sizeof(char));
    s = code_string;
  
    /* Set the first character to the "start bar" */
    *code_string++ = '1';

    while( x = toupper( *str++ ) ) {

     /* If an illegal character is found, set the first  */
     /* character of the code string to '*' but continue */
     /* processing to complete the string.               */
     if( ! isdigit( x ) ) {
      *s = '*';
      memcpy( code_string, "*****", 5 );

     } else {

      n = x - 48;  /* Convert char to int. */
      memcpy( code_string, postnet_chars[ n ], 5 );
      check_digit += n;
     }
     code_string += 5;
    }

    /* Calculate the check digit */
    check_digit = 10 - ( check_digit % 10 );

    /* Append check digit to code string */
    memcpy(code_string, postnet_chars[ check_digit ], 5 );
    code_string += 5;

    /* Append the "stop bar" to the code string */
    *code_string++ = '1';
    *code_string = '\0';

    return( s );

}


char *FENFORM_BC_EncodePlanet( FF_Handle *Hndl, char *str)
{
    char x,
         *s,
         *code_string;

    int n,
        check_digit = 0;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Starting function 'encode_planet'" );

    /* Allocate space for the code string                */
    /* Allow an extra 5 chars for the check digit plus   */
    /* 3 chars for the start, stop, and NULL terminator  */

    code_string = realloc( NULL, (5 * strlen(str) + 8) * sizeof(char));
    s = code_string;
  
    /* Set the first character to the "start bar" */
    *code_string++ = '1';

    while( x = toupper( *str++ ) ) {
     /* If an illegal character is found, set the first  */
     /* character of the code string to '*' but continue */
     /* processing to complete the string.               */
     if( ! isdigit( x ) ) {
      *s = '*';
      memcpy(code_string, "*****", 5);

     } else {

      n = x - 48;  /* Convert char to int. */
      memcpy( code_string, postnet_chars[ n ], 5 );
      check_digit += n;
     }
     code_string += 5;
    }

    /* Calculate the check digit */
    check_digit = 10 - ( check_digit % 10 );

    /* Append check digit to code string */
    memcpy( code_string, planet_chars[ check_digit ], 5 );
    code_string += 5;

    /* Append the "stop bar" to the code string */
    *code_string++ = '1';
    *code_string = '\0';

    return( s );

}


char *FENFORM_BC_EncodeRM4SCC( FF_Handle *Hndl, char *str )
{
    char x,
         *s,
         *code_string;

    int n,
        r = 0,
        c = 0;

    if( Hndl->verbose )
     syslog( LOG_INFO, "Starting function 'encode_rm4scc'" );

    /* Allocate space for the code string                */
    /* Allow an extra 4 chars for the check digit plus   */
    /* 3 chars for the start, stop, and NULL terminator  */

    code_string = realloc( NULL, ( 4 * strlen( str ) + 7 ) * sizeof( char ) );
    s = code_string;
  
    /* Set the first character to the "start bar" */
    *code_string++ = '1';

    while( x = toupper( *str++ ) ) {

     /* Ignore characters <= 32 */
     if( x <= 32 )
       continue;

     n = 0;
     while( n < 36 && rm4scc_alpha[ n ] != x )
      n++;

     /* Convert illegal characters to 'X' */
     if( n >= 36 )
      n = 33;

     c = c + (n + 1) % 6;
     r = r + n / 6 + 1;
     memcpy( code_string, rm4scc_chars[ n ], 4 );
     code_string += 4;
    }

    /* Append the check digit */
    c = c % 6;
    r = r / 6;
    n = r * 6 + c - 1;

    memcpy( code_string, rm4scc_chars[ n ], 4 );
    code_string += 4;
 
    /* Append the "stop bar" to the code string */
    *code_string++ = '0';
    *code_string = '\0';

    return( s );

}


float FENFORM_BC_PrintRM4SCC( FF_Handle *Hndl, char *str )
{
    return FENFORM_BC_Print_RM4SCC_Character( Hndl, FENFORM_BC_EncodeRM4SCC( Hndl, str ) );
}


float FENFORM_BC_PrintPostnet( FF_Handle *Hndl, char *str )
{
    return FENFORM_BC_PrintPostnetCharacter( Hndl, FENFORM_BC_EncodePostnet( Hndl, str ) );
}


float FENFORM_BC_PrintPlanet( FF_Handle *Hndl, char *str )
{
    return FENFORM_BC_PrintPostnetCharacter( Hndl, FENFORM_BC_EncodePlanet( Hndl, str ) );
}



/* Two dimensional barcode (PDF417) */

float FENFORM_BC_PrintPDF417( FF_Handle *Hndl, char *str )
{
    int cols,
        k;

    pdf417param p;

    pdf417init( &p );

    p.text = str;
    p.options = PDF417_INVERT_BITMAP;
    paintCode( &p );

    if ( p.error ) {
     pdf417free( &p );
     return 0.0;
    }

    fprintf( Hndl->ofile, "\n%% Printing PDF417 barcode\n" );

    cols = p.bitColumns / 8 + 1;

    fprintf( Hndl->ofile, "save\nstroke\n%f %f translate\n%g %g scale\n", Hndl->cur_col, Hndl->cur_line, p.bitColumns/2.0, p.codeRows * 3/2.0);
    fprintf( Hndl->ofile, "%d %d 1 [%d 0 0 %d 0 %d]{<", p.bitColumns, p.codeRows, p.bitColumns, -p.codeRows, p.codeRows);

    for( k = 0; k < p.lenBits; ++k ) {
     if ( ! ( k % cols ) )
      fprintf( Hndl->ofile, "\n" );
     fprintf( Hndl->ofile, "%02X", (int)p.outBits[ k ] & 0xff );
    }

    fprintf( Hndl->ofile, "\n>}image\nrestore\n" );

    fprintf( Hndl->ofile, "%% End PDF417 barcode\n\n\n" );
    
    pdf417free( &p );

    return( p.bitColumns );

}


void FENFORM_BC_InvalidBarcode( FF_Handle *Hndl )
{

    if( Hndl->verbose )
     syslog( LOG_INFO, "Invalid Barcode data found" );

    switch( Hndl->ofile_fmt ) {

     case PS:  FENFORM_BC_InvalidBarcode_PS( Hndl );
               break;
    }

    fprintf( Hndl->ofile, "%% End Invalid Barcode\n\n" );

}


void FENFORM_BC_InvalidBarcode_PS( FF_Handle *Hndl )
{

    float x,
          y;

    x = Hndl->cur_col + BARCODE_DEFAULT_MARGIN;
    y = Hndl->cur_line;

    PS_setlinewidth( Hndl->output_ps, 1 );

    PS_rect( Hndl->output_ps, x, y,
                              Hndl->bc_parm.width + BARCODE_DEFAULT_MARGIN,
                              Hndl->bc_parm.height );

    PS_stroke( Hndl->output_ps );
    PS_moveto( Hndl->output_ps, x, y );
    PS_lineto( Hndl->output_ps, x + Hndl->bc_parm.width + BARCODE_DEFAULT_MARGIN, y + Hndl->bc_parm.height );
    PS_stroke( Hndl->output_ps );
    PS_moveto( Hndl->output_ps, x, y + Hndl->bc_parm.height );
    PS_lineto( Hndl->output_ps, x + Hndl->bc_parm.width + BARCODE_DEFAULT_MARGIN, y );
    PS_stroke( Hndl->output_ps );

}


/*********************************************************************************/
/*                                                                               */
/*                                Barcodes Module                                */
/*                                                                               */
/*                         End of Source File fenform_bc.c                       */
/*                                                                               */
/*********************************************************************************/

