/*********************************************************************************/
/*                                                                               */
/*  FenForm - Document Printer                                                   */
/*  Vn 2.0                                                                       */
/*                                                                               */
/*  Project Leader - Mike Eggleston                                              */
/*                                                                               */
/*  Copyright 2005 - 2008 Fenland Software Ltd                                   */
/*  Date: 28/08/2008                                                             */
/*                                                                               */
/*  Main Module - fenform.c                                                      */
/*                                                                               */
/*********************************************************************************/
/*                                                                               */
/*  Licence: GPL Vn2                                                             */
/*                                                                               */
/*  This program is free software; you can redistribute it and/or modify         */
/*  it under the terms of the GNU General Public License as published by         */
/*  the Free Software Foundation; either version 2 of the License, or            */
/*  (at your option) any later version.                                          */
/*                                                                               */
/*  This program is distributed in the hope that it will be useful,              */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of               */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                */
/*  GNU General Public License for more details.                                 */
/*                                                                               */
/*  You should have received a copy of the GNU General Public License            */
/*  along with this program; if not, write to the Free Software                  */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    */
/*                                                                               */
/*                                                                               */
/*                                                                               */
/*********************************************************************************/

#include "fenform.h"
#include <getopt.h>


/******************************/
/*  Miscellaneous Functions   */
/******************************/

void FENFORM_VersionInfo( void )
{

    fprintf(stderr, "\n%s\n%s\n%s\n\n", PROGNAME, COPYRIGHT, LICENCE);
    fprintf(stderr, "Commercial support for this software is available from\n");
    fprintf(stderr, "%s    %s\n\n\n", COMPANY, WEB_SITE);

}


void FENFORM_ShowConfiguration( FF_Handle *Hndl )
{

  char *error = "***ERROR***",
       *no_error = " ",
       *error_flag;

  fprintf( stderr, "\n---------------------------------\n" );
  fprintf( stderr, " FenForm - Current Configuration\n" );
  fprintf( stderr, "---------------------------------\n" );

  fprintf( stderr, "Files & Directories -\n" );
  fprintf( stderr, "  Fontmap:          %s\n", Hndl->cfg.font_map );
  fprintf( stderr, "  Forms Directory:  %s\n", Hndl->cfg.forms_dir );
  fprintf( stderr, "  Paper Size File:  %s\n", Hndl->cfg.paper_size_file );

  fprintf( stderr, "\nMedia Parameters -\n" );

  error_flag = no_error;
  if( ( Hndl->cfg.paper_length < 0 ) || ( Hndl->cfg.paper_width < 0 ) )
   error_flag = error;

  fprintf( stderr, "  Paper Size:       %s  %s\n", Hndl->cfg.paper_size, error_flag );
  fprintf( stderr, "  Orientation:      %c\n", Hndl->cfg.paper_orientation );
  fprintf( stderr, "  Paper Length:     %f  %s\n", Hndl->cfg.paper_length, error_flag );
  fprintf( stderr, "  Paper Width:      %f  %s\n", Hndl->cfg.paper_width, error_flag );

  fprintf( stderr, "\nFonts -\n" );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.std_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Standard:         %s  %s\n", Hndl->cfg.std_font, error_flag );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Bold:             %s  %s\n", Hndl->cfg.bold_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Double Strike:    %s  %s\n", Hndl->cfg.dblstrike_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Italic:           %s  %s\n", Hndl->cfg.italic_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Bold Italic:      %s  %s\n", Hndl->cfg.bold_italic_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  D/Strike Italic:  %s  %s\n", Hndl->cfg.dblstrike_italic_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.nlq_font ) )
   error_flag = no_error;
  fprintf( stderr, "  NLQ:              %s  %s\n", Hndl->cfg.nlq_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_nlq_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Bold NLQ:         %s  %s\n", Hndl->cfg.bold_nlq_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_nlq_font ) )
   error_flag = no_error;
  fprintf( stderr, "  D/Strike NLQ:     %s  %s\n", Hndl->cfg.dblstrike_nlq_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.nlq_italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  NLQ Italc:        %s  %s\n", Hndl->cfg.nlq_italic_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.bold_nlq_italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  Bold NLQ Italic:  %s  %s\n", Hndl->cfg.bold_nlq_italic_font, error_flag  );

  error_flag = error;
  if( FENFORM_PS_FindFontFile( Hndl, Hndl->cfg.dblstrike_nlq_italic_font ) )
   error_flag = no_error;
  fprintf( stderr, "  D/S NLQ Italic:   %s  %s\n", Hndl->cfg.dblstrike_nlq_italic_font, error_flag  );

  fprintf( stderr, "\nMiscellaneous -\n" );
  
  if( Hndl->cfg.auto_lf )
   fprintf( stderr, "  Auto. LF:         Y\n" );
  else
   fprintf( stderr, "  Auto. LF:         N\n" );

  if( Hndl->cfg.full_colour )
   fprintf( stderr, "  Full colour:      Y\n" );
  else
   fprintf( stderr, "  Full colour:      N\n" );

  if( Hndl->cfg.allow_override )
   fprintf( stderr, "  User Override:    Y\n" );
  else
   fprintf( stderr, "  User Override:    N\n" );

  fprintf( stderr, "\n---------------------------------\n" );
  fprintf( stderr, " FenForm - End of Configuration\n" );
  fprintf( stderr, "---------------------------------\n" );

}


/*************************************************************/
/* Initialize the "printer".  This sets the state of Fenform */
/* to that of a printer immediately after power-up.          */
/*************************************************************/

void FENFORM_Initialize( FF_Handle *Hndl )
{

  int  n,
       tabpos = 0;
  char *x;

  if( Hndl->verbose ) {
   syslog( LOG_INFO, "Initializing Virtual Printer\n" );
  }

  Hndl->curr_font = Hndl->cfg.std_font;
  Hndl->curr_font_id = Hndl->cfg.std_font_id;
  Hndl->font_flags = FF_OFF;
  Hndl->h_tab = 0;
  Hndl->v_tab = 0;
  Hndl->curr_vtab_channel = 0;
  Hndl->cur_line = Hndl->frm.length - Hndl->frm.line_space;
  Hndl->start_col = Hndl->frm.l_margin;
  Hndl->cur_col = Hndl->start_col;
  Hndl->text_mode = FF_OFF;
  Hndl->multi_size = FF_OFF;
  Hndl->ms_one_line = FF_OFF;
  Hndl->LF_reverse = FF_OFF;
  Hndl->red = 0;
  Hndl->green = 0;
  Hndl->blue = 0;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Virtual Printer Initialized\n" );

}


/****************************************************/
/* Read and ignore characters                       */
/* This function effects the "print suppress mode"  */
/* by just reading bytes until a DC1 code (cancel   */ 
/* print suppress) is read.                         */
/****************************************************/

void FENFORM_PrintSuppress( FF_Handle *Hndl )
{

  if( Hndl->verbose )
   syslog( LOG_INFO, "Printing Suppressed\n" );

  while( fgetc( Hndl->ifile ) != DC1 );

}


void FENFORM_DefaultForm( FF_Handle *Hndl )
{

  if( Hndl->verbose )
   syslog( LOG_INFO, "Setting Default Form Parameters\n" );

  /* Page parameters */
  Hndl->frm.form_name = DFT_FORM_NAME;
  Hndl->frm.orientation = Hndl->cfg.paper_orientation;
  Hndl->frm.length = Hndl->cfg.paper_length;
  Hndl->frm.width = Hndl->cfg.paper_width;
  Hndl->frm.l_margin = 0;
  Hndl->frm.r_margin = Hndl->frm.width;
  Hndl->frm.line_space = LPI6;
  Hndl->frm.base_char_width = CPI10;
  Hndl->frm.curr_char_width = Hndl->frm.base_char_width;

}


int FENFORM_StartPage( FF_Handle *Hndl )
{
  int rtn = 0;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting Page %d\n", Hndl->page );

  /* Set ready for first line */
  switch( Hndl->ofile_fmt ) {

   case PS:  
             FENFORM_PS_StartPage( Hndl );
             break;

  }

  return( rtn );

}




/******************************************************************/
/* Read numeric (no_parms) parameters from file.                  */  
/* if no_parms = 0, keep reading until NULL or '\' or (max) read  */
/******************************************************************/

int FENFORM_ReadParameters( FF_Handle *Hndl, int *parms, int no_parms, int max )
{

  unsigned int value;
  int x,
      y;

  if( no_parms )
    max = no_parms;

  for( y = 0; y < max; y++ )
  {
    x = fgetc( Hndl->ifile );
    if( no_parms == 0 && x == '\0' )
      break;
    parms[ y ] = x;
  }
  return( y );  /* Return the number of parameters read */

}



/******************************/
/*  Print Control Functions   */
/******************************/

void FENFORM_PrintText( FF_Handle *Hndl )
{

  if( ( Hndl->text_mode == FF_ON ) && ( Hndl->line_of_text ) ) {

   switch( Hndl->ofile_fmt ) {

    case PS:  FENFORM_PS_PrintText( Hndl );
              break;

   }
  }

  Hndl->text_mode = FF_OFF;

}


void FENFORM_PrintShadow( FF_Handle *Hndl )
{

  switch( Hndl->ofile_fmt ) {

   case PS:  FENFORM_PS_PrintShadow( Hndl );
             break;

  }

}


void FENFORM_PrintReverseImage( FF_Handle *Hndl )
{

  switch( Hndl->ofile_fmt ) {

   case PS:  FENFORM_PS_PrintReverseImage( Hndl );
             break;

  }

}


void FENFORM_SetFont( FF_Handle *Hndl )
{

  switch( Hndl->ofile_fmt ) {

   case PS:  FENFORM_PS_SetFont( Hndl );
             break;

  }

}


void FENFORM_SetColour( FF_Handle *Hndl )
{
  float colour;

  colour = Hndl->red + Hndl->green + Hndl->blue;

  if( Hndl->cfg.full_colour )
   fprintf( Hndl->ofile, "%% Full Colour Mode\n" );

  if( Hndl->cfg.full_colour && ( colour > 0.0 ) )
   fprintf( Hndl->ofile, "%0.2f %0.2f %0.2f setrgbcolor\n", Hndl->red, Hndl->green, Hndl->blue);
  else
   fprintf( Hndl->ofile, "0.0 setgray\n" );

}


void FENFORM_HeightModulatedBarcode( FF_Handle *Hndl, char type)
{

  int param,
      code_chars;
  float code_length = 0;
  char *text;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting function 'FENFORM_HeightModulatedBarcode' param %c", type);

  if( FENFORM_ReadParameters( Hndl, &param, 1, 0 ) == 1 )
  {
   /* Allocate space and read the text to be converted */ 
   text = malloc( ( size_t ) ( param + 1 ) * sizeof( char ) );
   fgets( text, param + 1, Hndl->ifile );

   switch( type )
   {
    case 'C': code_length = FENFORM_BC_PrintRM4SCC( Hndl, text );
              break;

    case 'D': code_length = FENFORM_BC_PrintPostnet( Hndl, text );
              break;

    case 'E': code_length = FENFORM_BC_PrintPlanet( Hndl, text );
              break;

    case 'F': code_length = FENFORM_BC_PrintPDF417( Hndl, text );
              break;
   }

   code_chars = code_length / Hndl->frm.curr_char_width + 1;
   code_length = code_chars * Hndl->frm.curr_char_width;
   Hndl->cur_col += code_length;

   free( text );

  }

}


void FENFORM_SetBarcode( FF_Handle *Hndl )
{

  int rv,
      param[ 3 ];

  Hndl->bc_parm.flags = BARCODE_OUT_PS | BARCODE_OUT_NOHEADERS | BARCODE_NO_ASCII;

  if( FENFORM_ReadParameters( Hndl, param, 3, 0 ) == 3 )
  {
    Hndl->bc_parm.flags |= (unsigned int)param[ 0 ];
    Hndl->bc_parm.height = (unsigned int)param[ 1 ];
    Hndl->bc_parm.width = (unsigned int)param[ 2 ];
  }

}


void FENFORM_PrintBarcode( FF_Handle *Hndl )
{
  char *text;
  int x,
      y,
      text_length,
      code_chars,
      error_flag = 0;
  float code_length;

  x = ( int )Hndl->cur_col;
  y = ( int )Hndl->cur_line - BARCODE_DEFAULT_MARGIN;

  fprintf( Hndl->ofile, "\n\n" );

  if( FENFORM_ReadParameters( Hndl, &text_length, 1, 1 ) )
  {
   text = malloc( ( size_t ) ( text_length + 1 ) * sizeof( char ) );
   fgets( text, text_length + 1, Hndl->ifile );

   if( ( 2 * BARCODE_DEFAULT_MARGIN + Hndl->bc_parm.width + x ) > Hndl->frm.r_margin )
    error_flag = -9;
   else
    error_flag = Barcode_Encode_and_Print( text, Hndl->ofile, Hndl->bc_parm.width,
                                          Hndl->bc_parm.height, x, y, Hndl->bc_parm.flags);

   if(error_flag < 0)
   {
    /* Invalid Barcode */
    FENFORM_BC_InvalidBarcode( Hndl );
   }

   FENFORM_SetColour( Hndl );
   fprintf( Hndl->ofile, "%0.2f setlinewidth\n", Hndl->cfg.line_width );

   free( text );
  }

  Hndl->font_set = FF_OFF;

  code_length = 2 * BARCODE_DEFAULT_MARGIN + Hndl->bc_parm.width;
  code_chars = code_length / Hndl->frm.curr_char_width + 1;
  code_length = code_chars * Hndl->frm.curr_char_width;
  Hndl->cur_col += code_length;

}


int FENFORM_ESC_error( int action )
{

  syslog( LOG_INFO, "Invalid ESC Code (%c) found\n", action );

  return( 0 );

}


int FENFORM_FS_error( int action )
{

  syslog( LOG_INFO, "Invalid FS Code (%c) found\n", action );

  return( 0 );

}


void FENFORM_PrintUnderline( FF_Handle *Hndl )
{

  float line;

  if( Hndl->ul_end > Hndl->ul_start )
  {
   line = Hndl->cur_line - Hndl->frm.line_space / _UNDERLINE_OFFSET;

   fprintf( Hndl->ofile, "%0.2f %0.2f %0.2f %0.2f ls\n", Hndl->ul_start, line, Hndl->ul_end, line );

   Hndl->ul_start = Hndl->cur_col;
   Hndl->ul_end = 0;
  }

}


void FENFORM_PrintOverscore( FF_Handle *Hndl )
{

  float line;

  if( Hndl->os_end > Hndl->os_start )
  {
   line = Hndl->cur_line + Hndl->frm.line_space / _OVERSCORE_OFFSET;

   fprintf( Hndl->ofile, "%0.2f %0.2f %0.2f %0.2f ls\n", Hndl->os_start, line, Hndl->os_end, line );

   Hndl->os_start = Hndl->cur_col;
   Hndl->os_end = 0;
  }

}


void FENFORM_NewPage( FF_Handle *Hndl )
{

  switch( Hndl->ofile_fmt ) {

   case PS:  FENFORM_PS_NewPage( Hndl );
             break;

  }

}


void FENFORM_Backspace( FF_Handle *Hndl )
{

  if( Hndl->cur_col > Hndl->frm.curr_char_width )
  {
   Hndl->start_col = Hndl->cur_col - Hndl->frm.curr_char_width;
   Hndl->cur_col = Hndl->start_col;
  }

}


void FENFORM_LineFeed( FF_Handle *Hndl, unsigned int lf )
{

  Hndl->h_tab = 0;
  Hndl->start_col = Hndl->frm.l_margin;
  Hndl->cur_col = Hndl->start_col;

  if( lf == FF_ON || Hndl->cfg.auto_lf )
  {
   if( Hndl->LF_reverse == FF_OFF )
   {
    /* Standard (forward) line feed */
    if( Hndl->cur_line >= Hndl->frm.line_space )
     Hndl->cur_line -= Hndl->frm.line_space;
    else
     Hndl->cur_line = 0;
   }
   else
   {
    /* Reverse line feed */
    if( Hndl->cur_line <= ( Hndl->frm.length - Hndl->frm.line_space ) )
     Hndl->cur_line += Hndl->frm.line_space;
    else
     Hndl->cur_line = Hndl->frm.length - Hndl->frm.line_space;
   }
  }

  if( Hndl->font_flags & FF_UNDERLINE_FLAG )
   Hndl->ul_start = Hndl->cur_col;

  if( Hndl->font_flags & FF_OVERSCORE_FLAG )
   Hndl->os_start = Hndl->cur_col;

  if( Hndl->font_flags & FF_SHADOW_FLAG )
   Hndl->shadow_start = Hndl->cur_col;

  if( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG )
   Hndl->ri_start = Hndl->cur_col;


  /****************************/
  /* If "one line multi-size" */
  /* was selected, cancel it. */
  /****************************/

  if( Hndl->ms_one_line )
  {
   /* Cancel Multi Size */
   FENFORM_MultiSize( Hndl, 0 );
   Hndl->ms_one_line = FF_OFF;
  }

}


void FENFORM_Absolute_H_Tab( FF_Handle *Hndl, int *param )
{

  float pos;

  /* Calculate 'pos' in inches */
  pos = (float)( ( 256.0 * param[ 1 ] + param[ 0 ] ) / 60.0 );

  /* Convert to 'points' */
  pos *= POINTS_PER_INCH;
  
  /* Add the left margin */
  pos += Hndl->frm.l_margin;

  if( pos < Hndl->frm.r_margin )
  {
   Hndl->cur_col = pos ;
   Hndl->start_col = Hndl->cur_col;
  }

}


void FENFORM_HorizontalTab( FF_Handle *Hndl )
{

  if( Hndl->cur_col < Hndl->frm.width )
  {
   while( Hndl->frm.htabs[ Hndl->h_tab ] <= Hndl->cur_col )
    Hndl->h_tab++;

   Hndl->cur_col = Hndl->frm.htabs[ Hndl->h_tab ];
   Hndl->start_col = Hndl->cur_col;

   Hndl->h_tab++;
  }

}


void FENFORM_VerticalTab( FF_Handle *Hndl )
{

  if( Hndl->verbose ) {
   syslog( LOG_INFO, "Start FENFORM_VerticalTab");
   syslog( LOG_INFO, "cur_line %f, curr_vtab_channel %d, v_tab %d\n",
        Hndl->cur_line, Hndl->curr_vtab_channel, Hndl->v_tab );
  }

  if( Hndl->cur_line < ( Hndl->frm.vtabs[ Hndl->curr_vtab_channel ] [ Hndl->v_tab ] ) )
   Hndl->cur_line = 0;
  else
  {
   Hndl->cur_line = Hndl->frm.vtabs[ Hndl->curr_vtab_channel ] [ Hndl->v_tab ];
   Hndl->v_tab++;
   Hndl->h_tab = 0;
   Hndl->start_col = Hndl->frm.l_margin;
   Hndl->cur_col = Hndl->start_col;
  }

  if( Hndl->verbose )
   syslog( LOG_INFO, "End FENFORM_VerticalTab");

}


void FENFORM_SetItalics( FF_Handle *Hndl, int set )
{

  if( set == FF_ON )
  {
   if( Hndl->verbose )
    syslog( LOG_INFO, "Set Italics");

   Hndl->font_flags |= FF_ITALICS_FLAG;
   Hndl->curr_font = Hndl->cfg.italic_font;
   Hndl->curr_font_id = Hndl->cfg.italic_font_id;

   if( Hndl->font_flags & FF_BOLD_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_italic_font_id;
   }

   if( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_italic_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_italic_font_id;
   }

   if( ( Hndl->font_flags & FF_BOLD_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_italic_font_id;
   }

   if( ( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_italic_font_id;
   }

  } else {

   if( Hndl->verbose )
    syslog( LOG_INFO, "Cancel Italics");

   Hndl->font_flags &= ~FF_ITALICS_FLAG;
   Hndl->curr_font = Hndl->cfg.std_font;
   Hndl->curr_font_id = Hndl->cfg.std_font_id;

   if( Hndl->font_flags & FF_BOLD_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_font;
    Hndl->curr_font_id = Hndl->cfg.bold_font_id;
   }

   if( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.nlq_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_BOLD_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_font_id;
   }
  }

  Hndl->font_set = FF_OFF;

}


void FENFORM_SetBoldFont( FF_Handle *Hndl, int set )
{

  if( set == FF_ON )
  {
   if( Hndl->verbose )
    syslog( LOG_INFO, "Set Bold Font" );

   Hndl->font_flags |= FF_BOLD_FLAG;
   Hndl->curr_font = Hndl->cfg.bold_font;
   Hndl->curr_font_id = Hndl->cfg.bold_font_id;

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_italic_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_ITALICS_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_italic_font_id;
   }

  } else {

   if( Hndl->verbose )
    syslog( LOG_INFO, "Cancel Bold Font" );

   Hndl->font_flags &= ~FF_BOLD_FLAG;
   Hndl->curr_font = Hndl->cfg.std_font;
   Hndl->curr_font_id = Hndl->cfg.std_font_id;

   if( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_font_id;
   }

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.italic_font;
    Hndl->curr_font_id = Hndl->cfg.italic_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.nlq_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_ITALICS_FLAG )
    && ( Hndl->font_flags & FF_NLQ_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_italic_font_id;
   }
  }

  Hndl->font_set = FF_OFF;

}


void FENFORM_SetDoubleStrikeFont( FF_Handle *Hndl, int set )
{

  if( set == FF_ON )
  {
   if( Hndl->verbose )
    syslog( LOG_INFO, "Set Double Strike" );

   Hndl->font_flags |= FF_DOUBLESTRIKE_FLAG;
   Hndl->curr_font = Hndl->cfg.dblstrike_font;
   Hndl->curr_font_id = Hndl->cfg.dblstrike_font_id;

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_italic_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_NLQ_FLAG )
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_italic_font_id;
   }

  } else {

   if( Hndl->verbose )
    syslog( LOG_INFO, "Cancel Double Strike" );

   Hndl->font_flags &= ~FF_DOUBLESTRIKE_FLAG;
   Hndl->curr_font = Hndl->cfg.std_font;
   Hndl->curr_font_id = Hndl->cfg.std_font_id;

   if( Hndl->font_flags & FF_BOLD_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_font;
    Hndl->curr_font_id = Hndl->cfg.bold_font_id;
   }

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.italic_font;
    Hndl->curr_font_id = Hndl->cfg.italic_font_id;
   }

   if( Hndl->font_flags & FF_NLQ_FLAG ) {
    Hndl->curr_font = Hndl->cfg.nlq_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_NLQ_FLAG )
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_italic_font_id;
   }
  }

  Hndl->font_set = FF_OFF;

}


void FENFORM_SetNLQFont( FF_Handle *Hndl, int set )
{

  if( set == FF_ON )
  {
   if( Hndl->verbose )
    syslog( LOG_INFO, "Set NLQ Font");

   Hndl->font_flags |= FF_NLQ_FLAG; 
   Hndl->curr_font = Hndl->cfg.nlq_font;
   Hndl->curr_font_id = Hndl->cfg.nlq_font_id;

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.nlq_italic_font_id;
   }

   if( Hndl->font_flags & FF_BOLD_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_font_id;
   }

   if( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_font_id;
   }

   if( ( Hndl->font_flags & FF_BOLD_FLAG ) 
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.bold_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_nlq_italic_font_id;
   }

   if( ( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) 
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_nlq_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_nlq_italic_font_id;
   }

  } else {

   if( Hndl->verbose )
    syslog( LOG_INFO, "Cancel NLQ Font" );

   Hndl->font_flags &= ~FF_NLQ_FLAG;
   Hndl->curr_font = Hndl->cfg.std_font;
   Hndl->curr_font_id = Hndl->cfg.std_font_id;

   if( Hndl->font_flags & FF_ITALICS_FLAG ) {
    Hndl->curr_font = Hndl->cfg.italic_font;
    Hndl->curr_font_id = Hndl->cfg.italic_font_id;
   }

   if( Hndl->font_flags & FF_BOLD_FLAG ) {
    Hndl->curr_font = Hndl->cfg.bold_font;
    Hndl->curr_font_id = Hndl->cfg.bold_font_id;
   }

   if( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_font_id;
   }

   if( ( Hndl->font_flags & FF_BOLD_FLAG )
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.bold_italic_font;
    Hndl->curr_font_id = Hndl->cfg.bold_italic_font_id;
   }

   if( ( Hndl->font_flags & FF_DOUBLESTRIKE_FLAG ) 
    && ( Hndl->font_flags & FF_ITALICS_FLAG ) ) {
    Hndl->curr_font = Hndl->cfg.dblstrike_italic_font;
    Hndl->curr_font_id = Hndl->cfg.dblstrike_italic_font_id;
   }
  }

  Hndl->font_set = FF_OFF;

}


void FENFORM_SetCharacterWidth( FF_Handle *Hndl, float width )
{

  Hndl->prev_char_width = Hndl->frm.base_char_width;
  Hndl->frm.base_char_width = width;
  Hndl->frm.curr_char_width = Hndl->frm.base_char_width;

  if( Hndl->multi_size > 0 )
   Hndl->frm.curr_char_width *= (double) Hndl->multi_size;

  FENFORM_PrintText( Hndl );
  Hndl->font_set = FF_OFF;

}


void FENFORM_MultiSize( FF_Handle *Hndl, int set )
{

  if( set )
  {
   /***************************************/
   /* Don't allow multi-size to be nested */
   /* Reset the character width first     */
   /***************************************/
   if( Hndl->multi_size != FF_OFF )
    Hndl->frm.curr_char_width = Hndl->frm.base_char_width;

   /*********************************************/
   /* 1 activates double size, anything greater */
   /* than 1 is the multiplication factor       */
   /*********************************************/
   Hndl->multi_size = (set == 1) ? 2 : set;
   Hndl->frm.base_char_width = Hndl->frm.curr_char_width;
   Hndl->frm.curr_char_width *= (float) Hndl->multi_size;

  } else {

   Hndl->multi_size = FF_OFF;
   Hndl->frm.curr_char_width = Hndl->frm.base_char_width;
  }

  FENFORM_PrintText( Hndl );
  Hndl->font_set = FF_OFF;

}


void FENFORM_SetSub_SuperScript( FF_Handle *Hndl, int sub_super )
{

  FENFORM_PrintText( Hndl );

  Hndl->frm.curr_char_width /= _SUB_SUPER_SIZE;
  Hndl->font_set = FF_OFF;

  if( sub_super == 0 )
  {
   Hndl->sub_superscript = SUPERSCRIPT;
   Hndl->line_offset = Hndl->frm.line_space / _SUPERSCRIPT_OFFSET;

  } else {

   Hndl->sub_superscript = SUBSCRIPT;
   Hndl->line_offset = ( -1 * Hndl->frm.line_space / _SUBSCRIPT_OFFSET );
  }

}


void FENFORM_CancelSub_SuperScript( FF_Handle *Hndl )
{

  FENFORM_PrintText( Hndl );

  Hndl->frm.curr_char_width *= _SUB_SUPER_SIZE;
  Hndl->font_set = FF_OFF;

  Hndl->line_offset = 0;
  Hndl->sub_superscript = FF_OFF;

}


void FENFORM_SetUnderline( FF_Handle *Hndl, int set )
{

  if( set == FF_ON ) {
   if( ! ( Hndl->font_flags & FF_UNDERLINE_FLAG ) ) {
    Hndl->ul_start = Hndl->cur_col;
    Hndl->font_flags |= FF_UNDERLINE_FLAG;
   }

  } else {

   Hndl->font_flags &= ~FF_UNDERLINE_FLAG;

  }

}


void FENFORM_SetOverscore( FF_Handle *Hndl, int set )
{

  if( set == FF_ON ) {
   if( ! ( Hndl->font_flags & FF_OVERSCORE_FLAG ) ) {
    Hndl->os_start = Hndl->cur_col;
    Hndl->font_flags |= FF_OVERSCORE_FLAG;
   }

  } else {

   Hndl->font_flags &= ~FF_OVERSCORE_FLAG;

  }

}


void FENFORM_SetShadow( FF_Handle *Hndl, int set )
{

  if( set == FF_ON )
  {
   /***********************************/ 
   /* Shadow and Reverse Image cannot */
   /* both be on at the same time     */
   /***********************************/
   if( ( ! ( Hndl->font_flags & FF_SHADOW_FLAG ) )
    && ( ! ( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG ) ) )
   {
    Hndl->shadow_start = Hndl->cur_col;
    Hndl->font_flags |= FF_SHADOW_FLAG;
   }

  } else {

   Hndl->font_flags &= ~FF_SHADOW_FLAG;

  }

}


void FENFORM_SetColourValue( FF_Handle *Hndl, int colour )
{

  int rgb[ 3 ];

  FENFORM_PrintText( Hndl );

  /* Codes 0 - 6 give coarse colour control */
  switch( colour )
  {
   case 0:   /* Black */
             Hndl->red = 0.0;
             Hndl->green = 0.0;
             Hndl->blue = 0.0;
             break;

   case 1:   /* Red */
             Hndl->red = (float) (( _COLOUR_RED / 0x10000 ) / 255.0 );
             Hndl->green = (float) (((_COLOUR_RED / 0x100) % 0x100) / 255.0 );
             Hndl->blue = (float) ((_COLOUR_RED % 0x100) / 255.0);
             break;

   case 2:   /* Blue */
             Hndl->red = (float) ((_COLOUR_BLUE / 0x10000) / 255.0);
             Hndl->green = (float) (((_COLOUR_BLUE / 0x100) % 0x100) / 255.0);
             Hndl->blue = (float) ((_COLOUR_BLUE % 0x100) / 255.0);
             break;
  
   case 3:   /* Violet */
             Hndl->red = (float) ((_COLOUR_VIOLET / 0x10000) / 255.0);
             Hndl->green = (float) (((_COLOUR_VIOLET / 0x100) % 0x100) / 255.0);
             Hndl->blue = (float) ((_COLOUR_VIOLET % 0x100) / 255.0);
             break;
  
   case 4:   /* Yellow */
             Hndl->red = (float) ((_COLOUR_YELLOW / 0x10000) / 255.0);
             Hndl->green = (float) (((_COLOUR_YELLOW / 0x100) % 0x100) / 255.0);
             Hndl->blue = (float) ((_COLOUR_YELLOW % 0x100) / 255.0);
             break;
  
   case 5:   /* Orange */
             Hndl->red = (float) ((_COLOUR_ORANGE / 0x10000) / 255.0);
             Hndl->green = (float) (((_COLOUR_ORANGE / 0x100) % 0x100) / 255.0);
             Hndl->blue = (float) ((_COLOUR_ORANGE % 0x100) / 255.0);
             break;
  
   case 6:   /* Green */
             Hndl->red = (float) ((_COLOUR_GREEN / 0x10000) / 255.0);
             Hndl->green = (float) (((_COLOUR_GREEN / 0x100) % 0x100) / 255.0);
             Hndl->blue = (float) ((_COLOUR_GREEN % 0x100) / 255.0);
             break;
  
   case 255: /* Fine colour control */
             FENFORM_ReadParameters( Hndl, rgb, 3, 3 );
             Hndl->red = (float)rgb[ 0 ] / 255.0;
             Hndl->green = (float)rgb[ 1 ] / 255.0;
             Hndl->blue = (float)rgb[ 2 ] / 255.0;
             break;

   default:  /* Black */
             Hndl->red = 0.0;
             Hndl->green = 0.0;
             Hndl->blue = 0.0;
  
  }

  FENFORM_SetColour( Hndl );

}


void FENFORM_SetPrintMode( FF_Handle *Hndl, int modes )
{

  if( Hndl->verbose )
   syslog( LOG_INFO, "Setting print mode - %d", modes );

  /*******************************************************/
  /* "Bold" and "Double Strike" cannot both be selected  */
  /* If this is attempted, the later selection will      */
  /* override the first.                                 */
  /*******************************************************/

  /* First, clear everything */
  FENFORM_SetCharacterWidth( Hndl, CPI10 );
  FENFORM_SetNLQFont( Hndl, FF_OFF );
  FENFORM_SetBoldFont( Hndl, FF_OFF );
  FENFORM_SetDoubleStrikeFont( Hndl, FF_OFF );
  FENFORM_MultiSize( Hndl, 0 );
  FENFORM_SetItalics( Hndl, FF_OFF );
  FENFORM_SetUnderline( Hndl, FF_OFF );
  FENFORM_SetOverscore( Hndl, FF_OFF );

  /* Now set the selected options */

  /* 12 CPI */
  if( modes & '\x01' )
   FENFORM_SetCharacterWidth( Hndl, CPI12 );

  /* NLQ font */
  if( modes & '\x02' )
   FENFORM_SetNLQFont( Hndl, FF_ON );

  /* Condensed Text */
  if( modes & '\x04' )
  {
   if( Hndl->frm.base_char_width == (float)CPI10 )
    FENFORM_SetCharacterWidth( Hndl, CPI17 );
   else
    FENFORM_SetCharacterWidth( Hndl, CPI20 );
  }

  /* Bold */
  if( modes & '\x08' )
   FENFORM_SetBoldFont( Hndl, FF_ON );

  /* Double Strike */
  if( modes & '\x10' )
   FENFORM_SetDoubleStrikeFont( Hndl, FF_ON );

  /* Double Size */
  if( modes & '\x20' )
   FENFORM_MultiSize( Hndl, 1 );

  /* Italics */
  if( modes & '\x40' )
   FENFORM_SetItalics( Hndl, FF_ON );

  /* Underline */
  if( modes & '\x80' )
   FENFORM_SetUnderline( Hndl, FF_ON );  

}



/*************************/
/* Process Control Codes */
/*************************/

void FENFORM_ProcessCtrlCode( FF_Handle *Hndl, int action )
{

  unsigned int rv,
               cnt;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting function 'FENFORM_ProcessCtrlCode' param x%02X", action);

  if( ( Hndl->text_mode == FF_ON ) && ( Hndl->cur_col > Hndl->start_col ) )
   FENFORM_PrintText( Hndl );

  if( ( Hndl->new_page == FF_ON ) && ( action != '@' ) )
    FENFORM_StartPage( Hndl );

  switch( action )
  {
   case BS:   FENFORM_Backspace( Hndl );
              break;

   case HT:   FENFORM_HorizontalTab( Hndl );
              break;

   case LF:   FENFORM_LineFeed( Hndl, FF_ON );
              break;

   case VT:   FENFORM_VerticalTab( Hndl );
              break;

   case FF:   FENFORM_NewPage( Hndl );
              break;

   case CR:   /* Carriage Return */
              FENFORM_LineFeed( Hndl, FF_OFF );
              break;

   case SO:   /****************************************/
              /* Double Size - One line               */
              /*                                      */
              /* To maintain compatibility with other */
              /* printers , 1 or 2 sets double size   */
              /****************************************/
              FENFORM_MultiSize( Hndl, 1 );
              Hndl->ms_one_line = FF_ON;
              break;
         
   case SI:   /* Select Condensed */
              if( Hndl->frm.base_char_width == (float)CPI10 )
               FENFORM_SetCharacterWidth( Hndl, CPI17 );
              else
               FENFORM_SetCharacterWidth( Hndl, CPI20 );
              break;

    case DC2: /* Cancel Condensed */
              FENFORM_SetCharacterWidth( Hndl, Hndl->prev_char_width );
              break;

    case DC3: /* Print Suppress Mode */
              FENFORM_PrintSuppress( Hndl );
              break;

    case DC4: /* Cancel Multi Size */
              FENFORM_MultiSize( Hndl, 0 );
              Hndl->ms_one_line = FF_OFF;
              break;
  }

}



/****************************/
/* Process Escape sequences */
/****************************/

void FENFORM_ProcessESC( FF_Handle *Hndl, int action )
{
  unsigned int rv,
               cnt;
  int *param;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting function 'FENFORM_ProcessESC' param x%02X", action );

  if( ( Hndl->text_mode == FF_ON ) && ( Hndl->cur_col > Hndl->start_col ) )
   FENFORM_PrintText( Hndl );

  if( Hndl->new_page == FF_ON && action != '@' )
   FENFORM_StartPage( Hndl );

  switch( action )
  {

   case SO:   /****************************************/
              /* Double Size - One line               */
              /*                                      */
              /* To maintain compatibility with other */
              /* printers , 1 or 2 sets double size   */
              /****************************************/
              FENFORM_MultiSize( Hndl, 1 );
              Hndl->ms_one_line = FF_ON;
              break;

    case SI:  /* Select Condensed */
              if( Hndl->frm.base_char_width == (float) CPI10 )
               FENFORM_SetCharacterWidth( Hndl, CPI17 );
              else
              {
               if( Hndl->frm.base_char_width == (float) CPI12 )
                FENFORM_SetCharacterWidth( Hndl, CPI20 );
              }
              break;

    case DLE: /* Print Barcode  - Advanced Function */
              param = malloc( sizeof( int ) );
              rv = FENFORM_ReadParameters( Hndl, param, 1, 1 );

              switch( *param )
              {
               case 'A':  FENFORM_SetBarcode( Hndl );
                          break;

               case 'B':  FENFORM_PrintBarcode( Hndl );
                          break;

               case 'C':
               case 'D':
               case 'E':
               case 'F':  FENFORM_HeightModulatedBarcode( Hndl, *param );
                          break;
              }
              free( param );
              break;


    case EM:  /* Select Paper Bin - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'w':
    case US:  /* Multi Size Characters */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );

              /****************************************/
              /* To maintain compatibility with other */
              /* printers , 1 or 2 sets double size   */
              /****************************************/

              if( *param )
               FENFORM_MultiSize( Hndl, *param );
              else
               FENFORM_MultiSize( Hndl, 0 );

              /* This code overrides the "Multi-Size One */
              /* Line" code so set the one_line flag off */
              Hndl->ms_one_line = FF_OFF;

              free( param );
              break;


    case SP:  /* Set Character Spacing - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;

    case '!': /* Select Print Mode */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              FENFORM_SetPrintMode( Hndl, *param );
              free( param );
              break;


    case '#': /* Cancel 8 bit mode - IGNORED */
              break;


    case '$': /* Absolute Horizontal Tab */
              param = malloc( sizeof( int ) * 2 );
              FENFORM_ReadParameters( Hndl, param, 2, 0 );
              FENFORM_Absolute_H_Tab( Hndl, param );
              free( param );
              break;


    case '%': /* Select DLL font - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case '(': /* Set Scoring Style or NLQ/HS mode - IGNORED */
              param = malloc( sizeof( int ) * 6 );
              FENFORM_ReadParameters( Hndl, param, 1, 0 );

              if( *param == '-' ) /* Set Scoring Style - IGNORED */
               FENFORM_ReadParameters( Hndl, param, 5, 0 );
               
              free( param );
              break;


    case '-': /* Select or Cancel underline */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param )
               FENFORM_SetUnderline( Hndl, FF_ON );
              else
               FENFORM_SetUnderline( Hndl, FF_OFF );
              free( param );
              break;


    case '/': /* Select Vertical Tab Channel */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param < VTAB_CHANNELS )
               Hndl->curr_vtab_channel = *param;
              free( param );
              break;


    case '0': /* Set 8 LPI */
              Hndl->frm.line_space = LPI8;
              break;


    case '1': /* Set 7/72 LPI */
              Hndl->frm.line_space = LPI7_72;
              break;


    case '2': /* Set 6 LPI */
              Hndl->frm.line_space = LPI6;
              break;


    case '3':  /* Set n/180 LPI */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->frm.line_space = (float)( *param * LPIn_180 );
              free( param );
              break;


    case '4': FENFORM_SetItalics( Hndl, FF_ON );
              break;

              
    case '5': FENFORM_SetItalics( Hndl, FF_OFF );
              break;


    case '6': /* Select IBM Character Set - IGNORED */
              break;


    case '7':  /* Cancel IBM Character Set - IGNORED */
              break;

    case '8': /* Disable paper out detection - IGNORED */
              break;

    case '9': /* Enable paper out detection - IGNORED */
              break;

    case '<': /* One Line Unidirectional Print - IGNORED */
              break;

    case '=': /* Set bit 7 to 0 - IGNORED */
              break;

    case '>': /* Select 7 or 8 bit commands - IGNORED */
              break;

    case '@': /* Reset Printer */
              FENFORM_Initialize( Hndl );
              break;

    case 'A': /* Set n/60 LPI */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->frm.line_space = (float)( *param * LPIn_60 );
              free( param );
              break;

    case 'B': /* Set Vertical Tabs */
              param = malloc( sizeof( int ) * MAX_V_TABS );
              rv = FENFORM_ReadParameters( Hndl, param, 0, MAX_V_TABS );
              for( cnt = 0; cnt < rv; cnt++ )
               Hndl->frm.vtabs[ 0 ] [cnt] = (float)( Hndl->frm.length - param[ cnt ] * Hndl->frm.line_space );
              free( param );
              break;

    case 'C': /* Set page length by lines or inches - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param == 0 )
               FENFORM_ReadParameters( Hndl, param, 1, 1 );

              free( param );
              break;

    case 'D': /* Set Horizontal Tabs */
              param = malloc( sizeof( int ) * MAX_H_TABS );
              rv = FENFORM_ReadParameters( Hndl, param, 0, MAX_H_TABS );

              for(cnt = 0; cnt < rv; cnt++)
               Hndl->frm.htabs[ cnt ] = (float)( param[ cnt ] * Hndl->frm.base_char_width );

              for(cnt = rv; cnt < MAX_H_TABS; cnt++)
               Hndl->frm.htabs[ cnt ] = Hndl->frm.htabs[ cnt-1 ] + ( DFT_HTAB * Hndl->frm.base_char_width );

              free( param );
              break;

    case 'E': FENFORM_SetBoldFont( Hndl, FF_ON );
              break;

    case 'F': FENFORM_SetBoldFont( Hndl, FF_OFF );
              break;

    case 'G': FENFORM_SetDoubleStrikeFont( Hndl, FF_ON );
              break;

    case 'H': FENFORM_SetDoubleStrikeFont( Hndl, FF_OFF );
              break;

    case 'I': /* Select or Cancel Control Code Printing - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;

    case 'J': /* Select one-shot n/180 inch forward feed */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( Hndl->cur_line >= (float)( *param * LPIn_180 ) )
              {
                FENFORM_PrintText( Hndl );
                Hndl->cur_line -= (float)( *param * LPIn_180 );
              }
              else
                FENFORM_NewPage( Hndl ); 
              free( param );
              break;


    case 'M': FENFORM_SetCharacterWidth( Hndl, CPI12 );
              break;


    case 'N': /* Set Skip Perforation - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'O': /* Cancel Skip Perforation - IGNORED */
              break;


    case 'P': FENFORM_SetCharacterWidth( Hndl, CPI10 );
              break;


    case 'Q': /* Set Right Margin */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->frm.r_margin = (float) *param * Hndl->frm.base_char_width;
              free( param );
              break;


    case 'R': /* Select International Character Set - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'S': /* Set sub/super script */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              FENFORM_SetSub_SuperScript( Hndl, *param );
              free( param );
              break;


    case 'T': /* Cancel sub/super script */
              FENFORM_CancelSub_SuperScript( Hndl );
              break;


    case 'U': /* Select or Cancel Uni-directional printing - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'W': /* Double Size Characters */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );

              /****************************************/
              /* To maintain compatibility with other */
              /* printers , 1 or 2 sets double size   */
              /****************************************/

              if( *param )
               FENFORM_MultiSize( Hndl, 1 );
              else
               FENFORM_MultiSize( Hndl, 0 );

              /* This code overrides the Multi-Size One */
              /* Line code so set the one_line flag off */
              Hndl->ms_one_line = FF_OFF;

              free( param );
              break;


    case '+':
    case '[': /* Set n/360 LPI */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->frm.line_space = (float)( *param * LPIn_360 );
              free( param );
              break;


    case BACKSLASH: /* Relative Dot Position - IGNORED */
              param = malloc( sizeof( int ) * 2 );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case ']': /* Select one-shot n/360 inch forward feed */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( Hndl->cur_line >= (float)( *param * LPIn_360 ) )
              {
               FENFORM_PrintText( Hndl );
               Hndl->cur_line -= (float)( *param * LPIn_360 );
              }
              else
               FENFORM_NewPage( Hndl ); 
              free( param );
              break;



    case '_': /* Select or Cancel overscore */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param )
               FENFORM_SetOverscore( Hndl, FF_ON );
              else
               FENFORM_SetOverscore( Hndl, FF_OFF );
              free( param );
              break;


    case 'a': /* Auto justification - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'b': /* Set Vertical Tabs for a channel*/
              param = malloc( sizeof( int ) * ( MAX_V_TABS + 1 ) );
              rv = FENFORM_ReadParameters( Hndl, param, 0, MAX_V_TABS + 1 );
              for( cnt = 1; cnt < rv; cnt++ )
               Hndl->frm.vtabs[ param[ 0 ] ] [ cnt ] = (float)( Hndl->frm.length - param[ cnt ] * Hndl->frm.line_space );
              free( param );
              break;

    case 'e': /* Set Relative Tab Stops - IGNORED */
              param = malloc( sizeof( int ) * 2 );
              FENFORM_ReadParameters( Hndl, param, 2, 0 );
              free( param );
              break;


    case 'f': /* Set Relative Print Position - IGNORED */
              param = malloc( sizeof( int ) * 2 );
              FENFORM_ReadParameters( Hndl, param, 2, 0 );
              free( param );
              break;


    case 'g': FENFORM_SetCharacterWidth( Hndl, CPI15 );
              break;


    case 'i': /* Select or Cancel incremental print - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;


    case 'j': /* Select one-shot n/180 inch reverse feed */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->cur_line += (float)( *param * LPIn_180 );
              if( Hndl->cur_line >= Hndl->frm.length )
              {
               FENFORM_PrintText( Hndl );
               Hndl->cur_line = Hndl->frm.length - Hndl->frm.line_space;
              }
              free( param );
              break;


    case 'k': /* Select Typestyle - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;
 

    case 'l': /* Set Left Margin */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              Hndl->frm.l_margin = (float) *param * Hndl->frm.base_char_width;
              free( param );
              break;


    case 'm': /* Triple Size */

              FENFORM_MultiSize( Hndl, 3 );

              /******************************************/
              /* This code overrides the Multi-Size One */
              /* Line code so set the one_line flag off */
              /******************************************/
              Hndl->ms_one_line = FF_OFF;

              break;


    case 'q': /* Begin Outline or Shadow */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );

              if( *param )
               FENFORM_SetShadow( Hndl, FF_ON );
              else
               FENFORM_SetShadow( Hndl, FF_OFF );

              free( param );
              break;
 

    case 'r': /* Select Colour */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param == 7 )  /* Select reverse image */
              {
               /***********************************/ 
               /* Shadow and Reverse Image cannot */
               /* both be on at the same time     */
               /***********************************/
               if( ( ! ( Hndl->font_flags & FF_SHADOW_FLAG ) )
                && ( ! ( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG ) ) )
               {
                Hndl->ri_start = Hndl->cur_col;
                Hndl->font_flags |= FF_REVERSE_IMAGE_FLAG;
               }

              } else {

               if( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG )
                Hndl->font_flags &= ~FF_REVERSE_IMAGE_FLAG;

               FENFORM_SetColourValue( Hndl, *param);
              }
              free( param );
              break;

    case 's': /* Select or Cancel half speed - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;

    case 't': /* Select Extended Character Set - IGNORED */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              free( param );
              break;
 
    case 'p': /* Proportional font - treat as NLQ */
    case 'x': /* Set or Cancel NLQ */
              param = malloc( sizeof( int ) );
              FENFORM_ReadParameters( Hndl, param, 1, 1 );
              if( *param )
               FENFORM_SetNLQFont( Hndl, FF_ON );
              else
               FENFORM_SetNLQFont( Hndl, FF_OFF );
              free( param );
              break;


    default:  /* Unrecognized command received */
              FENFORM_ESC_error( action );
              break;
  }
}



/****************************/
/* Process FS sequences     */
/****************************/

void FENFORM_ProcessFS( FF_Handle *Hndl, int action )
{
  unsigned int rv,
               cnt;
  int *param;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting function 'FENFORM_ProcessFS' param x%02X", action);

  if( ( Hndl->text_mode == FF_ON ) && ( Hndl->cur_col > Hndl->start_col ) )
   FENFORM_PrintText( Hndl );

  if( Hndl->new_page == FF_ON && action != '@' )
   FENFORM_StartPage( Hndl );

  switch( action )
  {
   case '@': /* Reset Printer */
             FENFORM_Initialize( Hndl );
             break;


   case '3': /* Set n/360 LPI */
             param = malloc( sizeof( int ) );
             FENFORM_ReadParameters( Hndl, param, 1, 1 );
             Hndl->frm.line_space = (float)( *param * LPIn_360 );
             free( param );
             break;


   case 'E': /* Double Size Characters */
             param = malloc( sizeof( int ) );
             FENFORM_ReadParameters( Hndl, param, 1, 1 );

             /****************************************/
             /* To maintain compatibility with other */
             /* printers , 1 or 2 sets double size   */
             /****************************************/

             if( *param )
              FENFORM_MultiSize( Hndl, 1 );
             else
              FENFORM_MultiSize( Hndl, 0 );

             free( param );
             break;


   case 'F': /* Set Line Feed Direction Forwards */
             Hndl->LF_reverse = FF_OFF;
             break;


   case 'I': /* Select Extended Character Set - IGNORED */
             param = malloc( sizeof( int ) );
             FENFORM_ReadParameters( Hndl, param, 1, 1 );
             free( param );
             break;


   case 'R': /* Set Line Feed Direction Backwards */
             Hndl->LF_reverse = FF_ON;
             break;


   case 'V': /* Multi Size Characters */
             param = malloc( sizeof( int ) );
             FENFORM_ReadParameters( Hndl, param, 1, 1 );

             if( *param )
              FENFORM_MultiSize( Hndl, *param );
             else
              FENFORM_MultiSize( Hndl, 0 );

             /* This code overrides the Multi-Size One */
             /* Line code so set the one_line flag off */
             Hndl->ms_one_line = FF_OFF;

             free( param );
             break;


   default:  /* Unrecognized command received */
             FENFORM_FS_error( action );
             break;
  }

}


int FENFORM_ProcessInput( FF_Handle *Hndl )
{
  int x,
      chars_per_line;
  char *line_ptr;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Starting function 'FENFORM_ProcessInput'" );

  /* Set ready for top line */
  if( FENFORM_StartPage( Hndl ) )
   return( -1 );

  Hndl->curr_font = Hndl->cfg.std_font;
  Hndl->curr_font_id = Hndl->cfg.std_font_id;

  /* Process 'text file' to output */
  while( ( x = fgetc( Hndl->ifile ) ) != EOF )
  {

   if( Hndl->cur_col >= Hndl->frm.r_margin )
    FENFORM_ProcessCtrlCode( Hndl, LF ); /* Send a line feed command */

   switch( x )
   {
    case BS:   /* Back Space */
    case HT:   /* Horizontal Tab */
    case LF:   /* Line Feed */
    case VT:   /* Vertical Tab */
    case FF:   /* Form Feed */
    case CR:   /* Carriage Return */
    case SO:   /* Select Double Size */
    case SI:   /* Select Condensed */
    case DC2:  /* Cancel Condensed */
    case DC3:  /* Set Print Suppress Mode */
    case DC4:  /* Cancel Double Size */
               FENFORM_ProcessCtrlCode( Hndl, x );
               break;

    case ESC:  /* ESC code */
               FENFORM_ProcessESC( Hndl, fgetc( Hndl->ifile ) );
               break;

    case FS:   /* FS code */
               FENFORM_ProcessFS( Hndl, fgetc( Hndl->ifile ) );
               break;

    /* Silently ignore these codes */
    case BEL:  /* Sound alarm */
    case CAN:  /* Clear buffer */
    case DEL:  /* Delete last character from buffer */
               break;

    default:
        if( isprint( x ) )
        {
         if( Hndl->new_page == FF_ON )
          FENFORM_StartPage( Hndl );

         if( Hndl->font_set == FF_OFF )
          FENFORM_SetFont( Hndl );

         if( Hndl->text_mode == FF_OFF )
         {
          /* Allocate space for a full line of text using current font size */
          chars_per_line = (int) ceilf( Hndl->frm.width / Hndl->frm.base_char_width + 1 );
          Hndl->line_of_text = realloc( Hndl->line_of_text, (size_t) chars_per_line * sizeof( char ) );
          Hndl->line_of_text = memset( Hndl->line_of_text, 0, chars_per_line );
          line_ptr = Hndl->line_of_text;

          Hndl->start_col = Hndl->cur_col;
          Hndl->text_mode = FF_ON;
         }

         *line_ptr++ = x;

         if( ( Hndl->font_flags & FF_UNDERLINE_FLAG )
          && ( Hndl->ul_start == 0 ) && ( Hndl->ul_end == 0 ) )
          Hndl->ul_start = Hndl->cur_col;

         if( ( Hndl->font_flags & FF_OVERSCORE_FLAG )
          && ( Hndl->os_start == 0 ) && ( Hndl->os_end == 0 ) )
          Hndl->os_start = Hndl->cur_col;

         if( ( Hndl->font_flags & FF_SHADOW_FLAG )
          && ( Hndl->shadow_start == Hndl->start_col ) && ( Hndl->shadow_end == 0 ) )
          Hndl->shadow_start = Hndl->cur_col;

         if( ( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG )
          && ( Hndl->ri_start == Hndl->start_col ) && ( Hndl->ri_end == 0 ) )
          Hndl->ri_start = Hndl->cur_col;

         Hndl->cur_col += Hndl->frm.curr_char_width;

         if( Hndl->font_flags & FF_UNDERLINE_FLAG )
          Hndl->ul_end = Hndl->cur_col;

         if( Hndl->font_flags & FF_OVERSCORE_FLAG )
          Hndl->os_end = Hndl->cur_col;

         if( Hndl->font_flags & FF_SHADOW_FLAG )
          Hndl->shadow_end = Hndl->cur_col;

         if( Hndl->font_flags & FF_REVERSE_IMAGE_FLAG )
          Hndl->ri_end = Hndl->cur_col;
        }
        break;
   }

   /* Jump to next page if current page is full */
   if( Hndl->cur_line <= 0.0 )
    FENFORM_ProcessCtrlCode( Hndl, FF );
  }

  FENFORM_EndDocument( Hndl );

  return( 0 );

}


int FENFORM_StartDocument( FF_Handle *Hndl )
{

  char *x;
  int  i,
       ret = 0;
  time_t tim;
  struct tm *tm;

  if( Hndl->verbose )
   syslog( LOG_INFO, "Begin Document\n" );

  /* Create date string. */
  time( &tim );
  tm = (struct tm *)localtime( &tim );

  Hndl->date_string = strdup( asctime( tm ) );
  i = strlen ( Hndl->date_string );
  Hndl->date_string[ i - 1 ] = '\0';

  Hndl->user_name = Hndl->pswd->pw_gecos;
  if( *Hndl->user_name ) {

   if( Hndl->verbose )
    syslog( LOG_INFO, "Reading User's Name\n" );

   if( x = strchr( Hndl->user_name, ',' ) )
    *x = '\0';

  } else {

    if( Hndl->verbose )
     syslog( LOG_INFO, "Reading Username\n" );

    Hndl->user_name = Hndl->pswd->pw_name;
  }

  Hndl->page = 1;
  Hndl->new_page = FF_OFF;
  Hndl->line_of_text = NULL;
  Hndl->curr_font = Hndl->cfg.std_font;
  Hndl->curr_font_id = Hndl->cfg.std_font_id;

  FENFORM_DefaultForm( Hndl );

  switch( Hndl->ofile_fmt ) {

   case PS:  ret = FENFORM_PS_StartDocument( Hndl );
             break;

  }

  /* Initialize the "printer" */
  if( ret == 0 ) {
   FENFORM_Initialize( Hndl );
   FENFORM_ProcessInput( Hndl );
  }

  return( ret );
}


void FENFORM_EndDocument( FF_Handle *Hndl )
{

  switch( Hndl->ofile_fmt ) {

   case PS:  FENFORM_PS_EndDocument( Hndl );
             break;

  }
}


/*****************/
/* MAIN function */
/*****************/

int main( int argc, char *argv[ ] )
{

  FF_Handle Hndl;
  char *form_file;

  int opt,
      cfg_flg = 0,
      frm_flg = 0,
      error_flag = 0;

  struct option long_options[] = {
                                  { "help", no_argument, NULL, 'h' },
                                  { "version", no_argument, NULL, 'v' },
                                  { "verbose", no_argument, NULL, 'V' },
                                  { "test", no_argument, NULL, 't' },
                                  { "config-file", required_argument, NULL, 'c' },
                                  { "form", required_argument, NULL, 'f' },
                                  { "input-file", required_argument, NULL, 'i' },
                                  { "output-file", required_argument, NULL, 'o' }
                                 };

  /* Clear the structure */
  memset( &Hndl, 0, sizeof( FF_Handle ) );

  Hndl.config_file = _CFG_FILE;
  form_file = NULL;

  Hndl.run_mode = 0;
  Hndl.ifile = stdin;
  Hndl.ofile = stdout;
  Hndl.verbose = FF_OFF;

  Hndl.ofile_fmt = PS;

  /* Get user's passwd entry. */
  Hndl.pswd = getpwuid( getuid ( ) );
  if( Hndl.pswd == NULL )
  {
   syslog( LOG_INFO, "Couldn't find passwd entry for UID - %d", getuid( ) );
   exit( EXIT_FAILURE );
  }

  syslog( LOG_INFO, "Fenform started by %s\n", Hndl.pswd->pw_name );

  opterr = 0;
  while( ( opt = getopt_long( argc, argv, "c:f:hi:o:tvV", long_options, NULL ) ) != -1 )
  {
   switch( opt )
   {
    case 'c': if( !cfg_flg )
              {
               Hndl.config_file = malloc( (size_t) ( strlen( optarg ) + 2 ) * sizeof( char ) );
               strcpy( Hndl.config_file, optarg );
               cfg_flg = 1;

              }
              else
               error_flag = 2;
              break;

     case 'f': if( !frm_flg )
               {
                form_file = malloc( (size_t) ( strlen( optarg ) + 2 ) * sizeof( char ) );
                strcpy( form_file, optarg );
                frm_flg = 1;
               }
               else
                 error_flag = 4;
               break;

     case 'i': Hndl.ifile = fopen( optarg, "r" );
               if( !Hndl.ifile )
               {
                fprintf( stderr, "%s: %s: %s\n", argv[ 0 ], optarg, strerror( errno ) );
                syslog( LOG_INFO, "Fenform aborted with error - %s.\n", strerror( errno ) );
                exit( EXIT_FAILURE );
               }
                break;

     case 'o': Hndl.ofile = fopen( optarg, "w" );
               if( !Hndl.ofile )
               {
                fprintf( stderr, "%s: %s: %s\n", argv[ 0 ], optarg, strerror( errno ) );
                syslog( LOG_INFO, "Fenform aborted with error - %s.\n", strerror( errno ) );
                exit( EXIT_FAILURE );
               }
               break;

     case 't': Hndl.run_mode = 1;
               break;

     case '?': fprintf( stderr, "\n*** Unrecognized option - '%c' ***\n", optopt );
     case 'h':
     case 'v': FENFORM_VersionInfo( );
               syslog( LOG_INFO, "Fenform completed successfully." );
               exit( EXIT_SUCCESS );
               break;

     case 'V': Hndl.verbose = FF_ON;
               break;

     default:  error_flag = 99;
               break;

    }
  }

  if( error_flag )
  {
   syslog( LOG_INFO, "Invalid command line parameter - %d", error_flag );
   syslog( LOG_INFO, "Fenform aborted with error.\n" );
   exit( EXIT_FAILURE );
  }

  error_flag = FENFORM_CFG_GetConfig( &Hndl );
  if( Hndl.run_mode )
  {
   FENFORM_ShowConfiguration( &Hndl );
   syslog( LOG_INFO, "Fenform completed successfully." );
   exit( EXIT_SUCCESS );
  }

  if( error_flag < 0 )
  {
   syslog( LOG_INFO, "Fenform aborted with configuration error." );
   exit( EXIT_FAILURE );

  } else {

   /* Set up for default form */
   Hndl.curr_form = NULL;
   if( form_file )
   {
    syslog( LOG_INFO, "Fenform using %s form", form_file );

    Hndl.curr_form = malloc( (size_t)( strlen( Hndl.cfg.forms_dir ) + strlen( form_file ) + 2 ) * sizeof( char ) );
    sprintf( Hndl.curr_form, "%s/%s", Hndl.cfg.forms_dir, form_file );

    /* form_file no longer needed */
    /* so we can free the memory. */
    free( form_file );

   } else {

    if( Hndl.verbose )
     syslog( LOG_INFO, "No form file specified" );
   }

   error_flag = FENFORM_StartDocument( &Hndl );
  }

  if( Hndl.ifile != stdin )
   fclose( Hndl.ifile );

  if( Hndl.ofile != stdout )
   fclose( Hndl.ofile );

  if( error_flag == 0 ) {
   syslog( LOG_INFO, "Fenform completed successfully." );
   exit( EXIT_SUCCESS );
  } else {
   syslog( LOG_INFO, "Fenform ended with error(s)." );
   exit( EXIT_FAILURE );
  }

}


/*********************************************************************************/
/*                                                                               */
/*                         End of Source File - fenform.c                        */
/*                                                                               */
/*********************************************************************************/

