
/*

    File: tcpproxy/ip-lib.c
    Version: 1.0

    Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <signal.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include "lib.h"
#include "ip-lib.h"


int	ip_lib_error =		0;

struct {
    int		code;
    char	msg[40];
    } _error[] = {
    0,	"unknown error",
    1,	"socket allocation error",
    2,	"nameserver lookup error",
    3,	"connection refused",
    4,	"fdopen() failed",
    5,	"setvbuf() failed",
    6,	"timeout reached",
    0,	""
    };
    
char *get_error(int code)
{
	int	i;

	for (i=0; _error[i].msg[0] != 0; i++) {
		if (_error[i].code == code)
			return (_error[i].msg);
		}

	return (_error[i].msg);
}

static void alarm_handler()
{
	return;
}

FILE *ip_open(char *host, unsigned int port)
{
	int	socketd;
	struct sockaddr_in server;
	struct hostent *hostp, *gethostbyname();
	FILE	*fp;

	socketd = socket(AF_INET, SOCK_STREAM, 0);
	if (socketd < 0) {
		ip_lib_error = 1;
		return (NULL);
		}
  
	server.sin_family = AF_INET;
	hostp = gethostbyname(host);
	if (hostp == NULL) {
		ip_lib_error = 2;
		return (NULL);
		}
  
	memcpy(&server.sin_addr, hostp->h_addr, hostp->h_length);
	server.sin_port = htons(port);

	signal(SIGALRM, alarm_handler);
	alarm(10);
	if (connect(socketd, (struct sockaddr *) &server, sizeof(server)) < 0) {
		ip_lib_error = 3;
		if (errno == EINTR)
			ip_lib_error = 6;
			
		return (NULL);
		}

	alarm(0);
	signal(SIGALRM, SIG_DFL);
	
	if ((fp = fdopen(socketd, "w+")) == NULL) {
		ip_lib_error = 4;
		return (NULL);
		}

	if ((setvbuf(fp, NULL, _IONBF, 0)) != 0) {
		fclose (fp);

		ip_lib_error = 5;
		return(NULL);
		}

 	return (fp);
}	

unsigned int getportnum(char *name)
{
	unsigned int port;
	struct servent *portdesc;
	
	if (isdigit(*name) != 0)
		port = atol(name);
	else {
		portdesc = getservbyname(name, "tcp");
		if (portdesc == NULL) {
			fprintf (stderr, "%s: service not found: %s\n", program, name);
			exit (-1);
			}

		port = ntohs(portdesc->s_port);
		if (port == 0) {
			fprintf (stderr, "%s: port error: %s\n", program, name);
			exit (-1);
			}
		}
	
	return (port);
}

unsigned int get_port(char *server, unsigned int def_port)
{
	unsigned int port;
	char	*p;

	if ((p = strchr(server, ':')) == NULL)
		return (def_port);

	*p++ = 0;
	port = getportnum(p);

	return (port);
}

int bind_to_port(char *interface, unsigned int port)
{
	struct sockaddr_in saddr;
	int	sock;

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf (stderr, "%s: can't create socket\n", program);
		exit (-1);
		}
	else {
		int	opt;

		opt = 1;
		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
		}


	memset(&saddr, 0, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_port   = htons(port);
	
	if (interface == NULL  ||  *interface == 0)
		interface = "0.0.0.0";
	else {
		struct hostent *ifp;

		ifp = gethostbyname(interface);
		if (ifp == NULL) {
			fprintf (stderr, "%s: can't lookup %s\n", program, interface);
			exit (-1);
			}

		memcpy(&saddr.sin_addr, ifp->h_addr, ifp->h_length);
		}
		
		
	if (bind(sock, (struct sockaddr *) &saddr, sizeof(saddr))) {
		fprintf (stderr, "%s: can't bind to %s:%u\n", program, interface, port);
		exit (-1);
		}
		
		
	if (listen(sock, 5) < 0) {
		fprintf (stderr, "%s: listen error\n", program);
		exit (-1);
		}

	return (sock);
}

