/* XPPP
 *
 * File:      xppp_config.cpp
 * Purpose:   Configuration
 * Author:    Peter Hofmann (970522)
 *
 */

#include <ctype.h>
#include <pwd.h>

#include "xppp_config.h"
#include "xppp.h"

// *** exported vars

ConfigData configData;

// *** ConfigData implementation

ConfigData::ConfigData()
{
  String home = getenv("HOME");
  if (home == "")
  {
    passwd *pw = getpwuid(getuid());
    if (pw != (passwd*)0)
    {
      if (pw->pw_dir != (char*)0)
        home = pw->pw_dir;
    }
  }
  if (home != "") home += "/";
  _rcFile += home + ".xppprc";
}

String ConfigData::nextLine(ifstream &cf)
{
  char line[1000];
  char c;
  bool continued = true;
  String result;
  while (continued && ! cf.eof())
  {
    cf.get(line, 1000, '\n');
    _lineNo++;
    if (cf.get(c) && c != '\n')
    {
      wxMessageBox(removeConst((String)"Input line #" + itoa(_lineNo) +
                               " too long (> 1000) char)"),
                   "Config Error", wxOK);
      while (! cf.eof() && c != '\n') cf.get(c); // skip until eoln
      return "";
    }
    if (line[0] == '#') continue;
  
    char *pp = line;
    // remove white space at beginning and end
    while (isspace(*pp)) pp++;
    char *pq = pp + strlen(pp) - 1;
    while (pq > pp && isspace(*pq)) *pq-- = 0;
    if (strlen(pp) == 0) continue;
    
    String tmp = pp;
    if (result.length() > 0) result += "\n";
    result += tmp;

    cf.get(c);
    continued = c == '\t';
    if (! continued) cf.putback(c);
  }
  return result;
}

void ConfigData::reload()
{
  String rcFile = _rcFile;
  *this = ConfigData();
  _rcFile = rcFile;
  
  ifstream cf;
  cf.open(_rcFile, ios::in);
  _lineNo = 0;
  if (cf.bad())
  {
    wxMessageBox(removeConst((String)"Cannot open config file " + _rcFile),
                 "Config Error", wxOK);
    return;
  }

  Isp newIsp;
  bool ispSet = false;
  bool timeOutSet = false;
  bool maxTriesSet = false;
  
  while (1)
  {
    String line = nextLine(cf);
    if (line.length() == 0) break;
    /* cout << "line: " << line << endl; */
    
    char *pp = removeConst(line);

    bool doublySetOption = false;
    if (strncmp(pp, "isp", 3) == 0)
    {
      pp += 3;
      while (isspace(*pp)) pp++;
      newIsp._name = pp;
      newIsp._chatScript = "";
      newIsp._pppdOptions = "";
      newIsp._checkConnect = "";
      newIsp._timeOut = DEFAULT_TIMEOUT;
      newIsp._phoneNumbers = vector<String>();
      newIsp._maxTries = DEFAULT_MAXTRIES;
      push_back(newIsp);
      maxTriesSet = timeOutSet = false;
      ispSet = true;
    }
    else
      if (! ispSet)
      {
        wxMessageBox(removeConst((String)"Option outside ISP section " +
                                 "in line #" + itoa(_lineNo)),
                     "Config Error", wxOK);
        return;
      }
      else
        if (strncmp(pp, "chat-script", 11) == 0)
        {
          pp += 11;
          while (isspace(*pp)) pp++;
          Isp *isp = end() - 1;
          if (isp->_chatScript.length() > 0)
            doublySetOption = true;
          else
            isp->_chatScript = pp;
        }
        else if (strncmp(pp, "pppd-options", 12) == 0)
        {
          pp += 12;
          while (isspace(*pp)) pp++;
          Isp *isp = end() - 1;
          if (isp->_pppdOptions.length() > 0)
            doublySetOption = true;
          else
            isp->_pppdOptions = pp;
        }
        else if (strncmp(pp, "connect-script", 14) == 0)
        {
          pp += 14;
          while (isspace(*pp)) pp++;
          Isp *isp = end() - 1;
          if (isp->_checkConnect.length() > 0)
            doublySetOption = true;
          else
            isp->_checkConnect = pp;
        }
        else if (strncmp(pp, "timeout", 7) == 0)
        {
          pp += 7;
          while (isspace(*pp)) pp++;
          doublySetOption = timeOutSet;
          Isp *isp = end() - 1;
          if (! timeOutSet)
            isp->_timeOut = atoi(pp);
          timeOutSet = true;
        }
        else if (strncmp(pp, "max-tries", 9) == 0)
        {
          pp += 9;
          while (isspace(*pp)) pp++;
          doublySetOption = maxTriesSet;
          Isp *isp = end() - 1;
          if (! maxTriesSet)
            isp->_maxTries = atoi(pp);
          maxTriesSet = true;
        }
        else if (strncmp(pp, "phone-number", 12) == 0)
        {
          pp += 12;
          while (isspace(*pp)) pp++;
          Isp *isp = end() - 1;
          isp->_phoneNumbers.push_back(pp);
        }
        else
        {
          wxMessageBox(removeConst((String)"Invalid config file syntax " +
                                   "in line #" + itoa(_lineNo)),
                       "Config Error", wxOK);
          return;
        }
    if (doublySetOption)
    {
      wxMessageBox(removeConst((String)"Redefined option in line #" +
                               itoa(_lineNo)),
                   "Config Error", wxOK);
      return;
    }
  }
  cf.close();
}

void ConfigData::save()
{
  String tmpName = _rcFile + "$";
  String bakName = _rcFile + "~";
  ofstream cf;
  cf.open(tmpName, ios::out);
  if (cf.bad())
  {
    wxMessageBox(removeConst((String)"Could not open " + tmpName),
                 "File Error", wxOK);
    return;
  }

  for (ConfigData::iterator i = begin(); i != end(); i++)
  {
    String chatScript = i->_chatScript;
    String pppdOptions = i->_pppdOptions;
    String checkConnect = i->_checkConnect;
    chatScript.gsub("\n", "\n\t");
    pppdOptions.gsub("\n", "\n\t");
    checkConnect.gsub("\n", "\n\t");
    
    cf << "isp " << i->_name << endl
       << "chat-script " << chatScript << endl
       << "pppd-options " << pppdOptions << endl
       << "connect-script " << checkConnect << endl
       << "timeout " << i->_timeOut << endl
       << "max-tries " << i->_maxTries << endl;

    for (vector<String>::iterator j = i->_phoneNumbers.begin();
         j != i->_phoneNumbers.end(); j++)
    {
      cf << "phone-number " << (*j) << endl;
    }
    cf << endl
       << "#################################################################"
       << endl << endl;
    if (cf.bad())
    {
      wxMessageBox(removeConst((String)"Error writing " + tmpName),
                   "File Error", wxOK);
      return;
    }
  }

  cf.close();
  if (cf.bad())
  {
    wxMessageBox(removeConst((String)"Error closing " + tmpName),
                 "File Error", wxOK);
    return;
  }
  
  // do some renaming
  if (rename(_rcFile, bakName) == -1)
  {
    wxMessageBox(removeConst((String)"Could not rename " + _rcFile + " to " +
                             bakName),
                 "File Error", wxOK);
    return;
  }
  if (rename(tmpName, _rcFile) == -1)
  {
    wxMessageBox(removeConst((String)"Could not rename " + tmpName + " to " +
                             _rcFile),
                 "File Error", wxOK);
    return;
  }
  wxMessageBox(removeConst((String)"Saved configuration to " + _rcFile),
                 "File Saved", wxOK);
}

