/* XPPP
 *
 * File:      xppp.cpp
 * Purpose:   Main Program
 * Author:    Peter Hofmann (970522)
 *
 */

#ifdef __GNUG__
#pragma implementation "xppp_misc.h"
#endif

#include <GetOpt.h>
#include <iostream.h>
#include <fstream.h>
#include <String.h>
#include <stdlib.h>
#include <stdio.h>
#include <stl.h>
#include <assert.h>
#include <sys/wait.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>

#include "xppp_misc.h"
#include "xppp.h"
#include "xppp_config.h"
#include "xppp_config_dialog.h"
#include "xppp_pppd_output.h"
#include "xppp.xbm"

// *** menu constants

#define XPPP_FILE_QUIT 1
#define XPPP_CONFIG_CONFIG 2
#define XPPP_CONFIG_SAVE 3
#define XPPP_CONFIG_REVERT 4
#define XPPP_FILE_DISCONNECT 5
#define XPPP_CONFIG_CLEAR_PW 6
#define XPPP_CONFIG_HIDE_SHOW_PPPD_OUTPUT 7

// *** local vars

// some layout elements
static wxColour *buttonActiveColour, *buttonDefaultColour;
static wxColour *buttonDisconnectColour, *buttonForegroundColour;
static MyApp myApp;
static MyFrame *frame;
static wxMenuBar *menuBar;

// global state
int pppPid = 0;                 // id of dial process if status != connected
                                // invariant: process not running: pppPid == 0
XPPPStatus status = not_connected;
bool flashOn = false;
int workCount = 0;              // counter for work_proc
char chatFileName[20];          // name of chat file if status ==
                                // waiting_for_connection
String connectionName;          // current connection name and dial string
String checkConnect;            // current connection check script
ConfigData::iterator currentIsp; // only valid when status != connected
int timeOut;                    // timeout when waiting_for_connection
int killTimeout;                // timeout when waiting_for_disconnect
bool sentTermSignal = false;    // true if kill pppPid, SIGTERM sent
bool sentKillSignal = false;    // true if kill pppPid, SIGKILL sent
int pppdPipe[2] = {-1, -1};     // communication to pppd process
bool alarmReceived;             // set to true by the alarm signal handler
int connectionStartTime;        // time() value of connection start

// exported vars
XpppResources xpppResources;

// *** aux functions

// +++ get password, return NULL if CANCEL pressed

int pwButtonPressed;

// aux functions
void pwDialogCancelButton(wxButton& but, wxEvent& WXUNUSED(event))
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  // There is a possibility that buttons belong to a sub panel.
  // So, we must search the dialog.
  while (!dialog->IsKindOf(CLASSINFO(wxDialogBox)))
    dialog = (wxDialogBox*) ((wxPanel*)dialog)->GetParent() ;

  dialog->Show(FALSE);
  //  delete dialog;
  dialog->Close(TRUE);
  pwButtonPressed = wxCANCEL;
}

void pwDialogOkButton(wxButton& but, wxEvent&)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  // There is a possibility that buttons belong to a sub panel.
  // So, we must search the dialog.
  while (!dialog->IsKindOf(CLASSINFO(wxDialogBox)))
    dialog = (wxDialogBox*) ((wxPanel*)dialog)->GetParent() ;

  dialog->Show(FALSE);
  //  delete dialog;
  dialog->Close(TRUE);
  pwButtonPressed = wxOK;
}

void pwDialogReturn(wxButton& item, wxEvent& event)
{
  if (event.eventClass == wxEVENT_TYPE_TEXT_ENTER_COMMAND)
    pwDialogOkButton(item, event);
}

String getPassword(wxWindow *parent, char *passwordPrompt)
{
  wxBeginBusyCursor();
  
  wxDialogBox *dialog =
    new wxDialogBox(parent, "Password entry",
                    TRUE, -1, -1, 1000, 1000, wxDEFAULT_DIALOG_STYLE);

  wxMessage *mess = new wxMessage(dialog, passwordPrompt, -1, -1, 0,
                                  "passwordMessage");
    
  dialog->NewLine();

  wxText *textItem = new wxText(dialog, (wxFunction)pwDialogReturn,
                                NULL, "", -1, -1, 230, -1, wxTE_PASSWORD);

  dialog->NewLine();

#if (!defined(wx_xview) && USE_PANEL_IN_PANEL)
//  wxPanel *but_panel = new wxPanel(dialog) ;
  int cw,ch;
  dialog->Fit();
  dialog->GetClientSize(&cw,&ch);
  wxPanel *but_panel = new wxPanel(dialog,-1,ch);
#else
  // Until sub panels work in XView mode
  wxPanel *but_panel = dialog ;
#endif

  wxButton *ok = new wxButton(but_panel, (wxFunction)&pwDialogOkButton,
                              wxSTR_BUTTON_OK);
  (void)new wxButton(but_panel, (wxFunction)&pwDialogCancelButton,
                     wxSTR_BUTTON_CANCEL);

  ok->SetDefault();
  textItem->SetFocus();

#if (!defined(wx_xview) && USE_PANEL_IN_PANEL)
  but_panel->Fit() ;
#endif
  dialog->Fit();
#ifndef wx_xview
  but_panel->Centre(wxHORIZONTAL) ;
#endif
  mess->Centre();
  dialog->Centre(wxBOTH);

  wxEndBusyCursor();
  dialog->Show(TRUE);

  if (pwButtonPressed == wxOK)
    return textItem->GetValue();

  return (String)"";
}

// +++ get integer resource

void getResource(char *name, int deflt, int &value)
{
  if (!wxGetResource(RESOURCE_PREFIX, name, &value))
    value = deflt;
}

// +++ get boolean resource

void getResource(char *name, bool deflt, bool &value)
{
  char *tmp = (char*)0;
  if (!wxGetResource(RESOURCE_PREFIX, name, &tmp))
    value = deflt;
  else
  {
    String s = tmp;
    s = downcase(s);
    s.gsub(" ", "");
    value = (s == "yes") || (s == "true");
  }
}

// +++ get string resource

void getResource(char *name, String deflt, String &value)
{
  char *tmp = (char*)0;
  if (wxGetResource(RESOURCE_PREFIX, name, &tmp))
    value = tmp;
  else
    value = deflt;
}

void resetFlash()
{
  workCount = 100000;
  flashOn = false;
}

void updateStatusLine()
{
  switch (status)
  {
  case not_connected:
    frame->SetStatusText("not connected");
    break;
  case connected:
  {
    int connectTime = time((time_t)0) - connectionStartTime;
    String padZero;
    if (connectTime % 60 < 10) padZero = "0";
    frame->SetStatusText(removeConst(currentIsp->_name + " " +
                                     itoa(connectTime / 60) + ":" +
                                     padZero + itoa(connectTime % 60)  +
                                     " min"));
    break;
  }
  case waiting_for_connection:
    frame->SetStatusText(removeConst("trying " + connectionName + " #" +
                                     itoa(currentIsp->_tries)));
    break;
  case waiting_for_disconnect:
    frame->SetStatusText(removeConst("closing down " + currentIsp->_name));
    break;
  }
  if (myApp._configMenu != (wxMenu*)0)
  {
    myApp._configMenu->Enable(XPPP_CONFIG_CONFIG, status == not_connected);
    myApp._configMenu->Enable(XPPP_CONFIG_REVERT, status == not_connected);
    myApp._configMenu->Enable(XPPP_CONFIG_CLEAR_PW, status == not_connected);
  }
}

// +++ dummy signal handler for alarm

void signalAlarm(int)
{
  alarmReceived = true;
  /* cerr << "signalAlarm called" << endl; FIXME */
}

// +++ return true if pppd process is terminated, else false

bool pppdTerminated()
{
  if (pppPid == 0) return true;
  int pid = waitpid(pppPid, (int*)0, WNOHANG);
  if ( pid < 0 || pid == pppPid)
  {
    pppPid = 0;
    return true;
  }
  return false;
}

// +++ kill pppd process if running, display error message and return false
//     if not successful, else return true

bool killPppd(bool useKillSignal = false)
{
  if (pppPid == 0) return true; // process isn't running anyway

  if (kill(-pppPid, useKillSignal ? SIGKILL : SIGTERM) < 0)
  {
    perror ("kill");
    return false;
  }

  // I add 2 seconds here because xppp_wrapper also uses killTimeOut
  // if xppp_wrapper is killed too early it might leave pppd behind
  // I don't want to eliminate killing from xppp because some users
  // might use pppd directly instead of xppp_wrapper
  killTimeout = (xpppResources.killTimeout + 2) * xpppResources.flashRate;
  sentTermSignal = true;
  sentKillSignal = useKillSignal;
  return true;
}

// +++ check if kill timeout reached
//     if so return true

bool checkKillTimeout()
{
  if (killTimeout-- <= 0)
  {
    if (sentKillSignal)
    {
      wxMessageBox(removeConst((String)"\nCould not terminate pppd " +
                               "process PID " + itoa(pppPid)),
                   "PPP Error", wxOK);
      pppPid = 0;
      status = not_connected;
      currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
      updateStatusLine();
    }
    else
      killPppd(true);
    return true;
  }
  return false;
}

// +++ fork pppd process, return pid if successful, otherwise 0

int startPppdProcess(String cmd)
{
  int pid;

  // open pppd/chat output pipe
  close(pppdPipe[0]);
  close(pppdPipe[1]);
  int res = pipe(pppdPipe);
  if (res != -1)
  {
    if (fcntl(pppdPipe[0], F_SETFL,
              fcntl(pppdPipe[0], F_GETFL, 0) | O_NONBLOCK) < 0)
    {
      close(pppdPipe[0]);
      close(pppdPipe[1]);
      pppdPipe[0] = pppdPipe[1] = -1;
    }
  }

  // fork the process
  switch (pid = fork())
  {
  case 0:                     // child
  {
    setpgrp();
    /* cerr << "[" << getpid() << "] child" << endl; */
    if (pppdPipe[1] != -1)
    {
      // set environment variables
      String fds = itoa(pppdPipe[1]);
      String kts = itoa(xpppResources.killTimeout);
      setenv("PPPD_LOG_FD", (const char*)fds, 1);
      setenv("KILL_TIMEOUT", (const char*)kts, 1);
    }
    execl("/bin/sh", "/bin/sh", "-c" , (const char*)cmd, (char*)NULL);
    cerr << "ERROR: exec unsuccessful" << endl;
    exit(6);
    break;
  }
  case -1:                    // error occurred
      return 0;
  }
  // parent
  sentTermSignal = false;
  return pid;
}

// *** next try with currentIsp, return false if maximum retries other other
//     error
//     nextAttempt() also manages the current status (not_connected,
//     waiting_for_connection) and sets connectionName

bool nextAttempt()
{
  String phoneNumber;
  if (currentIsp->_tries > 0) unlink(chatFileName);
  if (currentIsp->_tries++ >= currentIsp->_maxTries)
  {
    wxMessageBox("\nMaximum retry count reached", "PPP Error", wxOK);
    status = not_connected;
    updateStatusLine();
    return false;
  }
  // write temp chat file
  strcpy(chatFileName, "/tmp/chatXXXXXX");
  int tempFileOk = mkstemp(chatFileName);
  if (tempFileOk != -1)
  {
    close(tempFileOk);
    ofstream chatf;
    chatf.open(chatFileName, ios::out);
    if (chatf.bad())
      tempFileOk = -1;
    else
    {
      String tmpChat = currentIsp->_chatScript;

      // process password substitution
      String tmp;
      bool escaped = false, percent = false;
      const char *ip = tmpChat;
      unsigned int passwordCount = 0;
      
      while (*ip)
      {
        if (*ip == '\\')
        {
          escaped = true;
          percent = false;
        }
        else
        {
          if (*ip == '%')
            if (escaped)
              tmp = tmp.at(0, tmp.length() - 1);
            else
              percent = true;
          else
          {
            if (*ip == 'p' && percent)
            {
              // process password substitution
              // prompt for password if not in array
              // store password if configured in the resources
              tmp = tmp.at(0, tmp.length() - 1);
              ip++;
              char passwordPrompt[strlen(ip) + 1];
              char *pp = passwordPrompt;
                
              while (*ip != 0 && *ip != '.')
                *pp++ = *ip++;
              *pp = 0;
              if (*ip == '.') ip++;

              String password;
              if (passwordCount < currentIsp->_passwords.size())
                password = currentIsp->_passwords[passwordCount];
              else
              {
                password = getPassword(frame, passwordPrompt);
                if (password.length() == 0)
                {
                  unlink(chatFileName);
                  status = not_connected;
                  updateStatusLine();
                  return false;
                }
                if (xpppResources.storePasswords)
                  currentIsp->_passwords.push_back(password);
              }
              tmp += password;
              passwordCount++;
            }
            percent = false;
          }
          escaped = false;
        }
        tmp += *ip++;
      }
      tmpChat = tmp;

      // process phonenumber substitution
      if (currentIsp->_phoneNumbers.size() > 0)
      {
        if (currentIsp->_phoneIndex == currentIsp->_phoneNumbers.end())
          currentIsp->_phoneIndex = currentIsp->_phoneNumbers.begin();
        phoneNumber = *currentIsp->_phoneIndex++;

        char tmp[phoneNumber.length() + tmpChat.length() + 10];
        bool escaped = false, percent = false;
        const char *ip = tmpChat;
        char *op = tmp;
        
        while (*ip)
        {
          if (*ip == '\\')
          {
            escaped = true;
            percent = false;
          }
          else
          {
            if (*ip == '%')
              if (escaped)
                op--;
              else
                percent = true;
            else
            {
              if (*ip == 'd' && percent)
              {                 // process phone number substitution
                op--;
                ip++;
                strcpy(op, phoneNumber);
                op += phoneNumber.length();
                percent = false;
                escaped = false;
                continue;
              }
              percent = false;
            }
            escaped = false;
          }
          *op++ = *ip++;
        }
        *op = 0;
        tmpChat = tmp;
      }
      
      tmpChat.gsub("\n", " ");
      
      chatf << tmpChat;
      /* cout << "chat-script: " << tmpChat << endl << endl; */
      chatf.close();
      if (chatf.bad()) tempFileOk = -1;
    }
  }
  if (!tempFileOk)
  {
    wxMessageBox(removeConst((String)"\nCannot open chat temp file\n" +
                             "connect aborted"),
                 "File Error", wxOK);
    unlink(chatFileName);
    status = not_connected;
    updateStatusLine();
    return false;
  }

  // start pppd script
  String cmd = xpppResources.pppdCommand + " ";

  if (xpppResources.connectOptionStart.length() > 0 ||
      xpppResources.connectOptionEnd.length() > 0)
    cmd += xpppResources.connectOptionStart + " " + chatFileName + " " +
      xpppResources.connectOptionEnd + " ";

  cmd += currentIsp->_pppdOptions;
  cmd.gsub("\n", " ");
  /* cout << "executing " << cmd << endl << endl; */
          
  pppPid = startPppdProcess(cmd);
  if (pppPid == 0)
  {
    wxMessageBox(removeConst("\nCould not execute command\n'" + cmd + "'"),
                 "PPP Error", wxOK);
    unlink(chatFileName);
    status = not_connected;
    updateStatusLine();
    return false;
  }
  
  connectionName = currentIsp->_name;
  if (phoneNumber.length() > 0) connectionName += " (" + phoneNumber + ")";

  timeOut = time((time_t*)0) + currentIsp->_timeOut;
  status = waiting_for_connection;
  updateStatusLine();
  return true;
}

// *** event handlers

void connectionWorkProc(wxApp*)
{
  // update status line if seconds change and connected
  static lastTime;
  if (status == connected)
  {
    if (lastTime != time((time_t)0))
      updateStatusLine();
    lastTime = time((time_t)0);
  }
  
  // get pppd output
  if (pppdPipe[0] != -1)
  {
    int nbytes;
    char buf[257];
    while ((nbytes = read(pppdPipe[0], buf, 256)) > 0)
    {
      buf[nbytes] = 0;
      frame->_pppdOutput->writeLog(buf);
    }
  }
  
  // work every 0.5 sec when not connected, every 2 sec when connected
  if (workCount++ >=
      (status == connected ? 5 * xpppResources.checkInterval
       : 50 / xpppResources.flashRate))
  {
    workCount = 0;
    //    cerr << "work proc called" << endl;
    switch (status)
    {
    case not_connected:
      break;

    case waiting_for_disconnect:
      if (pppdTerminated())
      {
        status = not_connected;
        currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
        updateStatusLine();
      }
      else
      {
        if (flashOn)
        {
          currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
          flashOn = false;
        }
        else
        {
          currentIsp->_btn->SetBackgroundColour(*buttonDisconnectColour);
          flashOn = true;
        }
        checkKillTimeout();
      }
      break;

    case waiting_for_connection:
      // check if process terminated, then don't wait for timeout 
      // when timeout reached, try next one
      if (pppdTerminated())
      {
        if (! nextAttempt())
        {
          currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
          break;
        }
      }
      else
        if (sentTermSignal)
        {
          checkKillTimeout();
        }
        else
          if (time((time_t*)0) >= timeOut)
            killPppd();

      // fall through
    case connected:
    {
      alarmReceived = false;
      if (xpppResources.checkConnectTimeout > 0)
      {
        /* cerr << "Setting alarm" << endl; FIXME*/
        signal(SIGALRM, signalAlarm); // timeout for checkConnect
        alarm(xpppResources.checkConnectTimeout);
      }
      if (system(checkConnect) || alarmReceived)
      {                         // connection test failed
        signal(SIGALRM, SIG_IGN); // switch off alarm
        alarm(0);
        
        if (status == waiting_for_connection)
          if (flashOn)
          {
            currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
            flashOn = false;
          }
          else
          {
            currentIsp->_btn->SetBackgroundColour(*buttonActiveColour);
            flashOn = true;
          }
        else
        {
          updateStatusLine();
          wxMessageBox(removeConst("\nLost connection to " + connectionName +
                                   (alarmReceived ? " (TIMEOUT)" : "")),
                       "PPP Error", wxOK);
          currentIsp->_btn->SetBackgroundColour(*buttonDefaultColour);
          flashOn = false;
          status = waiting_for_disconnect;
          killPppd();
        }
      }
      else                      // ping successful
      {
        signal(SIGALRM, SIG_IGN); // switch off alarm
        alarm(0);

        if (status == waiting_for_connection)
        {
          status = connected;
          connectionStartTime = time((time_t)0);
          updateStatusLine();
          currentIsp->_btn->SetBackgroundColour(*buttonActiveColour);
        }
        else
          if (pppPid != 0 && pppdTerminated())
            wxMessageBox(removeConst((String)"\npppd process terminated,\n" +
                                     "but connect script successful.\n" +
                                     "Please check your connect script."),
                         "PPP Error", wxOK);
      }
    }
    }
  }
  usleep(10000);              // so that work_proc doesn't use all resources
}  

static void disconnect()
{
  if (status == connected || status == waiting_for_connection)
  {
    unlink(chatFileName);
    killPppd();
    status = waiting_for_disconnect;
    updateStatusLine();
    resetFlash();
    if (status == waiting_for_connection) unlink(chatFileName);
  }
}

static void button_proc(wxButton& but, wxCommandEvent&)
{
  switch (status)
  {
  case not_connected:
    for (ConfigData::iterator i = configData.begin();
         i != configData.end(); i++)
      if (i->_btn == &but)
      {
        currentIsp = i;
        currentIsp->_tries = 0;
        currentIsp->_phoneIndex = currentIsp->_phoneNumbers.begin();
        if (nextAttempt())
        {
          checkConnect = i->_checkConnect;
          checkConnect.gsub("\n", " ");
          resetFlash();
        }
        return;
      }
    cerr << "button_proc(): invalid button" << endl;
    break;

  case connected:
  case waiting_for_connection:
    disconnect();
    break;
    
  case waiting_for_disconnect:
    // do nothing, continue waiting
    break;
  }
}

// myapp constructor

MyApp::MyApp() : _configMenu((wxMenu*)0)
{
  myApp.work_proc = connectionWorkProc;
}

// `Main program' equivalent, creating windows and returning main app frame
wxFrame *MyApp::OnInit(void)
{
  // determine if running setuid oder setgid
  bool suid = (getuid() != geteuid() || getgid() != getegid());
  if (suid)
    configData.setRcFile("/etc/xppprc");
  
  // load application resources
  getResource("configFile", "", xpppResources.configFile);
  getResource("background", "white", xpppResources.backgroundColour);
  getResource("buttonBackground", "white", xpppResources.buttonBackground);
  getResource("buttonForeground", "black", xpppResources.buttonForeground);
  getResource("connectionOkColour", "green",
              xpppResources.connectionOkColour);
  getResource("disconnectColour", "red", xpppResources.disconnectColour);
  getResource("flashRate", 1, xpppResources.flashRate);
  getResource("ispButtonHeight", 35, xpppResources.ispButtonHeight);
  getResource("mainWindowWidth", 150, xpppResources.mainWindowWidth);
  getResource("pppdCommand", "/usr/X11R6/bin/xppp_wrapper -detach",
              xpppResources.pppdCommand);
  getResource("connectOptionStart", "connect '/usr/sbin/chat -v -f",
              xpppResources.connectOptionStart);
  getResource("connectOptionEnd", "'", xpppResources.connectOptionEnd);
  getResource("killTimeout", 5, xpppResources.killTimeout);
  getResource("storePasswords", true, xpppResources.storePasswords);
  getResource("checkInterval", 20, xpppResources.checkInterval);
  getResource("checkConnectTimeout", 15, xpppResources.checkConnectTimeout);
  
  if (xpppResources.configFile.length() > 0)
    if (suid)
      cerr << "WARNING: suid/sgid execution, using /etc/xppprc" << endl;
    else
      configData.setRcFile(xpppResources.configFile);
  
  // process command line options
  GetOpt getopt(argc, argv, "c:v");
  char option;
  while ((option = getopt ()) != EOF)
    switch (option)
    {
    case 'c':
      if (suid)
        cerr << "WARNING: suid/sgid execution, using /etc/xppprc" << endl;
      else
        configData.setRcFile(getopt.optarg);
      break;

    case 'v':
      cerr << "XPPP version " << XPP_VERSION << " ("
           << XPP_BUILD_DATE << ")" << endl;
      exit(0);
      break;
      
    default:
      wxMessageBox("\nUnknown command line option", "Option Error", wxOK);
      exit(1);
    }
  
  // load config file
  configData.reload();
  
  // Create the main frame window
  frame = new MyFrame(NULL, "XPPP", 50, 50,
                      xpppResources.mainWindowWidth,
                      configData.size() * xpppResources.ispButtonHeight + 30);
  frame->SetAutoLayout(FALSE);

  // create pppd output dialog
  frame->_pppdOutput = new XpppPppdOutput(frame);
  frame->_pppdOutput->Centre();
  
  // Give it an icon
  frame->SetIcon(new wxIcon(xppp_bits, xppp_width, xppp_height));

  // add a status bar
  frame->CreateStatusLine(2);
  updateStatusLine();

  // add menu
  wxMenu *fileMenu = new wxMenu;
  fileMenu->Append(XPPP_FILE_DISCONNECT, "Disconnect", "Disconnect");
  fileMenu->Append(XPPP_FILE_QUIT, "Quit", "Quit program");

  _configMenu = new wxMenu;
  _configMenu->Append(XPPP_CONFIG_HIDE_SHOW_PPPD_OUTPUT, "Log Window",
                      "Show pppd and chat output");
  _configMenu->Append(XPPP_CONFIG_CONFIG, "Config",
                      "Open config dialog");
  _configMenu->Append(XPPP_CONFIG_SAVE, "Save", "Save config file");
  _configMenu->Append(XPPP_CONFIG_REVERT, "Revert",
                      "Revert to last saved config file");
  _configMenu->Append(XPPP_CONFIG_CLEAR_PW, "Clear Passwords",
                      "Clear all passwords entered so far");
  
  menuBar = new wxMenuBar;
  menuBar->Append(fileMenu, "File");
  // determine if running setuid and grey out configuration if so
  if (!suid)
    menuBar->Append(_configMenu, "Config");
  frame->SetMenuBar(menuBar);
  
  // Make a panel
  frame->panel = new wxPanel(frame, 0, 0, 1, 1, wxBORDER | wxUSER_COLOURS);

  wxLayoutConstraints *c = new wxLayoutConstraints;
  c->top.SameAs(frame, wxTop);
  c->bottom.SameAs(frame, wxBottom);
  c->left.SameAs(frame, wxLeft);
  c->right.SameAs(frame, wxRight);
  frame->panel->SetConstraints(c);

  static wxColour *panelBackgroundColour =
    new wxColour(xpppResources.backgroundColour);
  frame->panel->SetBackgroundColour(*panelBackgroundColour);

  // now add the buttons
  buttonActiveColour = new wxColour(xpppResources.connectionOkColour);
  buttonDisconnectColour = new wxColour(xpppResources.disconnectColour);
  buttonDefaultColour = new wxColour(xpppResources.buttonBackground);
  buttonForegroundColour = new wxColour(xpppResources.buttonForeground);
  
  frame->addButtons();
  
  // Show the frame
  frame->Show(TRUE);
  
  // Return the main frame window
  return frame;
}

// add buttons for ISPs
void MyFrame::addButtons()
{
  // remove old buttons
  for (vector<wxButton*>::iterator i = _oldButtons.begin();
       i != _oldButtons.end(); i++)
    delete *i;
  _oldButtons = vector<wxButton*>();
  
  // draw new ones
  static bool redraw = false;
  static int baseHeight;
    
  wxButton *last = (wxButton *)0;
  
  for (ConfigData::iterator i = configData.begin(); i != configData.end(); i++)
  {
    wxButton *btn = new wxButton(panel,
                                 (wxFunction)&button_proc,
                                 removeConst(i->_name),
                                 -1, -1, -1, -1, 0, "ispButton");
    btn->SetBackgroundColour(*buttonDefaultColour);
    btn->SetButtonColour(*buttonForegroundColour);
    btn->SetLabelColour(*buttonForegroundColour);
    
    wxLayoutConstraints *b = new wxLayoutConstraints;
    if (last)
      b->top.Below(last, 5);
    else
      b->top.SameAs(panel, wxTop, 5);

    b->left.SameAs(panel, wxLeft, 5);
    b->right.SameAs(panel, wxRight, 7);
    b->height.Absolute(30);
    btn->SetConstraints(b);

    last = btn;
    i->_btn = btn;
  }

  // (re)calculate layout
  //  SetSizeHints(-1, -1, -1, -1);
  if (redraw)
    SetClientSize(-1, baseHeight +
                  configData.size() * xpppResources.ispButtonHeight);
  Layout();
  int width, height;
  GetSize(&width, &height);
  //  SetSizeHints(width, height, width, height);
  if (! redraw)
  {
    GetClientSize(&width, &height);
    baseHeight = height - configData.size() * xpppResources.ispButtonHeight;
  }
  redraw = true;
}

// My frame constructor
MyFrame::MyFrame(wxFrame *frame, char *title, int x, int y, int w, int h):
  wxFrame(frame, title, x, y, w, h), _dialog((XpppConfigDialog*)0)
{}

bool MyFrame::reallyQuit()
{
  if (_dialog == (wxDialogBox*)0 ||
      ! _dialog->_changed ||
      wxMessageBox(removeConst((String)"\nConfiguration changed but not " +
                               "saved.\nReally quit?"),
                   "XPPP Confirm", wxYES_NO | wxCANCEL) == wxYES)
  {
    if (status == waiting_for_connection) unlink(chatFileName);
    Show(FALSE);
    return true;
  }
  return false;
}

Bool MyFrame::OnClose(void)
{
  return reallyQuit();
}

void MyFrame::OnMenuCommand(int id)
{
  switch (id)
  {
  case XPPP_FILE_QUIT:
    if (reallyQuit())
      exit(0);
    break;

  case XPPP_FILE_DISCONNECT:
    disconnect();
    break;

  case XPPP_CONFIG_HIDE_SHOW_PPPD_OUTPUT:
  {
    _pppdOutput->show();
    break;
  }
  
  case XPPP_CONFIG_CONFIG:
  {
    for (ConfigData::iterator i = configData.begin();
         i != configData.end(); i++)
      _oldButtons.push_back(i->_btn);
    
    if (_dialog == (wxDialogBox*)0)
    {
      _dialog = new XpppConfigDialog(this);
      _dialog->Centre();
      _dialog->Show(TRUE);
    }
    else
      _dialog->Show(TRUE);
    break;
  }

  case XPPP_CONFIG_SAVE:
    configData.save();
    if (_dialog != (wxDialogBox*)0)
      _dialog->_changed = false;
    break;

  case XPPP_CONFIG_REVERT:
    for (ConfigData::iterator i = configData.begin();
         i != configData.end(); i++)
    {
      delete i->_btn;
    }
    configData.reload();
    addButtons();
    if (_dialog != (wxDialogBox*)0)
    {
      _dialog->redisplayAll();
      _dialog->_changed = false;
    }
    break;

  case XPPP_CONFIG_CLEAR_PW:
    for (ConfigData::iterator i = configData.begin();
         i != configData.end(); i++)
      i->_passwords = vector<String>();
    break;
  }
  return;
}

void MyFrame::OnSize(int, int)
{
  Layout();
}

// called by configuration dialog when finished

void configFinished()
{
  frame->addButtons();
}
