/*
  Copyright (C) 1997-2002  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <sys/param.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <signal.h>
#include "common.h"

#ifdef SUNOS41x
extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;
extern int fputs(), fscanf(), fprintf(), fclose(), sscanf();
extern char *vsprintf();
extern void bcopy();
#endif

#ifdef RUNASEUID
int xisp_euidaccess();
#endif

#define MAXLEN_BUF MAXLEN_FNAME

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

char *scriptfname;						/* login script filename */
#if !ISPENV_USEVARS
char *envfname;							/* dialing environment filename */
#endif /* !ISPENV_USEVARS */
char Chat[MAXLEN_BUF+2];				/* complete path to chat utility */
char Pipe[MAXLEN_BUF+2];				/* named pipe report file */
char Terminal[MAXLEN_BUF+2];			/* complete path to xispterm */
char envLine[MAXLEN_BUF+2];				/* buf for reading non string vars */
int pipeFD = 0;							/* report file descriptor */
int childPID;							/* PID of running child */
int maxAttempts = MAXNUM_RETRY;			/* max dialing attempts */
int sleepDelay = MAXSEC_DELAY;			/* post-dial sleep delay */
int connectWait = MAXSEC_CNWAIT;		/* maximum wait for connection */
char account[MAXLEN_ACCOUNT+2];			/* account name */
char **phone;							/* phone numbers */
int numPhones;							/* how many there are to dial */
unsigned int mode;						/* connection mode */
char **sline;							/* script lines */
int numSlines;							/* how many there are to include */
char **CBsln;							/* call-back script lines */
int numCBSlns;							/* how many there are to include */
int CBDelay;							/* time wait for call-back */
char *modemInit;						/* modem init string */				
char modemUInit[MAXLEN_MDMCMD+2];		/* user specified modem init str */
char modemUReset[MAXLEN_MDMCMD+2];		/* user specified modem reset string */				
char modemUConn[MAXLEN_MDMSTR+2];		/* user specified modem connect str */
char modemDial[MAXLEN_DIALEXTRA+2+2];	/* dialing chars + method char */
char dispname[MAXLEN_TTPARM+2];			/* X11 display for auto login */
char geometry[MAXLEN_TTPARM+2];			/* and terminal window geometry */
char bgcol[16] = {0};					/* form-background color string */


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Utility routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 256

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR];

#ifdef HAVE_STRERROR
	sprintf(emsg, "xispdial: %s: %s\n", msg, strerror(errno));
#else
	if (errno < sys_nerr)
		sprintf(emsg, "xispdial: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xispdial: %s: error #%d\n", msg, errno);
#endif
	fputs(emsg, stderr);
	if (pipeFD) close(pipeFD);
	exit(1);
}


/* Open a named pipe for writing only */

int namedPipe(char *fname)
{
	struct stat st;
	int fd;

#ifdef RUNASEUID
	if (xisp_euidaccess(fname, F_OK) == -1)		/* check to see if it exists */
#else
	if (access(fname, F_OK) == -1)
#endif
		doErr("namedPipe: access");				/* nope, this is not right! */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode))				/* is it a FIFO? */
			doErr("namedPipe: stat");			/* nope, still not right! */
	}
#ifdef SUNOS5x
	fd = open(fname, O_WRONLY|O_NONBLOCK);		/* yes, open it for writing */
#elif (defined(BSD) && BSD >= 199306)
	fd = open(fname, O_WRONLY|O_NONBLOCK);
#else
	fd = open(fname, O_WRONLY|O_NDELAY);
#endif
	if (fd < 0)									/* error means no process has */
		doErr("namedPipe: open");				/* opened it for reading */
	return fd;									/* return the descriptor */
}


/* Write messages printf() style to the named pipe file descriptor. Note
   the implicit total length of MSGLEN_PIPE bytes for the resulting string. */

#define MSGLEN_PIPE 256

int pprintf(va_alist) va_dcl
{
	int iw, bw;
	va_list ap;
	char *fmt, msg[MSGLEN_PIPE];

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
#ifdef BROKEN_SPRINTF
	vsprintf(msg, fmt, ap);						/* pass rest to vsprintf() */
	iw = strlen(msg);
#else
	iw = vsprintf(msg, fmt, ap);
#endif
	va_end(ap);									/* end variable arg list */
	bw = write(pipeFD, msg, strlen(msg));		/* write buffer to pipe */
	if (bw < strlen(msg))						/* all bytes written? */
		doErr("xispdial: pprintf");				/* nope, bail out */
	return iw;									/* else return items written */
}


/* Execute a child process, attaching its stderr to a named pipe. The
   pipe output is read by xisp and displayed on its browser window.   */

#define MAXARGS_CHILD 16

int child(va_alist) va_dcl
{
	int i, stat;
	char *args[MAXARGS_CHILD+1];
	va_list ap;

	childPID = fork();							/* fork to create child */
	if (childPID < 0)							/* ret < 0 : fork failed */
		doErr("child: fork");
	if (childPID) {								/* in parrent process */
		i = waitpid(childPID, &stat, 0);		/* get child return status */
		if (i < 0)								/* ret < 0 : wait failed */
			doErr("child: waitpid");
		childPID = 0;							/* OK, indicate child done */
	}
	else {										/* in child process */
		dup2(pipeFD, 2);						/* tack stderr on pipe input */
		va_start(ap);							/* start variable arg list */
		for (									/* parse program arguments */
			i=0; i < MAXARGS_CHILD &&			/* to a max of MAXARGS_CHILD */
			(args[i]=va_arg(ap, char*)) !=
				(char *)0;
			i++
		);
		va_end(ap);								/* end variable arg list */
		execv(args[0], args);					/* exec child */
		doErr("child: execv");					/* return here means error */
	}
	if (WIFEXITED(stat))						/* parent returns child stat */
		return(WEXITSTATUS(stat));				/* unless something funny */
	else										/* has happened, in which */
		return -1;								/* case -1 is returned */
}

/* Prints out of memory error and exits */

void outOfMem(char *msg)
{
	fprintf(stderr, "%s: %s: out of memory!\n",	/* notify xisp */
			"xispdial", msg);
	if (pipeFD) close(pipeFD);					/* close the report pipe */
	exit(1);									/* and exit with error */
}

/* Initializes the dialer file names using the user home directory. */

void initFnames(void)
{
	struct passwd *user;

#ifdef RUNASEUID
	user = getpwuid(geteuid());
#else
	user = getpwuid(getuid());
#endif
	scriptfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(SCRIPTFNAME)+1);
#if !ISPENV_USEVARS
	envfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(ENVFNAME)+1);
	if (scriptfname != NULL && envfname != NULL) {
#else /* ISPENV_USEVARS */
	if (scriptfname != NULL) {
#endif /* ISPENV_USEVARS */
		strcpy(scriptfname, user->pw_dir); strcat(scriptfname, "/");
		strcat(scriptfname, SCRIPTFNAME);
#if !ISPENV_USEVARS
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
#endif /* !ISPENV_USEVARS */
	}
	else outOfMem("initFnames");
}


/* Print message indicating a corrupted environment */

void uEOF(void)
{
	if (pipeFD) {								/* if the pipe to xisp's */
		pprintf("%s: %s!\n", "xispdial",		/* browser is open, print */
				"corrupted environment");		/* a message there */
		close(pipeFD);							/* close the report pipe */
	}
	else										/* otherwise print the */
		fprintf(stderr,							/* message to pppd's script */
				"%s: %s!\n", "xispdial",		/* error log file */
				"corrupted environment");
	exit(1);									/* exit with error */
}

#if ISPENV_USEVARS

/* Functions for fetching the xispdial environment from an
   environment variable. */

static const char *eptr = NULL;		/* environment string pointer */

static void init_env()
{
	if (eptr == NULL) {
		eptr = getenv(ENVVAR);
		if (eptr == NULL)
			doErr("environment variable " ENVVAR " is not set");
	}
}

static char *egets(char *s, int size)
{
	register char *t = s;

	init_env();
	if (eptr == NULL || !*eptr)
		return NULL;
	while ((t-s) < (size-1) && *eptr && (*t++ = *eptr++) != '\n');
	*t = 0;
	return s;
}

static int escanf(const char *fmt, void *datap)
{
	init_env();
	if (eptr == NULL || !*eptr)
		return EOF;
	egets(envLine, MAXLEN_BUF+1);
	return sscanf(envLine, fmt, datap);
}

#define efinish() /* Nothing to do after parsing the environment var */

#else /* !ISPENV_USEVARS */

/* Functions for fetching the xispdial environment from a file.  */

FILE *envfp;  /* environment file pointer */

static void open_envfp()
{
	if (envfp == NULL)
	{
		envfp = fopen(envfname, "r");		/* open temp environment file */
		if (envfp == NULL)					/* if not there bail out */
			doErr("open_envfp: fopen");
	}
}

static char *egets(char *s, int size)
{
	open_envfp();
	return fgets(s, size, envfp);
}

static int escanf(const char *fmt, void *datap)
{
	open_envfp();
	fgets(envLine, MAXLEN_BUF+1, envfp);
	return sscanf(envLine, fmt, datap);
}

static void efinish()
{
	if (envfp != NULL)
	{
		fclose(envfp);
		envfp = NULL;
	}
	unlink(envfname);
}

#endif /* !ISPENV_USEVARS */

/* The readLine function reads a single line from the environment file
   prepared by xisp, and can be instructed to remove all '\x' escape
   sequences from the buffer before returning */

void readLine(char *param, int len, int allowEscape)
{
	char *p;
	int rest;

	if (egets(param, len+1)	== NULL)			/* read environment line */
		uEOF();									/* bail out on error */
	param[rest = strlen(param)-1] = 0;			/* kill the ending '\n' */
	for (p=param; *p;) {						/* scan the line */
		if (*p == '\\' && !allowEscape) {		/* if '\\' is found and */
#ifdef SUNOS41x									/* escape chars not allowed */
			bcopy(p+1, p, rest);				/* eat-up the '\\' by */
#else											/* decreasing string len by 1 */
			memmove(p, p+1, rest);
#endif
			--rest;								/* adjust remaining chars */
			if (*p) {							/* if it's not the last */
#ifdef SUNOS41x									/* char in the string, then */
				bcopy(p+1, p, rest);			/* eat-up the next char also */
#else
				memmove(p, p+1, rest);
#endif
				--rest;							/* adjust remaining chars */
			}
		}
		else {									/* char not a '\\' */
			++p;								/* so carry on */
			--rest;								/* adjust remaining chars */
		}
	}
}

/* getISPenv() reads the environment variable (or file) created by
   xISP and adjusts xispdial's operation mode flags; if a file was
   used for passing the environment, after processing it is deleted */

void getISPenv(void)
{
	int ir, i;

	readLine(Chat, MAXLEN_BUF, 0);				/* read all needed dialing */
	readLine(Pipe, MAXLEN_BUF, 0);				/* environment variables */
	readLine(Terminal, MAXLEN_BUF, 0);
	ir = escanf("%d ",&maxAttempts);
	ir += escanf("%d ",&sleepDelay);
	ir += escanf("%d ",&connectWait);
	ir += escanf("%d ",&numPhones);				/* read # of phone numbers */
	if (ir < 4)									/* all read so far? */
		uEOF();									/* nope, bail out */
	phone = (char **)							/* allocate space for */
			malloc(numPhones*sizeof(char *));	/* numPhones phone numbers */
	for (i=0; i<numPhones && phone!=NULL;		/* which follow */
		 i++)
	{
		phone[i] = (char *)						/* allocate phone number */
					malloc(MAXLEN_PHONE+2);
		if (phone[i] != NULL)
			*(phone[i]) = 0;					/* and initialize it */
		else
			phone = NULL;
	}
	if (phone == NULL)							/* allocation completed? */
		outOfMem("getISPenv");					/* nope, bail out */
	for (i=0; i<numPhones; i++)
		readLine(phone[i], MAXLEN_PHONE, 0);	/* read telephone numbers */
	if (escanf("%X ",&mode) < 1)				/* read connection method */
		uEOF();
	switch (mode) {

		case SCRIPT_DIALIN:
			readLine(account,MAXLEN_ACCOUNT,0);	/* read account name and */
			ir= escanf("%d ",&numSlines);		/* number of script lines */
			if (! ir)							/* bail out on error */
				uEOF();
			if (numSlines) {					/* 0 lines is possible */
				sline = (char **)				/* allocate space for */
				  malloc(numSlines*				/* numSlines scipt lines */
						 sizeof(char *));
				for (i=0;
					 i<numSlines&&sline!=NULL;	/* which follow */
					 i++)
				{
					sline[i] = (char *)			/* allocate script line */
					  malloc(4*MAXLEN_SLINE+10	/* double the size for */
						+5*MAXLEN_ACCOUNT+2);	/* automatic escaping */
					if (sline[i] != NULL)
						*(sline[i]) = 0;		/* and initialize it */
					else
						sline = NULL;
				}
				if (sline == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numSlines; i++)
					readLine(sline[i],			/* read script lines */
							 4*MAXLEN_SLINE+10+
							 5*MAXLEN_ACCOUNT, 1);
			}
		break;

		case SCRIPT_DIALIN|SCRIPT_CALLBACK:
			readLine(account,MAXLEN_ACCOUNT,0);	/* read account name and */
			ir= escanf("%d ",&numSlines);		/* number of script lines */
			if (! ir)							/* bail out on error */
				uEOF();
			if (numSlines) {					/* 0 lines is possible */
				sline = (char **)				/* allocate space for */
				  malloc(numSlines*				/* numSlines scipt lines */
						 sizeof(char *));
				for (i=0;
					 i<numSlines&&sline!=NULL;	/* which follow */
					 i++)
				{
					sline[i] = (char *)			/* allocate script line */
					  malloc(4*MAXLEN_SLINE+10	/* double the size for */
						+5*MAXLEN_ACCOUNT+2);	/* automatic escaping */
					if (sline[i] != NULL)
						*(sline[i]) = 0;		/* and initialize it */
					else
						sline = NULL;
				}
				if (sline == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numSlines; i++)
					readLine(sline[i],			/* read script lines */
							 4*MAXLEN_SLINE+10+
							 5*MAXLEN_ACCOUNT, 1);
			}
			ir = escanf("%d ",&CBDelay);		/* time wait for call-back */
			ir+=escanf("%d ",&numCBSlns);		/* number of call-back lines */
			if (ir < 2)							/* bail out on error */
				uEOF();
			if (numCBSlns) {					/* 0 lines is possible */
				CBsln = (char **)				/* allocate space for */
				 malloc(numSlines*				/* numCBSlns call-back lines */
						sizeof(char *));
				for (i=0; i<numCBSlns &&		/* which follow */
						  CBsln!=NULL;
					 i++)
				{
					CBsln[i] = (char *)			/* allocate call-back line */
					  malloc(4*MAXLEN_SLINE+
						10+5*MAXLEN_ACCOUNT+2);
					if (CBsln[i] != NULL)
						*(CBsln[i]) = 0;		/* and initialize it */
					else
						CBsln = NULL;
				}
				if (CBsln == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numCBSlns; i++)
					readLine(CBsln[i],			/* read script lines */
							 4*MAXLEN_SLINE+10+
							 5*MAXLEN_ACCOUNT, 1);
			}
		break;

		case SCRIPT_DIALIN|MANUAL_CALLBACK:
			readLine(account,MAXLEN_ACCOUNT,0);	/* read account name and */
			ir= escanf("%d ",&numSlines);		/* number of script lines */
			if (! ir)							/* bail out on error */
				uEOF();
			if (numSlines) {					/* 0 lines is possible */
				sline = (char **)				/* allocate space for */
				  malloc(numSlines*				/* numSlines scipt lines */
						 sizeof(char *));
				for (i=0;
					 i<numSlines&&sline!=NULL;	/* which follow */
					 i++)
				{
					sline[i] = (char *)			/* allocate script line */
					  malloc(4*MAXLEN_SLINE+10	/* double the size for */
						+5*MAXLEN_ACCOUNT+2);	/* automatic escaping */
					if (sline[i] != NULL)
						*(sline[i]) = 0;		/* and initialize it */
					else
						sline = NULL;
				}
				if (sline == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numSlines; i++)
					readLine(sline[i],			/* read script lines */
							 4*MAXLEN_SLINE+10+
							 5*MAXLEN_ACCOUNT, 1);
			}
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
			if (escanf("%d ",&CBDelay)<1)		/* time wait for call-back */
				uEOF();							/* bail out on error */
		break;

		case MANUAL_DIALIN:
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
		break;

		case MANUAL_DIALIN|SCRIPT_CALLBACK:
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
			ir = escanf("%d ",&CBDelay);		/* time wait for call-back */
			ir+=escanf("%d ",&numCBSlns);		/* number of call-back lines */
			if (ir < 2)							/* bail out on error */
				uEOF();
			if (numCBSlns) {					/* 0 lines is possible */
				CBsln = (char **)				/* allocate space for */
				 malloc(numSlines*				/* numCBSlns call-back lines */
						sizeof(char *));
				for (i=0; i<numCBSlns &&		/* which follow */
						  CBsln!=NULL;
					 i++)
				{
					CBsln[i] = (char *)			/* allocate call-back line */
					  malloc(4*MAXLEN_SLINE+
						10+5*MAXLEN_ACCOUNT+2);
					if (CBsln[i] != NULL)
						*(CBsln[i]) = 0;		/* and initialize it */
					else
						CBsln = NULL;
				}
				if (CBsln == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numCBSlns; i++)
					readLine(CBsln[i],			/* read script lines */
							 4*MAXLEN_SLINE+10+
							 5*MAXLEN_ACCOUNT, 1);
			}
		break;

		case MANUAL_DIALIN|MANUAL_CALLBACK:
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
			if (escanf("%d ",&CBDelay)<1)		/* time wait for call-back */
				uEOF();							/* bail out on error */
		break;

		case AUTH_DIALIN:
		case AUTH_DIALIN|NT_RAS_DIALIN:
		break;

		case AUTH_DIALIN|AUTH_CALLBACK:
		case AUTH_DIALIN|NT_RAS_CALLBACK:
			if (escanf("%d ",&CBDelay)<1)		/* time wait for call-back */
				uEOF();							/* bail out on error */
		break;

		case AUTH_DIALIN|MANUAL_DIALIN:
		case AUTH_DIALIN|MANUAL_DIALIN|
			 NT_RAS_DIALIN:
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
		break;

		case AUTH_DIALIN|MANUAL_DIALIN|
			 AUTH_CALLBACK:
			readLine(dispname,					/* read the X11 display */
					 MAXLEN_TTPARM, 0);
			readLine(bgcol,						/* read background color */
					 (sizeof bgcol)-2, 0);
			readLine(geometry,					/* read window geometry */
					 MAXLEN_TTPARM, 0);
			if (escanf("%d ",&CBDelay)<1)		/* time wait for call-back */
				uEOF();							/* bail out on error */
		break;

		default: break;
	}
	readLine(modemUReset, MAXLEN_MDMCMD, 1);	/* modem reset string */
	readLine(modemUInit, MAXLEN_MDMCMD, 1);		/* modem init string */
	readLine(modemDial, MAXLEN_DIALEXTRA+2, 0);	/* dial char(s) + method */
	readLine(modemUConn, MAXLEN_MDMSTR, 0);		/* modem connect string */

	efinish();
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Dialer routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#ifdef _SLOW_MODEM_
#define MODEM_SLEEP 2
#else
#define MODEM_SLEEP 1
#endif

int init_modem(char *initCmd)					/* initialize modem */
{
	int rcode;

	rcode = child(Chat,"-V","TIMEOUT","3",
				  "", modemUReset, "OK", initCmd,
				  (char *)0);
	if (rcode) {
		rcode = child(Chat,"-V","TIMEOUT","3",
					  "", "+++\\c", "OK", modemUReset,
					  "OK", initCmd, (char *)0);
		if (rcode)
			rcode = child(Chat,"-V","TIMEOUT","3",
						  "","AT","OK-AT-OK",
						  modemUReset, "OK",
						  initCmd, (char *)0);
	}
	sleep(MODEM_SLEEP);
	return rcode;
}

/* Dial the number given and process the connection mode selected */

#define MAXLEN_PREAMBLE 512
#define MAXLEN_CBTRAIL  256
#define MAXLEN_SCRIPT   (MAXLEN_PREAMBLE + MAXLEN_DIALEXTRA +MAXLEN_PHONE + \
						 MAXDIG_CNWAIT + MAXLEN_MDMSTR + \
                         2*MAXNUM_SLINES * (MAXLEN_SLINE + MAXLEN_ACCOUNT) + \
                         MAXLEN_CBTRAIL + MAXDIG_CNWAIT + MAXLEN_MDMSTR)

int callnumber(char *number)
{
	int i, rcode;
	FILE *scriptF;
	char preambleD[MAXLEN_PREAMBLE + 1] =
			"TIMEOUT\t\t3\nABORT\t\tBUSY\nABORT\t\t'NO CARRIER'\n"
			"ABORT\t\t'NO DIALTONE'\nABORT\t\tenied\nABORT\t\timeout\n"
			"''\t\t\t'AT %s %s'\nTIMEOUT\t\t%d\n%s\t\t\\c\n"
			"TIMEOUT\t\t5\n\\r\t\t\\c\n",
		 preambleCB[MAXLEN_PREAMBLE + 1] =
			"TIMEOUT\t\t3\nABORT\t\tBUSY\nABORT 'NO DIALTONE'\n"
			"ABORT\t\tenied\nABORT\t\timeout\n''\t\t\t'AT %s %s'\n"
			"TIMEOUT\t\t%d\n%s\t\t\\c\nTIMEOUT\t\t5\n\\r\t\t\\c\n",
		 cbtrail[MAXLEN_CBTRAIL+1] =
			"TIMEOUT\t\t%d\nRING\t\tATA\nTIMEOUT\t\t30\n%s\t\t\\c\n"
			"TIMEOUT\t\t5\n\\r\t\t\\c\n",
		 script[MAXLEN_SCRIPT + 1];

	umask(077);									/* read/write only by owner */
	scriptF = fopen(scriptfname, "w");			/* open the script file */
	if (scriptF == NULL)						/* failed to do so? */
		doErr("callnumber: fopen");				/* yup, bail out */
	switch (mode) {
		case SCRIPT_DIALIN:
			strcpy(script, preambleD);			/* normal dialing preamble */
			for (i=0; i<numSlines; i++) {		/* append the script lines */
				strcat(script, sline[i]);
				strcat(script, "\n");
			}
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case SCRIPT_DIALIN|SCRIPT_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			for (i=0; i<numSlines; i++) {		/* append the script lines */
				strcat(script, sline[i]);
				strcat(script, "\n");
			}
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			strcpy(script, cbtrail);			/* add the call-back trail */
			for (i=0; i<numCBSlns; i++) {		/* append call-back lines */
				strcat(script, CBsln[i]);
				strcat(script, "\n");
			}
			fprintf(scriptF, script,			/* using script as format */
					CBDelay, modemUConn);		/* insert all parameters */
			break;

		case SCRIPT_DIALIN|MANUAL_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			for (i=0; i<numSlines; i++) {		/* append the script lines */
				strcat(script, sline[i]);
				strcat(script, "\n");
			}
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case MANUAL_DIALIN:
			strcpy(script, preambleD);			/* normal dialing preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case MANUAL_DIALIN|SCRIPT_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case MANUAL_DIALIN|MANUAL_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case AUTH_DIALIN:
		case AUTH_DIALIN|NT_RAS_DIALIN:
			strcpy(script, preambleD);			/* normal dialing preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case AUTH_DIALIN|AUTH_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			strcpy(script, cbtrail);			/* add the call-back trail */
			fprintf(scriptF, script,			/* using script as format */
					CBDelay, modemUConn);		/* insert all parameters */
			break;

		case AUTH_DIALIN|NT_RAS_CALLBACK:
			strcpy(script, cbtrail);			/* copy call-back trail */
			fprintf(scriptF, script,			/* using script as format */
					CBDelay, modemUConn);		/* insert all parameters */
			break;

		case AUTH_DIALIN|MANUAL_DIALIN:
		case AUTH_DIALIN|MANUAL_DIALIN|
			 NT_RAS_DIALIN:
			strcpy(script, preambleD);			/* normal dialing preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		case AUTH_DIALIN|MANUAL_DIALIN|
			 AUTH_CALLBACK:
			strcpy(script, preambleCB);			/* copy call-back preamble */
			fprintf(scriptF, script, modemDial,	/* using script as format */
					number, connectWait,		/* insert all parameters */
					modemUConn);
			break;

		default:								/* no valid mode selected! */
			pprintf("\n%s: %s\n", "xispdial",	/* so report it */
					"invalid connection mode!");
			close(pipeFD);						/* close the report pipe */
			exit(1);							/* and bail out */
			break;
	}
	fclose(scriptF);							/* close the script file */
	rcode = child(Chat,"-V","-f",				/* hand it over to chat */
				  scriptfname,(char *)0);
	unlink(scriptfname);						/* done with script, remove */
	if (rcode == 0) {							/* chat terminated OK? */
		if (mode & NT_RAS_CALLBACK)				/* send "done" message */
			pprintf("\n%s: %s\n","xispdial",	/* according to type */
					"done RAS call-back");
		else
			pprintf("\n%s: %s %s\n","xispdial",
					"done dialing", number);
		if (mode & (MANUAL_DIALIN |				/* carry on if manual login */
					MANUAL_CALLBACK)) {			/* selected for either phase */
			if (mode & MANUAL_DIALIN) {			/* if manual login selected */
				if (child(Terminal, "-bgcol",	/* for dial-in phase, then */
						  bgcol, "-display",	/* start terminal app  */
						  dispname, "-geometry",
						  geometry, (char *)0))	/* if terminal aborted */
				{
					close(pipeFD);				/* then close report pipe */
					exit(1);					/* tell pppd: connect failed */
				}
			}
			if (mode & (SCRIPT_CALLBACK |		/* if some type of */
						MANUAL_CALLBACK |		/* call-back is selected */
						AUTH_CALLBACK)) {
				scriptF=fopen(scriptfname,"w");	/* create a new script file */
				if (scriptF == NULL)			/* failed to do so? */
					doErr("callnumber: fopen");	/* yup, bail out */
				strcpy(script, cbtrail);		/* use call-back trail only */
				if (mode & SCRIPT_CALLBACK) {	/* if call-back is scripted */
					for (i=0;i<numCBSlns;i++) {	/* append call-back lines */
					  strcat(script, CBsln[i]);
					  strcat(script, "\n");
					}
				}
				fprintf(scriptF, script, 		/* using script as format */
						CBDelay, modemUConn);	/* insert all parameters */
				fclose(scriptF);				/* close the script file */
				child(Chat,"TIMEOUT","1",		/* swallow whatever's left */
					  "NO CARRIER","\\c",		/* from the terminal session */
					  (char *)0);
				init_modem(modemInit);			/* re-initialize the modem */
				rcode = child(Chat,"-V","-f",	/* hand script over to chat */
							  scriptfname,
							  (char *)0);
				unlink(scriptfname);			/* done with script, remove */
			}
			if (rcode == 0 &&					/* if we're still OK and */
				(mode & MANUAL_CALLBACK)) {		/* manual login selected */
				if (child(Terminal, "-bgcol",	/* for call-back phase */
						  bgcol, "-display",	/* then start terminal app */
						  dispname, "-geometry",
						  geometry, (char *)0))	/* if terminal aborted */
				{
					close(pipeFD);				/* then close report pipe */
					exit(1);					/* tell pppd: connect failed */
				}
			}
			if (rcode != 0 ) {					/* check what happened */
				if (rcode < 0) {				/* problem with chat? */
					pprintf("\n%s: %s\n",		/* yes, report it */
					 "xispdial", "chat"
					 "execution failed!");
					close(pipeFD);				/* close the report pipe */
					exit(1);					/* and bail out */
				}
				return rcode;					/* else valid report code */
			}
		}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |  This is the only point in xispdial where we can exit with status 0,    |
  |  i.e the only case in which status returned to pppd indicates success.  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

		if (mode & NT_RAS_DIALIN)				/* NT-RAS call-back -> send */
			pprintf("\nxispdial: NT-RAS\n");	/* trigger for 2nd "Connect" */
		close(pipeFD);							/* close the report pipe */
		exit(rcode);							/* and exit */

	}
	if (rcode < 0) {							/* problem with chat? */
		pprintf("\n%s: %s\n", "xispdial",		/* yes, report it */
				"chat execution failed!");
		close(pipeFD);							/* and bail out */
		exit(1);
	}
	return rcode;								/* else valid report code */
}

void callall(int sn)							/* call the list of numbers */
{
	int i, rcode;

	pprintf("%s: %s #%d\n", "xispdial",			/* report dialing attempt */
			"dialing attempt", sn+1);
	for (i=0; i<numPhones; i++) {				/* try all phones given */
		pprintf("xispdial: dialing: %s\n",		/* report dialed number */
				phone[i]);
		if ((rcode = callnumber(phone[i])))		/* dial it */
			pprintf("\n");						/* not successful, print NL */
		if (rcode == 3) {						/* did chat timeout? */
			pprintf("TIMEOUT\n");				/* yup, report it */
			pprintf("%s: %s %s\n",
				"xispdial", "timeout dialing",
				phone[i]);
			child(Chat,"-V",""," ",(char *)0);	/* and hang-up */
		}
		sleep(sleepDelay);						/* sleep a while */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         SIGTERM trap and Main                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void Term(int signum)
{
	switch (signum) {
		case SIGTERM: {							/* SIGTERM signal received */
			pprintf("\nxispdial: SIGTERM\n");	/* return status via pipe */
			if (childPID)						/* if a child is in progress */
				kill(childPID, SIGTERM);		/* send it SIGTERM also */
			child(Chat,"-V",""," ",(char *)0);	/* try hanging-up */
#if !ISPENV_USEVARS
			unlink(envfname);					/* remove environment file */
#endif /* !ISPENV_USEVARS */
			unlink(scriptfname);				/* and script file */
			sleep(MODEM_SLEEP);					/* wait a sec before */
			init_modem("\\c");					/* reseting the modem */
			close(pipeFD);						/* close the report pipe */
			exit(1);							/* and exit */
			break;
		}
		default: exit(1);
	}
}

int main()
{
	int rcode, attempt;

	signal(SIGTERM, Term);			/* register SIGTERM handler */
	initFnames();					/* assemble file names used */
	getISPenv();					/* get info via dialing environment file */
	pipeFD = namedPipe(Pipe);		/* initialize named pipe to xISP */

	pprintf("xispdial: PID=%d.\n", getpid());	/* the '.' is important! */
	if (mode & AUTH_DIALIN)
		pprintf("xispdial: dialing ISP: using PAP/CHAP\n");
	else if (mode & MANUAL_DIALIN)
		pprintf("xispdial: dialing ISP: using xispterm\n");
	else
		pprintf("xispdial: dialing ISP: user: %s\n", account);

/* Prepare the modem init string
   ----------------------------- */
	modemInit = malloc(strlen(modemUInit)+4+1);
	if (modemInit == NULL)
		outOfMem("main");
	strcpy(modemInit, modemUInit);
	strcat(modemInit, " H0");

/* Initialize the modem for dialing
   -------------------------------- */
	if ((rcode=init_modem(modemInit))) {
		pprintf("xispdial: chat return status = %d!\n", rcode);
		pprintf("xispdial: modem init failed!\n");
		close(pipeFD);
		return 1;
	}

/* Loop dialing the numbers given maxAttempts times
   ------------------------------------------------ */
	for (attempt=0; attempt<maxAttempts; attempt++) {
		pprintf("\n");
		callall(attempt);
		if ((rcode=init_modem(modemInit))) {
			pprintf("xispdial: chat return status = %d!\n", rcode);
			pprintf("xispdial: modem re-init failed!\n");
			close(pipeFD);
			return 1;
		}
	}
	close(pipeFD);
	return 1;
}

