/*
  Copyright (C) 1997-2002  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <string.h>
#include <sys/param.h>
#include <unistd.h>

#ifdef SUNOS41x
 extern int socket(), ioctl(), sscanf(), printf(), pclose();
#elif defined(SUNOS5x)
 #include <sys/sockio.h>
#elif defined(linux)

/* Returns kernel revision number (10*major+minor). Only used on Linux to
   figure out what kind of parsing format to use for /proc/net/dev output */

static int kernelRevNo(void)
{
	int major, minor, ver = -1;
	FILE *pfp;

	pfp = popen("/bin/uname -r 2>&1", "r");
	if (pfp != NULL && fscanf(pfp, "%d.%d.%*d", &major, &minor) > 1)
		ver = major*10 + minor;
	pclose(pfp);
	return ver;
}
#endif


/* Gets interface address by reading /proc/net/route to make sure the
   desired interface exists, and then requesting its address. For
   SunOS-4.1.x/5.x, it parses the output of `netstat -nr` to avoid need
   for group suid privileges. Returns pointer to a string representation
   of the IP address, or NULL. */

char *IFAddr(char *IFName, int dest)
{
	int skfd = -1, request = SIOCGIFADDR;
	static char IP[16]={0};
	struct ifreq ifr;
	struct sockaddr_in *sinp;

	skfd = socket(AF_INET, SOCK_DGRAM, 0);			/* open net channel */
	if (skfd < 0)									/* make sure all is OK */
		return NULL;								/* if failed, bail out */
	do {
		strcpy(ifr.ifr_name, IFName);
		if (dest)									/* dest rather than src? */
			request = SIOCGIFDSTADDR;				/* yes, adjust request */
		if (ioctl(skfd, request, &ifr) < 0) {		/* get interface address */
			close(skfd);							/* if failed, close */
			return NULL;							/* socket and exit */
		}
		sinp = (struct sockaddr_in*)&ifr.ifr_addr;	/* point to data */
	} while (! sinp->sin_addr.s_addr);				/* sometimes 0 returned */
	close(skfd);									/* close the socket */
	strcpy(IP, inet_ntoa(sinp->sin_addr));			/* copy converted string */
	return IP;										/* return its address */
}

/* System dependent macros for pppAddr() */

#ifdef SUNOS41x
 #define AddrOpen   popen("/usr/ucb/netstat -nr", "r")
 #define AddrScanf  sscanf(routeLine, "%*s %*s %*s %*s %*s %s", IF)
 #define AddrClose  while (fgets(routeLine,128,infofp) != NULL); pclose(infofp)
#elif defined(SUNOS5x) || defined(__FreeBSD__)
 #define AddrOpen   popen("/usr/bin/netstat -nr", "r")
 #define AddrScanf  sscanf(routeLine, "%*s %*s %*s %*s %*s %s", IF)
 #define AddrClose  while (fgets(routeLine,128,infofp) != NULL); pclose(infofp)
#elif (defined(BSD) && BSD >= 199306)
 #define AddrOpen   popen("/usr/bin/netstat -nr", "r")
 #define AddrScanf  sscanf(routeLine, "%*s %*s %*s %*s %*s %*s %s", IF)
 #define AddrClose  while (fgets(routeLine,128,infofp) != NULL); pclose(infofp)
#else
 #define AddrOpen   fopen("/proc/net/route", "r")
 #define AddrScanf  sscanf(routeLine,"%s", IF)
 #define AddrClose  fclose(infofp)
#endif

char *pppAddr(char *IFName)
{
	int IFLen = strlen(IFName);
	FILE *infofp;
	char routeLine[129], IF[8];

	infofp = AddrOpen;								/* open route info stream */
	if (infofp == NULL)								/* failed to open? */
		return NULL;								/* return NULL string */
	while (fgets(routeLine, 128, infofp) != NULL) {	/* open OK, read line */
		if (AddrScanf)								/* parse interface */
			if (! strncmp(IF,IFName,IFLen)) {		/* desired interface? */
				AddrClose;							/* close info stream */
				return IFAddr(IF, 0);				/* return its address */
			}
	}
	AddrClose;										/* nothing found, close */
	return NULL;									/* node and return NULL */
}


/* Using the /prod/net/dev entry, it parses and returns the sum
   of TX and RX packets on the requested network interface. For
   SunOS-4.1.x/5.x it parses the output of `netstat -n -I<IF>`
   to avoid group suid privileges. */

/* System dependent macros for pppPkts() */

#ifdef SUNOS41x
 #define CMDLINE   "/usr/ucb/netstat -n -I%s 2> /dev/null"
 #define PktsOpen  popen(sprintf(line, CMDLINE, IFName), "r")
 #define PktsScanf sscanf(line, "%s %*s %*s %*s %u %*s %u", \
						  interface, &recv, &trans)
 #define PktsClose while (fgets(line,128,infofp) != NULL); pclose(infofp)
#elif defined(SUNOS5x)
 #define CMDLINE   "/usr/bin/netstat -n -I %s 2> /dev/null"
 #define PktsOpen  popen((sprintf(line, CMDLINE, IFName)) ? line:line, "r")
 #define PktsScanf sscanf(line, "%s %*s %*s %*s %u %*s %u", \
						  interface, &recv, &trans)
 #define PktsClose while (fgets(line,128,infofp) != NULL); pclose(infofp)
#elif (defined(BSD) && BSD >= 199306)
 #define CMDLINE   "/usr/bin/netstat -n -I %s 2> /dev/null"
 #define PktsOpen  popen((sprintf(line, CMDLINE, IFName)) ? line:line, "r")
 #define PktsScanf sscanf(line, "%s %*s %s %*s %u %*s %u", \
						  interface, network, &recv, &trans)
 #define PktsClose while (fgets(line,128,infofp) != NULL); pclose(infofp)
#else
 #define PktsOpen  fopen("/proc/net/dev", "r")
 #define PktsScanf sscanf(line, ((kernelRevNo()>20) ? \
				  " %[^ \t\r\n:] %*c %*s %u %*s %*s %*s %*s %*s %*s %*s %u" : \
				  " %[^ \t\r\n:] %*c %u %*s %*s %*s %*s %u"), \
				  interface, &recv, &trans)
 #define PktsClose fclose(infofp)
#endif

unsigned pppPkts(char *IFName, unsigned long *pTX, unsigned long *pRX)
{
    int IFLen = strlen(IFName);
    FILE *infofp;
    char line[129], interface[16];
#if (defined(BSD) && BSD >= 199306)
    char network[16];
#endif
    unsigned recv = 0, trans = 0;

	infofp = PktsOpen;							/* open IF stats info stream */
    while (fgets(line, 128, infofp) != NULL) {
		PktsScanf;								/* read line by line */
		if (!strncmp(interface,IFName,IFLen)) {	/* scanning for interface */
#if (defined(BSD) && BSD >= 199306)
			/* NetBSD: discard non-conforming multicast line. */
			if (strncmp(network,"<Link>", strlen(network))) {
#endif
			PktsClose;							/* found, close info stream */
			if (pTX) *pTX = trans;				/* if return storage avail */
			if (pRX) *pRX = recv;				/* save TXed/RXed separately */
			return (recv + trans);				/* return total I/O packets */
#if (defined(BSD) && BSD >= 199306)
			}
#endif
		}
	}
	PktsClose;									/* close info stream */
	if (pTX) *pTX = 0;							/* indicate nothing found */
	if (pRX) *pRX = 0;
	return -1;
}

