/*
  Copyright (C) 1997-2002  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

/*
   Call cost calculation and PTT utility module. Work included herein was
   inspired by pppcosts, written by Tillmann Steinbrecher <tst@bigfoot.com>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <errno.h>
#include <varargs.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include "common.h"
#include "logs.h"
#include "version.h"

#ifdef SUNOS41x
extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;
extern time_t timelocal(), time();
extern int fputs(), fprintf(), vfprintf(), fflush(), fclose(), rename();
extern int sscanf(), fscanf(), setvbuf(), toupper();
extern long strtol();
#else
/* exit() replacement for emulating on_exit() functionality */
#define exit(x) {extern int exitStatus; exitStatus = x; exit(x);}
#endif

#ifdef RUNASEUID
int xisp_euidaccess();
#endif

/* PTT table of initializer functions and structures */

void at_initPtt(),			/* Oesterreichische Telekom, Austria */
	 be_initPtt(),			/* BelgaCom, Belgium */
	 bt_initPtt(),			/* British Telecom, United Kingdom */
	 cambridge_initPtt(),	/* Cambridge Cable Phone Company, Cambridge, UK */
	 ch_initPtt(),			/* Swiss PTT */
	 cz_initPtt(),			/* SPT TELECOM, a.s., CZ */
	 dt_initPtt(),			/* Deutsche Telekom A.G., Germany */
	 netcologne_initPtt(),	/* NetCologne GmbH, Germany */
	 arcor_initPtt(),		/* Arcor Online, Germany */
	 bol_eur_initPtt(),		/* Business Online AG (EUR), Germany */
	 bol_dm_initPtt(),		/* Business Online AG (DM), Germany */
	 es_initPtt(),			/* Telefnica de Espaa, Spain */
	 ft_initPtt(),			/* France Telecom, France */
	 gr_initPtt(),			/* OTE A.E., Hellas */
	 gr_epak_initPtt(),		/* OTE A.E. (EAK), Hellas */
	 it_initPtt(),			/* Italian PTT */
	 matav_initPtt(),		/* MATAV, Hungary */
	 nl_initPtt(),			/* Dutch PTT, the Netherlands */
	 ntt_initPtt(),			/* NTT, Japan */
	 pl_initPtt(),			/* Telekom. Polska S.A., Poland */
	 pt_initPtt(),			/* Portuguese PTT */
	 sat_initPtt(),			/* Telkom, South Africa */
	 sit_initPtt(),			/* Slovene Telekom */
	 tdk_initPtt(),			/* TeleDanmark, Denmark */
	 carnet_initPtt(),		/* CARNet, Croatia */
	 hinet_initPtt(),		/* Hinet, Croatia */
	 iskon_initPtt(),		/* ISKON, Croatia */
	 ibmcr_initPtt(),		/* IBM Net, Croatia */
	 hpy_initPtt(),			/* HPY, Helsinki Telephone Corp. */
	 telenor_initPtt(),		/* Telenor, Norway */
	 inetserv_initPtt(),	/* Inetservice, Germany */
	 komtel_initPtt(),		/* Komtel, Germany */
	 mobilkom_initPtt(),	/* Mobilkom, Germany */
	 ipy_initPtt();			/* IPY, Finland */

void (*pttInit[MAXNUM_PTT])(ptt_t*) = {
	at_initPtt,
	be_initPtt,
	bt_initPtt,
	cambridge_initPtt,
	ch_initPtt,
	cz_initPtt,
	dt_initPtt,
	netcologne_initPtt,
	arcor_initPtt,
	bol_eur_initPtt,
	bol_dm_initPtt,
	es_initPtt,
	ft_initPtt,
	gr_initPtt,
	gr_epak_initPtt,
	it_initPtt,
	matav_initPtt,
	nl_initPtt,
	ntt_initPtt,
	pl_initPtt,
	pt_initPtt,
	sat_initPtt,
	sit_initPtt,
	tdk_initPtt,
	carnet_initPtt,
	hinet_initPtt,
	iskon_initPtt,
	ibmcr_initPtt,
	hpy_initPtt,
	telenor_initPtt,
	inetserv_initPtt,
	komtel_initPtt,
	mobilkom_initPtt,
	ipy_initPtt
};

/* Local storage */

static char *costfn;			/* pointers to cost and logging file name */
static char *logfn;				/* and logging directort name strings for */
static char *logdirn;			/* use by module routines */
static char *costfnroot;		/* saved pointers to root-names for both */
static char *logfnroot;			/* files, for on-the-fly file-name changing */
static char *bkupfn;			/* scratch area for backup file names */
static float lastTime;			/* quantized instant from last unit charge */
static unsigned long units;		/* number of units for current call */
static float currentCost;		/* remember this for per-minute calculations */
static time_t startTime;		/* time when communication started */
static char *pttfn;				/* local pointer to the PTT file name string */
static char *curfn;				/* name of file currently being read */

/* And some default variables, useful for resetting whole structures */

static ruledate_t reset_adate = {0, 1};
static ruledate_t reset_edate = {11, 31};
static discount_t reset_discount = {0.0,{0,0,0},{24,0,0}};
static rule_t reset_rule = {RULE_WEEKALL,{0,1},{11,31},0,{0,0,0},{24,0,0},0.0};


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                            Utility routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int get_Easter(struct tm *ct) /* return Easter Sunday day_of_year style */
{
	signed int a,b,m,q,w,p,n,tt,mm;
	struct tm ptm;
	struct tm htm;
	time_t htt;
	
	/* O'Beirne's Easter calculation algorithm, valid 1900-2099 */
	n = ct->tm_year;
	a = n % 19;
	b = floor((7*a+1)/19);
	m = (11*a+4-b) % 29;
	q = floor(n/4);
	w = (n+q+31-m) % 7;
	p = 25-m-w;
	if (p>0) {
		tt=p;
		mm=4;
	}
 	else {
		tt=p+31;
 		mm=3;
	}

	/* change d.m.y to day_of_year */
	htm.tm_year=ct->tm_year;
	htm.tm_mon=mm-1;
	htm.tm_mday=tt;
	htm.tm_hour=12;
	htm.tm_min=0;
	htm.tm_sec=0;
#ifdef SUNOS41x
	htt=timelocal(&htm);
#else
	htt=mktime(&htm);
#endif
	memcpy(&ptm, localtime(&htt), sizeof(struct tm));
	
	return(ptm.tm_yday);
}


/* Return length of one "unit", based on the current time, the state of the
   current call, the billing type and the minimum charges, using the rules
   for dates and times defined in the PTT structure */

float getUnitLength(ptt_t *ptt, time_t tt)
{
	struct tm cts, *ct = &cts, sts, *st = &sts;
	float result;
	int rc, rule_applies = 0;
	unsigned long rst, ret, rtt, rct;

	memcpy(ct, localtime(&tt), sizeof(struct tm));
	memcpy(st, localtime(&startTime), sizeof(struct tm));
	result = ptt->dflt_tariff[ptt->current_zone];

	for (rc=0; rc<ptt->num_categories; rc++) {
		switch ((ptt->rule[ptt->current_zone][rc].type) & 0x3F) {
			case RULE_WEEKALL:
				rule_applies = 1;
				break;

			case RULE_WEEKDAY:
				rule_applies = (ct->tm_wday > 0 && ct->tm_wday < 6);
				break;

			case RULE_SATURDAY:
				rule_applies = (ct->tm_wday == 6);
				break;

			case RULE_SUNDAY:
				rule_applies = (ct->tm_wday == 0);
				break;

			case RULE_WEEKEND:
				rule_applies = (ct->tm_wday == 6 || ct->tm_wday == 0);
				break;

			case RULE_RHOLIDAY:
				rule_applies = (ct->tm_yday ==
								(get_Easter(ct) +
								ptt->rule[ptt->current_zone][rc].rdate));
				break;

			case RULE_AHOLIDAY:
				rule_applies =
				  (ct->tm_mon == ptt->rule[ptt->current_zone][rc].adate.mon &&
				   ct->tm_mday == ptt->rule[ptt->current_zone][rc].adate.day);
				break;

			case RULE_SPECIAL_WEEKDAY:
				rule_applies =
				  (ct->tm_mon >= ptt->rule[ptt->current_zone][rc].adate.mon &&
				   ct->tm_mon <= ptt->rule[ptt->current_zone][rc].edate.mon &&
				   ct->tm_mday >= ptt->rule[ptt->current_zone][rc].adate.day &&
				   ct->tm_mday <= ptt->rule[ptt->current_zone][rc].edate.day &&
				   ct->tm_wday > 0 && ct->tm_wday < 6);
				break;

			case RULE_SPECIAL_WEEKEND:
				rule_applies =
				  (ct->tm_mon >= ptt->rule[ptt->current_zone][rc].adate.mon &&
				   ct->tm_mon <= ptt->rule[ptt->current_zone][rc].edate.mon &&
				   ct->tm_mday >= ptt->rule[ptt->current_zone][rc].adate.day &&
				   ct->tm_mday <= ptt->rule[ptt->current_zone][rc].edate.day &&
				   (ct->tm_wday == 6 || ct->tm_wday == 0));
				break;

			default:
				rule_applies = 0;
				break;
		}
		if (rule_applies) {
			rst = ptt->rule[ptt->current_zone][rc].start.h * 3600 +
				  ptt->rule[ptt->current_zone][rc].start.m * 60 +
				  ptt->rule[ptt->current_zone][rc].start.s;
			ret = ptt->rule[ptt->current_zone][rc].end.h * 3600 +
				  ptt->rule[ptt->current_zone][rc].end.m * 60 +
				  ptt->rule[ptt->current_zone][rc].end.s;
			rct = ct->tm_hour * 3600 + ct->tm_min * 60 + ct->tm_sec;
			if (rct >= rst && rct < ret) {
				result = ptt->rule[ptt->current_zone][rc].tariff;

				if (ptt->rule[ptt->current_zone][rc].type &
					DISCOUNT_PERMITTED) {
					rst = ptt->discount[ptt->current_zone].start.h * 3600 +
						  ptt->discount[ptt->current_zone].start.m * 60 +
						  ptt->discount[ptt->current_zone].start.s;
					ret = ptt->discount[ptt->current_zone].end.h * 3600 +
						  ptt->discount[ptt->current_zone].end.m * 60 +
						  ptt->discount[ptt->current_zone].end.s;
					rtt = st->tm_hour * 3600 + st->tm_min * 60 + st->tm_sec;
					if ((rtt < rst || rtt >= ret) && (rct < rst || rct >= ret))
						result *=
						((100.-ptt->discount[ptt->current_zone].percent)/100.);
				}
				break;	/* apply first matching rule only */
			}
			else
				rule_applies=0;
		}
	}

	/* Extra discount may apply even when no rule applies (i.e. for
	   the default tariff), if discount is > 0% for the given zone */

	if (!rule_applies && ptt->discount[ptt->current_zone].percent > 0) {
		rst = ptt->discount[ptt->current_zone].start.h * 3600 +
		  	ptt->discount[ptt->current_zone].start.m * 60 +
		  	ptt->discount[ptt->current_zone].start.s;
		ret = ptt->discount[ptt->current_zone].end.h * 3600 +
		  	ptt->discount[ptt->current_zone].end.m * 60 +
		  	ptt->discount[ptt->current_zone].end.s;
		rtt = st->tm_hour * 3600 + st->tm_min * 60 + st->tm_sec;
		rct = ct->tm_hour * 3600 + ct->tm_min * 60 + ct->tm_sec;
		if ((rtt < rst || rtt >= ret) && (rct < rst || rct >= ret))
			result *= ((100.-ptt->discount[ptt->current_zone].percent)/100.);
	}

	if (ptt->attribs &
		(PTT_PER_MINUTE | PTT_PER_SECS)) {			/* "time-length" charge */
		if ((ptt->attribs & PTT_NONLINEAR_MIN)		/* if min-charge time */
			&& (units <= ptt->min_units)) {			/* length is non-linear */
			ptt->cost_quantum = ptt->min_cost;		/* and call started now */
			if (ptt->rule[ptt->current_zone]
				[rc].type & DEFEAT_MIN_TIME_LEN)	/* unless rule flag */
				return 0;							/* defeats min length */
			else
				return ptt->						/* ret min time length */
					min_cost_len[ptt->current_zone];
		}
		else {										/* else, min-len passed */
			ptt->cost_quantum = result;				/* or min-charge time */
			if (ptt->attribs & PTT_PER_SECS)		/* length is linear */
				ptt->cost_quantum *=				/* scale per-min tariff */
					(ptt->charge_period / 60.);		/* if period < 1 minute */
			return (float)ptt->charge_period;		/* ret charge period val */
		}
	}
	else											/* charge "by unit", so */
		return result;								/* ret unit time length */
}


/* Utility routines for initializing the various rule types.
   Defined for code readability in the "known" PTT entries */

rule_t mk_Weekday(ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_WEEKDAY;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_Saturday(ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_SATURDAY;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_Sunday(ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_SUNDAY;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_Weekend(ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_WEEKEND;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start  = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_Weekall(ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_WEEKALL;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_RHoliday(int rdate, ruletime_t start, ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_RHOLIDAY;
	r.adate = reset_adate;
	r.edate = reset_edate;
	r.rdate = rdate;
	r.start = start;
	r.end  = end;
	r.tariff = val;
	return r;
}

rule_t mk_AHoliday(ruledate_t adate, ruletime_t start, ruletime_t end,
				   float val)
{
	rule_t r;

	r.type = RULE_AHOLIDAY;
	r.adate = adate;
	r.edate = reset_edate;
	r.rdate = 0;
	r.start  = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_SWeekday(ruledate_t sdate, ruledate_t edate, ruletime_t start,
				   ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_SPECIAL_WEEKDAY;
	r.adate = sdate;
	r.edate = edate;
	r.rdate = 0;
	r.start  = start;
	r.end = end;
	r.tariff = val;
	return r;
}

rule_t mk_SWeekend(ruledate_t sdate, ruledate_t edate, ruletime_t start,
				   ruletime_t end, float val)
{
	rule_t r;

	r.type = RULE_SPECIAL_WEEKEND;
	r.adate = sdate;
	r.edate = edate;
	r.rdate = 0;
	r.start  = start;
	r.end = end;
	r.tariff = val;
	return r;
}

discount_t mk_discount(discount_t dc)
{
	discount_t r;

	r.percent = dc.percent;
	r.start  = dc.start;
	r.end  = dc.end;
	return r;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |              Phone costs for the Oesterreichische Telekom               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void at_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Austrian Telekom", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 1;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.804;
	strncpy(ptt->currency, "ATS", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 72;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] =
		mk_Weekall((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 72);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Phone costs for the Belgian phone company BelgaCom            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void be_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "BelgaCom (Belgium)", MAXLEN_PTTNAME);
	ptt->num_zones = 3;
	ptt->current_zone = 0;
	ptt->num_categories = 17;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 6.05;		/* 5.00 BEF, VAT = 21% */
	strncpy(ptt->currency, "BEF", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 720;
	ptt->dflt_tariff[1] = 360;
	ptt->dflt_tariff[2] = 100;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Zonal)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (IntZonA)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (IntZonB)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;

	ptt->rule[0][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][1] = mk_AHoliday((ruledate_t){2, 31}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][2] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][3] = mk_AHoliday((ruledate_t){4, 8}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][4] = mk_AHoliday((ruledate_t){4, 19}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][5] = mk_AHoliday((ruledate_t){6, 21}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][6] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][7] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][8] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][9] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 720);
	ptt->rule[0][10] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){9, 0, 0}, 360);
	ptt->rule[0][11] =
		mk_Weekday((ruletime_t){9, 0, 0}, (ruletime_t){12, 0, 0}, 240);
	ptt->rule[0][12] =
		mk_Weekday((ruletime_t){12, 0, 0}, (ruletime_t){13, 30, 0}, 360);
	ptt->rule[0][13] =
		mk_Weekday((ruletime_t){13, 30, 0}, (ruletime_t){17, 0, 0}, 240);
	ptt->rule[0][14] =
		mk_Weekday((ruletime_t){17, 0, 0}, (ruletime_t){18, 30, 0}, 360);
	ptt->rule[0][15] =
		mk_Weekday((ruletime_t){18, 30, 0}, (ruletime_t){24, 0, 0}, 720);
	ptt->rule[0][16] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 720);

	ptt->rule[1][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][1] = mk_AHoliday((ruledate_t){2, 31}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][2] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][3] = mk_AHoliday((ruledate_t){4, 8}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][4] = mk_AHoliday((ruledate_t){4, 19}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][5] = mk_AHoliday((ruledate_t){6, 21}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][6] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][7] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][8] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][9] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 360);
	ptt->rule[1][10] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){9, 0, 0}, 180);
	ptt->rule[1][11] =
		mk_Weekday((ruletime_t){9, 0, 0}, (ruletime_t){12, 0, 0}, 150);
	ptt->rule[1][12] =
		mk_Weekday((ruletime_t){12, 0, 0}, (ruletime_t){13, 30, 0}, 180);
	ptt->rule[1][13] =
		mk_Weekday((ruletime_t){13, 30, 0}, (ruletime_t){17, 0, 0}, 150);
	ptt->rule[1][14] =
		mk_Weekday((ruletime_t){17, 0, 0}, (ruletime_t){18, 30, 0}, 180);
	ptt->rule[1][15] =
		mk_Weekday((ruletime_t){18, 30, 0}, (ruletime_t){24, 0, 0}, 360);
	ptt->rule[1][16] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 360);

	ptt->rule[2][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][1] = mk_AHoliday((ruledate_t){2, 31}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][2] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][3] = mk_AHoliday((ruledate_t){4, 8}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][4] = mk_AHoliday((ruledate_t){4, 19}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][5] = mk_AHoliday((ruledate_t){6, 21}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][6] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][7] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][8] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][9] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 100);
	ptt->rule[2][10] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){9, 0, 0}, 50);
	ptt->rule[2][11] =
		mk_Weekday((ruletime_t){9, 0, 0}, (ruletime_t){12, 0, 0}, 40);
	ptt->rule[2][12] =
		mk_Weekday((ruletime_t){12, 0, 0}, (ruletime_t){13, 30, 0}, 50);
	ptt->rule[2][13] =
		mk_Weekday((ruletime_t){13, 30, 0}, (ruletime_t){17, 0, 0}, 40);
	ptt->rule[2][14] =
		mk_Weekday((ruletime_t){17, 0, 0}, (ruletime_t){18, 30, 0}, 50);
	ptt->rule[2][15] =
		mk_Weekday((ruletime_t){18, 30, 0}, (ruletime_t){24, 0, 0}, 100);
	ptt->rule[2][16] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 100);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Phone costs for British Telecom                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void bt_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "British Telecom (UK)", MAXLEN_PTTNAME);
	ptt->num_zones = 4;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_MINUTE;
	ptt->min_units = 0;
	ptt->min_cost = 0.05;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "", MAXLEN_CURRENCY);
	ptt->decimals = 1;
	ptt->dflt_tariff[0] = 0.0165;
	ptt->dflt_tariff[1] = 0.0395;
	ptt->dflt_tariff[2] = 0.0465;
	ptt->dflt_tariff[3] = 0.45;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	ptt->min_cost_len[3] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Regional)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (National)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Zone 4 (Premium)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;

	ptt->rule[0][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 0.0165);
	ptt->rule[0][1] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){18, 0, 0}, 0.0395);
	ptt->rule[0][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 0.0165);
	ptt->rule[0][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 0.01);

	ptt->rule[1][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 0.0395);
	ptt->rule[1][1] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){18, 0, 0}, 0.0823);
	ptt->rule[1][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 0.0395);
	ptt->rule[1][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 0.0329);

	ptt->rule[2][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 0.0465);
	ptt->rule[2][1] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){18, 0, 0}, 0.0879);
	ptt->rule[2][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 0.0465);
	ptt->rule[2][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 0.0329);

	ptt->rule[3][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 0.45);
	ptt->rule[3][1] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){18, 0, 0}, 0.50);
	ptt->rule[3][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 0.45);
	ptt->rule[3][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 0.45);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |    Phone costs for Cambridge Cable Phone Company, Cambridge, England    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void cambridge_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Cambridge Cable Phone Co. (UK)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.069;
	strncpy(ptt->currency, "", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 6;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){8, 0, 0}, 6);
	ptt->rule[0][1] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){18, 0, 0}, 1.45);
	ptt->rule[0][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 6);
	ptt->rule[0][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 6);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Phone costs for the Swiss phone company SwissCom              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void ch_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "SwissCom (Switzerland)", MAXLEN_PTTNAME);
	ptt->num_zones = 2;
	ptt->current_zone = 0;
	ptt->num_categories = 8;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.1;
	strncpy(ptt->currency, "Fr.", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 360;
	ptt->dflt_tariff[1] = 96;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Nachbereich)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Fernbereich)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){6, 0, 0}, 360);
	ptt->rule[0][1] =
		mk_Weekday((ruletime_t){6, 0, 0}, (ruletime_t){8, 0, 0}, 180);
	ptt->rule[0][2] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){17, 0, 0}, 90);
	ptt->rule[0][3] =
		mk_Weekday((ruletime_t){17, 0, 0}, (ruletime_t){19, 0, 0}, 180);
	ptt->rule[0][4] =
		mk_Weekday((ruletime_t){19, 0, 0}, (ruletime_t){21, 0, 0}, 90);
	ptt->rule[0][5] =
		mk_Weekday((ruletime_t){21, 0, 0}, (ruletime_t){23, 0, 0}, 180);
	ptt->rule[0][6] =
		mk_Weekday((ruletime_t){23, 0, 0}, (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][7] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 360);

	ptt->rule[1][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){6, 0, 0}, 96);
	ptt->rule[1][1] =
		mk_Weekday((ruletime_t){6, 0, 0}, (ruletime_t){8, 0, 0}, 48);
	ptt->rule[1][2] =
		mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){17, 0, 0}, 24);
	ptt->rule[1][3] =
		mk_Weekday((ruletime_t){17, 0, 0}, (ruletime_t){19, 0, 0}, 48);
	ptt->rule[1][4] =
		mk_Weekday((ruletime_t){19, 0, 0}, (ruletime_t){21, 0, 0}, 24);
	ptt->rule[1][5] =
		mk_Weekday((ruletime_t){21, 0, 0}, (ruletime_t){23, 0, 0}, 48);
	ptt->rule[1][6] =
		mk_Weekday((ruletime_t){23, 0, 0}, (ruletime_t){24, 0, 0}, 96);
	ptt->rule[1][7] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 96);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for the SPT TELECOM, a.s.                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void cz_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "SPT Telecom A.S. (CZ)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 2.25;
	strncpy(ptt->currency, "CZK", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 360;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] =
		mk_Weekday((ruletime_t){0, 0, 0}, (ruletime_t){7, 0, 0}, 360);
	ptt->rule[0][1] =
		mk_Weekday((ruletime_t){7, 0, 0}, (ruletime_t){18, 0, 0}, 180);
	ptt->rule[0][2] =
		mk_Weekday((ruletime_t){18, 0, 0}, (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][3] =
		mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0}, 360);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for Deutsche Telekom A.G.                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void dt_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Deutsche Telekom A.G. (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 4;
	ptt->current_zone = 0;
	ptt->num_categories = 59;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.12;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 240;
	ptt->dflt_tariff[1] = 120;
	ptt->dflt_tariff[2] = 120;
	ptt->dflt_tariff[3] = 120;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	ptt->min_cost_len[3] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Up to 20km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (20 to 50km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (50 to 200km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Zone 4 (Over 200km)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;

	ptt->rule[0][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][1] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][2] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][4] = mk_RHoliday(-2, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][5] = mk_RHoliday(-2, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][7] = mk_RHoliday(0, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][8] = mk_RHoliday(0, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][9] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][10] = mk_RHoliday(1, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][11] = mk_RHoliday(1, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][12] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][13] = mk_RHoliday(39, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][14] = mk_RHoliday(39, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][15] = mk_RHoliday(49, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][16] = mk_RHoliday(49, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][17] = mk_RHoliday(49, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][18] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][19] = mk_RHoliday(50, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][20] = mk_RHoliday(50, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][21] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][22] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][23] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][24] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][25] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][26] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][27] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][28] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][29] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][30] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][31] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][32] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][33] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][34] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][35] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][36] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][37] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][38] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][39] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][40] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][41] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][42] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){2, 0, 0}, 240);
	ptt->rule[0][43] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){2, 0, 0},
								   (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][44] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){9, 0, 0}, 150);
	ptt->rule[0][45] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){9, 0, 0},
								   (ruletime_t){12, 0, 0}, 90);
	ptt->rule[0][46] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){12, 0, 0},
								   (ruletime_t){18, 0, 0}, 90);
	ptt->rule[0][47] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){18, 0, 0},
								   (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][48] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][49] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][50] = mk_Weekend((ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][51] = mk_Weekend((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][52] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){2, 0, 0}, 240);
	ptt->rule[0][53] = mk_Weekday((ruletime_t){2, 0, 0},
								  (ruletime_t){5, 0, 0}, 240);
	ptt->rule[0][54] = mk_Weekday((ruletime_t){5, 0, 0},
								  (ruletime_t){9, 0, 0}, 150);
	ptt->rule[0][55] = mk_Weekday((ruletime_t){9, 0, 0},
								  (ruletime_t){12, 0, 0}, 90);
	ptt->rule[0][56] = mk_Weekday((ruletime_t){12, 0, 0},
								  (ruletime_t){18, 0, 0}, 90);
	ptt->rule[0][57] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){21, 0, 0}, 150);
	ptt->rule[0][58] = mk_Weekday((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);

	ptt->rule[1][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][1] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][2] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][4] = mk_RHoliday(-2, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][5] = mk_RHoliday(-2, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][7] = mk_RHoliday(0, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][8] = mk_RHoliday(0, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][9] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][10] = mk_RHoliday(1, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][11] = mk_RHoliday(1, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][12] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][13] = mk_RHoliday(39, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][14] = mk_RHoliday(39, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][15] = mk_RHoliday(49, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][16] = mk_RHoliday(49, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][17] = mk_RHoliday(49, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][18] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][19] = mk_RHoliday(50, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][20] = mk_RHoliday(50, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][21] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][22] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][23] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][24] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][25] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][26] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][27] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][28] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][29] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][30] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][31] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][32] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][33] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][34] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][35] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][36] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][37] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][38] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][39] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][40] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][41] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][42] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){2, 0, 0}, 60);
	ptt->rule[1][43] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){2, 0, 0},
								   (ruletime_t){5, 0, 0}, 120);
	ptt->rule[1][44] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){9, 0, 0}, 45);
	ptt->rule[1][45] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){9, 0, 0},
								   (ruletime_t){12, 0, 0}, 36);
	ptt->rule[1][46] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){12, 0, 0},
								   (ruletime_t){18, 0, 0}, 36);
	ptt->rule[1][47] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){18, 0, 0},
								   (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][48] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][49] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 60);
	ptt->rule[1][50] = mk_Weekend((ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][51] = mk_Weekend((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 60);
	ptt->rule[1][52] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){2, 0, 0}, 60);
	ptt->rule[1][53] = mk_Weekday((ruletime_t){2, 0, 0},
								  (ruletime_t){5, 0, 0}, 120);
	ptt->rule[1][54] = mk_Weekday((ruletime_t){5, 0, 0},
								  (ruletime_t){9, 0, 0}, 45);
	ptt->rule[1][55] = mk_Weekday((ruletime_t){9, 0, 0},
								  (ruletime_t){12, 0, 0}, 26);
	ptt->rule[1][56] = mk_Weekday((ruletime_t){12, 0, 0},
								  (ruletime_t){18, 0, 0}, 30);
	ptt->rule[1][57] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){21, 0, 0}, 45);
	ptt->rule[1][58] = mk_Weekday((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 60);

	ptt->rule[2][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][1] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][2] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][4] = mk_RHoliday(-2, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][5] = mk_RHoliday(-2, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][7] = mk_RHoliday(0, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][8] = mk_RHoliday(0, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][9] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][10] = mk_RHoliday(1, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][11] = mk_RHoliday(1, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][12] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][13] = mk_RHoliday(39, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][14] = mk_RHoliday(39, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][15] = mk_RHoliday(49, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][16] = mk_RHoliday(49, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][17] = mk_RHoliday(49, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][18] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][19] = mk_RHoliday(50, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][20] = mk_RHoliday(50, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][21] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][22] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][23] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][24] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][25] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][26] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][27] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][28] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][29] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][30] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][31] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][32] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][33] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][34] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][35] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][36] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][37] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][38] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][39] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][40] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][41] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][42] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){2, 0, 0}, 36);
	ptt->rule[2][43] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){2, 0, 0},
								   (ruletime_t){5, 0, 0}, 120);
	ptt->rule[2][44] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){9, 0, 0}, 36);
	ptt->rule[2][45] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){9, 0, 0},
								   (ruletime_t){12, 0, 0}, 36);
	ptt->rule[2][46] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){12, 0, 0},
								   (ruletime_t){18, 0, 0}, 36);
	ptt->rule[2][47] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){18, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[2][48] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][49] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[2][50] = mk_Weekend((ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 22.5);
	ptt->rule[2][51] = mk_Weekend((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[2][52] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){2, 0, 0}, 36);
	ptt->rule[2][53] = mk_Weekday((ruletime_t){2, 0, 0},
								  (ruletime_t){5, 0, 0}, 120);
	ptt->rule[2][54] = mk_Weekday((ruletime_t){5, 0, 0},
								  (ruletime_t){9, 0, 0}, 22.5);
	ptt->rule[2][55] = mk_Weekday((ruletime_t){9, 0, 0},
								  (ruletime_t){12, 0, 0}, 13);
	ptt->rule[2][56] = mk_Weekday((ruletime_t){12, 0, 0},
								  (ruletime_t){18, 0, 0}, 14);
	ptt->rule[2][57] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){21, 0, 0}, 22.5);
	ptt->rule[2][58] = mk_Weekday((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);

	ptt->rule[3][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][1] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][2] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][4] = mk_RHoliday(-2, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][5] = mk_RHoliday(-2, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][7] = mk_RHoliday(0, (ruletime_t){5, 0, 0},
								  (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][8] = mk_RHoliday(0, (ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][9] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][10] = mk_RHoliday(1, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][11] = mk_RHoliday(1, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][12] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][13] = mk_RHoliday(39, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][14] = mk_RHoliday(39, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][15] = mk_RHoliday(49, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][16] = mk_RHoliday(49, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][17] = mk_RHoliday(49, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][18] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][19] = mk_RHoliday(50, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][20] = mk_RHoliday(50, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][21] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][22] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][23] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][24] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][25] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][26] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][27] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][28] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][29] = mk_AHoliday((ruledate_t){11, 24}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][30] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][31] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][32] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][33] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][34] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][35] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][36] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][37] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][38] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][39] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 36);
	ptt->rule[3][40] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][41] = mk_SWeekend((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][42] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){2, 0, 0}, 36);
	ptt->rule[3][43] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){2, 0, 0},
								   (ruletime_t){5, 0, 0}, 120);
	ptt->rule[3][44] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){5, 0, 0},
								   (ruletime_t){9, 0, 0}, 36);
	ptt->rule[3][45] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){9, 0, 0},
								   (ruletime_t){12, 0, 0}, 36);
	ptt->rule[3][46] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){12, 0, 0},
								   (ruletime_t){18, 0, 0}, 36);
	ptt->rule[3][47] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){18, 0, 0},
								   (ruletime_t){21, 0, 0}, 36);
	ptt->rule[3][48] = mk_SWeekday((ruledate_t){11, 27}, (ruledate_t){11, 30},
								   (ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 36);
	ptt->rule[3][49] = mk_Weekend((ruletime_t){0, 0, 0},
								   (ruletime_t){5, 0, 0}, 30);
	ptt->rule[3][50] = mk_Weekend((ruletime_t){5, 0, 0},
								   (ruletime_t){21, 0, 0}, 21.5);
	ptt->rule[3][51] = mk_Weekend((ruletime_t){21, 0, 0},
								   (ruletime_t){24, 0, 0}, 30);
	ptt->rule[3][52] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){2, 0, 0}, 30);
	ptt->rule[3][53] = mk_Weekday((ruletime_t){2, 0, 0},
								  (ruletime_t){5, 0, 0}, 120);
	ptt->rule[3][54] = mk_Weekday((ruletime_t){5, 0, 0},
								  (ruletime_t){9, 0, 0}, 21.5);
	ptt->rule[3][55] = mk_Weekday((ruletime_t){9, 0, 0},
								  (ruletime_t){12, 0, 0}, 12);
	ptt->rule[3][56] = mk_Weekday((ruletime_t){12, 0, 0},
								  (ruletime_t){18, 0, 0}, 13.5);
	ptt->rule[3][57] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){21, 0, 0}, 21.5);
	ptt->rule[3][58] = mk_Weekday((ruletime_t){21, 0, 0},
								  (ruletime_t){24, 0, 0}, 30);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                 Phone costs for NetCologne GmbH, Germany                |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void netcologne_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "NetCologne GmbH (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 4;
	ptt->current_zone = 0;
	ptt->num_categories = 15;
	ptt->attribs = PTT_PER_SECS | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;

	ptt->dflt_tariff[0] = 0.02;
	ptt->dflt_tariff[1] = 0.035;
	ptt->dflt_tariff[2] = 0.09;
	ptt->dflt_tariff[3] = 0.07;

	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	ptt->min_cost_len[3] = 0.0;

	strncpy(ptt->zone_name[0], "Interfonieren", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "City", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Deutschland", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Interfonieren + Timeline", MAXLEN_ZNAME);

	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;

	ptt->rule[0][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][1] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][2] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][4] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][5] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][6] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][7] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][8] = mk_RHoliday(-48, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][9] = mk_RHoliday(-52, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][10] = mk_SWeekend((ruledate_t){11, 25}, (ruledate_t){11, 26},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][11] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);
	ptt->rule[0][12] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.02);
	ptt->rule[0][13] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){18, 0, 0}, 0.04);
	ptt->rule[0][14] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.02);

	ptt->rule[1][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][1] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][2] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][4] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][5] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][6] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][7] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][8] = mk_RHoliday(-48, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][9] = mk_RHoliday(-52, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][10] = mk_SWeekend((ruledate_t){11, 25}, (ruledate_t){11, 26},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][11] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);
	ptt->rule[1][12] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.035);
	ptt->rule[1][13] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){18, 0, 0}, 0.06);
	ptt->rule[1][14] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.035);

	ptt->rule[2][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][1] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][2] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][4] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][5] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][6] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][7] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][8] = mk_RHoliday(-48, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][9] = mk_RHoliday(-52, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][10] = mk_SWeekend((ruledate_t){11, 25}, (ruledate_t){11, 26},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][11] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);
	ptt->rule[2][12] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.09);
	ptt->rule[2][13] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){18, 0, 0}, 0.19);
	ptt->rule[2][14] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.09);

	ptt->rule[3][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][1] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][2] = mk_AHoliday((ruledate_t){9, 3}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][3] = mk_RHoliday(-2, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][4] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][5] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][6] = mk_RHoliday(39, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][7] = mk_RHoliday(50, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][8] = mk_RHoliday(-48, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][9] = mk_RHoliday(-52, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][10] = mk_SWeekend((ruledate_t){11, 25}, (ruledate_t){11, 26},
								   (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][11] = mk_Weekend((ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
	ptt->rule[3][12] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.07);
	ptt->rule[3][13] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){18, 0, 0}, 0.09);
	ptt->rule[3][14] = mk_Weekday((ruletime_t){18, 0, 0},
								  (ruletime_t){24, 0, 0}, 0.07);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for Arcor Online, Germany                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void arcor_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Arcor Online (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 1;
	ptt->attribs = PTT_PER_MINUTE | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.06;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Arcor Internet by Call", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.06);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |          Phone costs (in EUR) for Business Online AG, Germany           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void bol_eur_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Business Online AG-EUR (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 6;
	ptt->attribs = PTT_PER_SECS | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "EUR", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.025;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Bundesweit", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){5, 0, 0}, 0.015);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){5, 0, 0},
								 (ruletime_t){9, 0, 0}, 0.025);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){9, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.041);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){21, 0, 0}, 0.025);
	ptt->rule[0][4] = mk_Weekday((ruletime_t){21, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.015);
	ptt->rule[0][5] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.025);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Phone costs (in DM) for Business Online AG, Germany           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void bol_dm_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Business Online AG-DM (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 6;
	ptt->attribs = PTT_PER_SECS | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.048;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Bundesweit", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){5, 0, 0}, 0.03);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){5, 0, 0},
								 (ruletime_t){9, 0, 0}, 0.048);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){9, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.08);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){21, 0, 0}, 0.048);
	ptt->rule[0][4] = mk_Weekday((ruletime_t){21, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.03);
	ptt->rule[0][5] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.048);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                 Phone costs for Telefnica de Espaa                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void es_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Telefonica de Espana (Spain)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 18;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 1;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 6.61;
	strncpy(ptt->currency, "Ptas", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 240;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Urbana e Infovia)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][1] = mk_AHoliday((ruledate_t){0, 6}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][2] = mk_AHoliday((ruledate_t){2, 28}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][3] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][4] = mk_AHoliday((ruledate_t){6, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][5] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][6] = mk_AHoliday((ruledate_t){9, 12}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][7] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][8] = mk_AHoliday((ruledate_t){11, 6}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][9] = mk_AHoliday((ruledate_t){11, 8}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][10] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][11] = mk_Sunday((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][12] = mk_Saturday((ruletime_t){0, 0, 0},
								   (ruletime_t){8, 0, 0}, 240);
	ptt->rule[0][13] = mk_Saturday((ruletime_t){8, 0, 0},
								   (ruletime_t){14, 0, 0}, 180);
	ptt->rule[0][14] = mk_Saturday((ruletime_t){14, 0, 0},
								   (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][15] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 240);
	ptt->rule[0][16] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){22, 0, 0}, 180);
	ptt->rule[0][17] = mk_Weekday((ruletime_t){22, 0, 0},
								  (ruletime_t){24, 0, 0}, 240);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Phone costs for France Telecom                     |
  |                         (last update June 1998)                         |
  | Changes :                                                               |
  |    * "Primaliste Internet" discount changed from -40% to -50%           |
  |    * "Primaliste" discount is added (-20% everytime)                    |
  |    * "Primaliste and Primaliste Internet" can now be cumulated          |
  | http://                                                                 |
  |    www.france-telecom.com/vfrance/produits/html/fiche.htsql?id=vprimint |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void ft_initPtt(ptt_t *ptt)
{
	int z;

	strncpy(ptt->name, "France Telecom", MAXLEN_PTTNAME);
	ptt->num_zones = 12;
	ptt->current_zone = 3;
	ptt->num_categories = 11;
	ptt->attribs = PTT_PER_SECS | PTT_CUR_AFTER_COST | PTT_NONLINEAR_MIN;
	ptt->min_units = 0;
	ptt->min_cost = 0.74;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "FF", MAXLEN_CURRENCY);
	ptt->decimals = 2;
 	ptt->dflt_tariff[0] = 0.140;		/* zone 1 - locale */
 	ptt->dflt_tariff[1] = 0.140 * 0.80;	/* zone 1 + primaliste */
 	ptt->dflt_tariff[2] = 0.140;		/* zone 1 + prim inet */
 	ptt->dflt_tariff[3] = 0.140 * 0.80;	/* zone 1 + prim + prim inet */
 	ptt->dflt_tariff[4] = 0.200;
 	ptt->dflt_tariff[5] = 0.200 * 0.80;
 	ptt->dflt_tariff[6] = 0.300;
 	ptt->dflt_tariff[7] = 0.300 * 0.80;
 	ptt->dflt_tariff[8] = 0.475;
 	ptt->dflt_tariff[9] = 0.475 * 0.80;
 	ptt->dflt_tariff[10] = 0.570;
 	ptt->dflt_tariff[11] = 0.570 * 0.80;
	ptt->min_cost_len[0] = 180.0;
	ptt->min_cost_len[1] = 180.0;
	ptt->min_cost_len[2] = 180.0;
	ptt->min_cost_len[3] = 180.0;
	ptt->min_cost_len[4] = 111.0;
	ptt->min_cost_len[5] = 111.0;
	ptt->min_cost_len[6] =  74.0;
	ptt->min_cost_len[7] =  74.0;
	ptt->min_cost_len[8] =  47.0;
	ptt->min_cost_len[9] =  47.0;
	ptt->min_cost_len[10] =  39.0;
	ptt->min_cost_len[11] =  39.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Locale)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 1 + Primaliste", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 1 + Primaliste Inet", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Zone 1 + Primaliste + Primaliste Inet",
			MAXLEN_ZNAME);
	strncpy(ptt->zone_name[4], "Zone 2 (Jusqu'a 25km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[5], "Zone 2 + Primaliste", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[6], "Zone 3 (25km - 30km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[7], "Zone 3 + Primaliste", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[8], "Zone 4 (30km - 50km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[9], "Zone 4 + Primaliste", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[10], "Zone 5 (Nationale)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[11], "Zone 5 + Primaliste", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = mk_discount((discount_t){50.0,{8,0,0},{22,0,0}});
	/*
	 * -37.5% to dflt_tariff[3] : 
	 * -37.5% to 0.14*0.80 FF == -50% on 0.14 FF
	 */
	ptt->discount[3] = mk_discount((discount_t){37.5,{8,0,0},{22,0,0}});
	ptt->discount[4] = reset_discount;
	ptt->discount[5] = reset_discount;
	ptt->discount[6] = reset_discount;
	ptt->discount[7] = reset_discount;
	ptt->discount[8] = reset_discount;
	ptt->discount[9] = reset_discount;
	ptt->discount[10] = reset_discount;
	ptt->discount[11] = reset_discount;

	for (z=0; z<12; z++) {
		ptt->rule[z][0] =
			mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][1] =
			mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][2] =
			mk_AHoliday((ruledate_t){4, 8}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0},ptt->dflt_tariff [z]);
		ptt->rule[z][3] =
			mk_RHoliday(39, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][4] =
			mk_AHoliday((ruledate_t){6, 14}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][5] =
			mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][6] =
			mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][7] =
			mk_AHoliday((ruledate_t){10, 11}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][8] =
			mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
						(ruletime_t){24, 0, 0}, ptt->dflt_tariff [z]);
		ptt->rule[z][9] =
			mk_Weekend((ruletime_t){0, 0, 0}, (ruletime_t){24, 0, 0},
					   ptt->dflt_tariff [z]);
		ptt->rule[z][10] =
			mk_Weekday((ruletime_t){8, 0, 0}, (ruletime_t){19, 0, 0},
					   ptt->dflt_tariff [z]*2);
	}
	/*
	 * Extra discount (aka primaliste internet)
	 * applies only to zone=2 and zone=3
	 */
	for (z=2; z<4; z++) {
		ptt->rule[z][0].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][1].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][2].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][3].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][4].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][5].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][6].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][7].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][8].type |= DISCOUNT_PERMITTED;
		ptt->rule[z][9].type |= DISCOUNT_PERMITTED;
	}
}

 
/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                   Phone costs for the Hellenic OTE                      |
  |            (tariff update: February 2001  18% VAT included)             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void gr_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Hellenic OTE (Hellas)", MAXLEN_PTTNAME);
	ptt->num_zones = 4;
	ptt->current_zone = 0;
	ptt->num_categories = 3;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 12.39;
	strncpy(ptt->currency, "GRD", MAXLEN_CURRENCY);
	ptt->decimals = 0;

	ptt->dflt_tariff[0] = 60;
	ptt->dflt_tariff[1] = 60;
	ptt->dflt_tariff[2] = 60;
	ptt->dflt_tariff[3] = 22.5;

	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	ptt->min_cost_len[3] = 0.0;

	strncpy(ptt->zone_name[0], "Zone 1 (Local/Same Province)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Up to 45 Km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (Over 45 Km, same switching center)",
			MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Zone 4 (Over 45 Km, other switching center)",
			MAXLEN_ZNAME);

	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 120);
	ptt->rule[0][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 60);
	ptt->rule[0][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 120);

	ptt->rule[1][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 120);
	ptt->rule[1][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 60);
	ptt->rule[1][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 120);

	ptt->rule[2][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 120);
	ptt->rule[2][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 60);
	ptt->rule[2][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 120);

	ptt->rule[3][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 22.5);
	ptt->rule[3][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 22.5);
	ptt->rule[3][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 22.5);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                Phone costs for the Hellenic OTE (EAK)                  |
  |            (tariff update: February 2001  18% VAT included)             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void gr_epak_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Hellenic OTE - EPAK (Hellas)", MAXLEN_PTTNAME);
	ptt->num_zones = 3;
	ptt->current_zone = 0;
	ptt->num_categories = 3;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 12.39;
	strncpy(ptt->currency, "GRD", MAXLEN_CURRENCY);
	ptt->decimals = 0;

	ptt->dflt_tariff[0] = 315;
	ptt->dflt_tariff[1] = 60;
	ptt->dflt_tariff[2] = 22.5;

	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;

	strncpy(ptt->zone_name[0], "Zone 1 (Same Province)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Other province, up to 45 Km)",
			MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (Other Province, over 45 Km)",
			MAXLEN_ZNAME);

	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 630);
	ptt->rule[0][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 315);
	ptt->rule[0][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 630);

	ptt->rule[1][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 630);
	ptt->rule[1][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 60);
	ptt->rule[1][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 630);

	ptt->rule[2][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 630);
	ptt->rule[2][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){22, 0, 0}, 22.5);
	ptt->rule[2][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 630);

}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for the Italian Telecom                    |
  |                         Updated December 1999                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void it_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Italian Telecom", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 17;
	ptt->attribs = PTT_PER_SECS | PTT_NONLINEAR_MIN;
	ptt->min_units = 0;
	ptt->min_cost = 100.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.354;
	strncpy(ptt->currency, "L.", MAXLEN_CURRENCY);
	ptt->decimals = 0;
	ptt->dflt_tariff[0] = 21.24;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][0].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][1] = mk_RHoliday(1, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][1].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][2] = mk_AHoliday((ruledate_t){3, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][2].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][3] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][3].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][4] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][4].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][5] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][5].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][6] = mk_AHoliday((ruledate_t){11, 8}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][6].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][7] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][7].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][8] = mk_AHoliday((ruledate_t){11, 26}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][8].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][9] = mk_AHoliday((ruledate_t){11, 31}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][9].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][10] = mk_Saturday((ruletime_t){0, 0, 0},
								   (ruletime_t){8, 0, 0}, 21.24);
	ptt->rule[0][10].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][11] = mk_Saturday((ruletime_t){8, 0, 0},
								   (ruletime_t){13, 0, 0}, 36.72);
	ptt->rule[0][11].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][12] = mk_Saturday((ruletime_t){13, 0, 0},
								   (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][12].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][13] = mk_Sunday((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][13].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][14] = mk_Weekday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 21.24);
	ptt->rule[0][14].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][15] = mk_Weekday((ruletime_t){8, 0, 0},
								  (ruletime_t){18, 30, 0}, 36.72);
	ptt->rule[0][15].type |= DEFEAT_MIN_TIME_LEN;

	ptt->rule[0][16] = mk_Weekday((ruletime_t){18, 30, 0},
								  (ruletime_t){24, 0, 0}, 21.24);
	ptt->rule[0][16].type |= DEFEAT_MIN_TIME_LEN;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |             Phone costs for the Hungarian phone company MATAV           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void matav_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "MATAV (Hungary)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 10;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 11.25;
	strncpy(ptt->currency, "Ft", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 420;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){5, 0, 0}, 420);
	ptt->rule[0][1] = mk_Weekend((ruletime_t){5, 0, 0},
								 (ruletime_t){22, 0, 0}, 230);
	ptt->rule[0][2] = mk_Weekend((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 420);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){5, 0, 0}, 420);
	ptt->rule[0][4] = mk_Weekday((ruletime_t){5, 0, 0},
								 (ruletime_t){7, 0, 0}, 230);
	ptt->rule[0][5] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){9, 0, 0}, 110);
	ptt->rule[0][6] = mk_Weekday((ruletime_t){9, 0, 0},
								 (ruletime_t){15, 0, 0}, 90);
	ptt->rule[0][7] = mk_Weekday((ruletime_t){15, 0, 0},
								 (ruletime_t){18, 0, 0}, 110);
	ptt->rule[0][8] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){22, 0, 0}, 230);
	ptt->rule[0][9] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 420);
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                Phone costs for the Dutch "PTT Telecom"                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void nl_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Dutch PTT (NL)", MAXLEN_PTTNAME);
	ptt->num_zones = 6;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_SECS;
	ptt->min_units = 0;
	ptt->min_cost = 0.10;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "HF", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.0325;
	ptt->dflt_tariff[1] = 0.1000;
	ptt->dflt_tariff[2] = 0.0250;
	ptt->dflt_tariff[3] = 0.0750;
	ptt->dflt_tariff[4] = 0.1000;
	ptt->dflt_tariff[5] = 0.3000;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	ptt->min_cost_len[3] = 0.0;
	ptt->min_cost_len[4] = 0.0;
	ptt->min_cost_len[5] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Basis, Local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Basis, Non-local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (Plus, Local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[3], "Zone 4 (Plus, Non-local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[4], "Zone 5 (Budget, Local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[5], "Zone 6 (Budget, Non-local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;
	ptt->discount[3] = reset_discount;
	ptt->discount[4] = reset_discount;
	ptt->discount[5] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0325);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.0325);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.0650);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0325);

	ptt->rule[1][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.1000);
	ptt->rule[1][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.1000);
	ptt->rule[1][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.2000);
	ptt->rule[1][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.1000);

	ptt->rule[2][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0250);
	ptt->rule[2][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.0250);
	ptt->rule[2][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.0650);
	ptt->rule[2][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0250);

	ptt->rule[3][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0750);
	ptt->rule[3][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.0750);
	ptt->rule[3][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.2000);
	ptt->rule[3][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.0750);

	ptt->rule[4][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.1000);
	ptt->rule[4][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.1000);
	ptt->rule[4][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.2000);
	ptt->rule[4][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.1000);

	ptt->rule[5][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.3000);
	ptt->rule[5][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.3000);
	ptt->rule[5][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.6000);
	ptt->rule[5][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.3000);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Phone costs for Japan's NTT                        |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void ntt_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "NTT (Japan)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 3;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 10.0;
	strncpy(ptt->currency, "JY", MAXLEN_CURRENCY);
	ptt->decimals = 0;
	ptt->dflt_tariff[0] = 240;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 240);
	ptt->rule[0][1] = mk_Weekall((ruletime_t){8, 0, 0},
								 (ruletime_t){23, 0, 0}, 180);
	ptt->rule[0][2] = mk_Weekall((ruletime_t){23, 0, 0},
								 (ruletime_t){24, 0, 0}, 240);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for the Portugal Telecom                   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void pt_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Portugal Telecom", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 8;
	ptt->attribs = PTT_BY_UNIT;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 13.5;
	strncpy(ptt->currency, "Esc", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 540;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 540);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 540);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){10, 0, 0}, 360);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){10, 0, 0},
								 (ruletime_t){13, 0, 0}, 180);
	ptt->rule[0][4] = mk_Weekday((ruletime_t){13, 0, 0},
								 (ruletime_t){14, 0, 0}, 360);
	ptt->rule[0][5] = mk_Weekday((ruletime_t){14, 0, 0},
								 (ruletime_t){18, 0, 0}, 180);
	ptt->rule[0][6] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){22, 0, 0}, 360);
	ptt->rule[0][7] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 540);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |             Phone costs for the South African PTT, "Telkom"             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void sat_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Telkom (South Africa)", MAXLEN_PTTNAME);
	ptt->num_zones = 3;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_SECS;
	ptt->min_units = 0;
	ptt->min_cost = 0.46;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "R", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.05;
	ptt->dflt_tariff[1] = 0.30;
	ptt->dflt_tariff[2] = 0.60;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Up to 50km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (50 to 100km)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (Over 100km)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;
	ptt->discount[2] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.05);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 0.05);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){19, 0, 0}, 0.16);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){19, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.05);

	ptt->rule[1][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.30);
	ptt->rule[1][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 0.30);
	ptt->rule[1][2] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){19, 0, 0}, 0.60);
	ptt->rule[1][3] = mk_Weekday((ruletime_t){19, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.30);

	ptt->rule[2][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.62);
	ptt->rule[2][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 0.62);
	ptt->rule[2][2] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){19, 0, 0}, 1.24);
	ptt->rule[2][3] = mk_Weekday((ruletime_t){19, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.62);

}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Phone costs for the Slovene Telekom                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void sit_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Slovene Telekom", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 4.25;
	strncpy(ptt->currency, "SIT", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 240;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 240);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 240);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){19, 0, 0}, 120);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){19, 0, 0},
								 (ruletime_t){24, 0, 0}, 240);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                    Phone costs for the TeleDanmark                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void tdk_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "TeleDanmark (DK)", MAXLEN_PTTNAME);
	ptt->num_zones = 3;
	ptt->current_zone = 0;
	ptt->num_categories = 7;
	ptt->attribs = PTT_PER_MINUTE | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.125;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "kr", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.160;
	ptt->dflt_tariff[1] = 0.235;
	ptt->dflt_tariff[2] = 0.350;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	ptt->min_cost_len[2] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Regional)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[2], "Zone 3 (National)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Sunday((ruletime_t){0, 0, 0},
								(ruletime_t){24, 0, 0}, 0.160);
	ptt->rule[0][1] = mk_Saturday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.160);
	ptt->rule[0][2] = mk_Saturday((ruletime_t){8, 0, 0},
								  (ruletime_t){19, 30, 0}, 0.320);
	ptt->rule[0][3] = mk_Saturday((ruletime_t){19, 30, 0},
								  (ruletime_t){24, 0, 0}, 0.160);
	ptt->rule[0][4] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.160);
	ptt->rule[0][5] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){19, 30, 0}, 0.320);
	ptt->rule[0][6] = mk_Weekday((ruletime_t){19, 30, 0},
								 (ruletime_t){24, 0, 0}, 0.160);

	ptt->rule[1][0] = mk_Sunday((ruletime_t){0, 0, 0},
								(ruletime_t){24, 0, 0}, 0.235);
	ptt->rule[1][1] = mk_Saturday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.235);
	ptt->rule[1][2] = mk_Saturday((ruletime_t){8, 0, 0},
								  (ruletime_t){19, 30, 0}, 0.470);
	ptt->rule[1][3] = mk_Saturday((ruletime_t){19, 30, 0},
								  (ruletime_t){24, 0, 0}, 0.235);
	ptt->rule[1][4] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.235);
	ptt->rule[1][5] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){19, 30, 0}, 0.470);
	ptt->rule[1][6] = mk_Weekday((ruletime_t){19, 30, 0},
								 (ruletime_t){24, 0, 0}, 0.235);

	ptt->rule[2][0] = mk_Sunday((ruletime_t){0, 0, 0},
								(ruletime_t){24, 0, 0}, 0.350);
	ptt->rule[2][1] = mk_Saturday((ruletime_t){0, 0, 0},
								  (ruletime_t){8, 0, 0}, 0.350);
	ptt->rule[2][2] = mk_Saturday((ruletime_t){8, 0, 0},
								  (ruletime_t){19, 30, 0}, 0.700);
	ptt->rule[2][3] = mk_Saturday((ruletime_t){19, 30, 0},
								  (ruletime_t){24, 0, 0}, 0.350);
	ptt->rule[2][4] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.350);
	ptt->rule[2][5] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){19, 30, 0}, 0.700);
	ptt->rule[2][6] = mk_Weekday((ruletime_t){19, 30, 0},
								 (ruletime_t){24, 0, 0}, 0.350);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |   Phone costs for the Polish phone company Telekomunikacja Polska SA.   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void pl_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Telekom. Polska S.A. (Poland)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 1;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.1926;
	strncpy(ptt->currency, "PLN", MAXLEN_PTTNAME);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 180;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 180);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |        Phone costs for the Croatian Academic and Research Network       |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void carnet_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "CARNet (Croatia)", MAXLEN_PTTNAME);
	ptt->num_zones = 2;
	ptt->current_zone = 0;
	ptt->num_categories = 14;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.205;
	strncpy(ptt->currency, "Kn", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 360;
	ptt->dflt_tariff[1] = 72;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Lokalni poziv)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Medjugradski poziv)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 360);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){16, 0, 0}, 180);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){16, 0, 0},
								 (ruletime_t){22, 0, 0}, 240);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][4] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][5] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][7] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][8] = mk_AHoliday((ruledate_t){4, 30}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][9] = mk_AHoliday((ruledate_t){5, 22}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][10] = mk_AHoliday((ruledate_t){7, 4}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][11] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][12] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][13] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);

	ptt->rule[1][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 72);
	ptt->rule[1][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){16, 0, 0}, 36);
	ptt->rule[1][2] = mk_Weekday((ruletime_t){16, 0, 0},
								 (ruletime_t){22, 0, 0}, 48);
	ptt->rule[1][3] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][4] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][5] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][7] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][8] = mk_AHoliday((ruledate_t){4, 30}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][9] = mk_AHoliday((ruledate_t){5, 22}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][10] = mk_AHoliday((ruledate_t){7, 4}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][11] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][12] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][13] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                   Phone costs for the Croatian Hinet                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void hinet_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Hinet (Croatia)", MAXLEN_PTTNAME);
	ptt->num_zones = 2;
	ptt->current_zone = 0;
	ptt->num_categories = 3;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.205;
	strncpy(ptt->currency, "Kn", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 120;
	ptt->dflt_tariff[1] = 30;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	strncpy(ptt->zone_name[0], "Ulaz za pretplatnike", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Anonimni ulaz", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 120);
	ptt->rule[0][1] = mk_Weekall((ruletime_t){7, 0, 0},
								 (ruletime_t){22, 0, 0}, 60);
	ptt->rule[0][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 120);

	ptt->rule[1][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 30);
	ptt->rule[1][1] = mk_Weekall((ruletime_t){7, 0, 0},
								 (ruletime_t){22, 0, 0}, 30);
	ptt->rule[1][2] = mk_Weekall((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 30);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                    Phone costs for ISKON in Croatia                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void iskon_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "ISKON (Croatia)", MAXLEN_PTTNAME);
	ptt->num_zones = 2;
	ptt->current_zone = 0;
	ptt->num_categories = 14;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.205;
	strncpy(ptt->currency, "Kn", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 360;
	ptt->dflt_tariff[1] = 72;
	ptt->min_cost_len[0] = 0.0;
	ptt->min_cost_len[1] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Lokalni poziv)", MAXLEN_ZNAME);
	strncpy(ptt->zone_name[1], "Zone 2 (Medjugradski poziv)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;
	ptt->discount[1] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 360);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){16, 0, 0}, 180);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){16, 0, 0},
								 (ruletime_t){22, 0, 0}, 240);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][4] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][5] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][7] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][8] = mk_AHoliday((ruledate_t){4, 30}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][9] = mk_AHoliday((ruledate_t){5, 22}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][10] = mk_AHoliday((ruledate_t){7, 4}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][11] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][12] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][13] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);

	ptt->rule[1][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 72);
	ptt->rule[1][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){16, 0, 0}, 36);
	ptt->rule[1][2] = mk_Weekday((ruletime_t){16, 0, 0},
								 (ruletime_t){22, 0, 0}, 48);
	ptt->rule[1][3] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][4] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][5] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][7] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][8] = mk_AHoliday((ruledate_t){4, 30}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][9] = mk_AHoliday((ruledate_t){5, 22}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][10] = mk_AHoliday((ruledate_t){7, 4}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][11] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][12] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
	ptt->rule[1][13] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 72);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                   Phone costs for IBM Net in Croatia                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void ibmcr_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "IBM Net (Croatia)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 14;
	ptt->attribs = PTT_BY_UNIT | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 0;
	ptt->cost_quantum = 0.205;
	strncpy(ptt->currency, "Kn", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 360;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Lokalni poziv)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 360);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){16, 0, 0}, 180);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){16, 0, 0},
								 (ruletime_t){22, 0, 0}, 240);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){22, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][4] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][5] = mk_AHoliday((ruledate_t){0, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][6] = mk_RHoliday(0, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][7] = mk_AHoliday((ruledate_t){4, 1}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][8] = mk_AHoliday((ruledate_t){4, 30}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][9] = mk_AHoliday((ruledate_t){5, 22}, (ruletime_t){0, 0, 0},
								  (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][10] = mk_AHoliday((ruledate_t){7, 4}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][11] = mk_AHoliday((ruledate_t){7, 15}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][12] = mk_AHoliday((ruledate_t){10, 1}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
	ptt->rule[0][13] = mk_AHoliday((ruledate_t){11, 25}, (ruletime_t){0, 0, 0},
								   (ruletime_t){24, 0, 0}, 360);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Phone costs for the Helsinki Telephone Corp.              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void hpy_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Helsinki Telephone Corp. (FI)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_MINUTE | PTT_CUR_AFTER_COST | PTT_NONLINEAR_MIN;
	ptt->min_units = 0;
	ptt->min_cost = 0.47;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "Mk", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.055;
	ptt->min_cost_len[0] = 1800.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){7, 0, 0}, 0.055);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){7, 0, 0},
								 (ruletime_t){17, 0, 0}, 0.055);
	ptt->rule[0][1].type |= DEFEAT_MIN_TIME_LEN;
	ptt->rule[0][2] = mk_Weekday((ruletime_t){17, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.055);
	ptt->rule[0][3] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.055);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Phone costs for Telenor, Norway                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void telenor_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Telenor (Norway)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_MINUTE | PTT_CUR_AFTER_COST;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "kr", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.14;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1 (Local)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.14);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){17, 0, 0}, 0.25);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){17, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.14);
	ptt->rule[0][3] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.14);
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                   Phone costs for Inetservice, Germany                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/
void inetserv_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Inetservice (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 5;
	ptt->attribs = PTT_PER_SECS;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.03;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Internet by Call", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){21, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.03);
	ptt->rule[0][1] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.03);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){18, 0, 0}, 0.08);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){18, 0, 0},
								 (ruletime_t){21, 0, 0}, 0.048);
	ptt->rule[0][4] = mk_Weekend((ruletime_t){8, 0, 0},
								 (ruletime_t){21, 0, 0}, 0.048);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Phone costs for Komtel, Germany                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/
void komtel_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Komtel (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 4;
	ptt->attribs = PTT_PER_SECS;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 1;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.06;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Internettarif (Deutschland)", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekend((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.06);
	ptt->rule[0][1] = mk_Weekday((ruletime_t){0, 0, 0},
								 (ruletime_t){8, 0, 0}, 0.06);
	ptt->rule[0][2] = mk_Weekday((ruletime_t){8, 0, 0},
								 (ruletime_t){20, 0, 0}, 0.1);
	ptt->rule[0][3] = mk_Weekday((ruletime_t){20, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.06);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                    Phone costs for Mobilkom, Germany                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/
void mobilkom_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "Mobilkom (Germany)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 1;
	ptt->attribs = PTT_PER_MINUTE;
	ptt->min_units = 0;
	ptt->min_cost = 0.0;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "DM", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.05;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Internet by Call", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.05);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Phone costs for IPY, Finland                       |
  |                                                                         |
  +-------------------------------------------------------------------------+*/
void ipy_initPtt(ptt_t *ptt)
{
	strncpy(ptt->name, "IPY (Finland)", MAXLEN_PTTNAME);
	ptt->num_zones = 1;
	ptt->current_zone = 0;
	ptt->num_categories = 1;
	ptt->attribs = PTT_PER_MINUTE | PTT_NONLINEAR_MIN;
	ptt->min_units = 0;
	ptt->min_cost = 0.49;
	ptt->charge_period = 60;
	ptt->cost_quantum = 0.0;
	strncpy(ptt->currency, "mk", MAXLEN_CURRENCY);
	ptt->decimals = 2;
	ptt->dflt_tariff[0] = 0.049;
	ptt->min_cost_len[0] = 0.0;
	strncpy(ptt->zone_name[0], "Zone 1", MAXLEN_ZNAME);
	ptt->discount[0] = reset_discount;

	ptt->rule[0][0] = mk_Weekall((ruletime_t){0, 0, 0},
								 (ruletime_t){24, 0, 0}, 0.049);
	ptt->rule[0][0].type |= DEFEAT_MIN_TIME_LEN;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Internal routines                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

static void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

#ifdef HAVE_STRERROR
	sprintf(emsg, "xISP: %s: %s\n", msg, strerror(errno));
#else
	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
#endif
	fputs(emsg, stderr);
	exit(1);
}

/* Prints error message if error occurs while processing
   xispcost or xispPTTs file. Prints out offending line */

static void inpLineError(char *function, char *line)
{
	fprintf(stderr, "xISP: %s: error reading %s\n", function, curfn);
	fprintf(stderr, "xISP: %s: offending line: [%s]\n", function, line);
	exit(1);
}

/* Reads parameters from xispcost or xispPTTs file, doing error checking.
   NOTE: it assumes lines less than 256 bytes and parameters less than 128 */

#define MAXLEN_LINE 256
#define MAXLEN_PARAM 128

static char *readParam(FILE *fp, char *name, char type,
					   int llimit, int ulimit, void *data)
{
	static char line[MAXLEN_LINE], NAME[MAXLEN_PARAM];
	char sparam[MAXLEN_PARAM] = {0}, *p, *P, *endp;
	int iparam, hex = 0, len;
	unsigned int uparam, th, tm, ts, dd, dm;
	unsigned long lparam;
	float fparam;
	ruletime_t *rt;
	ruledate_t *rd;

	if (fgets(line, MAXLEN_LINE, fp) == NULL) {
		fprintf(stderr, "xISP: readParam: %s: premature EOF\n", pttfn);
		exit(1);
	}
	line[strlen(line)-1] = 0;
	for (p=name, P=NAME; *p; p++, P++)
		*P = toupper(*p);
	*P = 0;
	if ((p=strchr(line,':')) == NULL)
		inpLineError("readParam", line);
	if (strncmp(name, line, (int)(p-line)) &&	/* accept all capitals */
		strncmp(NAME, line, (int)(p-line)))
	{
		fprintf(stderr, "xISP: readParam: expected [%s], got [%s]\n",
				name, line);
		exit(1);
	}
	for (++p; *p==' '; p++);
		strncpy(sparam, p, sizeof(sparam)-1);
	switch (type) {
		case 'S':
			len = strlen(sparam);
			if (ulimit<len || len<llimit)
				inpLineError("readParam", line);
			strcpy((char *)data, sparam);
			break;

		case 'H':
		case 'X':
			hex = 1;
		case 'B':
		case 'U':
			uparam = strtol(sparam, &endp, (hex)? 16:10);
			if ((!hex && (ulimit<uparam || uparam<llimit)) ||
				 endp == sparam || *endp )
				inpLineError("readParam", line);
			if (type == 'H' || type == 'B')
				*((unsigned char *)data) = uparam;
			else
				*((unsigned int *)data) = uparam;
			break;

		case 'I':
			iparam = strtol(sparam, &endp, 10);
			if (ulimit<iparam || iparam<llimit ||
				endp == sparam || *endp )
				inpLineError("readParam", line);
			*((int *)data) = iparam;
			break;

		case 'L':
			lparam = strtol(sparam, &endp, 10);
			if ((ulimit!=llimit && ulimit<lparam) || lparam<llimit ||
				endp == sparam || *endp )
				inpLineError("readParam", line);
			*((unsigned long *)data) = lparam;
			break;

		case 'F':
			fparam = strtod(sparam, &endp);
			if (fparam<llimit ||
				endp == sparam || *endp)
				inpLineError("readParam", line);
			*((float *)data) = fparam;
			break;

		case 'T':
			rt = data;
			if (sscanf(sparam, "%2u:%2u:%2u", &th, &tm, &ts) < 3 ||
				th > 24 || tm > 60 || ts > 60 ||
				(th > 23 && (tm != 0 || ts != 0)))
				inpLineError("readParam", line);
			rt->h = th; rt->m = tm; rt->s = ts;
			break;

		case 'D':
			rd = data;
			if (sscanf(sparam, "%2u/%2u", &dd, &dm) < 2 ||
				dm < 1 || dm > 12 || dd < 1 || dd > 31)
				inpLineError("readParam", line);
			rd->day = dd; rd->mon = dm - 1;
			break;

		default: break;
	}
	return(line);
}

/* Function for writing strings in the costs and logging files,
   as well as in xispPTTs, while checking for write errors */

static int safePrint(va_alist) va_dcl
{
	int bw;
	va_list ap;
	char *fmt;
	FILE *fp;

	va_start(ap);
	fp = va_arg(ap, FILE*);
	fmt = va_arg(ap, char*);
	bw = vfprintf(fp, fmt, ap);
	va_end(ap);
	if (bw <= 0)
		doErr("safePrint");
	fflush(fp);
	return bw;
}

/* Function for writing out the entire .xisprc file */

static void outputAllXispPTTs(ptt_t *p, FILE *pttfp, glob_t *global)
{
	int i, z, c;

	safePrint(pttfp, "xISP-%s\n", Version);
	safePrint(pttfp, "NumPTTs: %d\n", global->numPTTs);
	for (i=0; i<global->numPTTs; i++) {
		safePrint(pttfp, "[%d]\n", i);
		safePrint(pttfp, "Name: %s\n", p[i].name);
		safePrint(pttfp, "NumZones: %d\n", p[i].num_zones);
		safePrint(pttfp, "CurrentZone: %d\n", p[i].current_zone);
		safePrint(pttfp, "NumCategories: %d\n", p[i].num_categories);
		safePrint(pttfp, "Attributes: %X\n", p[i].attribs);
		safePrint(pttfp, "MinUnits: %d\n", p[i].min_units);
		safePrint(pttfp, "MinCost: %g\n", p[i].min_cost);
		safePrint(pttfp, "ChargePeriod: %d\n", p[i].charge_period);
		safePrint(pttfp, "CostQuantum: %g\n", p[i].cost_quantum);
		safePrint(pttfp, "Currency: %s\n", p[i].currency);
		safePrint(pttfp, "Decimals: %d\n", p[i].decimals);
		for (z=0; z<p[i].num_zones; z++) {
			safePrint(pttfp, "DfltTariff[%d]: %g\n", z, p[i].dflt_tariff[z]);
			safePrint(pttfp, "MinCostLen[%d]: %g\n", z, p[i].min_cost_len[z]);
			safePrint(pttfp, "ZoneName[%d]: %s\n", z, p[i].zone_name[z]);
			safePrint(pttfp, "DiscountPercent[%d]: %g\n", z,
					  p[i].discount[z].percent);
			safePrint(pttfp, "DiscountStart[%d]: %02d:%02d:%02d\n", z,
					  p[i].discount[z].start.h, p[i].discount[z].start.m,
					  p[i].discount[z].start.s);
			safePrint(pttfp, "DiscountEnd[%d]: %02d:%02d:%02d\n", z,
					  p[i].discount[z].end.h, p[i].discount[z].end.m,
					  p[i].discount[z].end.s);
			for (c=0; c<p[i].num_categories; c++) {
				safePrint(pttfp, "RuleType[%d][%d]: %X\n", z, c,
						  p[i].rule[z][c].type);
				safePrint(pttfp, "RuleADate[%d][%d]: %02d/%02d\n", z, c,
						  p[i].rule[z][c].adate.day,
						  1 + p[i].rule[z][c].adate.mon);
				safePrint(pttfp, "RuleEDate[%d][%d]: %02d/%02d\n", z, c,
						  p[i].rule[z][c].edate.day,
						  1 + p[i].rule[z][c].edate.mon);
				safePrint(pttfp, "RuleRDate[%d][%d]: %d\n", z, c,
						  p[i].rule[z][c].rdate);
				safePrint(pttfp, "RuleStart[%d][%d]: %02d:%02d:%02d\n", z, c,
						  p[i].rule[z][c].start.h, p[i].rule[z][c].start.m,
						  p[i].rule[z][c].start.s);
				safePrint(pttfp, "RuleEnd[%d][%d]: %02d:%02d:%02d\n", z, c,
						  p[i].rule[z][c].end.h, p[i].rule[z][c].end.m,
						  p[i].rule[z][c].end.s);
				safePrint(pttfp, "RuleTariff[%d][%d]: %g\n", z, c,
						  p[i].rule[z][c].tariff);
			}
		}
	}
}

/* Function for retrieving the xispPTTs file version. Note that
   it must be called after fopen() and before all other input */

static char *pttVersion(FILE *fp)
{
	static char version[8] = {0};

	if (fscanf(fp, "xISP-%7[.0-9] ", version) < 1) {
		fputs("xISP: pttVersion: error reading file version!\n", stderr);
		exit(1);
	}
	return version;
}

/* Function for retrieving the PTT sequence number. Note that
   it must be called at the beginning of each PTT record. */

static void pttSeq(FILE *fp, int nPTT)
{
	int n;

	if (fscanf(fp, "[%4d] ", &n) < 1) {
		fprintf(stderr, "xISP: pttSeq: error reading PTT number in %s\n",
				pttfn);
		exit(1);
	}
	if (n != nPTT) {
		fprintf(stderr, "xISP: pttSeq: PTT sequence number wrong in %s\n",
				pttfn);
		fprintf(stderr, "xISP: pttSeq: expected %d, got %d\n", nPTT, n);
		exit(1);
	}
}

/* Function for reading in the entire xispPTTs file */

static void inputAllXispPTTs(ptt_t *p, FILE *pttfp, glob_t *global)
{
	int i, z, c, testEaster;
	char pname[32] = {0}, *line;
	time_t tt = time(NULL);
	struct tm *ct = localtime(&tt);

	curfn = pttfn;
	for (i=0; i<global->numPTTs; i++) {
		pttSeq(pttfp, i);
		readParam(pttfp, "Name", 'S', 0, MAXLEN_PTTNAME, p[i].name);
		readParam(pttfp, "NumZones", 'B', 0, MAXNUM_ZONES, &(p[i].num_zones));
		readParam(pttfp, "CurrentZone", 'B', 0, MAXNUM_ZONES-1,
				  &(p[i].current_zone));
		readParam(pttfp, "NumCategories", 'B', 0, MAXNUM_CATEGORY,
				  &(p[i].num_categories));
		readParam(pttfp, "Attributes", 'X', 0, 0, &(p[i].attribs));
		readParam(pttfp, "MinUnits", 'B', 0, MAXVAL_MINUNITS,
				  &(p[i].min_units));
		readParam(pttfp, "MinCost", 'F', 0, 0, &(p[i].min_cost));
		if (p[i].attribs & (PTT_PER_MINUTE | PTT_PER_SECS))
			readParam(pttfp, "ChargePeriod", 'B', 1, 60, &(p[i].charge_period));
		else
			readParam(pttfp, "ChargePeriod", 'B', 0, 0, &(p[i].charge_period));
		readParam(pttfp, "CostQuantum", 'F', 0, 0, &(p[i].cost_quantum));
		readParam(pttfp, "Currency", 'S', 0, MAXLEN_CURRENCY, p[i].currency);
		readParam(pttfp, "Decimals", 'B', 0, MAXNUM_DECIMALS,
				  &(p[i].decimals));
		for (z=0; z<p[i].num_zones; z++) {
			sprintf(pname, "DfltTariff[%d]", z);
			readParam(pttfp, pname, 'F', 0, 0, &(p[i].dflt_tariff[z]));
			sprintf(pname, "MinCostLen[%d]", z);
			readParam(pttfp, pname, 'F', 0, MAXSEC_MINCOSTLEN,
					  &(p[i].min_cost_len[z]));
			sprintf(pname, "ZoneName[%d]", z);
			readParam(pttfp, pname, 'S', 0, MAXLEN_ZNAME, p[i].zone_name[z]);
			sprintf(pname, "DiscountPercent[%d]", z);
			readParam(pttfp, pname, 'F', 0, 99.0, &(p[i].discount[z].percent));
			sprintf(pname, "DiscountStart[%d]", z);
			readParam(pttfp, pname, 'T', 0, 0, &(p[i].discount[z].start));
			sprintf(pname, "DiscountEnd[%d]", z);
			readParam(pttfp, pname, 'T', 0, 0, &(p[i].discount[z].end));
			for (c=0; c<p[i].num_categories; c++) {
				sprintf(pname, "RuleType[%d][%d]", z, c);
				readParam(pttfp, pname, 'H', 0, 0, &(p[i].rule[z][c].type));
				sprintf(pname, "RuleADate[%d][%d]", z, c);
				readParam(pttfp, pname, 'D', 0, 0, &(p[i].rule[z][c].adate));
				sprintf(pname, "RuleEDate[%d][%d]", z, c);
				readParam(pttfp, pname, 'D', 0, 0, &(p[i].rule[z][c].edate));
				sprintf(pname, "RuleRDate[%d][%d]", z, c);
				line = readParam(pttfp, pname, 'I', MINVAL_RELDATE,
								 MAXVAL_RELDATE, &(p[i].rule[z][c].rdate));
				testEaster = p[i].rule[z][c].rdate + get_Easter(ct);
				if (testEaster < 1 || testEaster > 365)
					inpLineError("inputAllXispPTTs", line);
				sprintf(pname, "RuleStart[%d][%d]", z, c);
				readParam(pttfp, pname, 'T', 0, 0, &(p[i].rule[z][c].start));
				sprintf(pname, "RuleEnd[%d][%d]", z, c);
				readParam(pttfp, pname, 'T', 0, 0, &(p[i].rule[z][c].end));
				sprintf(pname, "RuleTariff[%d][%d]", z, c);
				readParam(pttfp, pname, 'F', 0, 0, &(p[i].rule[z][c].tariff));
			}
		}
	}
}

/* Function for calculating the week within a year */

int yweek(void)
{
	time_t tsec = time(NULL);
	struct tm* ct = localtime(&tsec);
	int yday = ct->tm_yday, wday1, isdst = ct->tm_isdst, yweek = 1;
	/*
	 * January 1st does not start W1 unless it is Monday,
	 * Tuesday, Wednesday or Thursday. In other words, if
	 * January 1st is Friday, Saturday or Sunday, it is
	 * assumed to belong to W52, together with all days
	 * 'till the following Monday. Apparently there's no
	 * commonly accepted logic here -- at least none I'm
	 * aware of; I just followed what my SHARP YO-370
	 * pocket organizer does :). W52 is the last week of
	 * the year, regardless of how many days it contains
	 */
	tsec -= (yday*86400);		/* what day was January 1st this year? */
	if (isdst)					/* take daylight savings into account! */
		tsec += 3600;
	ct = localtime(&tsec);
	wday1 = ct->tm_wday;
	if (--wday1 < 0)			/* convert January 1st from range */
		wday1 += 7;				/* Sunday-Saturday to Monday-Sunday */
	if (wday1 > 0)				/* if it wasn't Monday, don't start */
		yday -= (7-wday1);		/* 1st week 'till next Monday */
	if (yday < 0)				/* all days in between belong to */
		yweek = 52;				/* week 52, the year before our year */
	else {						/* else, all is OK */
		yweek = (1+yday)/7+		/* calculate the week number */
			(((1+yday)%7)?1:0);
		if (yweek > 52)			/* truncate to week 52 */
			yweek = 52;				
	}
	return yweek;
}

void RCSq1(void)
{
	char *rcsp;
	rcsp = RCSid;
	rcsp = PatchLevel;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Interface routines for connection costs and logging           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Assembles the file names for total costs and logging. Returns 0 if
   the file names did not change, or 1 if there was a change in the
   file suffixes due to crossing a logging period boundary */

int initLogFnames(glob_t *global)
{
	time_t tsec = time(NULL);
	struct tm* ct = localtime(&tsec);
	char monName[12][4] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
						   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"},
		 addon[6] = {0};
	int cmonth, cbimonth, changed;


	if (global->logOpts & LOG_WEEKLY)
		sprintf(addon, ".W%d", yweek());
	else if (global->logOpts & LOG_MONTHLY) {
		cmonth = ct->tm_mon;
		sprintf(addon, ".%s", monName[cmonth]);
	}
	else if (global->logOpts & LOG_BIMONTHLY) {
		cbimonth = ((ct->tm_mon+1)/2)+(((ct->tm_mon+1)%2)?1:0);
		sprintf(addon, ".B%d", cbimonth);
	}
	changed = strcmp(costfnroot, addon);
	strcpy(costfnroot, addon);
	strcpy(logfnroot, addon);
	return (changed)? 1:0;
}

/* Initializes the logging module */

void logs_init(glob_t *global, char *logdirname,
			   char *costfname, char *logfname, char *pttfname)
{
	extern void outofMem();

	logdirn = logdirname;
	costfn = costfname;
	logfn = logfname;
	pttfn = pttfname;
	costfnroot = &costfname[strlen(costfname)];
	logfnroot = &logfname[strlen(logfname)];
	bkupfn = (char *)malloc(strlen(costfn)+16);
	if (bkupfn == NULL)
		outofMem();
	initLogFnames(global);
}

/* Resets the cost calculation module before new cycle */

void costCalcReset(ptt_t *ptt, time_t upTime)
{
	startTime = upTime;
	lastTime = 0.0;
	currentCost = 0.0;
	if (ptt->min_units > 0)
		units = ptt->min_units;
	else
		units = 0;
}

/* Call cost calculation routine */

float callCost(ptt_t *ptt, unsigned long onLineSecs,
			   unsigned long *onLineUnits)
{
	time_t time0;

	time0 = time(NULL);
	while (lastTime <= onLineSecs)
	{
		lastTime += getUnitLength(ptt, time0-(onLineSecs-(int)lastTime));
		++units;	/* MUST be incremented AFTER call to getUnitLength() */
		if (ptt->attribs & (PTT_PER_MINUTE | PTT_PER_SECS))
			currentCost += ptt->cost_quantum;
		else
			currentCost = units * ptt->cost_quantum;
	}
	*onLineUnits = units;
	if (ptt->min_cost > 0 && currentCost < ptt->min_cost)
		return ptt->min_cost;
	else
		return currentCost;
}

/* Read the cost data from the total cost file, or create a new file if
   it's the first time the program is run. Rename one year old files to
   backup names and start new logs for totals, if appropriate */

void readXispCost(unsigned long *totalTime, float *totalCost)
{
	FILE *costfp;
	struct stat st;
	struct tm *ptm;
	int last, now, newLog;
	time_t tt = time(NULL);

	curfn = costfn;
#ifdef RUNASEUID
	if (! xisp_euidaccess(costfn, R_OK|W_OK)) {
#else
	if (! access(costfn, R_OK|W_OK)) {
#endif
		stat(costfn, &st);
		ptm = localtime(&st.st_mtime);
		last = ptm->tm_year;
		ptm = localtime(&tt);
		now = ptm->tm_year;
		if (last != now) {
			strcpy(bkupfn, costfn); strcat(bkupfn, ".bak");
			rename(costfn, bkupfn);
			strcpy(bkupfn, logfn); strcat(bkupfn, ".bak");
			rename(logfn, bkupfn);
			newLog = 1;
		}
		else {
			costfp = fopen(costfn, "r");
			readParam(costfp, "TotalTime", 'L', 0, 0, totalTime);
			readParam(costfp, "TotalCost", 'F', 0, 0, totalCost);
			fclose(costfp);
			newLog = 0;
		}
	}
	else
		newLog = 1;
	if (newLog) {
		*totalTime = 0;
		*totalCost = 0;
		if (stat(logdirn, &st) < 0) {
			if (mkdir(logdirn, 0777) < 0)
				doErr("readXispCost: mkdir");
			else
				stat(logdirn, &st);
		}
		if (! S_ISDIR(st.st_mode)) {
			fprintf(stderr, "xISP: readXispCost: %s not a directory!\n",
					logdirn);
			exit(1);
		}
		costfp = fopen(costfn, "w");
		if (costfp == NULL)
			doErr("readXispCost: creating: fopen");
		safePrint(costfp, "TotalTime: %lu\n", *totalTime);
		safePrint(costfp, "TotalCost: %.2f\n", *totalCost);
		fclose(costfp);
	}
}

/* Write the cost data to the logging file */

void writeXispCost(unsigned long totalTime, float totalCost)
{
	FILE *costfp;

	costfp = fopen(costfn, "w");
	if (costfp != NULL) {
		safePrint(costfp, "TotalTime: %lu\n", totalTime);
		safePrint(costfp, "TotalCost: %.2f\n", totalCost);
		fclose(costfp);
	}
	else
		doErr("writeXispCost: fopen");
}

void writeXispLog(int up, ptt_t *ptt, char *ts, char *ISPn, char *IPs,
				  char *speed, unsigned long onLineSecs, float onLineCost,
				  unsigned long pTXed, unsigned long pRXed)
{
	FILE *logfp;
	char format[MAXLEN_CURRENCY+5+1], coststr[32+MAXLEN_CURRENCY+5+1];
	struct stat st;

	if (stat(logdirn, &st) < 0) {
		if (mkdir(logdirn, 0777) < 0)
			doErr("writeXispLog: mkdir");
		else
			stat(logdirn, &st);
	}
	if (! S_ISDIR(st.st_mode)) {
		fprintf(stderr, "xISP: writeXispLog: %s not a directory!\n",
				logdirn);
		exit(1);
	}
	logfp = fopen(logfn, "a");
	if (logfp != NULL) {
		if (up)
			safePrint(logfp, "UP at %s, ISP: %s, IP: %s, Speed: %s\n",
					  ts, ISPn, IPs, speed);
		else {
			if (ptt->attribs & PTT_CUR_AFTER_COST)
				sprintf(format, "%%.%df %s", ptt->decimals, ptt->currency);
			else
				sprintf(format, "%s %%.%df", ptt->currency, ptt->decimals);
			sprintf(coststr, format, onLineCost);
			safePrint(logfp,
				"DOWN at %s, Packets TX/RX: %lu/%lu, Total: %lu secs, %s\n",
				ts, pTXed, pRXed, onLineSecs, coststr);
		}
		fclose(logfp);
	}
	else
		doErr("writeXispLog: fopen");
}

/* Reads in information from logging files, builds a table of costs
   and a table of seconds, returns logging type (0-2) and current
   logging period. Each table entry holds the value from the
   corresponding logging period. The first table entry is always the
   one for the current (according to the date) logging period. */

int buildStats(glob_t *global, unsigned int *current,
			   unsigned long *tblTime, float *tblCosts)
{
	int i, type = 0, tnf[3] = {52, 12, 6}, cur = 0;
	time_t tsec = time(NULL);
	struct tm* ct = localtime(&tsec);
	char monName[12][4] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
						   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"},
		 addon[6] = {0}, til[3][2] = {"W", "", "B"};
	FILE *costfp;

	if (global->logOpts & LOG_WEEKLY) {
		type = 0;
		cur = yweek();
	}
	else if (global->logOpts & LOG_MONTHLY) {
		type = 1;
		cur = ct->tm_mon+1;
	}
	else if (global->logOpts & LOG_BIMONTHLY) {
		type = 2;
		cur = ((ct->tm_mon+1)/2)+(((ct->tm_mon+1)%2)?1:0);
	}
	for (i=0; i<tnf[type]; i++) {
		if (type == 1)
			sprintf(addon, ".%s", monName[cur-1]);
		else
			sprintf(addon, ".%s%d", til[type], cur);
		strcpy(costfnroot, addon); strcpy(logfnroot, addon);
		
		costfp = fopen(costfn, "r");
		if (costfp != NULL) {
			readParam(costfp, "TotalTime", 'L', 0, 0, &tblTime[cur-1]);
			readParam(costfp, "TotalCost", 'F', 0, 0, &tblCosts[cur-1]);
			fclose(costfp);
		}
		else {
			tblTime[cur-1] = tblCosts[cur-1] = 0;
		}
		cur = cur % tnf[type] + 1;
	}
	if (type == 1)
		sprintf(addon, ".%s", monName[cur-1]);
	else
		sprintf(addon, ".%s%d", til[type], cur);
	strcpy(costfnroot, addon); strcpy(logfnroot, addon);
	*current = cur;
	return type;
}

/* Initializes the ptt structure */

void initXispPTT(ptt_t *p)
{
	int z, c;

	p->name[0] = 0;
	p->num_zones = 0;
	p->current_zone = 0;
	p->num_categories = 0;
	p->attribs = PTT_BY_UNIT;
	p->min_units = 0;
	p->min_cost = 0.0;
	p->charge_period = 0;
	p->cost_quantum = 0.0;
	p->currency[0] = 0;
	p->decimals = 0;
	for (z=0; z<MAXNUM_ZONES; z++) {
		p->dflt_tariff[z] = 0.0;
		p->min_cost_len[z] = 0.0;
		p->zone_name[z][0] = 0;
		p->discount[z] = reset_discount;
		for (c=0; c<MAXNUM_CATEGORY; c++)
			p->rule[z][c] = reset_rule;
	}
}

/* Read all PTT data records from the user file, or create
   a new file if it's the first time the program is run. */

void readXispPTTs(ptt_t **pttdataptr, glob_t *global)
{
	FILE *pttfp;
	struct stat st;
	int i;
	char opttfn[MAXLEN_FNAME+1] = {0}, msg[512] = {0};
	extern void outofMem(), alertMessage();

	pttfp = fopen(pttfn, "r");
	if (pttfp != NULL) {
		if (strcmp(pttVersion(pttfp),Version)) {
			/* || global->numPTTs <MAXNUM_PTT) { */
			fclose(pttfp);
			strncpy(opttfn, pttfn, MAXLEN_FNAME-4);
			strcat(opttfn, ".bak");
			sprintf(msg, "Your existing PTT database will be renamed\n"
					"to: %s.\n"
					"Any entries you have added since the last update\n"
					"will have to be added anew.", opttfn);
			alertMessage("PTT database:", 0, 0, msg);
			rename(pttfn, opttfn);
			*pttdataptr = (ptt_t *)malloc(MAXNUM_PTT*sizeof(ptt_t));
			if (*pttdataptr == NULL)
				outofMem();
			for (i=0; i<MAXNUM_PTT; i++)
				(*pttInit[i])(&((*pttdataptr)[i]));
#if 0
			fprintf(stderr,
					"xISP: creating updated PTT database. Please wait...");
#endif
			pttfp = fopen(pttfn, "w");
			if (pttfp == NULL)
				doErr("readAllXispPTTs: creating: fopen");
			global->numPTTs = MAXNUM_PTT;
			outputAllXispPTTs(*pttdataptr, pttfp, global);
			fclose(pttfp);
			alertMessage("PTT database:", 0, 1, "Update complete.");
		}
		else {
			readParam(pttfp, "NumPTTs", 'B', 1, 255, &(global->numPTTs));
			*pttdataptr = (ptt_t *)malloc((global->numPTTs)*sizeof(ptt_t));
			if (*pttdataptr == NULL)
				outofMem();
			for (i=0; i<global->numPTTs; i++)
				initXispPTT(&((*pttdataptr)[i]));
			inputAllXispPTTs(*pttdataptr, pttfp, global);
			fclose(pttfp);
		}
	}
	else {
		if (stat(logdirn, &st) < 0) {
			if (mkdir(logdirn, 0777) < 0)
				doErr("readXispPTTs: mkdir");
			else
				stat(logdirn, &st);
		}
		if (! S_ISDIR(st.st_mode)) {
			fprintf(stderr, "xISP: readXispPTTs: %s not a directory!\n",
					logdirn);
			exit(1);
		}
		*pttdataptr = (ptt_t *)malloc(MAXNUM_PTT*sizeof(ptt_t));
		if (*pttdataptr == NULL)
			outofMem();
		for (i=0; i<MAXNUM_PTT; i++)
			(*pttInit[i])(&((*pttdataptr)[i]));
		alertMessage("PTT database:", 0, 1,
					 "The default PTT database\nwill be created now...");
		pttfp = fopen(pttfn, "w");
		if (pttfp == NULL)
			doErr("readAllXispPTTs: creating: fopen");
		if (global->numPTTs != MAXNUM_PTT)
			global->numPTTs = MAXNUM_PTT;
		outputAllXispPTTs(*pttdataptr, pttfp, global);
		global->numPTTs = MAXNUM_PTT;
		fclose(pttfp);
		alertMessage("PTT database:", 0, 1, "Creation complete.");
	}
}

/* Write all xisprc data records to the user file. */

void writeXispPTTs(ptt_t *pttdata, glob_t *global)
{
	struct stat st;
	FILE *pttfp;

	if (stat(logdirn, &st) < 0) {
		if (mkdir(logdirn, 0777) < 0)
			doErr("writeXispPTTs: mkdir");
		else
			stat(logdirn, &st);
	}
	if (! S_ISDIR(st.st_mode)) {
		fprintf(stderr, "xISP: writeXispPTTs: %s not a directory!\n",
				logdirn);
		exit(1);
	}
	pttfp = fopen(pttfn, "w");
	if (pttfp != NULL) {
		outputAllXispPTTs(pttdata, pttfp, global);
		fclose(pttfp);
	}
	else
		doErr("writeXispPTTs: fopen");
}

