###############################################################
# TkNet - Time Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Flags
set gt_time_connected   0
set gt_saved_time       "000/00:00:00"
set gt_last_ping        0

# Current time maintenance.  The internal clock is
# set to the current time on startup, and is syncronised
# every gt_time_refresh seconds.  It ticks every
# gt_time_resolution seconds.
# Submitted by George M. Sipe (gsipe@mindspring.com)
set gt_time_resolution  1
set gt_time_last        0
set gt_time_refresh     120
set gt_time_counter     121

###############################################################
# The procedure for adding and subtracting times
proc AddSub_Time { time1 time2 addsub format } {
   #Debug "AddSub_Time"

   # Time2 is the larger time (more recent).   
   # Addsub must be 1 (Add) or -1 (Subtract)
   # Format must be 1 (Seconds) or 2 (Time)

   # Input formats must be as returned from 
   # Get_Current_Time, e.g. DDD/HH:MM:SS.

   # Oh, this routine will not handle years, nor
   # will it handle -ve differences at the moment.

   set secs1 [Time_To_Secs $time1]
   set secs2 [Time_To_Secs $time2]

   # If the first time is zero return null values
   if {$secs1 == 0} {
      # Return a null value
      set secs3 0
   } else {
      # Perform calculation
      set secs3 [expr $secs2 + ( $addsub * $secs1 )]
   }

   # Do output conversion
   if {$format == 1} {
      # Return as seconds
      return $secs3
   } else {
      # Return formatted
      return [Secs_To_Time $secs3]
   }
}

###############################################################
# The procedure for updating the current time counter
proc Time_Ticker {} {
   #Debug "Time_Ticker"

   # Globals
   global gt_time_resolution gt_time_counter

   # Bump time by period which this routine is scheduled
   incr gt_time_counter $gt_time_resolution
   after [expr $gt_time_resolution * 1000 ] Time_Ticker
}

###############################################################
# The procedure for finding the current time
proc Get_Current_Time {} {
   #Debug "Get_Current_Time"

   global test_nt gt_time_counter gt_time_last \
      gt_time_refresh gb_have_expect

   # If we have expect then use the built in timestamp command
   if {$gb_have_expect == 1 && [info commands "timestamp"] != ""} {
      set gt_time_counter [Time_To_Secs [timestamp -format "%j/%H:%M:%S"]]
   } else {
      # Don't have expect installed - use internal time ticker
      # Every gt_time_refresh synchronize gt_time_counter with real time
      if {$gt_time_counter > $gt_time_last + $gt_time_refresh} {

         # Test cases in here for NT
         if {$test_nt == 0} {
            set gt_time_counter [Time_To_Secs [exec date +%j/%H:%M:%S]]
         } else {
            set gt_time_counter 10637200
         }
         set gt_time_last $gt_time_counter
      }
   }

   # Format is DDD/HH:MM:SS
   # Debug "Get_Current_Time : $gt_time_counter"
   return [Secs_To_Time $gt_time_counter]
}

###############################################################
# The procedure for converting time format to secs format
proc Time_To_Secs {time} {
   #Debug "Time_To_Secs"

   # Format is DDD/HH:MM:SS
   set count [scan $time "%d/%d:%d:%d" day hour minute second]
   if {$count != 4} {return 0}
   return [expr $second + (60 * ($minute + (60 * ($hour + \
      (24 * $day)))))]
}

###############################################################
# The procedure for converting secs format to time format
proc Secs_To_Time {secs} {
   #Debug "Secs_To_Time"

   # Format is DDD/HH:MM:SS
   set day [expr $secs / [expr 60 * 60 * 24]]
   set secs [expr $secs - [expr $day * 60 * 60 * 24]]
   set hour [expr $secs / [expr 60 * 60]]
   set secs [expr $secs - [expr $hour * 60 * 60]]
   set min [expr $secs / 60]
   set secs [expr $secs - [expr $min * 60]]

   # Return formatted string
   # Debug "Secs_To_Time : $day/$hour:$min:$secs"
   return [format "%03d/%02d:%02d:%02d" $day $hour $min $secs]
}

###############################################################
# The procedure for stripping leading 0s etc from a time string
proc Strip_Time_String { time } {
   #Debug "Strip_Time_String"

   # Note that this routine should only be used for
   # display purposes.
   set secs [Time_To_Secs $time]

   # Format of secs is DDD/HH:MM:SS, however if the time
   # is minutes and seconds only, display as X min Y sec
   set return_string ""

   # If day exists just return - nothing more we can do
   set day [expr $secs / [expr 60 * 60 * 24]]
   if {$day > 0} {return $time}

   set hour [expr $secs / [expr 60 * 60]]
   if {$hour > 0} {
      append return_string [format "%02d:" $hour]
      set secs [expr $secs - [expr $hour * 60 * 60]]
      set min [expr $secs / 60]
      append return_string [format "%02d:" $min]
      set secs [expr $secs - [expr $min * 60]]
      append return_string [format "%02d" $secs]
   } else {
      set min [expr $secs / 60]
      if {$min > 0} {
         append return_string [format "%02d min " $min]
      }
      set secs [expr $secs - [expr $min * 60]]
      append return_string [format "%02d sec" $secs]
   }

   # Return formatted string
   return $return_string
}

###############################################################
# The procedure for updating the clock
proc Update_Clock {} {
   #Debug "Update_Clock"

   # Globals
   global gb_connection_down g_connection_time \
      gb_ping_connection gb_beep_on_connect g_ping_command \
      g_clock_refresh_freq g_connection_status GREEN RED \
      g_log_file gb_auto_reconnect g_disconnected_pixmap \
      gt_time_connected gt_saved_time g_throughput \
      gb_disconnect_on_user_request gt_last_stat_time \
      gt_last_ping

   # Only update clock if connected
   Set_Connection_Flag

   if {$gb_connection_down == 0} {
      set Time [Get_Current_Time]

      # Set into the text widget
      set connected_for [AddSub_Time $gt_time_connected \
         $Time -1 2]
      set connected_for [AddSub_Time $connected_for \
         $gt_saved_time 1 2]
      set connected_for [Strip_Time_String $connected_for]
      $g_connection_time configure -text $connected_for

      # Rerun after N secs
      if {$g_clock_refresh_freq > 0} {
         set run_after [ expr $g_clock_refresh_freq * 1000 ]
         after $run_after Update_Clock
      } else {
         set run_after 10000
         after $run_after Update_Clock
      }

      if {$gb_ping_connection == 1} {
         set secs [Time_To_Secs $Time]
         if {$secs > [expr $gt_last_ping + 100]} {
            # Keep the connection alive
            if [catch {eval exec $g_ping_command >>& $g_log_file &} \
               errortext] {
               Info_Dialog . "Ping failed : $errortext"
            }
            set gt_last_ping $secs
         }
      }
   } else {

      # Connection has gone down (use pixmap if specified)
      if {$g_disconnected_pixmap != ""} {
         set Image [Create_Image $g_disconnected_pixmap]
         if {$Image == -1} {
            # Image creation did not work, so turn it off!
            $g_connection_status configure -text  "Disconnected" \
               -background $GREEN -highlightbackground $GREEN
            set g_disconnected_pixmap ""
         } else {
            $g_connection_status configure -image $Image \
               -highlightbackground $GREEN
         }
      } else {
            $g_connection_status configure -text  "Disconnected" \
               -background $GREEN -highlightbackground $GREEN
      }
         
      Set_Message "Connection lost!" $RED
      wm iconname . "Net Down"

      # Reset Statistics variables
      set gt_last_stat_time 0
      # Update string
      Graph_Initialise $g_throughput

      # Beep if option set
      if {$gb_beep_on_connect == 1} {
         bell
         bell
      }

      # If a post-command is specified to run after a disconnection
      # then run it!
      global g_connect_postcmd g_connect_cond
      if {[string compare $g_connect_postcmd ""] != 0} {
         if {$g_connect_cond == 3} {
            # A post-command exists to be run now
            set cmd $g_connect_postcmd
            set g_connect_postcmd ""
            set g_connect_cond 0
            Set_Message "Running post-command $cmd" $GREEN
            Run_Script $cmd
         }
      }

      # If the option is set, attempt to reconnect 
      # immediately
      if {$gb_auto_reconnect == 1 && $gb_disconnect_on_user_request == 0} {
         Net_Connect
      }

      # Set the icon
      # wm iconbitmap . @/usr/include/X11/pixmaps/netdn.xpm
   }
}


