###############################################################
# TkNet - Script Setup Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Constants
set SCRIPT_ITEM_LENGTH  15

# Flags
set g_any_script_changed    0

# Session variables
set g_script_row_number     0
set gi_default_row          0
set g_current_script_row    0

# Stored session variables
set gi_default_session      0
set gi_current_session      0
set g_number_sessions       0

# Geometry
set TKNET_SCRIPT_SETUP_GEOMETRY "+408+51"

# Files
set TKNET_GLOBAL_SCRIPT_FILE "$GLOBAL_INCLUDE/tknet.scripts"
set TKNET_LOCAL_SCRIPT_FILE "$TKNET_USER_HOME/.tknet_scripts"

###############################################################
# The procedure to read the Script setup files
# Argument must be 0=Both 1=Global or 2=Local 
proc Source_Script_Setup { file } {
   Debug "Source_Script_Setup"

   # Globals
   global TKNET_GLOBAL_SCRIPT_FILE \
      TKNET_LOCAL_SCRIPT_FILE g_current_script_row \
      gi_default_session gi_current_session

   # Read the global setup file
   if {$file != 2 && [file exists $TKNET_GLOBAL_SCRIPT_FILE]} {
      source $TKNET_GLOBAL_SCRIPT_FILE
   }
   
   # Read the user's own config file if it exists
   if {$file != 1 && [file exists $TKNET_LOCAL_SCRIPT_FILE]} {
      # Override with users own
      source $TKNET_LOCAL_SCRIPT_FILE
   }

   # And set the current default script
   set gi_current_session $gi_default_session
}

###############################################################
# Show the setup panel 
proc Show_Script_Setup { } {
   Debug "Show_Script_Setup"

   # Globals
   global FONT_NORMAL FONT_BOLD DEFAULT_PADDING RIDGE_BORDER \
      g_script_row_number g_any_script_changed \
      TKNET_SCRIPT_SETUP_GEOMETRY g_number_sessions \
      gi_default_session gi_default_row

   # Create the setup toplevel as a transient from the main window
   set window .script_setup
   if [winfo exists $window] {destroy $window}
   toplevel $window
   wm title $window "TkNet Script Setup"
   wm transient $window .
   wm geometry $window $TKNET_SCRIPT_SETUP_GEOMETRY
   wm protocol $window WM_DELETE_WINDOW Close_Script_Setup

   ###########################################################################
   # Create Menu Bar
   set menu [frame $window.mbar -relief raised -bd 2]
   pack $menu -side top -fill x

   menubutton $menu.file -text File -underline 0 -menu $menu.file.menu
   menubutton $menu.edit -text Edit -underline 0 -menu $menu.edit.menu
   menubutton $menu.setup -text Setup -underline 0 -menu $menu.setup.menu
   menubutton $menu.help -text Help -underline 0 -menu $menu.help.menu
   pack $menu.file $menu.edit $menu.setup -side left
   pack $menu.help -side right

   menu $menu.file.menu -tearoff 0
      $menu.file.menu add command -label "Save" \
         -command { Save_Script_Setup } -underline 0
      $menu.file.menu add command -label \
         "Set Defaults from Global" -command \
         { Source_Script_Setup 1 ; Show_Script_Setup ; \
         set g_setup_changed 1 } -underline 4
      $menu.file.menu add separator
      $menu.file.menu add command -label "Close" \
         -command { Close_Script_Setup } -underline 0

   menu $menu.edit.menu -tearoff 0
      $menu.edit.menu add command -label "Add to end" \
         -command {New_Script_Session $g_script_row_number; \
         incr g_script_row_number; set g_any_script_changed 1} \
         -underline 0
      $menu.edit.menu add command -label \
         "Add before current" \
         -command {New_Script_Session $g_current_script_row; \
         incr g_script_row_number; set g_any_script_changed 1} \
         -underline 0
      $menu.edit.menu add separator
      $menu.edit.menu add command -label "Delete current" \
         -command {Delete_Script_Session; \
         set g_any_script_changed 1} -underline 0

   menu $menu.setup.menu -tearoff 0
      $menu.setup.menu add command -label \
         "Scripts..." -command {Show_Interactive_Scripts} \
         -underline 0

   menu $menu.help.menu -tearoff 0
      $menu.help.menu add command -underline 0 -label \
         "Show Help..." -command {MH_MiniHelp ScriptSetupWindow}

   tk_menuBar $menu $menu.file $menu.edit $menu.setup $menu.help

   ###############################################################
   # Create the instructions
   button .script_setup.label -text \
      "Enter the connection scripts in the fields below.\n\
To obtain help click on this label." -relief flat \
      -command "MH_MiniHelp ScriptSetupWindow"
   frame .script_setup.dummy -width 10c -height 3m
   pack .script_setup.label .script_setup.dummy \
      -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING -side top -fill x

   ###############################################################
   # Create the column headers
   label .script_setup.column1 -text "Default"
   place .script_setup.column1 -in .script_setup.label \
      -rely 1.0 -y 5 -x 0
   label .script_setup.column2 -text "Row"
   place .script_setup.column2 -in .script_setup.label \
      -rely 1.0 -y 5 -x 70 
   label .script_setup.column3 -text "Label"
   place .script_setup.column3 -in .script_setup.label \
      -rely 1.0 -y 5 -x 170 
   label .script_setup.column4 -text "Connect"
   place .script_setup.column4 -in .script_setup.label \
      -rely 1.0 -y 5 -x 320 
   label .script_setup.column5 -text "Disconnect"
   place .script_setup.column5 -in .script_setup.label \
      -rely 1.0 -y 5 -x 500
   label .script_setup.column6 -text "Timeout"
   place .script_setup.column6 -in .script_setup.label \
      -rely 1.0 -y 5 -x 620

   ###############################################################
   # Create the main panel
   frame .script_setup.setup_fr -borderwidth $RIDGE_BORDER \
      -relief groove 
   pack .script_setup.setup_fr -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING -side top -expand true -fill both
   canvas .script_setup.setup_fr.c -yscrollcom \
      ".script_setup.setup_fr.sy set" -scrollregion \
      "0 0 700 0" -width 700 -highlightthickness 0
   pack [scrollbar .script_setup.setup_fr.sy -orient v \
      -com ".script_setup.setup_fr.c yview"] \
      -side right -fill y
   pack .script_setup.setup_fr.c -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side top -fill both \
      -expand true -exp 1

   ###############################################################
   # Create the individual settings
   set total $g_number_sessions
   set g_script_row_number 0
   for {set count 1} {$count <= $total} {incr count} \
   {
      # Define globals
      global g_script_$count

      # Check for default flag
      if {$gi_default_session == $count} {
         set gi_default_row $g_script_row_number
      }
               
      # Get the number of rows
      set rows [subst $[subst g_script_${count}(number)]]
      for {set script 1} {$script <= $rows} {incr script} \
      {
         # Create a row for the item
         New_Script_Session $g_script_row_number
         set frame .script_setup.setup_fr.c.$g_script_row_number

         # Define globals for the row
         global g_script_row_type_$g_script_row_number \
            g_script_connect_$g_script_row_number \
            g_script_disconnect_$g_script_row_number \
            g_script_timeout_$g_script_row_number

         if {$script == 1} {
            set g_script_row_type_${g_script_row_number} NEW
            set text [subst $[subst g_script_${count}(label)]]
            $frame.label insert end $text
            Select_Row_Type NEW $g_script_row_number $frame
         } else {
            set g_script_row_type_${g_script_row_number} THEN
            Select_Row_Type THEN $g_script_row_number $frame
         }

         # Write Script item (checking that each of the
         # scripts exists)
         set row [expr $g_script_row_number + 1]
         set text [subst $[subst g_script_${count}(connect$script)]]
         if [Script_Exists $text] {
            set g_script_connect_$g_script_row_number $text
         } else {
            Info_Dialog . "Connect script for row $row reset -
No such script '$text' exists!"
            set g_any_script_changed 1
         }
         set text [subst $[subst g_script_${count}(disconnect$script)]]
         if [Script_Exists $text] {
            set g_script_disconnect_$g_script_row_number $text
         } else {
            Info_Dialog . "Disconnect script for row $row reset -
No such script '$text' exists!"
            set g_any_script_changed 1
         }
         set text [subst $[subst g_script_${count}(timeout$script)]]
         set g_script_timeout_$g_script_row_number $text

         # Increment the number of displayed rows
         incr g_script_row_number
      }
   }

   .script_setup.setup_fr.c yview moveto 0
   frame .script_setup.button_frame -borderwidth $DEFAULT_PADDING
   pack .script_setup.button_frame -side bottom -fill x
   button .script_setup.button_frame.close -font $FONT_NORMAL \
       -text Close -command { Close_Script_Setup }
   pack .script_setup.button_frame.close
}

###############################################################
# Does the script exist as a PPP script or a command
proc Script_Exists { text } {
   Debug "Script_Exists"

   regsub -all {[\ ]+} $text {_} search
   set scripts [info globals "gs_ppp_name_$search"]
   if {[llength $scripts] > 0} {
      return 1
   }
   set scripts [info globals "gs_cmd_name_$search"]
   if {[llength $scripts] > 0} {
      return 1
   }
   return 0
}

###############################################################
# Create a button, entry pair
# Number is the item to insert after
proc New_Script_Session { number } {
   Debug "New_Script_Session"

   # Globals
   global FONT_NORMAL DEFAULT_PADDING \
      SCRIPT_ITEM_LENGTH g_any_script_changed \
      g_current_script_row \
      g_script_row_number gi_default_row \
      RIDGE_BORDER

   # A new row involves the creation of a number of
   # new global variables to hold the information
   # pertaining to the row.  These are:
   # 
   # The row type (NEW or THEN)
   #   g_script_row_type_${g_script_row_number}
   # The connect script
   #   g_script_connect_${g_script_row_number}
   # The disconnect script
   #   g_script_disconnect_${g_script_row_number}
   # The timeout for this script
   #   g_script_timeout_${g_script_row_number}

   # Work out the canvas height
   set canvas_height [expr $g_script_row_number * 55]
   .script_setup.setup_fr.c configure -scrollregion \
      "0 0 700 $canvas_height"

   # Create the row - Radiobutton. When set the default
   # session is the physical number of the row.
   set frame [frame .script_setup.setup_fr.c.$g_script_row_number]
   set item [radiobutton $frame.default \
      -variable gi_default_row \
      -value $g_script_row_number -anchor w]
   bind $item <ButtonPress> "set g_any_script_changed 1"
   bind $item <FocusIn> "set g_current_script_row \
      $g_script_row_number"

   # Name
   set item [entry $frame.label -width $SCRIPT_ITEM_LENGTH]
   bind $item <ButtonPress> "set g_any_script_changed 1"
   bind $item <FocusIn> "set g_current_script_row \
      $g_script_row_number"

   # Menu giving the row type (continuation or new)
   global g_script_row_type_${g_script_row_number}
   menubutton $frame.row_type -menu \
      $frame.row_type.menu -textvariable \
      "g_script_row_type_${g_script_row_number}" \
      -width 4 -borderwidth $RIDGE_BORDER \
      -relief raised -indicatoron True
   menu $frame.row_type.menu -tearoff 0
      $frame.row_type.menu add command -label NEW \
      -command "Select_Row_Type NEW $g_script_row_number $frame"
      $frame.row_type.menu add command -label THEN \
      -command "Select_Row_Type THEN $g_script_row_number $frame"
   bind $frame.row_type <ButtonPress> "set g_any_script_changed 1 ; \
      set g_current_script_row $g_script_row_number"

   # Get the current list of available scripts
   # These begin with gs_ppp_name_ and gs_cmd_name_
   set list ""
   set search "gs_ppp_name_*"
   set globals_list1 [info globals $search]
   foreach item $globals_list1 {
      global $item
      set name [subst $[subst $item]]
      lappend list $name
   }
   set search "gs_cmd_name_*"
   set globals_list2 [info globals $search]
   foreach item $globals_list2 {
      global $item
      set name [subst $[subst $item]]
      lappend list $name
   }

   # Check that the list is not empty!
   if {$list == ""} {
      Info_Dialog . "No scripts yet created.  Please select
Setup / Scripts... and create some scripts."
      return
   }
   
   # The connect script
   global g_script_connect_$g_script_row_number
   set item [eval tk_optionMenu $frame.connect \
      "g_script_connect_$g_script_row_number" \
      $list]
   $frame.connect configure -width $SCRIPT_ITEM_LENGTH
   bind $frame.connect <ButtonPress> "set g_any_script_changed 1 ; \
      set g_current_script_row $g_script_row_number"

   # The disconnect script
   global g_script_connect_$g_script_row_number
   set item [eval tk_optionMenu $frame.disconnect \
      "g_script_disconnect_$g_script_row_number" \
      $list]
   $frame.disconnect configure -width $SCRIPT_ITEM_LENGTH
   bind $frame.disconnect <ButtonPress> "set g_any_script_changed 1 ; \
      set g_current_script_row $g_script_row_number"

   # The timeout
   global g_script_timeout_$g_script_row_number
   set item [tk_optionMenu $frame.timeout \
      "g_script_timeout_$g_script_row_number" \
      10 20 30 40 50 60 70 80 90 100 110 120]
   $frame.timeout configure -width 3
   bind $frame.timeout <ButtonPress> "set g_any_script_changed 1 ; \
      set g_current_script_row $g_script_row_number"

   # Pack the row
   pack $frame.default $frame.row_type $frame.label \
      $frame.connect $frame.disconnect $frame.timeout \
      -side left \
      -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING

   set offset [expr (45 * $g_script_row_number)]
   .script_setup.setup_fr.c create window 10 $offset \
      -window .script_setup.setup_fr.c.$g_script_row_number \
      -anchor nw

   # Scroll to the new item
   if {$g_script_row_number > 0} {
      set perc [expr ( 1.0 * $number ) / $g_script_row_number]
      .script_setup.setup_fr.c yview moveto $perc
   }

   # If number is less that the current total, user
   # has requested an insert.  Move all items below
   # number down one.
   if {$number < $g_script_row_number} {

      set last [expr $g_script_row_number -1]
      for {set count $last} {$count >= $number} {incr count -1} \
      {
         # Copy the text from the current item to the next item
         set next [expr $count + 1]
         set frame .script_setup.setup_fr.c.$next
   
         # The default
         $frame.default configure -state normal
         if {$gi_default_row == $count} {
            incr gi_default_row
         }

         # The label
         $frame.label configure -state normal
         set text [.script_setup.setup_fr.c.$count.label get]
         .script_setup.setup_fr.c.$next.label delete 0 end
         .script_setup.setup_fr.c.$next.label insert end $text

         # The row_type
         global g_script_row_type_$next g_script_row_type_$count
         set g_script_row_type_$next [subst $[subst \
            g_script_row_type_$count]]
         Select_Row_Type [subst $[subst g_script_row_type_$next]]\
            $next $frame

         # The connect script
         global g_script_connect_$next g_script_connect_$count
         set g_script_connect_$next [subst $[subst \
            g_script_connect_$count]]

         # The disconnect script
         global g_script_disconnect_$next g_script_disconnect_$count
         set g_script_disconnect_$next [subst $[subst \
            g_script_disconnect_$count]]

         # The timeout
         global g_script_timeout_$next g_script_timeout_$count
         set g_script_timeout_$next [subst $[subst \
            g_script_timeout_$count]]
      }

      # Blank the new entry
      .script_setup.setup_fr.c.$number.label configure \
         -state normal
      .script_setup.setup_fr.c.$number.label delete 0 end
   }
   # Do not increment option number - responsibility of caller
   set g_current_script_row $number
}

###############################################################
# Deletes the script item given by g_current_script_row
proc Delete_Script_Session {} {
   Debug "Delete_Script_Session"

   # Globals
   global g_any_script_changed g_current_script_row \
      g_script_row_number g_any_script_changed \
      gi_default_row

   set last [expr $g_script_row_number - 1]
   for {set count $g_current_script_row} {$count < $last} \
      {incr count} \
   {
      # Copy the text from the next item to the current item
      set next [expr $count + 1]
      set frame .script_setup.setup_fr.c.$count

      # Check for bounds
      if {$next > $last} {
         continue
      }

      # The default
      $frame.default configure -state normal
      if {$gi_default_row == $count} {
         incr gi_default_row -1
      }

      # The label
      $frame.label configure -state normal
      set text [.script_setup.setup_fr.c.$next.label get]
      .script_setup.setup_fr.c.$count.label delete 0 end
      .script_setup.setup_fr.c.$count.label insert end $text

      # The row_type
      global g_script_row_type_$next g_script_row_type_$count
      set g_script_row_type_$count [subst $[subst \
         g_script_row_type_$next]]
      Select_Row_Type [subst $[subst g_script_row_type_$count]]\
         $count $frame

      # The connect script
      global g_script_connect_$next g_script_connect_$count
      set g_script_connect_$count [subst $[subst \
         g_script_connect_$next]]

      # The disconnect script
      global g_script_disconnect_$next g_script_disconnect_$count
      set g_script_disconnect_$count [subst $[subst \
         g_script_disconnect_$next]]

      # The timeout
      global g_script_timeout_$next g_script_timeout_$count
      set g_script_timeout_$count [subst $[subst \
         g_script_timeout_$next]]
   }

   # Now delete the last row
   destroy .script_setup.setup_fr.c.$last

   # And decrease the count
   incr g_script_row_number -1
   set g_any_script_changed 1
}

###############################################################
# The procedure for quitting the setup window
proc Close_Script_Setup {} {
   Debug "Close_Script_Setup"

   # Globals
   global g_any_script_changed

   # Check to see whether anything has changed.
   if {$g_any_script_changed == 1 && ![Question_Dialog .script_setup \
      "Do you wish to save changes?" "Save" "Reset Changes"]} {

      # Save requested
      Save_Script_Setup
   } else {

      # Re-read defaults from file
      Source_Script_Setup 0
      set g_any_script_changed 0
   }

   # Destroy the window
   destroy .script_setup 
}

###############################################################
# The procedure for saving the configuration
proc Save_Script_Setup {} {
   Debug "Save_Script_Setup"

   # Globals
   global g_script_row_number g_setup_file g_any_script_changed \
      TKNET_USER TKNET_GLOBAL_SCRIPT_FILE \
      TKNET_LOCAL_SCRIPT_FILE GREEN \
      gb_script_window gi_default_session \
      gi_default_row

   # Save the setup to file
   # For each of the piars of items in the window,
   # ascertain the variable name being modified, then
   # write the output to the setup file specified.
   # 
   # This file contains the following information:
   # 
   # g_number_sessions - The number of sessions
   # gi_default_script - the default session to run
   # g_script_N - the session details

   # If root save to global settings, else save to users
   # local ~/.tknet_setup file
   if {$TKNET_USER == "root" && [Question_Dialog \
      . "Where do you want to save script setup to?" \
      "Local" "Global"]} {
         set g_setup_file $TKNET_GLOBAL_SCRIPT_FILE
   } else {
      set g_setup_file $TKNET_LOCAL_SCRIPT_FILE
   }

   # Open the file
   set file_id [ open $g_setup_file w ]
   puts $file_id "#########################################################
### This is the TkNet Scripts file.
### This file contains details of connection and 
### disconnection sequences used by TkNet. It is
### maintained by the Script Setup screen. 
###
### PLEASE DO NOT EDIT BY HAND UNLESS YOU ARE QUITE
### SURE WHAT YOU ARE DOING!
#########################################################"
   puts $file_id " "

   # Write all information
   puts $file_id "## Sessions"

   # Initialise session count
   set g_number_sessions 0
   set canvas .script_setup.setup_fr.c
   set gi_default_session 0

   # Analyse each row in turn
   for {set count 0} {$count < $g_script_row_number} \
      {incr count} \
   {
      # Set up variables
      set frame $canvas.$count

      # If this is the start of a new session then start anew
      global g_script_row_type_$count
      if {[subst $[subst g_script_row_type_$count]] == "NEW"} {
         # New session
         if {$g_number_sessions != 0} {
            # Close the previous session
            puts $file_id "set g_script_${g_number_sessions}(number) $script_count"
            puts $file_id " "
         }
         set script_count 1
         incr g_number_sessions
         puts $file_id "global g_script_${g_number_sessions}"
         set label [$frame.label get]
         puts $file_id "set g_script_${g_number_sessions}(label) \"$label\""

         # Check whether the default flag is set for this 
         # new session
         if {$gi_default_row == $count} {
            set gi_default_session $g_number_sessions
         }
      } else {
         # Continuation
         incr script_count
      }

      # For the line, get the script details and timeout
      # and write to the file
      global g_script_connect_$count g_script_disconnect_$count \
         g_script_timeout_$count
      set connect [subst $[subst g_script_connect_$count]]
      set disconnect [subst $[subst g_script_disconnect_$count]]
      set timeout [subst $[subst g_script_timeout_$count]]

      puts $file_id "set g_script_${g_number_sessions}(connect${script_count}) \"$connect\""
      puts $file_id "set g_script_${g_number_sessions}(disconnect${script_count}) \"$disconnect\""
      puts $file_id "set g_script_${g_number_sessions}(timeout${script_count}) \"$timeout\""
   }

   # Close the last session
   puts $file_id "set g_script_${g_number_sessions}(number) $script_count"
   puts $file_id " "

   # Write the number of items in the script as determined
   # by the above loop
   puts $file_id "## Number of items"
   puts $file_id "global g_number_sessions"
   puts $file_id "set g_number_sessions $g_number_sessions"
   puts $file_id "global gi_default_session"
   puts $file_id "set gi_default_session $gi_default_session"

   # Close file
   flush $file_id
   close $file_id

   # Display a dialog
   Set_Message "Script Items Saved to $g_setup_file" $GREEN
   set g_any_script_changed 0

   # Re-read from file
   Source_Script_Setup 0

   # Show the script again
   Show_Script_Setup

   # Re-create both menus
   Create_File_Menu
   Create_Popup_Menu
}


###############################################################
# Toggles the status of the label field depending on 
# whether the row is a continuation or a new script
proc Select_Row_Type { text row frame } {
   Debug "Select_Row_Type"

   # Globals
   global g_script_row_type_$row \
      gi_default_row

   # Set the selected value into the menu.  Note that
   # this is being performed manually because tk_optionmenu
   # doesn't give us the flexibility we need to call this
   # routine on selection.
   set g_script_row_type_$row $text

   # If this is a continuation grey out the 
   # label and default button fields.
   if {$text == "THEN"} {
      $frame.label delete 0 end
      $frame.label configure -state disabled
      $frame.default configure -state disabled
      if {$gi_default_row == $row} {
        set gi_default_row 0
      }
   } else {
      $frame.label configure -state normal
      $frame.default configure -state normal
   }      
}
