###############################################################
# TkNet - PPP Setup Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Flags
set gl_ppp_windows      [list "0"]

# Geometry
set TKNET_PPP_GEOMETRY "+500+150"

# Files
set TKNET_GLOBAL_NETWORK_FILE "$GLOBAL_INCLUDE/tknet.network"
set TKNET_LOCAL_NETWORK_FILE "$TKNET_USER_HOME/.tknet_network"

###############################################################
# The procedure to read the network setup files
proc PPP_Source_Scripts { file } {
   Debug "PPP_Source_Scripts"

   # Argument must be 0=Both 1=Global or 2=Local 

   # Globals
   global TKNET_GLOBAL_NETWORK_FILE \
      TKNET_LOCAL_NETWORK_FILE

   # Read the global setup file
   if {$file != 2 && [file exists $TKNET_GLOBAL_NETWORK_FILE]} {
      source $TKNET_GLOBAL_NETWORK_FILE
   }
   
   # Read the user's own config file if it exists
   if {$file != 1 && [file exists $TKNET_LOCAL_NETWORK_FILE]} {
      # Override with users own
      source $TKNET_LOCAL_NETWORK_FILE
   }
}

##################################################################
# Procedure for editing and creating ppp sessions using 
# ppp and chat. 
proc PPP_Create_Session { parent edit_type name } {
   Debug "PPP_Create_Session"

   # parent - Calling window
   # edit_type = EDIT/NEW
   # name = Name of script to edit

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING TKNET_PPP_GEOMETRY \
      gl_ppp_windows SCRIPT_ITEM_LENGTH

   # The global variable gi_number_ppp_windows gives the 
   # number of ppp edit windows currently in use.  Each
   # currently used window is checked to see whether it
   # is the edit session for the desired script. If no
   # current session is found, a new session is created.
   set win_num [PPP_Find_Script $name]
   if {$win_num != 0} {
      # Session found, pop it up!
      wm deiconify .ppp_window_$win_num
      raise .ppp_window_$win_num
      update
      return
   }

   # Set local constants
   set LABEL_LENGTH 15
   set RELIEF flat

   # Create a dialogue, read the last number from gl_ppp_windows
   # and increment it.
   set win_num [expr [lindex $gl_ppp_windows end] + 1]
   lappend gl_ppp_windows $win_num
   set window [toplevel .ppp_window_$win_num]
   wm title $window "PPP Setup : $name"
   wm iconname $window "PPP : $name"
   wm geometry $window $TKNET_PPP_GEOMETRY
   wm protocol $window WM_DELETE_WINDOW "PPP_Close_Window \
      $parent $win_num 0"

   # Set instance global variables
   global gs_ppp${win_num}_interface 
   global gs_ppp${win_num}_edit_type
   global gs_ppp${win_num}_modified
   set gs_ppp${win_num}_edit_type $edit_type
   set gb_ppp${win_num}_modified False

   # Change to internal representation of name - all spaces
   # converted to _'s.
   set orig_name $name
   regsub -all {[\ ]+} $name {_} name

   # A PPP session is completely described by:
   #
   # (1) A name
   # (2) An interface (e.g. ppp0, sl0)
   # (3) A pppd command string
   # (4) A command to run before hand
   # (5) A command to run afterwards
   # (6) Conditions for running post command
   global gs_ppp_name_${name}
   global gs_ppp_interface_${name}
   global gs_ppp_script_${name}
   global gs_ppp_precmd_${name}
   global gs_ppp_postcmd_${name}
   global gs_ppp_cond_${name}
   set gs_ppp_name_${name} $orig_name

   # Note that the parameter edit_type may be either
   # NEW or EDIT.  However if the script name is
   # modified in any way the edit type becomes NEW.
   
   ###########################################################################
   # Create Menu Bar
   frame $window.mbar -relief raised -bd 2
   pack $window.mbar -side top -fill x

   # Create the buttons   
   menubutton $window.mbar.file -text File -underline 0 -menu \
      $window.mbar.file.menu
   menubutton $window.mbar.edit -text Edit -underline 0 \
      -menu $window.mbar.edit.menu
   menubutton $window.mbar.help -text Help -underline 0 \
      -menu $window.mbar.help.menu
   pack $window.mbar.file $window.mbar.edit \
      -side left
   pack $window.mbar.help -side right

   # Create each menu item
   menu $window.mbar.file.menu -tearoff 0
      $window.mbar.file.menu add command -label "Save" \
         -command "PPP_Save_Script $win_num $name" -underline 0
      $window.mbar.file.menu add command -label "Return" \
         -command "PPP_Save_Script $win_num $name ; \
         PPP_Close_Window $parent $win_num 1" -underline 0
      $window.mbar.file.menu add command -label "Close" \
         -command "PPP_Close_Window $parent $win_num 0" \
         -underline 0
   menu $window.mbar.edit.menu -tearoff 0
      $window.mbar.edit.menu add command -label "Add to end" \
         -command "PPP_New_Item $win_num END" -underline 0
      $window.mbar.edit.menu add command -label "Add before current" \
         -command "PPP_New_Item $win_num CURRENT" -underline 0
      $window.mbar.edit.menu add separator
      $window.mbar.edit.menu add command -label "Delete current" \
         -command "PPP_Delete_Item $win_num" -underline 0
   menu $window.mbar.help.menu -tearoff 0
      $window.mbar.help.menu add command -label "Show Help..." \
         -command "MH_MiniHelp PPPWindow" -underline 0

   # Create the menu
   tk_menuBar $window.mbar $window.mbar.file \
      $window.mbar.edit $window.mbar.help

   # Top frame contains the selected script details
   frame $window.sel_fr -borderwidth $RIDGE_BORDER -relief groove
   pack $window.sel_fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill x
   label $window.sel_fr.label -text "Selected Script" \
      -width 20 -anchor nw
   entry $window.sel_fr.entry 
   $window.sel_fr.entry insert end $orig_name
   pack $window.sel_fr.label $window.sel_fr.entry \
      -side left -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING
   bind $window.sel_fr.entry <KeyPress> "set \
      gs_ppp${win_num}_edit_type NEW"
   tk_optionMenu $window.sel_fr.interface \
      gs_ppp${win_num}_interface "ppp0" "ppp1" "sl0" "sl1" 
   $window.sel_fr.interface configure -width 5
   pack $window.sel_fr.interface -side left -padx \
      $DEFAULT_PADDING -pady $DEFAULT_PADDING

   #################################################################
   # Create a frame for the pre and post commands.
   set frame [frame $window.precmd_fr -borderwidth \
      $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING  \
      -side top -fill x

   # Now create the two option menus which contain all currently
   # available command scripts which may be used as pre and post 
   # commands.
   set search "gs_cmd_name_*"
   set globals_list [info globals $search]
   set list "-none-"
   foreach item $globals_list {
      global $item
      set lscr [subst \$$item]
      lappend list $lscr
   }
   global g_ppp${win_num}_precmd g_ppp${win_num}_postcmd \
      g_ppp${win_num}_cond 
   set frame1 [frame $frame.f1]
   button $frame1.l -text "Pre-command:" -relief flat -command \
      "MH_MiniHelp PrePostCommands" -width 15 -justify left
   eval tk_optionMenu $frame1.precmd g_ppp${win_num}_precmd $list
   $frame1.precmd configure -width $SCRIPT_ITEM_LENGTH
   pack $frame1.l $frame1.precmd -side left
   set frame2 [frame $frame.f2]
   button $frame2.l -text "Post-command:" -relief flat -command \
      "MH_MiniHelp PrePostCommands" -width 15 -justify left
   eval tk_optionMenu $frame2.postcmd g_ppp${win_num}_postcmd $list
   $frame2.postcmd configure -width $SCRIPT_ITEM_LENGTH
   pack $frame2.l $frame2.postcmd -side left
   set frame3 [frame $frame.f3]
   set g_ppp${win_num}_cond 1
   radiobutton $frame3.r1 -variable g_ppp${win_num}_cond \
      -value 1 -text "Immediate"
   radiobutton $frame3.r2 -variable g_ppp${win_num}_cond \
      -value 2 -text "Connect"
   radiobutton $frame3.r3 -variable g_ppp${win_num}_cond \
      -value 3 -text "Disconnect"
   pack  $frame3.r1 $frame3.r2 $frame3.r3 -side left \
      -padx $DEFAULT_PADDING
   pack $frame1 $frame2 $frame3 -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side top -fill x

   # OPTIONS
   # Now create the first part of the interface, the PPPD options
   # Supported options are:
   # 
   # crtscts - Hardware flow control (if off then xonxoff is used)
   # defaultroute - Add PPP route to system routine tables
   # lock - Use a uucp like lock on the modem
   # passive - Enables passive option (see man page)
   # debug - Enables debugging
   # noipdefault - Not sure what this does!
   #
   # And the configurable items:
   # 
   # Tty-name - e.g /dev/ppp0
   # Speed - port speed
   # mru - Set maximum received unit
   # mtu - Set maximum transfer unit
   # netmask - Set net mask.
   # Local/Remote IP addresses
   # File - read options from file
   # asyncmap - how to escape control characters
   # Other command line options
   # 

   ##########################################################
   # Create flag frame and help
   set frame [frame $window.flag_fr -borderwidth \
       $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING  \
      -side top -fill x

   # Help button and instructions
   button $frame.label -text "Select the flags to pass to PPPD.\n\
To obtain help click on this label." -relief flat -command \
      "MH_MiniHelp PPPDFlags"
   pack $frame.label -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING\
      -side top -fill x
   
   # Scrollable canvas region
   set canvas [canvas $frame.c -scrollregion "0 0 400 0" \
      -width 400 -height 80 -highlightthickness 0 -yscrollcom \
      "$frame.sy set"]
   pack [scrollbar $frame.sy -orient v -com "$frame.c yview"] \
      -side right -fill y
   pack $frame.c -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill both -expand true

   # Create flags
   set offset 0
   global g_ppp${win_num}_flag_crtscts
   checkbutton $canvas.crtscts -text \
      "Use hardware flow-control (crtscts)" \
      -variable g_ppp${win_num}_flag_crtscts
   $canvas create window 5 $offset -window $canvas.crtscts -anchor nw

   incr offset 30
   global g_ppp${win_num}_flag_defaultroute
   checkbutton $canvas.defaultroute -text \
      "Add PPP route to system routing tables" \
      -variable g_ppp${win_num}_flag_defaultroute
   $canvas create window 5 $offset -window $canvas.defaultroute -anchor nw

   incr offset 30
   global g_ppp${win_num}_flag_lock
   checkbutton $canvas.lock -text \
      "Lock the modem when in use" \
      -variable g_ppp${win_num}_flag_lock
   $canvas create window 5 $offset -window $canvas.lock -anchor nw

   incr offset 30
   global g_ppp${win_num}_flag_passive
   checkbutton $canvas.passive -text \
      "Wait for peer to connect (Passive option)" \
      -variable g_ppp${win_num}_flag_passive
   $canvas create window 5 $offset -window $canvas.passive -anchor nw

   incr offset 30
   global g_ppp${win_num}_flag_debug
   checkbutton $canvas.debug -text \
      "Enable PPPD debugging" \
      -variable g_ppp${win_num}_flag_debug
   $canvas create window 5 $offset -window $canvas.debug -anchor nw

   incr offset 30
   global g_ppp${win_num}_flag_noipdefault
   checkbutton $canvas.noipdefault -text \
      "Do not use default IP Addresses" \
      -variable g_ppp${win_num}_flag_noipdefault
   $canvas create window 5 $offset -window $canvas.noipdefault -anchor nw

   # And configure the scroll region for the canvas
   incr offset 30
   $canvas configure -scrollregion "2 2 400 $offset"

   ##########################################################
   # Create options frame and help
   # Create flag frame and help
   set frame [frame $window.options_fr -borderwidth \
       $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -side top -fill x

   # Help buttons and instructions
   label $frame.label -text \
      "Select the Options to pass to PPPD.\n\
To obtain help click on the labels below."
   pack $frame.label -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING -side top -fill x

   # Scrollable canvas region
   set canvas [canvas $frame.c -scrollregion "0 0 400 0" \
      -width 400 -height 80 -highlightthickness 0 -yscrollcom \
      "$frame.sy set"]
   pack [scrollbar $frame.sy -orient v -com "$canvas yview"] \
      -side right -fill y
   pack $canvas -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill x -expand true -fill y

   # Create entry for TTY
   set offset 0
   set xoffset 200
   set label [button $canvas.l_tty -text "Device" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsDevice"]
   set entry [entry $canvas.tty -width 11]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for SPEED
   incr offset 30
   set label [button $canvas.l_speed -text "Baud Rate" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsBaudRate"]
   set entry [entry $canvas.speed -width 7]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for MRU
   incr offset 30
   set label [button $canvas.l_mru -text "Maximum Receive Unit" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsMTU"]
   set entry [entry $canvas.mru -width 7]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for MTU
   incr offset 30
   set label [button $canvas.l_mtu -text "Maximum Transmit Unit" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsMRU"]
   set entry [entry $canvas.mtu -width 7]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for Netmask
   incr offset 30
   set label [button $canvas.l_netmask -text "Net Mask" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsNetMask"]
   set entry [entry $canvas.netmask -width 15]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for Options file
   incr offset 30
   set label [button $canvas.l_file -text "Options File" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsOptionsFile"]
   set entry [entry $canvas.file -width 15]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for local IP address
   incr offset 30
   set label [button $canvas.l_localip -text "Local IP Address" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp PPPDOptionsLocalIP"]
   set entry [entry $canvas.localip -width 15]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for remote IP address
   incr offset 30
   set label [button $canvas.l_remoteip -text "Remote IP Address" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp g_ppp_remoteip_text_help"]
   set entry [entry $canvas.remoteip -width 15]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for control mask
   incr offset 30
   set label [button $canvas.l_asyncmap -text "Control character xmit mask" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp g_ppp_asyncmap_text_help"]
   set entry [entry $canvas.asyncmap -width 20]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Create entry for extra flags
   incr offset 30
   set label [button $canvas.l_flags -text "Other Flags" \
      -relief flat -anchor nw \
      -command "MH_MiniHelp g_ppp_flags_text_help"]
   set entry [entry $canvas.flags -width 20]
   $canvas create window 5 $offset -window $label -anchor nw
   $canvas create window $xoffset $offset -window $entry -anchor nw

   # Configure the scrollregion
   incr offset 30
   $canvas configure -scrollregion "2 2 400 $offset"

   ##########################################################
   # Create expect/send region and canvas
   set frame [frame $window.chat_fr -borderwidth \
       $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill both -expand true

   # Help button and instructions
   button $frame.label -text "Create the connect script.\n\
To obtain help click on this label." -relief flat -command \
      "MH_MiniHelp ChatScript"
   pack $frame.label -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING\
      -side top -fill x
   
   # Scrollable canvas region
   set canvas [canvas $frame.c -scrollregion "0 0 400 0" \
      -width 400 -height 100 -highlightthickness 0 -yscrollcom \
      "$frame.sy set"]
   pack [scrollbar $frame.sy -orient v -com "$frame.c yview"] \
      -side right -fill y
   pack $canvas -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill both -expand true

   # The entries within this region are maintained by
   # global variables, which are reset here.
   # The number of rows in the canvas
   global gi_ppp${win_num}_number_rows
   set gi_ppp${win_num}_number_rows 0
   # The current row in the canvas
   global gi_ppp${win_num}_current_row
   set gi_ppp${win_num}_current_row 0

   # Bind the right mouse button to pop-up the edit menu
   bind $window <Button-3> "Popup_Menu $frame \
      $window.mbar.edit.menu"
   bind $window <Button-2> "Popup_Menu $frame \
      $window.mbar.edit.menu"

   # Now load the data associated with the given script
   if {$edit_type == "EDIT"} {
      PPP_Display_Script $win_num $name
      $canvas yview moveto 0
   } else {
      # Reset interface
      set g_ppp${win_num}_flag_crtscts 0
      set g_ppp${win_num}_flag_defaultroute 0
      set g_ppp${win_num}_flag_lock 0
      set g_ppp${win_num}_flag_debug 0
      set g_ppp${win_num}_flag_noipdefault 0
   }
}

##################################################################
# Procedure for adding a new send/expect sequence
proc PPP_New_Item { win_num item } {
   Debug "PPP_New_Item"

   # Win_num is the id of the PPP edit window.
   # Item may be END or CURRENT

   # Globals
   global gi_ppp${win_num}_number_rows \
      gi_ppp${win_num}_current_row \
      gb_ppp${win_num}_modified
   set canvas .ppp_window_${win_num}.chat_fr.c

   # Locals
   set ROW_SIZE 40

   # Work out new height for canvas
   incr gi_ppp${win_num}_number_rows
   set rows [subst $[subst gi_ppp${win_num}_number_rows]]
   set current [subst $[subst gi_ppp${win_num}_current_row]]
   set offset [expr ( ( $rows - 1 ) * $ROW_SIZE )]

   # Create an option menu plus entry field
   global g_ppp${win_num}_option_$rows
   tk_optionMenu $canvas.o$rows g_ppp${win_num}_option_$rows \
      "SEND" "EXPECT" "ABORT" "TIMEOUT" "FILE"
   $canvas.o$rows configure -width 10
   $canvas create window 5 $offset -window $canvas.o$rows -anchor nw
   bind $canvas.o$rows <ButtonPress> "set \
      gb_ppp${win_num}_modified 1 ; \
      set gi_ppp${win_num}_current_row $rows"
   entry $canvas.e$rows -width 20
   $canvas create window 205 $offset -window $canvas.e$rows -anchor nw
   bind $canvas.e$rows <KeyPress> "set \
      gb_ppp${win_num}_modified 1"
   bind $canvas.e$rows <FocusIn> "set \
      gi_ppp${win_num}_current_row $rows"

   # Configure the scroll region for the canvas
   incr offset $ROW_SIZE
   $canvas configure -scrollregion "0 0 400 $offset"

   # Scroll to the new item, if placed at the end
   if ![string compare $item "END"] {
      $canvas yview moveto 1
   }

   # If number is less that the current total, user
   # has requested an insert.  Move all items below
   # number down one.
   if {[string compare $item "END"] != 0 && $rows > 1} {

      set last [expr $rows -1]
      for {set count $last} {$count >= $current} {incr count -1} \
      {
         # Copy the text from the current item to the next item
         set next [expr $count + 1]
   
         # The option menu
         global g_ppp${win_num}_option_$next \
            g_ppp${win_num}_option_$count
         set g_ppp${win_num}_option_$next [subst $[subst \
            g_ppp${win_num}_option_$count]]

         # The type
         set text [$canvas.e$count get]
         $canvas.e$next delete 0 end
         $canvas.e$next insert end $text
      }

      # Blank the new entry
      set g_ppp${win_num}_option_$current "SEND"
      $canvas.e$current delete 0 end
   }
   # Set the current row to the inserted row
   set gi_ppp${win_num}_current_row [expr $current + 1]
}

##################################################################
# Procedure for closing a PPP edit session
proc PPP_Delete_Item { win_num } {
   Debug "PPP_Delete_Item"

   # Globals
   global gi_ppp${win_num}_number_rows
   global gi_ppp${win_num}_current_row
   set canvas .ppp_window_${win_num}.chat_fr.c

   # Locals
   set ROW_SIZE 40

   # Work out new height for canvas
   set rows [subst $[subst gi_ppp${win_num}_number_rows]]
   set current [subst $[subst gi_ppp${win_num}_current_row]]

   set last [expr $rows -1]
   for {set count $current} {$count <= $last} {incr count} \
   {
      # Copy the text from the current item to the next item
      set next [expr $count + 1]
   
      # The option menu
      global g_ppp${win_num}_option_$next \
         g_ppp${win_num}_option_$count
      set g_ppp${win_num}_option_$count [subst $[subst \
         g_ppp${win_num}_option_$next]]

      # The type
      set text [$canvas.e$next get]
      $canvas.e$count delete 0 end
      $canvas.e$count insert end $text
   }
   # Now destroy the last item
   destroy $canvas.e$rows
   destroy $canvas.o$rows

   # Configure the scroll region for the canvas
   incr gi_ppp${win_num}_number_rows -1
   set rows [subst $[subst gi_ppp${win_num}_number_rows]]
   set offset [expr ( ( $rows - 1 ) * $ROW_SIZE )]
   incr offset $ROW_SIZE
   $canvas configure -scrollregion "0 0 400 $offset"
}

##################################################################
# Procedure for closing a PPP edit session
proc PPP_Close_Window { parent win_num return_value} {
   Debug "PPP_Close_Window $win_num"

   # Globals
   global gl_ppp_windows

   # Destroy the window and remove the number from
   # the list of active windows.
   # Info_Dialog . $gl_ppp_windows
   set list_item [lsearch $gl_ppp_windows $win_num]
   set gl_ppp_windows [lreplace $gl_ppp_windows \
      $list_item $list_item]

   # Destroy any global variables created by this window
   set search "*_ppp${win_num}_*"
   set globals_list [info globals $search]
   foreach global $globals_list {
      global $global
      unset $global
   }

   # Check to see whether the list window exists and
   # if the user has requested to return the value to 
   # the list window
   if [winfo exists $parent] {

      # Trigger a re-retrieve of the list window
      Display_Interactive_Scripts $parent.list_fr.list

      if {$return_value == 1} {

         # Set the current item into the window
         $parent.sel_fr.entry configure -state normal
         $parent.sel_fr.entry delete 0 end
         $parent.sel_fr.entry insert end [ \
            .ppp_window_${win_num}.sel_fr.entry get]
         $parent.sel_fr.entry configure -state disabled
      }
   }

   # Destroy the window
   destroy .ppp_window_$win_num
}

##################################################################
# Procedure for finding current PPP edit sessions
proc PPP_Find_Script { name } {
   Debug "PPP_Find_Script"

   # Globals
   global gl_ppp_windows

   # This script searches through the current windows to find
   # one which is editing the currently selected script $name.
   # If the window is found it's name is returned, otherwise
   # zero is returned and the window is created.
   foreach count $gl_ppp_windows \
   {
      # Check window contents
      # Info_Dialog . $count
      if {[winfo exists .ppp_window_$count] == 1 && [string compare \
         [.ppp_window_$count.sel_fr.entry get] $name] == 0} {
         return $count
      }
   }
   # No match found, return 0
   return 0
}

##################################################################
# Procedure for displaying a given script
proc PPP_Display_Script { win_num name } {
   Debug "PPP_Display_Script"

   # Globals
   global g_chat_command g_pppd_command

   # The script is given by the agrument name.
   # It is decomposed into parts and displayed 
   # using the window win_num.
   
   # Global variable give by 'name' is gs_ppp_script_$name.
   # The internal formats separates options using the ^
   # character.
   global gs_ppp_script_$name
   if {[info exists gs_ppp_script_$name] == 0} {
      # Change the status to NEW and pop-up a dialogue
      global gs_ppp${win_num}_edit_type
      set gs_ppp${win_num}_edit_type NEW
      Info_Dialog . "Script not found : Creating new script"
      return
   }
   set script [subst $[subst gs_ppp_script_$name]]
   #Info_Dialog . "gs_ppp_script_$name : '$script'"

   # Load the interface definition (sl0, ppp0 etc)
   global gs_ppp_interface_$name gs_ppp${win_num}_interface \
      gs_ppp_precmd_${name} g_ppp${win_num}_precmd \
      gs_ppp_postcmd_${name} g_ppp${win_num}_postcmd \
      gs_ppp_cond_${name} g_ppp${win_num}_cond 
   if [catch {set gs_ppp${win_num}_interface [subst \$gs_ppp_interface_$name]}] {
      set gs_ppp${win_num}_interface "-none-"
   }
   if [catch {set g_ppp${win_num}_precmd [subst \$gs_ppp_precmd_$name]}] {
      set g_ppp${win_num}_precmd "-none-"
   }
   if [catch {set g_ppp${win_num}_postcmd [subst \$gs_ppp_postcmd_$name]}] {
      set g_ppp${win_num}_postcmd "-none-"
   }
   if [catch {set g_ppp${win_num}_cond [subst \$gs_cmd_cond_$name]}] {
     set g_ppp${win_num}_cond 1
   }

   # OPTIONS
   # Search for the options
   # Now create the first part of the interface, the PPPD options
   # Supported options are:
   # 
   # crtscts - Hardware flow control (if off then xonxoff is used)
   # defaultroute - Add PPP route to system routine tables
   # lock - Use a uucp like lock on the modem
   # passive - Enables passive option (see man page)
   # debug - Enables debugging
   # noipdefault - Not sure what this does!
   #
   # And the configurable items:
   # 
   # Tty-name - e.g /dev/ppp0
   # Speed - port speed
   # mru - Set maximum received unit
   # mtu - Set maximum transfer unit
   # netmask - Set net mask.
   # Local/Remote IP addresses
   # File - read options from file
   # Other command line options
   # asyncmap - how to escape control characters
   # 
   
   # Break the command line up into a list.  Note
   # that the separator used is the ^ symbol.
   # regsub -all {[\ ]+} $script { } script
   set script [split $script ^]

   # Now split off the connect command
   set list_item [lsearch -exact $script "connect"]
   set connect [lrange $script [expr $list_item + 1] end]
   set script [lrange $script 0 [expr $list_item -1]]
   # Info_Dialog . $script
   # Info_Dialog . $connect

   # OPTIONS
   # Get the canvas and the options, and try to match
   # the options from the list
   set canvas .ppp_window_${win_num}.flag_fr.c
   set list [list crtscts defaultroute lock passive \
      noipdefault debug]
   foreach item $list {
      set list_item [lsearch -exact $script $item]
      set flag $canvas.$item
      if {$list_item != -1} {
         $flag select
         set script [lreplace $script $list_item $list_item]
      } else {$flag deselect}
   }

   # Get the canvas and the options, and try to match
   # the options from the list
   set canvas .ppp_window_${win_num}.options_fr.c

   # Try to match the device
   set list_item [lsearch $script "/dev/*"]
   set option $canvas.tty
   if {$list_item != -1} {
      $option delete 0 end
      $option insert end [lindex $script $list_item]
      set script [lreplace $script $list_item $list_item]
   } else {$option delete 0 end}

   set list [list mtu mru netmask file asyncmap]
   foreach item $list {
      # Try to match the mtu
      set list_item [lsearch -exact $script $item]
      set option $canvas.$item
      if {$list_item != -1} {
         $option delete 0 end
         $option insert end [lindex $script [expr $list_item + 1]]
         set script [lreplace $script $list_item \
            [expr $list_item + 1]]
      } else {$option delete 0 end}
   }

   # Delete the pppd and modem strings (these are 
   # compulsory)
   set list_item [lsearch -exact $script $g_pppd_command]
   set script [lreplace $script $list_item $list_item]
   set list_item [lsearch -exact $script "modem"]
   set script [lreplace $script $list_item $list_item]

   # Get the local and remote IP addresses
   set list_item [lsearch $script "*:*"]
   if {$list_item != -1} {
      set ip [lindex $script $list_item]
      set script [lreplace $script $list_item $list_item]

      # Split up into local and remote
      set ip [split $ip :]
      # Print local IP
      set option $canvas.localip
      $option delete 0 end
      $option insert end [lindex $ip 0]
      # Print remote IP
      set option $canvas.remoteip
      $option delete 0 end
      $option insert end [lindex $ip 1]
   }

   # Whatever is left in $script contains the connection
   # speed plus other flags not catered for by the MMI
   # Speed is found by looking for the first occurrence
   # of two zeroes.  Anyone have a better idea?
   set list_item [lsearch $script "*00"]
   set option $canvas.speed
   if {$list_item != -1} {
      $option delete 0 end
      $option insert end [lindex $script $list_item]
      set script [lreplace $script $list_item $list_item]
   } else {$option delete 0 end}

   # And anything else goes in the other options entry
   set script [join $script]
   set option $canvas.flags
   $option delete 0 end
   $option insert end $script

   # Now the options and flags have been set, we move
   # on to parsing of the connect script, currently
   # stored in the $connect variable.  Strings built 
   # using TkNet will have chat -v at the front.  This
   # is removed.
   set list [list $g_chat_command "-v"]
   foreach item $list {
      set list_item [lsearch $connect $item]
      if {$list_item != -1} {
         set connect [lreplace $connect $list_item $list_item]
      }
   }
   # And search for and delete the surrounding "'s
   set list_item [lsearch $connect "@"]
   if {$list_item != -1} {
      set connect [lreplace $connect $list_item $list_item]
   }
   set list_item [lsearch $connect "@"]
   if {$list_item != -1} {
      set connect [lreplace $connect $list_item $list_item]
   }
   #Info_Dialog . $connect

   # The remainder of the list comprises EXPECT/SEND sequences
   # which are parsed.  Special sequences are TIMEOUT and
   # ABORT, other pairs are treated as EXPECT/SEND pairs.
   set length [expr [llength $connect] -1]
   set canvas .ppp_window_${win_num}.chat_fr.c

   # Loop round each pair in the list.  
   # Info_Dialog . $connect
   global gi_ppp${win_num}_number_rows
   for {set count 0} {$count < $length} {incr count 2} \
   {
      # Get the expect/send pair
      set expect [lindex $connect $count]
      set send [lindex $connect [expr $count + 1]]
      # Clean them up by removing all 's
      regsub -all {'} $expect {} expect
      regsub -all {'} $send {} send
      # Info_Dialog . "Expect : '$expect', Send : '$send'"

      # Create a new item
      PPP_New_Item $win_num END

      # The option menu
      set rows [subst $[subst gi_ppp${win_num}_number_rows]]
      global g_ppp${win_num}_option_$rows 

      # Switch on the expect statement
      switch -- $expect {
         ABORT {
            # Create an abort pair with value $send
            set g_ppp${win_num}_option_$rows "ABORT"
            $canvas.e$rows delete 0 end
            $canvas.e$rows insert end $send
         }
         TIMEOUT {
            # Create an timeout pair with value $send
            set g_ppp${win_num}_option_$rows "TIMEOUT"
            $canvas.e$rows delete 0 end
            $canvas.e$rows insert end $send
         }
         "-f" {
            # Create a file pair
            set g_ppp${win_num}_option_$rows "FILE"
            $canvas.e$rows delete 0 end
            $canvas.e$rows insert end $send
         }
         default {
            # Create an expect pair with value $expect
            set g_ppp${win_num}_option_$rows "EXPECT"
            $canvas.e$rows delete 0 end
            $canvas.e$rows insert end $expect
            # Create a send pair with value $send
            PPP_New_Item $win_num END
            set rows [subst $[subst gi_ppp${win_num}_number_rows]]
            set g_ppp${win_num}_option_$rows "SEND"
            $canvas.e$rows delete 0 end
            $canvas.e$rows insert end $send
         }
      }
   }
}

##################################################################
# Procedure for saving a displayed/edited script
proc PPP_Save_Script { win_num name } {
   Debug "PPP_Save_Script"

   # Globals
   global TKNET_USER TKNET_GLOBAL_NETWORK_FILE \
      TKNET_LOCAL_NETWORK_FILE g_chat_command \
      g_pppd_command

   # Firstly the chat part of the configuration
   # is checked for syntactic correctness.
   # The return value is the chat string - if no
   # chat string is returned a syntax error has 
   # occurred and the save is aborted.
   set chat [PPP_Check_Script_Syntax $win_num $name]
   if {$chat == ""} {
      # Error dialogue has already been displayed
      return
   }

   # Now the full pppd string is built up, using ^'s
   # in place of spaces.  Start with the basics...
   set pppd "$g_pppd_command^modem^"

   # Now for each flag - only add the option if set
   set canvas .ppp_window_${win_num}.flag_fr.c
   set list [list crtscts defaultroute lock passive \
      noipdefault debug]
   foreach item $list {
      global g_ppp${win_num}_flag_$item
      if {[subst $[subst g_ppp${win_num}_flag_$item]] == 1} {
         # Flag is selected
         append pppd "$item^"
      }
   }
   
   # OPTIONS
   # Now for each option - only add the option if set
   set canvas .ppp_window_${win_num}.options_fr.c
   # tty speed mtu mru flags netmask file localip remoteip
   set text [$canvas.tty get]
   if {$text != ""} {
      # Check value.  Tty must begin /dev/.  If it
      # doesn't, prepend /dev/
      
      # Add to pppd string
      append pppd "$text^"
   }
   set text [$canvas.speed get]
   if {$text != ""} {
      # Check value.
      
      # Add to pppd string
      append pppd "$text^"
   }
   set text [$canvas.mtu get]
   if {$text != ""} {
      # Check value.
      
      # Add to pppd string
      append pppd "mtu^$text^"
   }
   set text [$canvas.file get]
   if {$text != ""} {
      # Check value.  If a file is specified warn
      # the user that this will override the flags
      # set in the MMI.
      
      # Add to pppd string
      append pppd "file^$text^"
   }
   set text [$canvas.mru get]
   if {$text != ""} {
      # Check value.
      
      # Add to pppd string
      append pppd "mru^$text^"
   }
   set text [$canvas.asyncmap get]
   if {$text != ""} {
      # Check value.
      
      # Add to pppd string
      append pppd "asyncmap^$text^"
   }
   set text [$canvas.flags get]
   if {$text != ""} {
      # Assume user defined flags ok
      # Add to pppd string
      append pppd "$text^"
   }
   set text [$canvas.netmask get]
   if {$text != ""} {
      # Check value.
      
      # Add to pppd string
      append pppd "netmask^$text^"
   }
   set text [$canvas.localip get]
   set text2 [$canvas.remoteip get]
   if {$text != "" || $text2 != ""} {
      # Check values.
      
      # Add to pppd string
      if {$text != ""} {append pppd "$text:"}
      if {$text2 != "" && $text == ""} {
         append pppd ":$text2^"
      } elseif {$text2 != ""} {
         append pppd "$text2^"
      }
   }

   # Now add the chat command to the end of the pppd
   # string.  There is a space at the end of the chat
   # command and the pppd command already.
   append pppd "connect^@^$g_chat_command^-v^$chat@"
   # Info_Dialog . $pppd

   # Now save the name and the string to the setup file
   # Get the final name from the entry field
   set window .ppp_window_${win_num}
   set frame .ppp_window_${win_num}.sel_fr

   # Use thes globals for the save
   global gs_ppp${win_num}_interface gs_ppp${win_num}_edit_type \
      g_ppp${win_num}_precmd g_ppp${win_num}_postcmd \
      g_ppp${win_num}_cond 

   # Get the new script name from the edit field (if new)
   set name [$frame.entry get]
   regsub -all {[\ ]+} $name {_} name

   # Save as these names
   global gs_ppp_name_${name}
   global gs_ppp_interface_${name}
   global gs_ppp_script_${name}
   global gs_ppp_precmd_${name}
   global gs_ppp_postcmd_${name}
   global gs_ppp_cond_${name}

   # Set the information
   set gs_ppp_name_$name [$frame.entry get]
   set gs_ppp_interface_$name [subst $[subst gs_ppp${win_num}_interface]]
   set gs_ppp_script_$name $pppd
   set gs_ppp_precmd_$name [subst \$g_ppp${win_num}_precmd]
   set gs_ppp_postcmd_$name [subst \$g_ppp${win_num}_postcmd]
   set gs_ppp_cond_$name [subst \$g_ppp${win_num}_cond]

   # Get the useful information before writing (simplifies
   # the syntax!)
   set title_variable [subst gs_ppp_name_$name]
   set title [subst $[subst gs_ppp_name_$name]]
   set interface_variable [subst gs_ppp_interface_$name]
   set interface [subst $[subst gs_ppp_interface_$name]]
   set script_variable [subst gs_ppp_script_$name]
   set script [subst $[subst gs_ppp_script_$name]]
   set precmd_variable gs_ppp_precmd_$name
   set precmd [subst \$gs_ppp_precmd_$name]
   set postcmd_variable gs_ppp_postcmd_$name
   set postcmd [subst \$gs_ppp_postcmd_$name]
   set cond_variable gs_ppp_cond_$name
   set cond [subst \$gs_ppp_cond_$name]

   ###############################################################
   # Now write it to file.  If the edit type is NEW, append to the
   # end of the file.  Otherwise, find the current entry in the
   # file and replace it.
   ###############################################################

   # File information:  The file, tknet.network or ~/.tknet_network
   # Contains information about *all* scripts, not just PPP scripts.
   # The format is as follows.  On a line, ### $gs_ppp_name_name ##.  
   # The following three lines contain the script information, up 
   # to the next record, which begins in the same manner.

   # If root save to global settings, else save to users
   # local ~/.tknet_network file
   if {$TKNET_USER == "root" && [Question_Dialog \
      . "Where do you want to save the network script to?" \
      "Local" "Global"]} {
         set setup_file $TKNET_GLOBAL_NETWORK_FILE
   } else {
      set setup_file $TKNET_LOCAL_NETWORK_FILE
   }

   # Check for existence of the file.
   if {[file exists $setup_file] != 1} {
      # Create New file
      set file_text "#########################################################
### This is the TkNet Network Scripts file.
### This file contains details of all the scripts defined
### using the script editors.
###
### PLEASE DO NOT EDIT BY HAND UNLESS YOU ARE QUITE SURE
### WHAT YOU ARE DOING!
#########################################################\n"
      set fd [open $setup_file w]
      puts -nonewline $fd $file_text
      flush $fd
      close $fd
   }

   # Write the script to the end of the file
   set fd [open $setup_file r]
   set file_text [read $fd]
   close $fd

   # If editing, ensure that the old version is deleted
   if {[subst $[subst gs_ppp${win_num}_edit_type]] == "EDIT"} {
      set start [string first "### $title_variable ##" $file_text]
      if {$start != -1} {
         set rest_file [string range $file_text [expr \
            $start + 4] end]
         set end [string first "###" $rest_file]
         set file_text [string range $file_text \
            0 [expr $start -1]]
         if {$end != -1} {
            set rest_file [string range $rest_file $end end]
            append file_text $rest_file
         }
      }
   }

   # Check to see that a script with the given name
   # does not already exist.  If it does, ask the user
   # if he wishes to replace the existing entry.  If
   # the user says no, then the function returns
   # immediately (with a dialog).
   if {[string first "### $title_variable ##" $file_text] != -1} {
      # Match found on title - ask user
      if {[Question_Dialog . \
         "Network Script already exists.  Overwrite?" \
         "Yes" "No"]} {
         Info_Dialog . "Save operation cancelled."
         return
      } else {
         # User has asked to overwrite the saved script
         set start [string first "### $title_variable ##" $file_text]
         if {$start != -1} {
            set rest_file [string range $file_text [expr \
               $start + 4] end]
            set end [string first "###" $rest_file]
            set file_text [string range $file_text \
               0 [expr $start -1]]
            if {$end != -1} {
               set rest_file [string range $rest_file $end end]
               append file_text $rest_file
            }
         }
      }
   }

   # Create the new script details
   append file_text "\n"
   append file_text "### $title_variable ##
global $title_variable $interface_variable $script_variable \\
   $precmd_variable $postcmd_variable $cond_variable
set $title_variable \"$title\"
set $interface_variable \"$interface\"
set $script_variable \"$script\"
set $precmd_variable \"$precmd\"
set $postcmd_variable \"$postcmd\"
set $cond_variable \"$cond\"\n"

   # And then overwrite the existing file
   set fd [open $setup_file w 755]
   puts -nonewline $fd $file_text
   flush $fd
   close $fd
   Info_Dialog $window "Network script '$name' saved"
}

##################################################################
# Procedure for saving a displayed/edited script
proc PPP_Check_Script_Syntax { win_num name } {
   Debug "PPP_Check_Script_Syntax"

   # This function attempts to analyse what the 
   # user has chosen for his chat script.  It uses
   # very basic rules to determine whether the 
   # chat script will work, but does not guarantee
   # to be either correct or comprehensive in its
   # checking!  
   # 
   # As it does this checking it also builds up the
   # chat script, which is returned.  If an error
   # is found a dialogue is displayed and "" is 
   # returned.
   set chat ""

   global gi_ppp${win_num}_number_rows
   set rows [subst $[subst gi_ppp${win_num}_number_rows]]
   set window .ppp_window_${win_num}
   set canvas .ppp_window_${win_num}.chat_fr.c

   if {$rows == 0} {
      Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
No rows specified for the chat script!"
      return ""
   }

   # Set the boolean statii
   set b_send 0
   set b_expect 0
   
   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Get the next row
      global g_ppp${win_num}_option_$count
      set type [subst $[subst g_ppp${win_num}_option_$count]]
      set text [$canvas.e$count get]

      # The rules of the game:
      # (1) If FILE is used, there must be no other entries
      # (2) EXPECT and SENDs must go in pairs
      # (3) Either the EXPECT or the SEND argument may be NULL,
      #     but not both!
      # (4) ABORTs must go before the SEND/EXPECT sequence.
      #     Produces a warning only.  Must not be null.

      switch $type {
         FILE {
            # Ok, check that no other elements are present
            # and that the specified file exists.
            if {$rows > 1} {
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
If you use the FILE option you may have at\n\
most one row.  Delete the other rows and add\n\
the functionality into the file if desired."
               return ""
            } elseif {$text == "" || [file exists $text] == 0} {
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
The file you have chosen is not a valid file.\n\
Please check that you have put in the full\n\
pathname and that you have permission to\n\
read the file."
               return ""
            } else {
               # Add to chat command
               append chat "-f^$text^"
               return $chat
            }
         }
         ABORT {
            # Check that the send/expect sequence has not
            # yet started.  If it has, issue a warning but
            # continue anyway.  Also check that a string
            # has been specified in the text
            if {$b_send != 0 || $b_expect != 0} {
               Info_Dialog $window "WARNING IN CHAT SCRIPT:\n\n\
You have an ABORT statement after an EXPECT/\n\
SEND sequence.  Usually all ABORT statements\n\
come before the start of the EXPECT/SEND sequence.\n\
The save will continue, but you may wish to \n\
look at the script again."
            }
            if {$text == ""} {
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
An ABORT statement needs an argument which\n\
is usually a return from your modem.  Examples\n\
are BUSY, 'NO CARRIER', ERROR, etc."
               return ""
            } else {
               # Append to the chat command
               append chat "ABORT^'$text'^"
            }
         }
         TIMEOUT {
            # The timeout is in seconds.  Check that the 
            # value in the field is indeed a number!
            # If no number is found return an error.
            set num [scan $text "%d" text]
            if {$num == -1 || $num == 0} {
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
TIMEOUT must be an integer number of seconds.\n\
Values such as 20 and 60 are reasonable."
               return ""
            } else {
               # Append to the chat command
               append chat "TIMEOUT^$text^"
            }
         }
         EXPECT {
            # Expect commands must preceed a send
            # command, and these must come in pairs.
            incr b_expect
            if {$b_expect <= $b_send || $b_expect > \
               [expr $b_send + 1]} {
               # Send before expect error
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
You have two EXPECTs in a row, or a SEND\n\
followed by an EXPECT.  This is not allowed.\n\
You must have an EXPECT before a SEND."
               return ""
            } else {
               # Append to the chat command.  Note that
               # a null value is allowed here.
               append chat "'$text'^"
            }
         }
         SEND {
            # Send commands must come after an expect
            # command, and these must come in pairs.
            incr b_send
            if {$last_type != "EXPECT"} {
               # Send before expect error
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
A SEND command must be preceded by an EXPECT command."
               return ""
            } elseif {$text == "" && $last_text == ""} {
               # Blank EXPECT/SEND pair
               Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
A single blank SEND or EXPECT command is allowed,\n\
but not two together.  Please correct the error."
               return ""
            } else {
               # Append to the chat command.  Note that
               # a null value is allowed here.
               append chat "'$text'^"
            }
         }
         default {
            Info_Dialog $window "ERROR IN CHAT SCRIPT:\n\n\
Unrecognised option type for chat script.\n\
Please contact the system administrator\n\
or the maintainer of this software for a\n\
bug fix!"
            return ""
         }
      }
      # Save the last type and text
      set last_type $type
      set last_text $text
   }

   # Return the built string
   return $chat
}
