###############################################################
# TkNet - Network Statistics Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

# Note that for best results ip_accounting should be turned 
# on in the kernel.  To do this the kernel must be recompiled
# with the switch enabled.  Once this has been done, your
# rc.local file should be modified to add accounting rules
# for your interfaces.  Thus, as my local IP address is
# 123.456.78.90 (not really, you understand), I add the rules:
#
# /sbin/ipfwadm -A -a -I 123.456.78.90 -S 123.456.78.90 -D 0/0
# /sbin/ipfwadm -A -a -I 123.456.78.90 -S 0/0 -D 123.456.78.90
#
# The accounting information then resides in /proc/net/ip_acct


###############################################################
# Globals for this module

# Flags
set gt_last_stat_time   0
set g_previous_stat     0
set g_idle_time         0
set g_throughput        0
# This flag is set in the main module on startup
set g_have_ip_acct      0

# Network information
set  g_hostname			 "Unknown"
set  g_local_ip_addr 	 "Unknown"
set  g_remote_ip_addr	 "Unknown"
set  g_remote_mask		 "Unknown"
set  g_local_mask 		 "Unknown"
set  g_remote_mtu 		 "Unknown"

###############################################################
# Show the network settings panel
proc Show_Network { } {
	Debug "Show_Network"

	# Globals
	global FONT_NORMAL FIXED_FONT_SMALL RIDGE_BORDER \
		DEFAULT_PADDING g_log_messages TKNET_NET_GEOMETRY \
		FONT_BOLD g_gethostname g_network_interface \
      g_hostname g_local_ip_addr g_remote_ip_addr \
      g_remote_mask g_local_mask g_remote_mtu

	set  g_hostname			 "Unknown"
	set  g_local_ip_addr 	 "Unknown"
	set  g_remote_ip_addr	 "Unknown"
   set  g_remote_mask       "Unknown"
   set  g_local_mask        "Unknown"
   set  g_remote_mtu        "Unknown"

   set offset 5
   set offset_incr 30

   # Popup a selection window
   set window .net_window
   if [winfo exists $window] {
       # Pop it up!
       wm deiconify $window
       raise $window
       update
       return
   }

   # Otherwise create the window
   toplevel $window
   wm title $window "Network Information"
   wm transient $window .
   wm geometry $window $TKNET_NET_GEOMETRY
   wm protocol $window WM_DELETE_WINDOW "destroy $window"
   set frame [frame $window.fr]
   pack $frame

   ############################################################
   # Create a frame for the options
   set frame [canvas $frame.type -borderwidth $RIDGE_BORDER -relief \
      groove -width 350]

   label $frame.label -text "Connection Information"
   $frame create window 15 $offset -window $frame.label -anchor nw
   incr offset $offset_incr

   # Hostname
   catch {eval exec $g_gethostname}  g_hostname
   label $frame.l$offset -text "Hostname"
   entry $frame.e$offset -width 20 -textvariable  g_hostname -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Read the routing table for IP Address data
   set fp [open /proc/net/route r]
   set route_info [read $fp]
   close $fp
   # Break into lines
   set route_info [split $route_info '\n']
   # ... and discard the first line
   set route_info [lreplace $route_info 0 0]
   # Find the pertinent information
   foreach route_line $route_info {
      # File is split by tabs (I hope!)
      set route_line [split $route_line '\t']
      # We are interested in lo and g_network_interface
      if {[lindex $route_line 0] == "lo"} {
         # Found local - check for address
         set  g_local_ip_addr [Hex_To_IP [lindex $route_line 1]]
         #set local_gateway [Hex_To_IP [lindex $route_line 2]]
         set  g_local_mask [Hex_To_IP [lindex $route_line 7]]
         #set local_mtu [lindex $route_line 8]
      }
      if {[lindex $route_line 0] == $g_network_interface} {
         # Found remote
         set dest [Hex_To_IP [lindex $route_line 1]]
         if {$dest != "0.0.0.0"} {set  g_remote_ip_addr $dest}
         set gateway [Hex_To_IP [lindex $route_line 2]]
         if {$gateway != "0.0.0.0"} {set remote_gateway $gateway}
         set  g_remote_mask [Hex_To_IP [lindex $route_line 7]]
         set  g_remote_mtu [lindex $route_line 8]
      }
   }
   #Info_Dialog . "$g_local_ip_addr, $g_local_mask"
   #Info_Dialog . "$g_remote_ip_addr, $remote_gateway, $g_remote_mask, $g_remote_mtu"

   # Interface
   label $frame.l$offset -text "Current Interface"
   entry $frame.e$offset -width 20 -textvariable g_network_interface -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Local IP Address
   label $frame.l$offset -text "Local Address"
   entry $frame.e$offset -width 20 -textvariable g_local_ip_addr -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Local Mask
   label $frame.l$offset -text "Local Mask"
   entry $frame.e$offset -width 20 -textvariable g_local_mask -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Remote IP Address
   label $frame.l$offset -text "Remote Address"
   entry $frame.e$offset -width 20 -textvariable  g_remote_ip_addr -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Remote Mask
   label $frame.l$offset -text "Remote Mask"
   entry $frame.e$offset -width 20 -textvariable  g_remote_mask -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # MTU
   label $frame.l$offset -text "Maximum Transfer Unit"
   entry $frame.e$offset -width 20 -textvariable  g_remote_mtu -state disabled
   $frame create window 15 $offset -window $frame.l$offset -anchor nw
   $frame create window 170 $offset -window $frame.e$offset -anchor nw
   incr offset $offset_incr

   # Pack the frames
   $frame configure -height $offset
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -fill y -side left

   # Now for a close button
   set frame [frame $window.button_fr]
   pack $frame -side bottom -fill x
   button $frame.close -text Close -command "destroy $window"
   button $frame.help -text Help -command "MH_MiniHelp NetworkWindow"
   pack $frame.close $frame.help -side right -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING
}

###############################################################
# The procedure for updating the throughput statistics
proc Update_Statistics {} {
   Debug "Update_Statistics"

   # Globals
   global gb_connection_down g_link_stats \
      gt_last_stat_time g_previous_stat \
      g_throughput g_idle_disconnect_time \
      RED g_network_interface g_idle_time \
      g_have_ip_acct

   # Only update statistics if connected
   Set_Connection_Flag
   if {$gb_connection_down == 0} {

      # Two times are used.  A previous time indication
      # stored in gt_last_stat_time along with the 
      # result of the ifconfig command, g_previous_stat.
      # These are compared against the new values which 
      # are stored in t_stat_time and stat_value

      # Retrieve current throughput data
      # Submitted by George M. Sipe (gsipe@mindspring.com)
      if {$g_have_ip_acct == 0} {
         # No IP Accounting
         set fd [open /proc/net/dev]
         regsub -all {[\ ]+} [read $fd] { } proc_net_dev
         close $fd
         set proc_net_dev [split $proc_net_dev]
         set entry [lsearch -exact $proc_net_dev \
            "${g_network_interface}:"]
         set xmit [lindex $proc_net_dev [expr $entry + 6]]
         set rec [lindex $proc_net_dev [expr $entry + 1]]
         if [catch {set stat_value [expr $xmit + $rec]}] {
            Set_Message "Error reading link statistics" $RED
            set g_link_stats 60
            set stat_value 0
         }
      } else {
         # IP Accounting enabled
         set fd [open /proc/net/ip_acct]
         regsub -all {[\ ]+} [read $fd] { } proc_net_dev
         close $fd
         set proc_net_dev [split $proc_net_dev]
         set xmit [lindex $proc_net_dev 9]
         set rec [lindex $proc_net_dev 26]
         if [catch {set stat_value [expr $xmit + $rec]}] {
            set stat_value 0
            Set_Message "Error reading link statistics" $RED
            set g_link_stats 60
         }
      }

      # Get current time stamp
      set t_stat_time [Get_Current_Time]

      # If this is the first time round, store
      # the time and value and return without 
      # calculating.
      if {$gt_last_stat_time == 0} {
         set g_previous_stat $stat_value
         set gt_last_stat_time $t_stat_time
         after 2000 Update_Statistics
         return
      }

      # Perform the throughput calculation - note the
      # introduction of a floating point number to 
      # improve the accuracy of the result
      set difftime [AddSub_Time $gt_last_stat_time \
         $t_stat_time -1 1]
      # Check for divide by zero
      if {$difftime == 0} {
         set g_previous_stat $stat_value
         set gt_last_stat_time $t_stat_time
         after 5000 Update_Statistics
         return
      }
      if [catch {set throughput [expr ( $stat_value - \
         $g_previous_stat ) / ( $difftime + 0.0 )]}] {
         set throughput 0
      }
      Graph_Create_Line $g_throughput $t_stat_time $throughput

      # If idle checking is enabled, perform that
      # here (as we've already done the work)
      if {$g_idle_disconnect_time > 0} {
         if {$throughput == 0} {
            incr g_idle_time $difftime
            if {[expr $g_idle_time / 60.0] > $g_idle_disconnect_time} {
               Net_Disconnect
               after 5000 Set_Message "Link idle - diconnected" $RED
            }
         } else {
            set g_idle_time 0
         }
      }

      # Copy the new stat time and value to the
      # old variables
      set gt_last_stat_time $t_stat_time
      set g_previous_stat $stat_value

   }

   # Rerun after N secs if time is > 0
   if {$g_link_stats > 0} {
      set run_after [ expr $g_link_stats * 1000 ]
      after $run_after Update_Statistics
   } else {
      # Run after 1 minute for the idle check
      after 60000 Update_Statistics
   }
}


