###############################################################
# TkNet - Launcher Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Constants
set LAUNCHER_COLOUR     Grey60
set LAUNCHER_FONT       -adobe-helvetica-medium-r-*-*-*-*-*-*-*-*-*-*
set LAUNCHER_ITEM_LENGTH 20

# Flags
set g_launcher_changed  0
set g_current_launcher  0
set g_launcher_number   0

# Geometry
set TKNET_LAUNCHER_SETUP_GEOMETRY "+408+51"
set TKNET_LAUNCHER_GEOMETRY "+100+100"

# Files
set TKNET_GLOBAL_LAUNCHER_FILE "$GLOBAL_INCLUDE/tknet.launcher"
set TKNET_LOCAL_LAUNCHER_FILE "$TKNET_USER_HOME/.tknet_launcher"


###############################################################
# The procedure to read the launcher setup files
# Argument must be 0=Both 1=Global or 2=Local 
proc Source_Launcher_Setup { file } {
   Debug "Source_Launcher_Setup"

   # Globals
   global TKNET_GLOBAL_LAUNCHER_FILE \
      TKNET_LOCAL_LAUNCHER_FILE

   # Read the global setup file
   if {$file != 2 && [file exists $TKNET_GLOBAL_LAUNCHER_FILE]} {
      source $TKNET_GLOBAL_LAUNCHER_FILE
   }
   
   # Read the user's own config file if it exists
   if {$file != 1 && [file exists $TKNET_LOCAL_LAUNCHER_FILE]} {
      # Override with users own
      source $TKNET_LOCAL_LAUNCHER_FILE
   }

   # Note that if neither exist the defaults are
   # taken from the internal settings corresponding
   # to my home settings.
}

###############################################################
# Show the setup panel
proc Show_Launcher_Setup { } {
   Debug "Show_Launcher_Setup"

   # Globals
   global FONT_NORMAL FONT_BOLD DEFAULT_PADDING RIDGE_BORDER \
      g_launcher_number g_launcher_changed \
      TKNET_LAUNCHER_SETUP_GEOMETRY

   # Create the setup toplevel as a transient from the main window
   set window .launcher_setup
   if [winfo exists $window] {destroy $window}
   toplevel $window
   wm title $window "TkNet Launcher Setup"
   wm transient $window .
   wm geometry $window $TKNET_LAUNCHER_SETUP_GEOMETRY
   wm protocol $window WM_DELETE_WINDOW Close_Launcher_Setup

   ###########################################################################
   # Create Menu Bar
   set menu [frame $window.mbar -relief raised -bd 2]
   pack $menu -side top -fill x

   menubutton $menu.file -text File -underline 0 -menu $menu.file.menu
   menubutton $menu.edit -text Edit -underline 0 -menu $menu.edit.menu
   menubutton $menu.setup -text Setup -underline 0 -menu $menu.setup.menu
   menubutton $menu.help -text Help -underline 0 -menu $menu.help.menu
   pack $menu.file $menu.edit $menu.setup -side left
   pack $menu.help -side right

   menu $menu.file.menu -tearoff 0
      $menu.file.menu add command -label "Save" -command \
         {Save_Launcher_Setup} -underline 0
      $menu.file.menu add command -label \
         "Set Defaults from Global" -command \
         { Source_Launcher_Setup 1 ; Show_Launcher_Setup ; \
         set g_setup_changed 1 }   -underline 4
      $menu.file.menu add separator
      $menu.file.menu add command -label "Close" \
         -command { Close_Launcher_Setup } -underline 0
   menu $menu.edit.menu -tearoff 0
      $menu.edit.menu add command -label "Add to end" \
         -command {New_Launcher_Item $g_launcher_number; \
         incr g_launcher_number; set g_launcher_changed 1} \
         -underline 0
      $menu.edit.menu add command -label \
         "Add before current" \
         -command {New_Launcher_Item $g_current_launcher; \
         incr g_launcher_number; set g_launcher_changed 1} \
         -underline 0
      $menu.edit.menu add separator
      $menu.edit.menu add command -label "Delete current" \
         -command {Delete_Launcher_Item; \
         set g_launcher_changed 1} \
         -underline 0
   menu $menu.setup.menu -tearoff 0
      $menu.setup.menu add command -label \
         "Scripts..." -command {Show_Interactive_Scripts} \
         -underline 0
   menu $menu.help.menu -tearoff 0
      $menu.help.menu add command -label "Show Help..." \
         -underline 0 -command {MH_MiniHelp Launcher}

   tk_menuBar $menu $menu.file $menu.edit $menu.setup $menu.help
   bind $window <Button-2> "Popup_Menu $window $menu.edit.menu"
   bind $window <Button-3> "Popup_Menu $window $menu.edit.menu"

   ###############################################################
   # Create the instructions
   button .launcher_setup.label -text \
      "Enter the launcher items in the fields below.\n\
To obtain help click on this label." \
      -relief flat -command "MH_MiniHelp Launcher"
   frame .launcher_setup.dummy -width 10c -height 3m
   pack .launcher_setup.label .launcher_setup.dummy \
      -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING -side top -fill x

   ###############################################################
   # Create the column headers
   label .launcher_setup.column1 -text "Label"
   place .launcher_setup.column1 -in .launcher_setup.label \
      -rely 1.0 -y 5 -x 80 
   label .launcher_setup.column2 -text "Bitmap/Pixmap"
   place .launcher_setup.column2 -in .launcher_setup.label \
      -rely 1.0 -y 5 -x 230 
   label .launcher_setup.column3 -text "Command to Execute"
   place .launcher_setup.column3 -in .launcher_setup.label \
      -rely 1.0 -y 5 -x 400

   ###############################################################
   # Create the main panel
   frame .launcher_setup.setup_fr -borderwidth $RIDGE_BORDER \
      -relief groove 
   pack .launcher_setup.setup_fr -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING -side top -expand true -fill both
   canvas .launcher_setup.setup_fr.c -yscrollcom \
      ".launcher_setup.setup_fr.sy set" -scrollregion \
      "0 0 600 0" -width 600 -highlightthickness 0
   pack [scrollbar .launcher_setup.setup_fr.sy -orient v \
      -com ".launcher_setup.setup_fr.c yview"] \
      -side right -fill y
   pack .launcher_setup.setup_fr.c -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side top -fill both \
      -expand true -exp 1

   ###############################################################
   # Create the individual settings
   set total $g_launcher_number
   set g_launcher_number 0
   for {set count 0} {$count < $total} {incr count} \
   {
      # Create a row for the item
      New_Launcher_Item $count
      incr g_launcher_number

      # Define globals
      global g_launcher_label_$count g_launcher${count}_command \
         g_launcher_bitmap_$count g_launcher_command_$count

      # Write launcher item
      set text [eval subst {\$g_launcher_label_$count}]
      .launcher_setup.setup_fr.c.$count.label insert end $text
      set text [eval subst {\$g_launcher_bitmap_$count}]
      .launcher_setup.setup_fr.c.$count.bitmap insert end $text
      set text [eval subst {\$g_launcher_command_$count}]
      set g_launcher${count}_command $text
   }


   frame .launcher_setup.button_frame -borderwidth $DEFAULT_PADDING
   pack .launcher_setup.button_frame -side bottom -fill x
   button .launcher_setup.button_frame.close -text Close \
      -command { Close_Launcher_Setup }
   pack .launcher_setup.button_frame.close
}

###############################################################
# Create a button, entry pair
# Number is the item to insert after
proc New_Launcher_Item { number } {
   Debug "New_Launcher_Item"

   # Globals
   global FONT_NORMAL DEFAULT_PADDING \
      LAUNCHER_ITEM_LENGTH SCRIPT_ITEM_LENGTH \
      g_launcher_changed g_current_launcher \
      g_launcher_number

   # Work out the canvas height
   set canvas_height [expr $g_launcher_number * 50]
   set canvas .launcher_setup.setup_fr.c
   $canvas configure -scrollregion \
      "0 0 600 $canvas_height"

   # Get the current list of available scripts and commands
   # These begin with gs_ppp_name_ and gs_cmd_name_
   set list ""
   set search "gs_ppp_name_*"
   set globals_list [info globals $search]
   foreach item $globals_list {
      global $item
      set name [subst \$$item]
      lappend list $name
   }
   set search "gs_cmd_name_*"
   set globals_list [info globals $search]
   foreach item $globals_list {
      global $item
      set name [subst \$$item]
      lappend list $name
   }

   # Check that the list is not empty!
   if {$list == ""} {
      Info_Dialog . "No scripts yet created.  Please select
Setup / Scripts... and create some scripts."
      return
   }
   
   # Create a label/textentry pair
   frame $canvas.$g_launcher_number
   entry $canvas.$g_launcher_number.label \
      -width $LAUNCHER_ITEM_LENGTH
   bind $canvas.$g_launcher_number.label \
      <KeyPress> "set g_launcher_changed 1"
   bind $canvas.$g_launcher_number.label \
      <FocusIn> "set g_current_launcher $g_launcher_number"
   entry $canvas.$g_launcher_number.bitmap \
      -width $LAUNCHER_ITEM_LENGTH
   bind $canvas.$g_launcher_number.bitmap \
      <KeyPress> "set g_launcher_changed 1"
   bind $canvas.$g_launcher_number.bitmap \
      <FocusIn> "set g_current_launcher $g_launcher_number"
   button $canvas.$g_launcher_number.button \
       -text "..." -command "Field_File_Select \
       $canvas.$g_launcher_number.bitmap"
   bind $canvas.$g_launcher_number.button \
       <ButtonPress> "set g_launcher_changed 1"
   bind $canvas.$g_launcher_number.button \
       <FocusIn> "set g_current_launcher $g_launcher_number"
   # Script stored in global variable
   global g_launcher${g_launcher_number}_command
   set item [eval tk_optionMenu $canvas.$g_launcher_number.command \
      g_launcher${g_launcher_number}_command $list]
   $canvas.$g_launcher_number.command configure -width $SCRIPT_ITEM_LENGTH
   bind $canvas.$g_launcher_number.command <ButtonPress> \
      "set g_launcher_changed 1 ; set g_current_launcher $g_launcher_number"
   pack $canvas.$g_launcher_number.label $canvas.$g_launcher_number.bitmap \
      $canvas.$g_launcher_number.button  $canvas.$g_launcher_number.command \
      -side left -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING

   set offset [expr (40 * $g_launcher_number)]
   $canvas create window 10 $offset \
      -window $canvas.$g_launcher_number \
      -anchor nw

   # If number is less that the current total, user
   # has requested an insert.  Move all items below
   # number down one.
   if {$number < $g_launcher_number} {

      set last [expr $g_launcher_number -1]
      for {set count $last} {$count >= $number} {incr count -1} \
      {
         # Copy the text from the current item to the next item
         set next [expr $count + 1]
   
         # The label
         set text [$canvas.$count.label get]
         $canvas.$next.label delete 0 end
         $canvas.$next.label insert end $text
   
         # The bitmap
         set text [$canvas.$count.bitmap get]
         $canvas.$next.bitmap delete 0 end
         $canvas.$next.bitmap insert end $text
   
         # The command
         global g_launcher${count}_command g_launcher${next}_command
         set g_launcher${next}_command [subst \$g_launcher${count}_command]
      }

      # Blank the new entry
      $canvas.$number.label delete 0 end
      $canvas.$number.bitmap delete 0 end
      set g_launcher${nunber}_command ""
   }
   # Do not increment option number - responsibility of caller
}

###############################################################
# Deletes the launcher item given by g_current_launcher
proc Delete_Launcher_Item {} {
   Debug "Delete_Launcher_Item"

   # Globals
   global g_launcher_changed g_current_launcher \
      g_launcher_number g_launcher_changed

   set canvas .launcher_setup.setup_fr.c
   set last [expr $g_launcher_number - 1]
   for {set count $g_current_launcher} {$count < $last} \
      {incr count} \
   {
      # Copy the text from the next item to the current item
      set next [expr $count + 1]

      # Check for bounds
      if {$next > $last} {
         continue
      }

      # The label
      set text [$canvas.$next.label get]
      $canvas.$count.label delete 0 end
      $canvas.$count.label insert end $text

      # The bitmap
      set text [$canvas.$next.bitmap get]
      $canvas.$count.bitmap delete 0 end
      $canvas.$count.bitmap insert end $text

      # The command
      global g_launcher${count}_command g_launcher${next}_command
      set g_launcher${count}_command [subst \$g_launcher${next}_command]
   }

   # Now delete the last row
   destroy $canvas.$last

   # And decrease the count
   incr g_launcher_number -1
   set g_launcher_changed 1
}

###############################################################
# The procedure for quitting the setup window
proc Close_Launcher_Setup {} {
   Debug "Close_Launcher_Setup"

   # Globals
   global g_launcher_changed

   # Check to see whether anything has changed.
   if {$g_launcher_changed == 1 && ![Question_Dialog .launcher_setup \
      "Do you wish to save changes?" "Save" "Reset Changes"]} {

      # Save requested
      Save_Launcher_Setup
   } else {

      # Re-read defaults from file
      Source_Launcher_Setup 0
      set g_launcher_changed 0
   }

   # Destroy the window
   destroy .launcher_setup 
}

###############################################################
# The procedure for saving the configuration
proc Save_Launcher_Setup {} {
   Debug "Save_Launcher_Setup"

   # Globals
   global g_launcher_number g_setup_file g_launcher_changed \
      TKNET_USER TKNET_GLOBAL_LAUNCHER_FILE \
      TKNET_LOCAL_LAUNCHER_FILE GREEN \
      gb_launcher_window

   # Save the setup to file
   # For each of the piars of items in the window,
   # ascertain the variable name being modified, then
   # write the output to a temporary file
   set canvas .launcher_setup.setup_fr.c

   # If root save to global settings, else save to users
   # local ~/.tknet_setup file
   if {$TKNET_USER == "root" && [Question_Dialog \
      . "Where do you want to save lancher setup to?" \
      "Local" "Global"]} {
         set g_setup_file $TKNET_GLOBAL_LAUNCHER_FILE
   } else {
      set g_setup_file $TKNET_LOCAL_LAUNCHER_FILE
   }

   # Open the file
   set file_id [ open $g_setup_file w ]
   puts $file_id "## TkNet Launcher Information"
   puts $file_id " "

   # Write the number of items in the launcher
   puts $file_id "## Number of items"
   puts $file_id "global g_launcher_number"
   puts $file_id [join [list "set g_launcher_number" \
      "$g_launcher_number"]]

   # Write all information
   puts $file_id " "
   puts $file_id "## Launcher Items"
   for {set count 0} {$count < $g_launcher_number} \
      {incr count} \
   {
      # Write launcher item
      puts $file_id "global g_launcher_label_$count g_launcher_bitmap_$count \\
         g_launcher_command_$count"
      set text [$canvas.$count.label get]
      puts $file_id [join [list "set g_launcher_label_" $count \
         " \"" $text \"] ""]
      set text [$canvas.$count.bitmap get]
      puts $file_id [join [list "set g_launcher_bitmap_" $count \
         " \"" $text \"] ""]
      global g_launcher${count}_command
      set text [subst \$g_launcher${count}_command]
      puts $file_id [join [list "set g_launcher_command_" $count \
         " \"" $text \"] ""]
         puts $file_id " "
   }

   # Close file
   close $file_id

   # Display a dialog
   Set_Message "Launcher Items Saved to $g_setup_file" $GREEN
   set g_launcher_changed 0

   # Re-read from file
   Source_Launcher_Setup 0

   # Show the launcher again
   Show_Launcher $gb_launcher_window
}

###############################################################
# The procedure for creating and displaying the launcher
# Argument must be 0=Embed 1=External
proc Show_Launcher { type } {
   Debug "Show_Launcher"

   # Globals
   global g_launcher_number gi_launcher_button_style \
      LAUNCHER_FONT RIDGE_BORDER LAUNCHER_COLOUR \
      TKNET_LAUNCHER_GEOMETRY gb_launcher_vertical \
      GREEN g_num_launcher_rows

   # If the window exists, destroy it
   if [winfo exists .launcher_window] {
      destroy .launcher_window
      wm geometry . ""
      update
   }

   # Manage window
   if {$type == 0} {

      # Create the window from the options specified
      frame .launcher_window -borderwidth $RIDGE_BORDER \
         -relief flat
      pack .launcher_window -after .mbar -side top -fill x

   } else {
   
      # Create as new window
      toplevel .launcher_window
      wm title .launcher_window "Launcher"
      wm geometry .launcher_window $TKNET_LAUNCHER_GEOMETRY
   }

   # Create the rows
   for {set count 0} {$count < $g_num_launcher_rows} \
      {incr count} \
   {
      frame .launcher_window.f$count -relief flat
      if {$type != 0 && $gb_launcher_vertical == 1} {
         pack .launcher_window.f$count -side left \
            -padx 0 -pady 0 -anchor nw
      } else {
         pack .launcher_window.f$count -side top \
            -padx 0 -pady 0 -anchor nw
      }
   }

   # Bind the options menu
   bind .launcher_window <Button-3> {Popup_Menu \
      .launcher_window .mbar.launcher.menu}
   bind .launcher_window <Button-2> {Popup_Menu \
      .launcher_window .mbar.launcher.menu}

   for {set count 0} {$count < $g_launcher_number} \
      {incr count} \
   {
      # Get the row number for the item
      set row_num [expr $count * $g_num_launcher_rows \
         / $g_launcher_number ]

      # Get the current information
      global g_launcher_label_$count \
         g_launcher_bitmap_$count \
         g_launcher_command_$count
      set label [eval subst {\$g_launcher_label_$count}]
      set bitmap [eval subst {\$g_launcher_bitmap_$count}]
      set command [eval subst {\$g_launcher_command_$count}]

      set cmd [subst -nocommands {Run_Script \"$command\"; \
         Set_Message \"Script $label run successfully.\" $GREEN;}]

      if {$gi_launcher_button_style == 1} {

         # Labels are specified
         button .launcher_window.f$row_num.$count \
            -text $label -font $LAUNCHER_FONT \
            -background $LAUNCHER_COLOUR -command $cmd

      } elseif {$gi_launcher_button_style == 2} {

         # Bitmaps are specified
         set image [Create_Image $bitmap]
         if {$image == -1} {
            # Display label instead
            button .launcher_window.f$row_num.$count \
               -text $label -font $LAUNCHER_FONT \
               -background $LAUNCHER_COLOUR \
               -command $cmd
         } else {
            button .launcher_window.f$row_num.$count \
               -image $image -command $cmd
         }
      } else {

         # Display both bitmaps and images
         set image [Create_Image $bitmap]
         if {$image == -1} {
            # Display label instead
            button .launcher_window.f$row_num.$count \
               -text $label -font $LAUNCHER_FONT \
               -background $LAUNCHER_COLOUR \
               -command $cmd
         } else {
            frame .launcher_window.f$row_num.$count -relief raised \
               -borderwidth $RIDGE_BORDER -highlightthickness 0
            button .launcher_window.f$row_num.$count.i \
               -command $cmd -image $image -relief flat
            button .launcher_window.f$row_num.$count.l \
               -command $cmd -text $label -font $LAUNCHER_FONT \
               -relief flat
            pack .launcher_window.f$row_num.$count.i \
               -side top -fill both -padx 0 -pady 0
            pack .launcher_window.f$row_num.$count.l \
               -side bottom -fill both -padx 0 -pady 0
            # Bind button down and up events in each
            # button to invoke the corresponding events
            # in the other
            bind .launcher_window.f$row_num.$count.l \
               <ButtonPress> "eval .launcher_window.f$row_num.$count configure -relief sunken"
            bind .launcher_window.f$row_num.$count.l \
               <ButtonRelease> "eval .launcher_window.f$row_num.$count configure -relief raised"
            bind .launcher_window.f$row_num.$count.i \
               <ButtonPress> "eval .launcher_window.f$row_num.$count configure -relief sunken"
            bind .launcher_window.f$row_num.$count.i \
               <ButtonRelease> "eval .launcher_window.f$row_num.$count configure -relief raised"
         }
      }

      # Pack buttons according to the orientation
      if {$type != 0 && $gb_launcher_vertical == 1} {
         pack .launcher_window.f$row_num.$count -side top \
            -fill both -padx 0 -pady 0 -anchor nw
      } else {
         pack .launcher_window.f$row_num.$count -side left \
            -fill both -padx 0 -pady 0 -anchor nw
      }
   }
}

