###############################################################
# TkNet - Graphing Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Flags
set g_graph_last_time 0
set g_graph_last_offset 0

# Geometry
set TKNET_GRAPH_OPTIONS_GEOMETRY "+500+150"

###############################################################
# Routine to initialise the graph + popup
proc Graph_Initialise { graph } {

   # Globals
   global RIDGE_BORDER g_graph_bg g_graph_fg g_graph_max \
      g_graph_units g_graph_lineincr g_graph_markline \
      g_graph_height g_graph_width FONT_SMALL

   # Raise the canvas
   $graph configure -relief raised -width $g_graph_width -height \
      $g_graph_height -bg $g_graph_bg -borderwidth 3 \
      -highlightthickness 0

   # Delete all items in the canvas
   $graph addtag delall all
   $graph delete delall

   # Re-initialise variables
   set g_graph_last_time 0
   set g_graph_last_offset 0

   # If options require it, draw the lines...
   if {$g_graph_lineincr > 0} {
      set pixel_height 100
      set line 0
      while {$pixel_height > 0} {
         set pixel_height [expr $g_graph_height - ( $g_graph_height * \
            $line * $g_graph_lineincr / $g_graph_max )]
         set id [$graph create line 0 $pixel_height $g_graph_width \
            $pixel_height]
         # Keep a record of all scale items
         $graph addtag scale withtag $id
         if {$g_graph_markline == 1} {
            # Add a text string
            set text [expr $line * $g_graph_lineincr ]
            set id [$graph create text 3 $pixel_height -text \
               $text -anchor sw -font $FONT_SMALL]
            $graph addtag scale withtag $id
         }
         incr line
      }
   }

   # Raise the scale items above anything else
   $graph raise scale

   # Add binding to pop-up graph options dialogue
   bind $graph <ButtonPress> "Graph_Options_Popup $graph"
   bind $graph <ButtonPress> "+$graph configure -relief sunken"
   bind $graph <ButtonRelease> "$graph configure -relief raised"
}

###############################################################
# Routine to initialise the graph + popup
proc Graph_Options_Popup { graph } {

   # This pops up the dialogue which maintains the
   # following information about the graph:
   #
   # g_graph_bg - Background colour
   # g_graph_fg - Foregroud colour
   # g_graph_max - Maximum display value
   # g_graph_units - Display units
   # g_graph_lineincr - Display horiz lines at intervals.
   # g_graph_markline - Display text on marker lines
   # g_graph_height - Height in pixels
   # g_graph_width - Width in pixels
   # g_graph_pixelsec - Number of pixels per second

   # Globals
   global TKNET_GRAPH_OPTIONS_GEOMETRY DEFAULT_PADDING \
      RIDGE_BORDER g_graph_bg g_graph_fg g_graph_max \
      g_graph_units g_graph_lineincr g_graph_markline \
      g_graph_height g_graph_width DEFAULT_COLOUR \
      g_graph_pixelsec

   # Popup a selection window
   set window .graph_options_window
   if [winfo exists $window] {
       # Pop it up!
       wm deiconify $window
       raise $window
       update
       return
   }

   # Otherwise create the window
   toplevel $window
   wm title $window "Graph Options"
   wm transient $window .
   wm geometry $window $TKNET_GRAPH_OPTIONS_GEOMETRY

   ############################################################
   # Create a frame for the options
   set frame [canvas $window.type -borderwidth $RIDGE_BORDER -relief \
      groove -width 300 -height 290 -highlightthickness 0]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING -fill x

   # And now the options
   label $frame.l1 -text "Background Colour"
   entry $frame.bg -width 10 -state disabled -textvariable \
      g_graph_bg -bg $DEFAULT_COLOUR
   button $frame.bgcol -text "..." -command "Choose_Colour $frame.bg"
   $frame create window 15 15 -window $frame.l1 -anchor nw
   $frame create window 150 15 -window $frame.bg -anchor nw
   $frame create window 250 15 -window $frame.bgcol -anchor nw

   label $frame.l2 -text "Foreground Colour"
   entry $frame.fg -width 10 -state disabled -textvariable \
      g_graph_fg -bg $DEFAULT_COLOUR
   button $frame.fgcol -text "..." -command "Choose_Colour $frame.fg"
   $frame create window 15 45 -window $frame.l2 -anchor nw
   $frame create window 150 45 -window $frame.fg -anchor nw
   $frame create window 250 45 -window $frame.fgcol -anchor nw

   label $frame.l3 -text "Maximum Value"
   entry $frame.max -width 10 -textvariable g_graph_max
   $frame create window 15 75 -window $frame.l3 -anchor nw
   $frame create window 150 75 -window $frame.max -anchor nw

   label $frame.l4 -text "Display Units"
   entry $frame.units -width 10 -state disabled -textvariable \
      g_graph_units -bg $DEFAULT_COLOUR
   $frame create window 15 105 -window $frame.l4 -anchor nw
   $frame create window 150 105 -window $frame.units -anchor nw

   label $frame.l5 -text "Horiz. Line intervals"
   entry $frame.lineincr -width 10 -textvariable g_graph_lineincr
   $frame create window 15 135 -window $frame.l5 -anchor nw
   $frame create window 150 135 -window $frame.lineincr -anchor nw

   label $frame.l6 -text "Graph Height"
   entry $frame.height -width 10 -textvariable g_graph_height
   $frame create window 15 165 -window $frame.l6 -anchor nw
   $frame create window 150 165 -window $frame.height -anchor nw

   label $frame.l7 -text "Graph Width"
   entry $frame.width -width 10 -textvariable g_graph_width
   $frame create window 15 195 -window $frame.l7 -anchor nw
   $frame create window 150 195 -window $frame.width -anchor nw

   label $frame.l8 -text "Pixels/sec"
   entry $frame.pixsec -width 10 -textvariable g_graph_pixelsec
   $frame create window 15 225 -window $frame.l8 -anchor nw
   $frame create window 150 225 -window $frame.pixsec -anchor nw

   checkbutton $frame.markline -variable g_graph_markline \
      -text "Show Text on Markers"
   $frame create window 15 255 -window $frame.markline -anchor nw

   # Now for a close button
   set frame [frame $window.button_fr]
   pack $frame -side bottom -fill x
   button $frame.close -text Close -command "destroy $window; \
      Graph_Initialise $graph"
   button $frame.help -text Help -command "MH_MiniHelp GraphSettingsWindow"
   pack $frame.close $frame.help -side right -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING
}

###############################################################
# Routine to add a new line to the end of the graph
proc Graph_Create_Line { graph time value } {

   # Globals, the current end location
   global g_graph_last_time g_graph_last_offset g_graph_max \
      g_graph_height g_graph_width g_graph_fg g_graph_pixelsec

   # Create a new polygon at the end of the current range
   # with value 'value'

   # Convert time to seconds.  This becomes the end of the 
   # current range
   set secs [Time_To_Secs $time]
   set offset [expr $g_graph_width - ($secs * $g_graph_pixelsec)]   
   set move [expr $offset - $g_graph_last_offset]
   if {$g_graph_last_offset == 0} {
      set g_graph_last_offset $offset
      return
   }
   set g_graph_last_offset $offset
   $graph move graph_items $move 0

   # New co-ordinates
   set x1 [expr $g_graph_last_time + $offset]
   # Info_Dialog . "Time:$time or $secs, Offset:$offset, Move:$move, X1:$x1"

   # y-coords are a flat line average
   set y [expr $g_graph_height - ( $value * \
            $g_graph_height / $g_graph_max )]

   # Create new polygon
   set id [$graph create polygon $x1 $g_graph_height $x1 $y \
      $g_graph_width $y $g_graph_width $g_graph_height \
      -fill $g_graph_fg]
   # All graphics are tagged separately to the lines
   $graph addtag graph_items withtag $id
   $graph lower graph_items scale

   # Check for overrun
   $graph addtag del enclosed 0 $g_graph_height  -10000 0
   $graph delete del

   # Set stored values
   set g_graph_last_time [expr $secs * $g_graph_pixelsec]
}
