###############################################################
# TkNet - Connection Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Flags
set gb_connection_down       1
set gb_connection_down_prev  1
set g_wait_time              0
set g_max_wait_time          0
set g_connect_process        0
set gb_waiting_for_connection 0
set gb_disconnect_on_user_request 0
set g_process                0

# Variables to keep track of the current pre and post commands.  
# Postcmd contain the names of the commands to run and cond 
# contains the run time of the post comand which may 
# be immediate (1), connect (2) or disconnect (3).  Note that 
# running a new script will result in these values being overwritten.
set g_connect_postcmd        ""
set g_connect_cond           0

# The current script in use within the session
set gi_current_script        1
# The last script name run
set gs_last_script_name      ""

###############################################################
# The procedure for connecting
proc Toggle_Connection { } {
   Debug "Toggle_Connection"

   # Globals
   global gb_connection_down gb_waiting_for_connection

   # If a connection attempt is already being made,
   # ask the user if they wish to terminate the 
   # current attempt.  Otherwise, connect/disconnect
   # on the basis of the flag.

   if {$gb_waiting_for_connection ==1} {
      if ![Question_Dialog . \
         "You are in the process of connecting.\n\
Terminate this connection attempt?" "Yes" "No"] {
         # Terminate connection attempt
         Net_Disconnect
      }
   } elseif {$gb_connection_down == 1} {
      # Connect
      Net_Connect
   } else {
      if ![Question_Dialog . \
         "Terminate connection?" "Yes" "No"] {
         # Terminate connection attempt
         Net_Disconnect
      }
   }
}

###############################################################
# The procedure for connecting
proc Run_Script { name } {
   Debug "Run_Script"

   # Globals
   global g_network_interface gs_last_script_name g_log_file \
      DEFAULT_PADDING gb_have_expect RIDGE_BORDER g_process \
      g_connect_postcmd g_connect_cond GREEN

   # Info_Dialog . "Running script $name"
   regsub -all {[\ ]+} $name {_} script

   # We first need to determine whether the script to
   # be run is a PPP script or a command (expect) script.
   set exist [info globals "gs_ppp_name_${script}"]
   if {[lindex $exist 0] != ""} {
      # Run the PPP script
      global gs_ppp_name_${script} gs_ppp_interface_${script} \
         gs_ppp_script_${script} gs_ppp_precmd_${script} \
         gs_ppp_postcmd_${script} gs_ppp_cond_${script}

      # Allow a script to accept the current interface setting
      set interface [eval subst \$gs_ppp_interface_${script}]
      if {$interface != "-none-"} {
         set g_network_interface $interface
      }

      # If a precommand exists, run it now!
      if {[string compare [subst \$gs_ppp_precmd_${script}] \
         "-none-"] != 0} {
         # A pre-command exists, so run it!
         set cmd [subst \$gs_ppp_precmd_${script}]
         Set_Message "Running pre-command $cmd" $GREEN
         Run_Script $cmd
      }

      # If a postcommand exists, set it up now...
      if {[string compare [subst \$gs_ppp_postcmd_${script}] \
         "-none-"] != 0} {
         # A post-command exists, so set it into the variables...
         set g_connect_postcmd [subst \$gs_ppp_postcmd_${script}]
         set g_connect_cond [subst \$gs_ppp_cond_${script}]
      }

      # Now run the current script
      set gs_last_script_name $name
      set command [eval subst \$gs_ppp_script_${script}]
      regsub -all {[\^]+} $command " " command
      regsub -all {[@]+} $command "\"" command
      # Info_Dialog . $command
      if [catch {eval exec $command >> $g_log_file &} g_process] {
         Info_Dialog . "Unable to execute command '$command'
Command failed with error:

$g_process"
      }

      # If a post-command is specified to run immediately
      # then run it!
      if {[string compare $g_connect_postcmd ""] != 0} {
         if {$g_connect_cond == 1} {
            # A post-command exists to be run now
            set cmd $g_connect_postcmd
            set g_connect_postcmd ""
            set g_connect_cond 0
            Set_Message "Running post-command $cmd" $GREEN
            Run_Script $cmd
         }
      }

   } else {
      set exist [info globals "gs_cmd_name_${script}"]
      if {[lindex $exist 0] > ""} {
         # Run the command
         global gs_cmd_interface_${script} gs_cmd_num_prompt_${script} \
            gs_cmd_script_${script} gs_cmd_precmd_${script} \
         gs_cmd_postcmd_${script} gs_cmd_cond_${script}

         # If a precommand exists, run it now!
         if {[string compare [subst \$gs_cmd_precmd_${script}] \
            "-none-"] != 0} {
            # A pre-command exists, so run it!
            set cmd [subst \$gs_cmd_precmd_${script}]
            Set_Message "Running pre-command $cmd" $GREEN
            Run_Script $cmd
         }

         # If a postcommand exists, set it up now...
         if {[string compare [subst \$gs_cmd_postcmd_${script}] \
            "-none-"] != 0} {
            # A post-command exists, so set it into the variables...
            set g_connect_postcmd [subst \$gs_cmd_postcmd_${script}]
            set g_connect_cond [subst \$gs_cmd_cond_${script}]
         }

         # If there is a new interface to be examined, change the global
         if {[string compare [subst \$gs_cmd_interface_${script}] \
            "-none-"] != 0} {
            set g_network_interface [eval subst \$gs_cmd_interface_${script}]
         }

         # Run the command, either using exec or using expect       
         if {$gb_have_expect == 0 || [eval subst \
            \$gs_cmd_num_prompt_${script}] == 0} {
            # We don't have expect or no arguments are given - 
            # run the command
            set command [eval subst \$gs_cmd_script_${script}]
            if [catch {eval exec $command >> $g_log_file &} g_process] {
               Info_Dialog . "Unable to execute command '$command'
Command failed with error:

$g_process"
            }
         } else {
            # We do have expect, and there are items to be
            # prompted for.
            Run_Expect_Script $name
         }

         # If a post-command is specified to run immediately
         # then run it!
         if {[string compare $g_connect_postcmd ""] != 0} {
            if {$g_connect_cond == 1} {
               # A post-command exists to be run now
               set cmd $g_connect_postcmd
               set g_connect_postcmd ""
               set g_connect_cond 0
               Set_Message "Running post-command $cmd" $GREEN
               Run_Script $cmd
            }
         }

      } else {
         Info_Dialog . "No script found for '$name'"
      }
   }
}

###############################################################
# The procedure for connecting
proc Run_Expect_Script { name } {
   Debug "Run_Expect_Script"

   # Globals
   global g_network_interface gs_last_script_name g_log_file \
      DEFAULT_PADDING RIDGE_BORDER g_max_wait_time RED

   # Info_Dialog . "Running script $name"
   regsub -all {[\ ]+} $name {_} script

   # Popup a selection window
   set window .connect_options_window
   if [winfo exists $window] {
       # Pop it up!
       wm deiconify $window
       raise $window
       update
       return
   }

   # Otherwise create the window
   toplevel $window
   wm title $window "TkNet Script Prompt"
   wm transient $window .
   wm protocol $window WM_DELETE_WINDOW "destroy $window"
   
   ############################################################
   # Create a frame for the options
   set frame [frame $window.connect -borderwidth $RIDGE_BORDER \
      -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING
   label $frame.label -text "Script - $name"
   pack $frame.label -side top -fill x -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING
   
   global gs_cmd_num_prompt_$script gs_cmd_prompt_$script \
      gs_cmd_expect_$script gs_cmd_script_${script} \
      gs_cmd_timeout_${script}
   set rows [subst $[subst gs_cmd_num_prompt_$script]]
   
   # Create a prompt -
   # The send strings are stored in an array similar
   # to the expect strings, called cmd_send()
   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Label
      set text [subst $[subst gs_cmd_prompt_${script}($count)]]
      # Check for given SEND string (begins with %%)
      if {[string first "%%" $text] == 0} {
         # Prompt specifies an autoreply - do not display
         global g_cmd_send_$count
         set g_cmd_send_$count [string range $text 2 end]
      } else {
         # Prompt for text string
         label $frame.l$count -text $text
         global g_cmd_send_$count
         entry $frame.e$count -width 25 -show * -textvariable \
            g_cmd_send_$count
         pack $frame.l$count $frame.e$count -side top \
            -fill x -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING
      }
   }
   
   # And a little button at the bottom.  Then wait for the
   # button to be pressed before continuing.
   set frame [frame $window.button_fr]
   pack $frame -side bottom -fill x
   global go_script
   set go_script 0
   button $frame.close -text "OK" -command "destroy $window; set go_script 1"
   button $frame.cancel -text "Cancel" -command "destroy $window; set go_script 0"
   pack $frame.close $frame.cancel -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side right
               
   # Centre the dialog
   Centre_Dialog $window
   tkwait window $window

   # Check response
   if {$go_script == 0} {
      # Unset the globals and return
      for {set count 1} {$count <= $rows} {incr count} {unset g_cmd_send_$count}
      unset go_script
      return
   }
   unset go_script

   # The window has now been destroyed.  Spawn the process
   # and begin the expect/send sequence.  Set the expect
   # timeout variable too, and the log_file for debugging
   set command [eval subst \$gs_cmd_script_${script}]
   set timeout $g_max_wait_time
   if [catch {eval spawn $command} error] {
      Info_Dialog . "Unable to spawn command '$command'
Command failed with error:

$error"
      # Unset the globals and return
      for {set count 1} {$count <= $rows} {incr count} {unset g_cmd_send_$count}
      return
   }

   # Internal logging disabled to prevent passwords being reported
   # exp_internal -f "/tmp/expect_log" 1
   # log_file -noappend "/tmp/expect_log"
   # log_file $g_log_file
   # Begin expect/send sequence

   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Label
      set expect_string [eval subst \$gs_cmd_expect_${script}($count)]
      set timeout [eval subst \$gs_cmd_timeout_${script}($count)]
      set send_string [eval subst \$g_cmd_send_$count]
      # Or should this be expect_background
      eval expect -timeout $timeout {"$expect_string" {
                 exp_send "$send_string\r"
                 send_log "expect: Got ${expect_string}, sent $send_string"
              } timeout {
                 # Abort the attempt
                 Set_Message "Command sequence failed - timeout" $RED
                 Info_Dialog . "Command failed with output:\n\n$expect_out(buffer)"
                 catch {close}
              } eof {
                 # Task has exited
                 Set_Message "Command sequence failed - exited" $RED
                 Info_Dialog . "Command failed with output:\n\n$expect_out(buffer)"
                 catch {close}
              } abort {
                 # Abort the attempt
                 Set_Message "Command sequence failed - abort" $RED
                 Info_Dialog . "Command failed with output:\n\n$expect_out(buffer)"
                 catch {close}
              }
      }
   }

   # Unset the globals
   for {set count 1} {$count <= $rows} {incr count} {unset g_cmd_send_$count}
}

###############################################################
# The procedure for connecting
proc Net_Connect { } {
   Debug "Net_Connect"

   # Globals
   global gb_connection_down g_connection_status \
      g_connection_time g_log_file g_connect_process GREEN RED \
      gb_waiting_for_connection g_wait_time gi_current_session \
      gi_current_script g_max_wait_time gb_disconnect_on_user_request

   # Set an hourglass
   WatchCursor

   # Start the connection process in the background
   if {$gb_connection_down == 1} {

      # If a connection process is already running, don't
      # start another one!
      set process_running [ Is_Process_Running $g_connect_process ]
      if {$process_running == 1} {
         Set_Message "Already Connecting ..." $RED
      } else {
         # Start connection process

         # Get the script label and command string
         global "g_script_$gi_current_session"
         set label [eval subst {\$g_script_${gi_current_session}(label)}]
         set script [eval subst {\$g_script_${gi_current_session}(connect1)}]
         set g_max_wait_time [eval subst {\$g_script_${gi_current_session}(timeout1)}]
         Set_Message "Connecting using $label" $GREEN

         # Now script is the name of the script to run, which
         # itself may be either a PPP script or an expect script.
         # Call Run_Script to distinguish and run the script.
         Run_Script $script

         # Set the flag
         set gb_waiting_for_connection 1
         set g_wait_time 0

         # And reset the user requested disconnect flag
         set gb_disconnect_on_user_request 0

         # Reset script flag to first script in the session
         set gi_current_script 1
      }
   } else {
      Set_Message "Already Connected!" $RED
   }

   # Reset the hourglass
   NormalCursor
}

###############################################################
# The procedure for disconnecting
proc Net_Disconnect { } {
   Debug "Net_Disconnect"

   # Globals
   global g_connection_time g_log_file \
      g_disconnect_process gb_delivering_mail GREEN RED \
      gb_waiting_for_connection gi_current_session \
      gi_current_script gb_disconnect_on_user_request \
      g_process

   # Set an hourglass
   WatchCursor

   # If in the middle of an auto-retrieve mail, stop it
   if {$gb_delivering_mail == 1} {

      # reset the flag
      set gb_delivering_mail 0
   }

   # Reset the waiting for connection flag
   set gb_waiting_for_connection 0

   # And set the user requested disconnect flag
   set gb_disconnect_on_user_request 1

   # Start the disconnection process in the background
   # Get the script label and command string
   global "g_script_$gi_current_session"
   set label [eval subst {\$g_script_${gi_current_session}(label)}]
   set script [eval subst {\$g_script_${gi_current_session}(disconnect$gi_current_script)}]
   Set_Message "Attempting disconnection from $label" $GREEN

   # Now script is the name of the script to run, which
   # itself may be either a PPP script or an expect script.
   # Call Run_Script to distinguish and run the script.
   set old_process $g_process
   Run_Script $script

   # If process is running, destroy it...
   #if {$old_process != 0} {
   #   set run [Is_Process_Running $old_process]
   #   Set_Message "Is process running? $run" $RED
   #   if {$run == 1} {
   #      Kill_Process $old_process
   #   }
   #   set g_process 0
   #}

   # Reset the hourglass
   NormalCursor
}

###############################################################
# The procedure for quitting
proc Quit { } {
   Debug "Quit"

   # Globals
   global gb_connection_down RED gb_confirm_quit

   # Set_Message "Quitting ..." $RED
   if {$gb_connection_down == 0} {

      # Ask the user whether to disconnect and
      # quit
      if ![Question_Dialog . \
         "You are still connected!  Do you want to \
disconnect and quit?" "Yes" "No"] {

         # Disconnect from network
         Net_Disconnect

         # Exit program
         exit
      }
   } else {

      if {$gb_confirm_quit && \
         ![Question_Dialog . "Do you really want to quit?" \
         "Yes" "No"]} {
         
         # Exit program
         exit
      }
      
      if {! $gb_confirm_quit} {
         # Exit program
         exit
      }
   }
}

###############################################################
# The procedure for setting the connection flag
proc Set_Connection_Flag {} {
   # Debug "Set_Connection_Flag"

   # Globals
   global gb_connection_down g_network_interface

   # g_network_interface is set from the interface
   # specified from the individual scripts.

   # Submitted by George M. Sipe (gsipe@mindspring.com) 23/2/96
   set fd [open /proc/net/route]
   set gb_connection_down [expr [lsearch -exact [split [read $fd]] \
      $g_network_interface] == -1]
   close $fd
}

###############################################################
# The procedure for checking whether the network is connected
proc Check_For_Connection {} {
   Debug "Check_For_Connection"

   # Globals
   global gb_connection_down gb_connection_down_prev \
      gb_waiting_for_connection g_network_check_freq \
      g_connection_status g_wait_time g_max_wait_time \
      g_connect_process GREEN RED gb_beep_on_connect \
      gb_reset_clock_on_connect g_connection_time \
      HELP_COLOUR g_connected_pixmap gt_time_connected \
      gt_saved_time gi_current_session

   # If the connected flag is set, start the clock update,
   # otherwise call yourself again in a few seconds.   If
   # a cetain amount of time has elapsed since the request
   # to connect to the network, disconnect, display an
   # error and retry.
   Set_Connection_Flag

   if {$gb_connection_down == 0 && \
      $gb_connection_down_prev == 1} {

      # Set the initial connect time (if not specified)
      set gt_time_connected [Get_Current_Time]

      # Set the big green label
      if {$g_connected_pixmap != ""} {
         set Image [Create_Image $g_connected_pixmap]
         if {$Image == -1} {
            # Image creation  did not work, so turn it off
            $g_connection_status configure -text "Connected" \
               -background $RED -highlightbackground $RED
            set g_connected_pixmap "" 
         } else {
            $g_connection_status configure -image $Image \
               -highlightbackground $RED
         }
      } else {
         $g_connection_status configure -text "Connected" \
            -background $RED -highlightbackground $RED
      }

      # Beep if option set
      if {$gb_beep_on_connect == 1} {
         bell
         bell
      }

      # Save the time if option set
      if {$gb_reset_clock_on_connect == 0} {
         set $gt_time_connected [Get_Current_Time]
         set gt_saved_time [$g_connection_time cget -text]
      } else {
         set gt_saved_time "000/00:00:00"
      }

      # Set the icon
      # wm iconbitmap . @/usr/include/X11/pixmaps/netup.xpm

      # Reset the waiting for connection flag
      set gb_waiting_for_connection 0
      set g_wait_time 0

      # If a post-command is specified to run on connection
      # then run it!
      global g_connect_postcmd g_connect_cond
      if {[string compare $g_connect_postcmd ""] != 0} {
         if {$g_connect_cond == 2} {
            # A post-command exists to be run now
            set cmd $g_connect_postcmd
            set g_connect_postcmd ""
            set g_connect_cond 0
            Set_Message "Running post-command $cmd" $GREEN
            Run_Script $cmd
         }
      }

      # Start the clock update
      Update_Clock

      # Set a connected message
      Set_Message "New connection detected" $GREEN
      wm iconname . "Net Up"
   } 

   if {$gb_connection_down == 1 && \
      $gb_waiting_for_connection == 1} {

      # Set a message
      # Debug "Waiting for connection..."
      set message "Waiting for connection ($g_wait_time secs)"
      Set_Message $message $HELP_COLOUR

      # Check against max allowable wait time
      set g_wait_time [ expr $g_wait_time + $g_network_check_freq ]
      if {$g_wait_time > $g_max_wait_time} {

         # Call disconnect script
         Net_Disconnect

         # If there is a further script to use, start it...
         global "g_script_$gi_current_session" gi_current_script
         set num_scripts [eval subst \$g_script_${gi_current_session}(number)]
         if {$gi_current_script < $num_scripts} {
            # Try the next script
            incr gi_current_script
            set script [eval subst \$g_script_${gi_current_session}(connect$gi_current_script)]
            set g_max_wait_time [eval subst \$g_script_${gi_current_session}(timeout$gi_current_script)]
            set g_wait_time 0
            set gb_waiting_for_connection 1
            Run_Script $script

            # Display a message
            Set_Message "Connection timed out - trying $script" $RED
         } else {
            # No script left to try - give up!
            Set_Message "Connection timed out - no more scripts!" $RED
            set gb_waiting_for_connection 0
         }
      } 
   }

   # Call routine again after N secs
   if {$g_network_check_freq > 0} {
      set run_after [ expr $g_network_check_freq * 1000 ]
   } else {
      set run_after 10000
   }
   after $run_after Check_For_Connection

   # Set the flag which remembers the previous state
   set gb_connection_down_prev $gb_connection_down
}


