###############################################################
# TkNet - Command Setup Module
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc/tknet.htm
# Version 1.1
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Globals for this module

# Flags
set gl_cmd_windows      [list "0"]

# Geometry
set TKNET_CMD_GEOMETRY "+500+150"

# Files
set TKNET_GLOBAL_NETWORK_FILE "$GLOBAL_INCLUDE/tknet.network"
set TKNET_LOCAL_NETWORK_FILE "$TKNET_USER_HOME/.tknet_network"

###############################################################
# The procedure to read the network setup files
proc CMD_Source_Scripts { file } {
   Debug "CMD_Source_Scripts"

   # Argument must be 0=Both 1=Global or 2=Local 

   # Globals
   global TKNET_GLOBAL_NETWORK_FILE \
      TKNET_LOCAL_NETWORK_FILE

   # Read the global setup file
   if {$file != 2 && [file exists $TKNET_GLOBAL_NETWORK_FILE]} {
      source $TKNET_GLOBAL_NETWORK_FILE
   }
   
   # Read the user's own config file if it exists
   if {$file != 1 && [file exists $TKNET_LOCAL_NETWORK_FILE]} {
      # Override with users own
      source $TKNET_LOCAL_NETWORK_FILE
   }
}

##################################################################
# Procedure for editing and creating cmd sessions using 
# cmd and chat. 
proc CMD_Create_Session { parent edit_type name } {
   Debug "CMD_Create_Session"

   # parent - Calling window
   # edit_type = EDIT/NEW
   # name = Name of script to edit

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING TKNET_CMD_GEOMETRY \
      gl_cmd_windows SCRIPT_ITEM_LENGTH

   # The global variable gi_number_cmd_windows gives the 
   # number of cmd edit windows currently in use.  Each
   # currently used window is checked to see whether it
   # is the edit session for the desired script. If no
   # current session is found, a new session is created.
   set win_num [CMD_Find_Script $name]
   if {$win_num != 0} {
      # Session found, pop it up!
      wm deiconify .cmd_window_$win_num
      raise .cmd_window_$win_num
      update
      return
   }

   # Set local constants
   set LABEL_LENGTH 15
   set RELIEF flat

   # Create a dialogue, read the last number from gl_cmd_windows
   # and increment it.
   set win_num [expr [lindex $gl_cmd_windows end] + 1]
   lappend gl_cmd_windows $win_num
   set window [toplevel .cmd_window_$win_num]
   wm title $window "Command Setup : $name"
   wm iconname $window "Command : $name"
   wm geometry $window $TKNET_CMD_GEOMETRY
   wm protocol $window WM_DELETE_WINDOW "CMD_Close_Window \
      $parent $win_num 0"

   # Set instance global variables
   global gs_cmd${win_num}_interface 
   global gs_cmd${win_num}_edit_type
   global gs_cmd${win_num}_modified
   set gs_cmd${win_num}_edit_type $edit_type
   set gb_cmd${win_num}_modified False

   # Change to internal representation of name - all spaces
   # converted to _'s.
   set orig_name $name
   regsub -all {[\ ]+} $name {_} name

   # A CMD session is completely described by:
   #
   # (1) A name
   # (2) An interface (e.g. cmd0, sl0)
   # (3) A cmd command string
   # (4) A sequnce of expect/prompt sequences + timeout
   # (5) A number indicating the number of expect/prompt sequemces
   # (6) A command to run before hand
   # (7) A command to run afterwards
   # (8) Conditions for running post command
   

   # A bit more about the expect/prompt sequences.  There is an array
   # of size gs_cmd_num_prompts_${name} which contains the details
   # of the sequence.  For each element of the array, a portion of
   # a dialogue box is created, which prompts the user for information
   # using the text of 'prompt'.  Then, during the command execution,
   # we wait for stdout of the form given by 'expect'.  When this is
   # received, the user input from the dialogue box is sent to the 
   # application.  Expect/send sequences may not be saved as this
   # could potentially result in a user saving his/her password in
   # plain text to a file - no, no...

   global gs_cmd_name_${name}
   global gs_cmd_interface_${name}
   global gs_cmd_script_${name}
   global gs_cmd_expect_${name}
   global gs_cmd_prompt_${name}
   global gs_cmd_num_prompt_${name}
   global gs_cmd_precmd_${name}
   global gs_cmd_postcmd_${name}
   global gs_cmd_cond_${name}
   set gs_cmd_name_${name} $orig_name

   # Note that the parameter edit_type may be either
   # NEW or EDIT.  However if the script name is
   # modified in any way the edit type becomes NEW.
   
   ###########################################################################
   # Create Menu Bar
   frame $window.mbar -relief raised -bd 2
   pack $window.mbar -side top -fill x

   # Create the buttons   
   menubutton $window.mbar.file -text File -underline 0 -menu \
      $window.mbar.file.menu
   menubutton $window.mbar.edit -text Edit -underline 0 \
      -menu $window.mbar.edit.menu
   menubutton $window.mbar.help -text Help -underline 0 \
      -menu $window.mbar.help.menu
   pack $window.mbar.file $window.mbar.edit \
      -side left
   pack $window.mbar.help -side right

   # Create each menu item
   menu $window.mbar.file.menu -tearoff 0
      $window.mbar.file.menu add command -label "Save" \
         -command "CMD_Save_Script $win_num $name" -underline 0
      $window.mbar.file.menu add command -label "Return" \
         -command "CMD_Save_Script $win_num $name ; \
         CMD_Close_Window $parent $win_num 1" -underline 0
      $window.mbar.file.menu add command -label "Close" \
         -command "CMD_Close_Window $parent $win_num 0" \
         -underline 0
   menu $window.mbar.edit.menu -tearoff 0
      $window.mbar.edit.menu add command -label "Add to end" \
         -command "CMD_New_Item $win_num END" -underline 0
      $window.mbar.edit.menu add command -label "Add before current" \
         -command "CMD_New_Item $win_num CURRENT" -underline 0
      $window.mbar.edit.menu add separator
      $window.mbar.edit.menu add command -label "Delete current" \
         -command "CMD_Delete_Item $win_num" -underline 0
   menu $window.mbar.help.menu -tearoff 0
      $window.mbar.help.menu add command -label "Show Help..." \
         -command "MH_MiniHelp CommandWindow" -underline 0

   # Create the menu
   tk_menuBar $window.mbar $window.mbar.file \
      $window.mbar.edit $window.mbar.help

   # Top frame contains the selected script details
   frame $window.sel_fr -borderwidth $RIDGE_BORDER -relief groove
   pack $window.sel_fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill x
   label $window.sel_fr.label -text "Selected Script" \
      -width 20 -anchor nw
   entry $window.sel_fr.entry 
   $window.sel_fr.entry insert end $orig_name
   pack $window.sel_fr.label $window.sel_fr.entry \
      -side left -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING
   bind $window.sel_fr.entry <KeyPress> "set \
      gs_cmd${win_num}_edit_type NEW"
   tk_optionMenu $window.sel_fr.interface \
      gs_cmd${win_num}_interface "-none-" "sl0" "sl1" "ppp0" "ppp1"
   $window.sel_fr.interface configure -width 5
   pack $window.sel_fr.interface -side left -padx \
      $DEFAULT_PADDING -pady $DEFAULT_PADDING

   #################################################################
   # Create a frame for the pre and post commands.
   set frame [frame $window.precmd_fr -borderwidth \
      $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING  \
      -side top -fill x

   # Now create the two option menus which contain all currently
   # available command scripts which may be used as pre and post 
   # commands.
   set search "gs_cmd_name_*"
   set globals_list [info globals $search]
   set list "-none-"
   foreach item $globals_list {
      global $item
      set lscr [subst \$$item]
      lappend list $lscr
   }
   global g_cmd${win_num}_precmd g_cmd${win_num}_postcmd \
      g_cmd${win_num}_cond 
   set frame1 [frame $frame.f1]
   button $frame1.l -text "Pre-command:" -relief flat -command \
      "MH_MiniHelp PrePostCommands" -width 15 -justify left
   eval tk_optionMenu $frame1.precmd g_cmd${win_num}_precmd $list
   $frame1.precmd configure -width $SCRIPT_ITEM_LENGTH
   pack $frame1.l $frame1.precmd -side left
   set frame2 [frame $frame.f2]
   button $frame2.l -text "Post-command:" -relief flat -command \
      "MH_MiniHelp PrePostCommands" -width 15 -justify left
   eval tk_optionMenu $frame2.postcmd g_cmd${win_num}_postcmd $list
   $frame2.postcmd configure -width $SCRIPT_ITEM_LENGTH
   pack $frame2.l $frame2.postcmd -side left
   set frame3 [frame $frame.f3]
   set g_cmd${win_num}_cond 1
   radiobutton $frame3.r1 -variable g_cmd${win_num}_cond \
      -value 1 -text "Immediate"
   radiobutton $frame3.r2 -variable g_cmd${win_num}_cond \
      -value 2 -text "Connect"
   radiobutton $frame3.r3 -variable g_cmd${win_num}_cond \
      -value 3 -text "Disconnect"
   pack  $frame3.r1 $frame3.r2 $frame3.r3 -side left \
      -padx $DEFAULT_PADDING
   pack $frame1 $frame2 $frame3 -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side top -fill x

   ##########################################################
   # Create flag frame and help
   set frame [frame $window.command_fr -borderwidth \
       $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING  \
      -side top -fill x

   # Help button and instructions
   button $frame.label -text "Enter the command to run here.\n\
To obtain help click on this label." -relief flat -command \
      "MH_MiniHelp CommandLine"
   entry $frame.entry -textvariable gs_cmd_script_${name}
   pack $frame.label $frame.entry -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -side top -fill x

   ##########################################################
   # Create expect/send region and canvas
   set frame [frame $window.prompt_fr -borderwidth \
       $RIDGE_BORDER -relief groove]
   pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill both -expand true

   # Help button and instructions
   button $frame.label -text "Enter any prompt information here.\n\
To obtain help click on this label." -relief flat -command \
      "MH_MiniHelp ExpectScript"
   pack $frame.label -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING\
      -side top -fill x
   set canvas [canvas $frame.headings -width 440 -height 25 \
      -highlightthickness 0]
   label $canvas.l1 -text "Expect String"
   label $canvas.l2 -text "User Prompt"
   label $canvas.l3 -text "Timeout"
   $canvas create window 25 0 -window $canvas.l1 -anchor nw
   $canvas create window 170 0 -window $canvas.l2 -anchor nw
   $canvas create window 370 0 -window $canvas.l3 -anchor nw
   pack $canvas -padx $DEFAULT_PADDING -side top -fill x
   
   # Scrollable canvas region
   set canvas [canvas $frame.c -scrollregion "0 0 400 0" \
      -width 440 -height 100 -highlightthickness 0 -yscrollcom \
      "$frame.sy set"]
   pack [scrollbar $frame.sy -orient v -com "$frame.c yview"] \
      -side right -fill y
   pack $canvas -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill both -expand true

   # The entries within this region are maintained by
   # global variables, which are reset here.
   # The number of rows in the canvas
   global gi_cmd${win_num}_number_rows
   set gi_cmd${win_num}_number_rows 0
   # The current row in the canvas
   global gi_cmd${win_num}_current_row
   set gi_cmd${win_num}_current_row 0

   # Bind the right mouse button to pop-up the edit menu
   bind $window <Button-3> "Popup_Menu $frame $window.mbar.edit.menu"
   bind $window <Button-2> "Popup_Menu $frame $window.mbar.edit.menu"

   # Now load the data associated with the given script
   if {$edit_type == "EDIT"} {
      CMD_Display_Script $win_num $name
      $canvas yview moveto 0
   }
}

##################################################################
# Procedure for adding a new send/expect sequence
proc CMD_New_Item { win_num item } {
   Debug "CMD_New_Item"

   # Win_num is the id of the CMD edit window.
   # Item may be END or CURRENT

   # Globals
   global gi_cmd${win_num}_number_rows \
      gi_cmd${win_num}_current_row \
      gb_cmd${win_num}_modified \
      gb_have_expect
   set canvas .cmd_window_${win_num}.prompt_fr.c

   # Locals
   set ROW_SIZE 40

   # Check that the user has expect installed on the system
   if {$gb_have_expect == 0} {
      # Expect not installed - disallow operation
      if ![Question_Dialog . "You must have expectk installed for this operation!" \
         "Help" "Dismiss"] {
         MH_MiniHelp ExpectPackage
      }
      return
   }

   # Work out new height for canvas
   incr gi_cmd${win_num}_number_rows
   set rows [subst $[subst gi_cmd${win_num}_number_rows]]
   set current [subst $[subst gi_cmd${win_num}_current_row]]
   set offset [expr ( ( $rows - 1 ) * $ROW_SIZE ) + 5]

   # Create the new row, comprising an expect string, a
   # prompt and a timeout.
   entry $canvas.expect$rows -width 14
   bind $canvas.expect$rows <KeyPress> "set gb_cmd${win_num}_modified 1"
   bind $canvas.expect$rows <FocusIn> "set gi_cmd${win_num}_current_row $rows"
   $canvas create window 5 $offset -window $canvas.expect$rows -anchor nw
   entry $canvas.prompt$rows -width 25
   bind $canvas.prompt$rows <KeyPress> "set gb_cmd${win_num}_modified 1"
   bind $canvas.prompt$rows <FocusIn> "set gi_cmd${win_num}_current_row $rows"
   $canvas create window 150 $offset -window $canvas.prompt$rows -anchor nw
   tk_optionMenu $canvas.timeout$rows g_cmd${win_num}_timeout$rows \
      1 2 5 10 20 30 40 50 60 70 80 90 100 110 120
   #$canvas.timeout$rows configure -width 3
   $canvas create window 370 [expr $offset - 5] -window \
      $canvas.timeout$rows -anchor nw

   # Configure the scroll region for the canvas
   incr offset $ROW_SIZE
   $canvas configure -scrollregion "0 0 400 $offset"

   # Scroll to the new item, if placed at the end
   if ![string compare $item "END"] {
      $canvas yview moveto 1
   }

   # If number is less that the current total, user
   # has requested an insert.  Move all items below
   # number down one.
   if {[string compare $item "END"] != 0 && $rows > 1} {

      set last [expr $rows -1]
      for {set count $last} {$count >= $current} {incr count -1} \
      {
         # Copy the text from the current item to the next item
         set next [expr $count + 1]
   
         # The expect string
         set text [$canvas.expect$count get]
         $canvas.expect$next delete 0 end
         $canvas.expect$next insert end $text

         # The prompt string
         set text [$canvas.prompt$count get]
         $canvas.prompt$next delete 0 end
         $canvas.prompt$next insert end $text

         # The timeout
         global g_cmd${win_num}_timeout$count g_cmd${win_num}_timeout$next
         set g_cmd${win_num}_timeout$next [subst \$g_cmd${win_num}_timeout$count]
      }

      # Blank the new entry
      $canvas.expect$current delete 0 end
      $canvas.prompt$current delete 0 end
      global g_cmd${win_num}_timeout$current
      set g_cmd${win_num}_timeout$current 10
   }
   # Set the current row to the inserted row
   set gi_cmd${win_num}_current_row [expr $current + 1]
}

##################################################################
# Procedure for closing a CMD edit session
proc CMD_Delete_Item { win_num } {
   Debug "CMD_Delete_Item"

   # Globals
   global gi_cmd${win_num}_number_rows
   global gi_cmd${win_num}_current_row
   set canvas .cmd_window_${win_num}.prompt_fr.c

   # Locals
   set ROW_SIZE 40

   # Work out new height for canvas
   set rows [subst $[subst gi_cmd${win_num}_number_rows]]
   set current [subst $[subst gi_cmd${win_num}_current_row]]

   set last [expr $rows -1]
   for {set count $current} {$count <= $last} {incr count} \
   {
      # Copy the text from the current item to the next item
      set next [expr $count + 1]
   
      # The expect string
      set text [$canvas.expect$next get]
      $canvas.expect$count delete 0 end
      $canvas.expect$count insert end $text

      # The prompt string
      set text [$canvas.prompt$next get]
      $canvas.prompt$count delete 0 end
      $canvas.prompt$count insert end $text

      # The timeout
      global g_cmd${win_num}_timeout$count g_cmd${win_num}_timeout$next
      set g_cmd${win_num}_timeout$count [subst \$g_cmd${win_num}_timeout$next]
   }
   # Now destroy the last item
   destroy $canvas.expect$rows
   destroy $canvas.prompt$rows
   destroy $canvas.timeout$rows
   global g_cmd${win_num}_timeout$rows
   unset g_cmd${win_num}_timeout$rows

   # Configure the scroll region for the canvas
   incr gi_cmd${win_num}_number_rows -1
   set rows [subst $[subst gi_cmd${win_num}_number_rows]]
   set offset [expr ( ( $rows - 1 ) * $ROW_SIZE )]
   incr offset $ROW_SIZE
   $canvas configure -scrollregion "0 0 400 $offset"
}

##################################################################
# Procedure for closing a CMD edit session
proc CMD_Close_Window { parent win_num return_value} {
   Debug "CMD_Close_Window $win_num"

   # Globals
   global gl_cmd_windows

   # Destroy the window and remove the number from
   # the list of active windows.
   # Info_Dialog . $gl_cmd_windows
   set list_item [lsearch $gl_cmd_windows $win_num]
   set gl_cmd_windows [lreplace $gl_cmd_windows \
      $list_item $list_item]

   # Destroy any global variables created by this window
   set search "*_cmd${win_num}_*"
   set globals_list [info globals $search]
   foreach global $globals_list {
      global $global
      unset $global
   }

   # Check to see whether the list window exists and
   # if the user has requested to return the value to 
   # the list window
   if [winfo exists $parent] {

      # Trigger a re-retrieve of the list window
      Display_Interactive_Scripts $parent.list_fr.list

      if {$return_value == 1} {

         # Set the current item into the window
         $parent.sel_fr.entry configure -state normal
         $parent.sel_fr.entry delete 0 end
         $parent.sel_fr.entry insert end [ \
            .cmd_window_${win_num}.sel_fr.entry get]
         $parent.sel_fr.entry configure -state disabled
      }
   }

   # Destroy the window
   destroy .cmd_window_$win_num
}

##################################################################
# Procedure for finding current CMD edit sessions
proc CMD_Find_Script { name } {
   Debug "CMD_Find_Script"

   # Globals
   global gl_cmd_windows

   # This script searches through the current windows to find
   # one which is editing the currently selected script $name.
   # If the window is found it's name is returned, otherwise
   # zero is returned and the window is created.
   foreach count $gl_cmd_windows \
   {
      # Check window contents
      # Info_Dialog . $count
      if {[winfo exists .cmd_window_$count] == 1 && [string compare \
         [.cmd_window_$count.sel_fr.entry get] $name] == 0} {
         return $count
      }
   }
   # No match found, return 0
   return 0
}

##################################################################
# Procedure for displaying a given script
proc CMD_Display_Script { win_num name } {
   Debug "CMD_Display_Script"

   # The script is given by the agrument name.
   # It is decomposed into parts and displayed 
   # using the window win_num.
   
   # Global variable give by 'name' is gs_cmd_script_$name.
   # The internal formats separates options using the ^
   # character.
   global gs_cmd_script_$name
   if {[info exists gs_cmd_script_$name] == 0} {
      # Change the status to NEW and pop-up a dialogue
      global gs_cmd${win_num}_edit_type
      set gs_cmd${win_num}_edit_type NEW
      Info_Dialog . "Script not found : Creating new script"
      return
   }
   set script [subst $[subst gs_cmd_script_$name]]
   #Info_Dialog . "gs_cmd_script_$name : '$script'"

   # Load the interface definition (sl0, cmd0 etc)
   global gs_cmd_interface_$name gs_cmd${win_num}_interface \
      gs_cmd_precmd_${name} g_cmd${win_num}_precmd \
      gs_cmd_postcmd_${name} g_cmd${win_num}_postcmd \
      gs_cmd_cond_${name} g_cmd${win_num}_cond 
   if [catch {set gs_cmd${win_num}_interface [subst \$gs_cmd_interface_$name]}] {
      set gs_cmd${win_num}_interface "-none-"
   }
   if [catch {set g_cmd${win_num}_precmd [subst \$gs_cmd_precmd_$name]}] {
      set g_cmd${win_num}_precmd "-none-"
   }
   if [catch {set g_cmd${win_num}_postcmd [subst \$gs_cmd_postcmd_$name]}] {
      set g_cmd${win_num}_postcmd "-none-"
   }
   if [catch {set g_cmd${win_num}_cond [subst \$gs_cmd_cond_$name]}] {
     set g_cmd${win_num}_cond 1
   }

   # And the prompt/expect sequence
   global gs_cmd_num_prompt_$name gs_cmd_prompt_$name \
      gs_cmd_expect_$name gs_cmd_timeout_$name
   set canvas .cmd_window_${win_num}.prompt_fr.c
   set rows [subst $[subst gs_cmd_num_prompt_$name]]
   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Create a new row
      CMD_New_Item $win_num END

      # And insert the information
      set text [subst $[subst gs_cmd_expect_${name}($count)]]
      $canvas.expect$count insert end $text
      set text [subst $[subst gs_cmd_prompt_${name}($count)]]
      $canvas.prompt$count insert end $text
      global g_cmd${win_num}_timeout$count
      set g_cmd${win_num}_timeout$count \
         [subst $[subst gs_cmd_timeout_${name}($count)]]
   }
}

##################################################################
# Procedure for saving a displayed/edited script
proc CMD_Save_Script { win_num name } {
   Debug "CMD_Save_Script"

   # Globals
   global TKNET_USER TKNET_GLOBAL_NETWORK_FILE \
      TKNET_LOCAL_NETWORK_FILE

   # Now save the name and the string to the setup file
   # Get the final name from the entry field
   set window .cmd_window_${win_num}
   set frame $window.sel_fr

   # Use thes globals for the save
   global gs_cmd${win_num}_interface gs_cmd${win_num}_edit_type \
      gi_cmd${win_num}_number_rows g_cmd${win_num}_precmd \
      g_cmd${win_num}_postcmd g_cmd${win_num}_cond 

   # Get the new script name from the edit field (if new)
   set name [$frame.entry get]
   regsub -all {[\ ]+} $name {_} name

   # Save as these names
   global gs_cmd_name_${name}
   global gs_cmd_interface_${name}
   global gs_cmd_script_${name}
   global gs_cmd_precmd_${name}
   global gs_cmd_postcmd_${name}
   global gs_cmd_cond_${name}

   # Set the information
   set gs_cmd_name_$name [$frame.entry get]
   set gs_cmd_interface_$name [subst $[subst gs_cmd${win_num}_interface]]
   set frame $window.command_fr
   set canvas $window.prompt_fr.c
   set gs_cmd_script_$name [$frame.entry get]
   set gs_cmd_precmd_$name [subst \$g_cmd${win_num}_precmd]
   set gs_cmd_postcmd_$name [subst \$g_cmd${win_num}_postcmd]
   set gs_cmd_cond_$name [subst \$g_cmd${win_num}_cond]

   # Get the expect/prompt pairs
   set rows [subst $[subst gi_cmd${win_num}_number_rows]]
   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Get the data for that row
      set gs_cmd_expect_${name}($count) \
         [$canvas.expect$count get]
      set gs_cmd_prompt_${name}($count) \
         [$canvas.prompt$count get]
      global g_cmd${win_num}_timeout$count
      set gs_cmd_timeout_${name}($count) \
         [subst \$g_cmd${win_num}_timeout$count]

      # Check that data has been input for each line.
      # If not, display an error dialogue and return w/o saving.
      if {[$canvas.expect$count get] == "" || \
         [$canvas.prompt$count get] == ""} \
      {
         Info_Dialog . "Invalid data entered for row $count.
All entry fields must contain text.
Save aborted."
         return
      }
   }

   # Get the useful information before writing (simplifies
   # the syntax!)
   set title_variable [subst gs_cmd_name_$name]
   set title [subst $[subst gs_cmd_name_$name]]
   set interface_variable [subst gs_cmd_interface_$name]
   set interface [subst $[subst gs_cmd_interface_$name]]
   set script_variable [subst gs_cmd_script_$name]
   set script [subst $[subst gs_cmd_script_$name]]
   set expect_variable [subst gs_cmd_expect_$name]
   set prompt_variable [subst gs_cmd_prompt_$name]
   set timeout_variable [subst gs_cmd_timeout_$name]
   set num_prompt_variable [subst gs_cmd_num_prompt_$name]
   set precmd_variable gs_cmd_precmd_$name
   set precmd [subst \$gs_cmd_precmd_$name]
   set postcmd_variable gs_cmd_postcmd_$name
   set postcmd [subst \$gs_cmd_postcmd_$name]
   set cond_variable gs_cmd_cond_$name
   set cond [subst \$gs_cmd_cond_$name]

   ###############################################################
   # Now write it to file.  If the edit type is NEW, append to the
   # end of the file.  Otherwise, find the current entry in the
   # file and replace it.
   ###############################################################

   # File information:  The file, tknet.network or ~/.tknet_network
   # Contains information about *all* scripts, not just CMD scripts.
   # The format is as follows.  On a line, ### $gs_cmd_name_name ##.  
   # The following lines contain the script information, up 
   # to the next record, which begins in the same manner.

   # If root save to global settings, else save to users
   # local ~/.tknet_network file
   if {$TKNET_USER == "root" && [Question_Dialog \
      . "Where do you want to save the network script to?" \
      "Local" "Global"]} {
         set setup_file $TKNET_GLOBAL_NETWORK_FILE
   } else {
      set setup_file $TKNET_LOCAL_NETWORK_FILE
   }

   # Check for existence of the file.
   if {[file exists $setup_file] != 1} {
      # Create New file
      set file_text "#########################################################
### This is the TkNet Network Scripts file.
### This file contains details of all the scripts defined
### using the script editors.
###
### PLEASE DO NOT EDIT BY HAND UNLESS YOU ARE QUITE SURE
### WHAT YOU ARE DOING!
#########################################################\n"
      set fd [open $setup_file w]
      puts -nonewline $fd $file_text
      flush $fd
      close $fd
   }

   # Write the script to the end of the file
   set fd [open $setup_file r]
   set file_text [read $fd]
   close $fd

   # If editing, ensure that the old version is deleted
   if {[subst $[subst gs_cmd${win_num}_edit_type]] == "EDIT"} {
      set start [string first "### $title_variable ##" $file_text]
      if {$start != -1} {
         set rest_file [string range $file_text [expr \
            $start + 4] end]
         set end [string first "###" $rest_file]
         set file_text [string range $file_text \
            0 [expr $start -1]]
         if {$end != -1} {
            set rest_file [string range $rest_file $end end]
            append file_text $rest_file
         }
      }
   }

   # Check to see that a script with the given name
   # does not already exist.  If it does, ask the user
   # if he wishes to replace the existing entry.  If
   # the user says no, then the function returns
   # immediately (with a dialog).
   if {[string first "### $title_variable ##" $file_text] != -1} {
      # Match found on title - ask user
      if {[Question_Dialog . \
         "Network Script already exists.  Overwrite?" \
         "Yes" "No"]} {
         Info_Dialog . "Save operation cancelled."
         return
      } else {
         # User has asked to overwrite the saved script
         set start [string first "### $title_variable ##" $file_text]
         if {$start != -1} {
            set rest_file [string range $file_text [expr \
               $start + 4] end]
            set end [string first "###" $rest_file]
            set file_text [string range $file_text \
               0 [expr $start -1]]
            if {$end != -1} {
               set rest_file [string range $rest_file $end end]
               append file_text $rest_file
            }
         }
      }
   }

   # Create the new script details
   append file_text "\n"
   append file_text "### $title_variable ##
global $title_variable $interface_variable $script_variable \\
   $expect_variable $prompt_variable $num_prompt_variable \\
   $timeout_variable $precmd_variable $postcmd_variable \\
   $cond_variable
set $title_variable \"$title\"
set $interface_variable \"$interface\"
set $script_variable \"$script\"
set $num_prompt_variable \"$rows\"
set $precmd_variable \"$precmd\"
set $postcmd_variable \"$postcmd\"
set $cond_variable \"$cond\"\n"

   # Add the expect/prompt pairs to this information
   for {set count 1} {$count <= $rows} {incr count} \
   {
      # Get the data for that row
      set text [subst $[subst gs_cmd_expect_${name}($count)]]
      append file_text "set ${expect_variable}($count) \"$text\"\n"
      set text [subst $[subst gs_cmd_prompt_${name}($count)]]
      append file_text "set ${prompt_variable}($count) \"$text\"\n"
      set text [subst $[subst gs_cmd_timeout_${name}($count)]]
      append file_text "set ${timeout_variable}($count) \"$text\"\n"
   }   

   # And then overwrite the existing file
   set fd [open $setup_file w 755]
   puts -nonewline $fd $file_text
   flush $fd
   close $fd
   Info_Dialog $window "Network script '$name' saved"
}
