#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"

###############################################################
# Minihelp - Help Editor
# Charlie KEMPSON - charlie@siren.demon.co.uk
# http://public.logica.com/~kempsonc
###############################################################

###############################################################
#
#    Copyright (c) 1995-1996 Charlie Kempson
#
#    This program is free software; you can redistribute it 
#    and/or modify it under the terms of the GNU General 
#    Public License as published by the Free Software 
#    Foundation (version 2 of the License).
#
#    This program is distributed in the hope that it will 
#    be useful, but WITHOUT ANY WARRANTY; without even the 
#    implied warranty of MERCHANTABILITY or FITNESS FOR A 
#    PARTICULAR PURPOSE.  See the GNU General Public License 
#    for more details.
#
#    For a copy of the GNU General Public License, write to the 
#    Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
#    MA 02139, USA.
###############################################################

###############################################################
# Set include path
set GLOBAL_INCLUDE "/usr/local/lib/TkNet/modules"

###############################################################
# The procedure to read other tcl files stored in
# /usr/local/lib/TkNet/modules
proc MH_SourceOther { filename } {

   # Globals
   global GLOBAL_INCLUDE

   # Look in the global area for files
   if [file exists "$GLOBAL_INCLUDE/$filename"] {
      if [catch {uplevel #0 source "$GLOBAL_INCLUDE/$filename"}] {
         Info_Dialog . "Error reading file : this doesn't look like a Tcl archive"
      }
   } else {
      # FILE MISSING
      Info_Dialog . "File $filename missing from $GLOBAL_INCLUDE"
   }
}

###############################################################
# The procedure for popping up a popup menu
proc MH_PopupMenu { parent window } {

   # Get current mouse position
   set x [ winfo pointerx $parent ]
   set y [ winfo pointery $parent ]

   # Popup the menu
   tk_popup $window $x $y
}

###############################################################
# Change cursor to an hourglass and back again
proc MH_WatchCursor {} {

   # Loop through children setting the cursor
        set busy ""
        foreach w [winfo children .] {
                lappend busy [list $w [lindex [$w config -cursor] 4]]
        }
        foreach w $busy {catch {[lindex $w 0] config -cursor watch}}
        update idletasks
}
proc MH_NormalCursor {} {

   # Loop through children setting the cursor
        set busy ""
        foreach w [winfo children .] {
                lappend notbusy [list $w [lindex [$w config -cursor] 4]]
        }
        foreach w $notbusy {catch {[lindex $w 0] config -cursor hand2}}
        update idletasks
}

###############################################################
# Procedure to open a help file
proc MH_OpenProject { } {

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING TEXT_COLOUR \
      gb_current_page_changed gb_project_changed \
      gs_current_project

   # Check for changes to the project
   if {$gb_current_page_changed || $gb_project_changed} {
      if [Question_Dialog . "The project has changed.  Do
you wish to discard changes?" "Discard" "Cancel"] {
         # User requested cancel
         return
      }
   }

   # Prompt for the project to open
   set gs_current_project [File_Select "Open Project"]
   if {$gs_current_project != ""} {MH_ReadProject}
}

###############################################################
# Procedure to open a help file
proc MH_ReadProject { } {

   # Globals
   global gas_pages gs_current_project gas_history \
      gt_mini_help gb_current_page_changed \
      gb_project_changed gs_image_base_dir

   # Get the file name
   set file $gs_current_project
   if {[file exists $file] == 1} {
      # If current information exists, delete it.
      set current_pages [info globals "gs_HELPTEXT_*"]
      foreach page $current_pages {
         global $page
         unset $page
      }

      # Blank old information
      # $gt_mini_help configure -state normal
      $gt_mini_help delete 1.0 end
      # $gt_mini_help configure -state disabled
      .name_fr.name configure -state normal
      .name_fr.name delete 0 end
      .name_fr.name configure -state disabled
      set gs_image_base_dir ""

      # Source the new file
      if [catch {uplevel #0 "source $file"}] {
         # Error reading file - probably not a help archive
         Info_Dialog . "File does not have the correct format.
Are you sure that this is a valid halp archive?"
         set gs_current_project ""
         return
      }

      # Check the image base
      if {$gs_image_base_dir == ""} {
         set gs_image_base_dir [file dirname $file]
      }

      # Build up a list of new pages
      set gas_pages [info globals "gs_HELPTEXT_*"]
      set gas_history ""
      set gb_current_page_changed 0
      set gb_project_changed 0

      # Display the page selector
      MH_SelectHelpPage FULL

      # Save the project name
      wm title . "Edit Help - $file"

   } else {
      # File does not exist
      Info_Dialog . " File $file does not exist"
      set gs_current_project ""
   }
}

###############################################################
# Procedure to save a help file
proc MH_SaveProject { {name ""} } {

   # Globals
   global gas_pages gb_project_changed gb_page_autocommit \
      gb_current_page_changed gs_current_project \
      gs_image_base_dir g_image_path

   # Check the name for syntactic correctness
   if {[llength [split $name]] > 1 || [regexp -nocase \
      {[`|!$%^&*()|~<>,]+} $name] || $name == ""} {
      Info_Dialog . "Chosen name is not valid.
The project has not been saved."
      return
   }

   # If project does not have a name pop
   # up the save project as dialog.
   if {$name == ""} {
      MH_SaveProjectAs
      return
   }

   # If the current page has been changed, ask the user if
   # (s)he wishes to save the changes to the current page
   # before saving.
   if {$gb_current_page_changed == 1} {
      if {$gb_page_autocommit == 1 || [Question_Dialog . \
         "The current page has changed.  Do you wish
 to save these changes with the project?" "No" "Yes"]} {
         MH_SaveHelpPage
      }
   }

   # Open the file
   if [catch {set fp [open $name w]}] {
      Info_Dialog . "Unable to open file $name for writing"
      return
   }

   # Write the header
   puts $fp "###############################################################
#
# THIS FILE WAS PRODUCES USING MINI HELP VERSION 1.0
#
# The file may be edited by hand but is best and most 
# easily editied using the mini-help editor.
#
# The help text uses a minor subset of the html command
# language.  The following constructs are supported:
# 
# <TITLE> Text </TITLE> - Centre's, bolds and underlines
# <XREF gs_help_string_ref> Text </XREF> - Hyperlinks
# <IMG SRC=filename> - Image
#
# <B> Text </B> - Bold
# <I> Text </I> - Italic
# <TT> Text </TT> - Sans serif
# <UL> Text </UL> - Underline
#
# <CENTER> Text </CENTER> - Centre Justify
# <LEFT> Text </LEFT> - Left Justify
# <RIGHT> Text </RIGHT> - Right Justify
#
# &dollar; - Dollar
# &lbrace; - Left brace
# &rbrace; - Right brace
# &lsquare; - Left square bracket
# &rsquare; - Right square bracket
# &quot; - Double quotes
# &squot; - Single quotes
# &backslash; - Backslash
#
###############################################################

# Set the base directory for images
global gs_image_base_dir 
set gs_image_base_dir \"$gs_image_base_dir\"
global g_image_path
set g_image_path \"$g_image_path\"

# The Help Pages...

"
   # Now write the help text itself
   foreach page $gas_pages {
      global $page
      set text [string trimright [subst \$$page] "\n"]
      puts $fp "global $page
set $page \\
\\
\"$text\"
"
   }

   # Flush the file
   flush $fp
   close $fp

   # Inform the user
   Info_Dialog . "Help saved to file $name"
   set gb_project_changed 0
   set gs_current_project $name
   wm title . "Edit Help - $name"
}

###############################################################
# Procedure to save a help file
proc MH_SaveProjectAs { } {

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING gs_new_project_name \
      TEXT_COLOUR

   # Prompt for the project to save as
   set gs_new_project_name [File_Select "Save Project As..."]
   if [file exists $gs_new_project_name] {
      if [Question_Dialog . "Are you sure you wish to overwrite
file $gs_new_project_name ?" "Yes" "No"] {
         Info_Dialog . "Help not saved"
         return
      }
   }

   # Check validity of name
   if {$gs_new_project_name != ""} {
      MH_SaveProject $gs_new_project_name
   } else {
      Info_Dialog . "You must choose a name for
your new help archive!"
   }
}

###############################################################
# Procedure to close a help file
proc MH_CloseProject { b_exit } {

   # Globals
   global gb_current_page_changed gb_project_changed gas_pages \
      gs_current_project gs_new_project_name gs_MH_NewHelpPage \
      gt_mini_help

   # Check for changes to the project
   if {$gb_current_page_changed || $gb_project_changed} {
      if [Question_Dialog . "The project has changed.  Do
you wish to discard changes?" "Discard" "Cancel"] {
         # User requested cancel
         return
      }
   }

   # Rest all variables, and destroy all globals
   foreach page $gas_pages {
      global $page
      unset $page
   }
   set gas_pages ""
   set gb_current_page_changed 0
   set gb_project_changed      0
   set gs_current_project      ""
   set gs_new_project_name     ""
   set gs_MH_NewHelpPage        ""
   wm title . "Edit Help - <no project>"

   # Reset all user fields
   # $gt_mini_help configure -state normal
   $gt_mini_help delete 1.0 end
   # $gt_mini_help configure -state disabled
   .name_fr.name configure -state normal
   .name_fr.name delete 0 end
   .name_fr.name configure -state disabled
   
   # If the argument is 1, the user is trying to exit
   if {$b_exit == 1} {destroy .}
}

###############################################################
# The procedure to change the base directory for images
proc MH_ImageChangeBase { } {

   # Global
   global gs_image_base_dir gs_current_project

   # If the current image base directory is "" then
   # default to the save directory
   if {$gs_image_base_dir == ""} {
      if {$gs_current_project != ""} {
         # Default to save directory
         set gs_image_base_dir [file dirname $gs_current_project]
      } else {
         # Default to current directory
         set gs_image_base_dir [pwd]
      }
   }

   # Popup a file selection dialogue in directory mode
   # to prompt for selection
   set gs_image_base_dir [File_Select "Select Image Directory" \
      "*" $gs_image_base_dir 1]
   set gs_image_base_dir [string trimright $gs_image_base_dir "/\\"]
   # Info_Dialog . $gs_image_base_dir
}

###############################################################
# The procedure to let you select a page
proc MH_SelectHelpPage { type } {

   # Type may be HISTORY or FULL

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING TEXT_COLOUR \
      gas_history gas_pages

   # Test for window
   # Popup a selection window
   set window .select_help
   if [winfo exists $window] {
       # Pop it up!
       wm deiconify $window
       raise $window
       update
       return
   }

   toplevel $window
   if {$type == "HISTORY"} {
      wm title $window "History"
      set as_list $gas_history
	} else {
		wm title $window "Select Page"
		set as_list $gas_pages
	}
	wm transient $window .

	# Create a frame containing a list
	frame $window.sel_fr -borderwidth $RIDGE_BORDER -relief groove
	pack $window.sel_fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
		-side top -fill x
	set list [ScrolledList $window.list_fr 0 20 browse 0]
	$list configure -bg $TEXT_COLOUR
	pack forget $window.list_fr
	pack $window.list_fr -side top -fill both -padx $DEFAULT_PADDING \
		-pady $DEFAULT_PADDING -expand true

	# Add the help pages title to the list
	foreach item $as_list {
		# Get the help text title for the page
		set text "Unknown Help Page"
		global $item
		set local_text [eval subst {\$$item}]
		set start [string first "<title>" $local_text]
		if {$start == -1} {set start [string first "<TITLE>" $local_text]}
		set end [string first "</title>" $local_text]
		if {$end == -1} {set end [string first "</TITLE>" $local_text]}

		if {$start != -1} {
			set start [expr [string wordend $local_text [expr $start + 1]] +1]
			set end [expr [string wordstart $local_text $end] -1]
			set text [string range $local_text $start $end]
			set text [string trimleft $text]
		} else {
			Info_Dialog . "Error in help text (${item})!  
All pages must begin with a title e.g. 
<title>Contents</title> or
<TITLE>Contents</TITLE>"
		}
		$list insert end $text
	}

	# Now for a goto, close and contents button
	set frame [frame $window.button_fr]
	pack $frame -side bottom -fill x
	button $frame.close -text Close -command "destroy $window"
	button $frame.new -text New -command "destroy $window; MH_NewHelpPage"
	button $frame.goto -text Goto -command "MH_SelectPageFromList $type"
	bind $list <Double-ButtonRelease-1> "MH_SelectPageFromList $type"
	pack $frame.close $frame.new $frame.goto -side right \
		-padx $DEFAULT_PADDING -pady $DEFAULT_PADDING

	# And now centre it on its parent
	Centre_Dialog $window widget .
}

###############################################################
# The procedure to navigate backwards through the history
proc MH_SelectPageFromList { type } {

   # Globals
   global gas_history gas_pages

	set page [.select_help.list_fr.list curselection]
	if {$page == ""} {
		return
	} else {

		# Type may be HISTORY or FULL
		if {$type == "HISTORY"} {
			set as_list $gas_history
      } else {
         set as_list $gas_pages
      }

      set name [lindex $as_list $page]
      if {$type == "HISTORY"} {set gas_history \
         [lreplace $gas_history $page end]}
      MH_DisplayHelpPage $name
      after 400 destroy .select_help
   }
}

###############################################################
# The procedure to navigate backwards through the history
proc MH_EditNavigateBack {} {

   # Globals
   global gas_history

   # The list gas_history contains a list of all help pages
   # visited.  Simply extract the second to last page, and
   # delete the last item in the list, then dislpay the 
   # page.
   # Info_Dialog . $gas_history
   if {[llength $gas_history] < 2} {
      # Nowhere to return to
      return
   }
   
   # Get the last page
   set help [lindex $gas_history [expr [llength $gas_history] - 2]]

   # Delete the two last items (note that one of them is recreated
   # when we call MH_DisplayHelpPage).
   set gas_history [lreplace $gas_history [expr [llength $gas_history] \
      - 2] end]

   # And show the help page
   MH_DisplayHelpPage $help
}

###############################################################
# Procedure for displaying a help page
proc MH_DisplayHelpPage { text } {

   # Globals
   global gt_mini_help gas_pages gb_current_page_changed \
      gas_history

   # Check for the existence of the page before going
   # ahead...
   if {[info globals $text] == ""} {
      Info_Dialog . "Page does not exist!"
      return
   }

   # Display the associated help text
   # $gt_mini_help configure -state normal
   $gt_mini_help delete 1.0 end
   global $text 

   # Perform substitution of invalid tcl characters
   # e.g. ${}[]"'
   set display_text [eval subst \$$text]
   set charlist [list \$ \{ \} \[ \] \" \' \\ ]
   set replacelist [list \&dollar\; \&lbrace\; \&rbrace\; \&lsquare\; \
      \&rsquare\; \&quot\; \&squot\; \&backslash\; ]
   set count 0
   set char [lindex $charlist $count]
   set repl [lindex $replacelist $count]
   while {$count < [llength $charlist]} {
      # Search and replace
      # Info_Dialog . "Char: $char   Repl: $repl"
      regsub -all $repl $display_text $char display_text
      # Info_Dialog . $display_text

      # Get the next item
      incr count
      set char [lindex $charlist $count]
      set repl [lindex $replacelist $count]
   }

   $gt_mini_help insert end $display_text
   # Parse the help (0=Edit rather than run time)
   MH_ParseHelp 0
   set gb_current_page_changed 0

   # Set the focus
   focus $gt_mini_help

   # Display the page title in the entry
   .name_fr.name configure -state normal
   .name_fr.name delete 0 end
   .name_fr.name insert end [string range $text 12 end]
   .name_fr.name configure -state disabled
   .name_fr.save configure -state disabled

   # Add to the history list (removing any other references to
   # the page on the way - there will be at most one given the
   # nature of the check)
   set item [lsearch -exact $gas_history $text]
   if {$item != -1} {set gas_history [lreplace $gas_history $item $item]}
   lappend gas_history $text
}

###############################################################
# Procedure for displaying a new help page
proc MH_NewHelpPage { } {

   # If the variable keep_old_text is set, then 
   # the text that is currently being displayed
   # is retained.

   # Globals
   global RIDGE_BORDER DEFAULT_PADDING gs_MH_NewHelpPage \
      gt_mini_help gas_pages TEXT_COLOUR gb_keep_old_text \
      gas_history

   # Popup a selection window
   set window .new_page
   if [winfo exists $window] {
       # Pop it up!
       wm deiconify $window
       raise $window
       return
   }

   # Otherwise create the window
   toplevel $window
   wm title $window "New Help Page"
   wm transient $window .

   # Create a frame containing a list and an entry field
   frame $window.sel_fr -borderwidth $RIDGE_BORDER -relief groove
   pack $window.sel_fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill x
   label $window.sel_fr.label -text "Page Name"
   entry $window.sel_fr.entry -width 30 -bg $TEXT_COLOUR -textvariable \
      gs_MH_NewHelpPage
   pack $window.sel_fr.label $window.sel_fr.entry -side left \
      -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING

   # And add buttons to the window
   set frame [frame $window.button_fr]
   pack $frame -side bottom -fill x
   button $frame.close -text Cancel -command "wm withdraw $window"
   button $frame.ok -text OK -command {
      # Check the name for syntactic correctness
      if {[llength [split $gs_MH_NewHelpPage]] > 1 || [regexp -nocase \
         {[`"#;:',.?/\|!$%^&*()|~<>,-]+} $gs_MH_NewHelpPage] ||
         $gs_MH_NewHelpPage == ""} {
         Info_Dialog . "Chosen name is not valid.  Please input 
another name.  It must contain only alphanumeric
characters."
      } else {
         # Create a new global
         set variable "gs_HELPTEXT_$gs_MH_NewHelpPage"
         global $variable
         set $variable ""
         lappend gas_pages $variable
         lappend gas_history $variable

         # Clear the text area
         # $gt_mini_help configure -state normal
         if {$gb_keep_old_text == 0} {
            $gt_mini_help delete 1.0 end
         } else {
            # Reset flag
            set gb_keep_old_text 0
         }

         # Display the page title in the entry
         .name_fr.name configure -state normal
         .name_fr.name delete 0 end
         .name_fr.name insert end $gs_MH_NewHelpPage
         .name_fr.name configure -state disabled

         # Set the focus
         focus $gt_mini_help

         # Page should be saved (project has changed)
         .name_fr.save configure -state normal
         set gb_current_page_changed 1

         # Withdraw this window
         wm withdraw .new_page
      }
   }
   bind $window.sel_fr.entry <Return> "$frame.ok invoke"
   pack $frame.ok $frame.close -side right -padx $DEFAULT_PADDING -pady \
      $DEFAULT_PADDING

   # And now centre it on its parent
   Centre_Dialog $window widget .
   focus $window.sel_fr.entry
}

###############################################################
# The procedure for finding tags in the help
proc MH_ShowHyperMenu { tag } {

   # Globals
   global gt_mini_help

   # Create the menu - Check for existence
   set window .hyper_popup_menu
   if [winfo exists $window] {destroy $window}

   # Find the start and end of the current tag
   set x [ winfo pointerx $gt_mini_help ]
   set y [ winfo pointery $gt_mini_help ]
   set curpos [$gt_mini_help index @$x,$y]
   set ranges [$gt_mini_help tag ranges $tag]
   set list_len [llength $ranges]
   for {set count 0} {$count < $list_len} {incr count 2} {
      if {$curpos >= [lindex $ranges $count]} {
         set start [lindex $ranges $count]
         set end [lindex $ranges [expr $count + 1]]
         #Info_Dialog . "Tag found $start $end"
      }
   }

   # Create the popup menu
   menu .hyper_popup_menu -tearoff 0
      $window add checkbutton -label [string range $tag 12 end] \
         -indicatoron False
      $window add separator
      $window add command -label "Open Link" -command "MH_OpenLink \
         $tag" -underline 0
      $window add command -label "Bind Link" -command "MH_BindLink \
         $tag $start $end" -underline 0

   # Now popup the window in place
   MH_PopupMenu . $window
}

###############################################################
# The procedure for finding tags in the help
proc MH_BindLink { tag start end } {

   # Globals
   global gas_pages RIDGE_BORDER \
      DEFAULT_PADDING TEXT_COLOUR gt_mini_help

   # Test for NULL arguments.  If so, use current
   # selection...
   if {$start == ""} {
      # Info_Dialog . "Using current selection"
      if [catch {$gt_mini_help get sel.first}] {return}
      set start [$gt_mini_help index sel.first]
      set end [$gt_mini_help index sel.last]

      # Ensure that the current selection has not
      # already got a hyperlink tag
      MH_RemoveTags XREF
   }
   if {$tag == ""} {set tag NONE}

   # Test for window
   # Popup a selection window
   set window .bind_hyperlink
   if [winfo exists $window] {destroy $window}
   toplevel $window
   wm title $window "Select Hyperlink"
   wm transient $window .

   # Create a frame containing a list
   frame $window.sel_fr -borderwidth $RIDGE_BORDER -relief groove
   pack $window.sel_fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING \
      -side top -fill x
   set list [ScrolledList $window.list_fr 0 20 browse 0]
   $list configure -bg $TEXT_COLOUR
   pack forget $window.list_fr
   pack $window.list_fr -side top -fill both -padx $DEFAULT_PADDING \
      -pady $DEFAULT_PADDING -expand true

   # Info_Dialog . $gas_pages
   # Add the help pages title to the list
   foreach item $gas_pages {
      # Get the help text title for the page
      set text "Unknown Help Page"
      global $item
      set local_text [eval subst {\$$item}]
      set p1 [string first "<title>" $local_text]
      if {$p1 == -1} {set p1 [string first "<TITLE>" $local_text]}
      set p2 [string first "</title>" $local_text]
      if {$p2 == -1} {set p2 [string first "</TITLE>" $local_text]}

      if {$p1 != -1} {
         set p1 [expr [string wordend $local_text [expr $p1 + 1]] +1]
         set p2 [expr [string wordstart $local_text $p2] -1]
         set text [string range $local_text $p1 $p2]
         set text [string trimleft $text]
      } else {
         Info_Dialog . "Error in help text (${item})!  
All pages must begin with a title e.g. 
<title>Contents</title> or
<TITLE>Contents</TITLE>"
      }
      $list insert end "$text"
   }

   # Now for a goto, close and contents button
   set frame [frame $window.button_fr]
   pack $frame -side bottom -fill x
   button $frame.cancel -text Cancel -command "destroy $window"
   button $frame.bind -text OK -command "MH_CreateLink $tag $start $end"
   pack $frame.cancel $frame.bind -side right \
      -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING

   # And now centre it on its parent
   Centre_Dialog $window widget .
}

###############################################################
# The procedure for opening a tag in the help
proc MH_CreateLink { tag start end } {

   # Globals
   global gt_mini_help gas_pages RED GREEN FONT_BOLD \
      gb_current_page_changed gb_project_changed

   set page [.bind_hyperlink.list_fr.list curselection]
   if {$page == ""} {
      return
   } else {
      set name [lindex $gas_pages $page]
      if {$tag != "NONE"} {
         $gt_mini_help tag remove $tag $start $end
      }
      $gt_mini_help tag add $name $start $end
      $gt_mini_help tag configure $name -foreground \
         $GREEN -font $FONT_BOLD
      $gt_mini_help tag bind $name <Enter> \
         "$gt_mini_help tag configure $name -foreground $RED"
      $gt_mini_help tag bind $name <Leave> \
         "$gt_mini_help tag configure $name -foreground $GREEN"
      $gt_mini_help tag bind $name <ButtonPress> "MH_ShowHyperMenu $name"

      # Page has changed
      set gb_project_changed      1
      set gb_current_page_changed 1
      .name_fr.save configure -state normal

      # Destroy the popup
      destroy .bind_hyperlink
   }
}

###############################################################
# The procedure for opening a tag in the help
proc MH_OpenLink { tag } {

   # Globals
   global gb_current_page_changed gb_page_autocommit

   # Check for changed
   if {$gb_current_page_changed == 0 || $gb_page_autocommit == 1} {
      MH_SaveHelpPage
      MH_DisplayHelpPage $tag
   } else {
      if ![Question_Dialog . "Help text on this page has changed!" \
         "Discard Changes" "Cancel"] {
         MH_DisplayHelpPage $tag
      }
   }
}

###############################################################
# The procedure for finding tags in the help
proc MH_RenamePage { } {

   # Global
   global gb_keep_old_text gas_history gas_pages

   # Popup a warning dialogue
   Info_Dialog . "All references to this page must be regenerated!"

   # All we need do is to call MH_NewHelpPage, asking the
   # routine to retain the current information.  Then
   # we delete the old variable reference.
   set gb_keep_old_text 1
   set name [.name_fr.name get]
   set name "gs_HELPTEXT_$name"
   global $name
   unset $name

   # Delete from history, and delete from list of pages
   set location [lsearch -exact $gas_pages $name]
   if {$location != -1} {
      set gas_pages [lreplace $gas_pages $location $location]
   }
   set location [lsearch -exact $gas_history $name]
   if {$location != -1} {
      set gas_history [lreplace $gas_history $location $location]
   }

   # Create the new page
   MH_NewHelpPage
   # Fill the field with default information, and
   # disable the cancel button
   wm title .new_page "Rename Page"
   .new_page.sel_fr.entry delete 0 end
   .new_page.sel_fr.entry insert end [.name_fr.name get]
   pack forget .new_page.button_fr.close
}

###############################################################
# The procedure for finding tags in the help
proc MH_CreateHelp { } {

   # Globals
   global gt_mini_help search FONT_BOLD FONT_ITALIC \
      FIXED_FONT TEXT_COLOUR

   # Format list
   set format_list [list TITLE BOLD ITALIC FIXED UNDERLINE CENTER \
      LEFT RIGHT]

   # Keyworks
   set TITLE "TITLE"
   set BOLD  "B"
   set ITALIC "I"
   set FIXED "TT"
   set UNDERLINE "U"
   set CENTER "CENTER"
   set LEFT "LEFT"
   set RIGHT "RIGHT"

   # During save of the images, save the image tags so that
   # we don't pick them up during parsing for hyperlinks.
   set tags [$gt_mini_help tag names]
   lappend whole_list sel

   # Now for any embedded images
   set image_list [$gt_mini_help window names]
   foreach image $image_list {

      # Get the location of the displayed image
      set name [$image.name cget -text]
      set tag "<IMG SRC=$name>"
      set window_num [string range $image [expr [string last . \
         $image] +1] end]
      # Info_Dialog . "$image | $tag | $window_num"

      # Get the location of the displayed image
      lappend whole_list image$window_num
      set location [$gt_mini_help tag nextrange image$window_num 1.0]
      if {$location != ""} {
         # Insert the tag (just after the image - this ensures
         # that other formatting instructions are picked up 
         # correctly.
         destroy $image
         $gt_mini_help tag delete image$window_num
         $gt_mini_help delete [lindex $location 0]
         $gt_mini_help insert [lindex $location 1] $tag
      } else {
         Info_Dialog . "Unable to find embedded image location!"
      }
   }

   # Loop round the declared types
   foreach type $format_list {
      lappend whole_list $type
      # Note that location here is a two element list
      set location [$gt_mini_help tag nextrange $type 1.0]
      set format [eval subst \$$type]
      # Info_Dialog . $location
      while {$location != ""} {
         # Replace the tagged area with the formatting
         # instructions
         $gt_mini_help tag remove $type [lindex $location 0] \
            [lindex $location 1]
         $gt_mini_help insert [lindex $location 1] "</$format>"
         $gt_mini_help insert [lindex $location 0] "<$format>"
         set location [$gt_mini_help tag nextrange $type 1.0]
      }
   }

   # Now save the hyperlinks - get a tag list and extract
   # all non-hyperlink tags.
   foreach format $whole_list {
      while {[lsearch -exact $tags $format] != -1} {
         set position [lsearch -exact $tags $format]
         set tags [lreplace $tags $position $position]
      }
   }
   #Info_Dialog . $tags
   foreach type $tags {
      # For each range in question, tag the selection
      set location [$gt_mini_help tag nextrange $type 1.0]
      # Info_Dialog . $location
      while {$location != ""} {
         # Replace the tagged area with the formatting
         # instructions
         $gt_mini_help tag remove $type [lindex $location 0] \
            [lindex $location 1]
         $gt_mini_help insert [lindex $location 1] "</XREF>"
         $gt_mini_help insert [lindex $location 0] "<XREF $type>"
         set location [$gt_mini_help tag nextrange $type 1.0]
      }
   }

   # Disable text by default
   # $gt_mini_help configure -state disabled
}

###############################################################
# The procedure for finding tags in the help
proc MH_SaveHelpPage { } {

   # Globals
   global gt_mini_help gb_current_page_changed \
      gb_show_formatted

   # Create the tagged text
   MH_CreateHelp

   # Now save to the variable
   set name [.name_fr.name get]
   set name "gs_HELPTEXT_$name"
   global $name
   set text [$gt_mini_help get 1.0 end]

   # Perform substitution of invalid tcl characters
   # e.g. ${}[]"'
   set charlist [list \$ \{ \} \[ \] \" \' \\ ]
   set replacelist [list \\\&dollar\; \\\&lbrace\; \\\&rbrace\; \\\&lsquare\; \
      \\\&rsquare\; \\\&quot\; \\\&squot\; \\\&backslash\; ]
   set count 0
   set char [lindex $charlist $count]
   set repl [lindex $replacelist $count]
   while {$count < [llength $charlist]} {
      # Search and replace
      # Info_Dialog . "Char: $char   Repl: $repl"
      regsub -all \\$char $text $repl text
      # Info_Dialog . $text

      # Get the next item
      incr count
      set char [lindex $charlist $count]
      set repl [lindex $replacelist $count]
   }
   set $name $text

   # If show is true, redisplay the text in formatted form.
   # Otherwise, leave unformatted.
   if {$gb_show_formatted == 1} {
      # Now redisplay the text and disable the save button
      # $gt_mini_help configure -state normal
      MH_ParseHelp 0
   }

   # Reset status
   .name_fr.save configure -state disabled
   set gb_current_page_changed 0
}

###############################################################
# The procedure for finding tags in the help
proc MH_DeleteHelpPage { name } {

   # Globals
   global gas_pages gt_mini_help gb_current_page_changed \
      gb_project_changed gas_history
   set page "gs_HELPTEXT_$name"
   global $page
   
   # Ask the user for confirmation
   if ![Question_Dialog . "Do you really want to delete page '$name'?" \
      "Delete" "Cancel"] {
      # Perform deletion.
      unset $page
      set item [lsearch -exact $gas_pages $page]
      set gas_pages [lreplace $gas_pages $item $item]
      set gas_history [lreplace $gas_history $item $item]

      # And blank out the edit areas of the screen
      # $gt_mini_help configure -state normal
      $gt_mini_help delete 1.0 end
      # $gt_mini_help configure -state disabled
      set gb_current_page_changed 0
      set gb_project_changed 1

      # Display the page title in the entry
      .name_fr.name configure -state normal
      .name_fr.name delete 0 end
      .name_fr.name configure -state disabled
      .name_fr.save configure -state disabled
   }
}

###############################################################
# The procedure to goto a specified page from the history
proc MH_CopyText { delete } {

   # Global
   global gt_mini_help gs_cut_buffer

   # Copy the selected text into the cut buffer, then
   # delete if the flag 'delete' is set.
   if [catch {$gt_mini_help get sel.first}] {return}
   set gs_cut_buffer [$gt_mini_help get sel.first sel.last]

   # And delete/allow a save if data deleted
   if {$delete == 1} { 
      $gt_mini_help delete sel.first sel.last
      set gb_project_changed      1
      set gb_current_page_changed 1
      .name_fr.save configure -state normal
   }
}

###############################################################
# The procedure to goto a specified page from the history
proc MH_PasteText { } {

   # Global
   global gt_mini_help gs_cut_buffer

   # Perform paste if data exists in the cut buffer
   if {$gs_cut_buffer != ""} {
      set position [$gt_mini_help index insert]
      $gt_mini_help insert $position $gs_cut_buffer
   }
}

###############################################################
# The procedure to add a tag to the current selection
proc MH_AddTag { type } {

   # Globals
   global gt_mini_help search FONT_BOLD FONT_ITALIC \
      FIXED_FONT TEXT_COLOUR RED GREEN gb_project_changed \
      gb_current_page_changed

   # Format list
   set format_list [list TITLE BOLD ITALIC FIXED UNDERLINE CENTER \
      LEFT RIGHT]

   # If the input argument is not in this list then it is 
   # not a valid type.
   if {[lsearch -exact $format_list $type] == -1} {
      Info_Dialog . "Invalid/unknown tag type specified!"
      return
   }

   # Keyworks and formatting
   set TITLE "TITLE"
   set TITLE_FORMAT "-font $FONT_BOLD -underline True -justify center"
   set BOLD  "B"
   set BOLD_FORMAT "-font $FONT_BOLD"
   set ITALIC "I"
   set ITALIC_FORMAT "-font $FONT_ITALIC"
   set FIXED "TT"
   set FIXED_FORMAT "-font $FIXED_FONT"
   set UNDERLINE "U"
   set UNDERLINE_FORMAT "-underline True"
   set CENTER "CENTER"
   set CENTER_FORMAT "-justify center"
   set LEFT "LEFT"
   set LEFT_FORMAT "-justify left"
   set RIGHT "RIGHT"
   set RIGHT_FORMAT "-justify right"

   # Get the selected text, and format with the 
   # characteristic given, flagging the block with
   # the tag.  Check for a selection first.
   if [catch {$gt_mini_help get sel.first}] {return}

   # Certain tags preclude other tags (e.g. justification)
   if {$type == "CENTER" || $type == "LEFT" || $type == "RIGHT"} {
      MH_RemoveTags JUSTIFY
   }

   # Add the new tag type
   $gt_mini_help tag add $type sel.first sel.last
   set format [eval subst \$${type}_FORMAT]
   eval $gt_mini_help tag configure $type $format
   $gt_mini_help tag raise $type

   # And allow a save
   set gb_project_changed      1
   set gb_current_page_changed 1
   .name_fr.save configure -state normal
}

###############################################################
# The procedure to remove all tags from the current selection
proc MH_RemoveTags { type } {

   # Global
   global gt_mini_help FIXED_FONT gb_project_changed \
      gb_current_page_changed

   # Allowable values for type are:
   # XREF - Hyperlink references
   # CHARACTER - Character/font formatting
   # JUSTIFY - Justification
   # ALL - All formatting

   set as_character [list BOLD UNDERLINE TITLE ITALIC FIXED]
   set as_justify [list CENTER LEFT RIGHT]
   set as_non_xref [list BOLD UNDERLINE TITLE ITALIC FIXED \
      CENTER LEFT RIGHT sel]
   set delete_xref 0
   set al_del_list ""

   # Determine what needs to be done
   switch -exact $type {
      XREF {set delete_xref 1}
      CHARACTER {set al_del_list $as_character}
      JUSTIFY {set al_del_list $as_justify}
      ALL {set al_del_list [list [join $as_character $as_justify]]
         set delete_xref 1}
      default {
         Info_Dialog . "Error: Unrecognised clear command"
         return
      }
   }

   # Get the selected text, and unformat
   if [catch {$gt_mini_help get sel.first}] {return}
   set this [$gt_mini_help index sel.first]
   set last [$gt_mini_help index sel.last]
   while {$this != $last} {

      # Delete selected formatting commands
      foreach tag $al_del_list {
         #Info_Dialog . "Removing $tag from position $this"
         $gt_mini_help tag remove $tag $this
      }
      # If delete_xref selected find the xrefs and delete
      if {$delete_xref == 1} {
         set tags [$gt_mini_help tag names $this]
         foreach tag $as_non_xref {
            set loc [lsearch -exact $tags $tag]
            if {$loc != -1} {set tags [lreplace $tags $loc $loc]}
         }
         foreach xref $tags {
            #Info_Dialog . "Removing xref $tag from position $this"
            $gt_mini_help tag remove $xref $this
         }
      }

      # Increment to the next character
      set this [$gt_mini_help index "$this + 1 chars"]
   }

   # And allow a save
   set gb_project_changed      1
   set gb_current_page_changed 1
   .name_fr.save configure -state normal
}

###############################################################
# Show the main interface
###############################################################

# Source other files
MH_SourceOther global.tcl
MH_SourceOther help.tcl
MH_SourceOther dialog.tcl
MH_SourceOther utils.tcl

###############################################################
# Globals for this module
set g_selected_file ""
set g_selected_file_filter ""

# Flags
set gas_pages ""
set gas_history ""
set gb_current_page_changed 0
set gb_project_changed      0
set gs_current_project      ""
set gs_new_project_name     ""
set gs_MH_NewHelpPage        ""
set gs_cut_buffer           ""
set gb_show_formatted       1
set gb_keep_old_text        0
set gb_page_autocommit      1
set g_image_path            "Relative"
set gs_image_base_dir       ""

# Source global setup

###############################################################
# Set application defaults

# Fonts
option add *font $FONT_NORMAL startupFile

# Highlight
option add *highlightThickness 0

# Colours
option add *background  $DEFAULT_COLOUR startupFile
option add *Text.background  $TEXT_COLOUR startupFile
option add *Entry.background  $TEXT_COLOUR startupFile

###########################################################################
# Create the mini-help screen
wm title . "Edit Help - <no project>"
wm protocol . WM_DELETE_WINDOW "MH_CloseProject 1"

###########################################################################
# Create Menu Bar
frame .mbar -relief raised -bd 2
pack .mbar -side top -fill x

# Create the buttons   
menubutton .mbar.file -text File -underline 0 -menu .mbar.file.menu
menubutton .mbar.edit -text Edit -underline 0 -menu .mbar.edit.menu
menubutton .mbar.format -text Format -underline 0 -menu .mbar.format.menu
menubutton .mbar.image -text Image -underline 0 -menu .mbar.image.menu
menubutton .mbar.justify -text Justify -underline 0 -menu .mbar.justify.menu
menubutton .mbar.page -text Page -underline 0 -menu .mbar.page.menu
menubutton .mbar.navigate -text Navigate -underline 0 -menu .mbar.navigate.menu
menubutton .mbar.options -text Options -underline 0 -menu .mbar.options.menu
pack .mbar.file .mbar.edit .mbar.format .mbar.image .mbar.justify .mbar.page \
   .mbar.navigate .mbar.options -side left

# Create each menu item
menu .mbar.file.menu -tearoff 0
   .mbar.file.menu add command -label "Open" -command \
      "MH_OpenProject" -underline 0 -accelerator "Ctrl-O"
   .mbar.file.menu add command -label "Close" -command \
      "MH_CloseProject 0" -underline 0
   .mbar.file.menu add separator
   .mbar.file.menu add command -label "Save" -command \
      {MH_SaveProject $gs_current_project} -underline 0
   .mbar.file.menu add command -label "Save As..." -command "MH_SaveProjectAs" -underline 0
   .mbar.file.menu add separator
   .mbar.file.menu add command -label "Exit" -command "MH_CloseProject 1" -underline 0
menu .mbar.edit.menu -tearoff 0
   .mbar.edit.menu add command -label "Cut" -command \
      "MH_CopyText 1" -underline 2 -accelerator "Ctrl-X"
   .mbar.edit.menu add command -label "Copy" -command \
      "MH_CopyText 0" -underline 0 -accelerator "Ctrl-C"
   .mbar.edit.menu add command -label "Paste" -command \
      "MH_PasteText" -underline 0 -accelerator "Ctrl-V"
   .mbar.edit.menu add separator
   .mbar.edit.menu add command -label "Remove Link" -command \
      "MH_RemoveTags XREF" -underline 0
   .mbar.edit.menu add command -label "Remove Formatting" -command \
      "MH_RemoveTags CHARACTER" -underline 0
   .mbar.edit.menu add command -label "Remove Justification" -command \
      "MH_RemoveTags JUSTIFY" -underline 0
   .mbar.edit.menu add command -label "Remove All" -command \
      "MH_RemoveTags ALL" -underline 0
menu .mbar.format.menu -tearoff 0
   .mbar.format.menu add command -label Bold -command "MH_AddTag BOLD" \
      -underline 0 -accelerator "Ctrl-B"
   .mbar.format.menu add command -label Italic -command \
      "MH_AddTag ITALIC" -underline 0 -accelerator "Ctrl-I"
   .mbar.format.menu add command -label Underline -command \
      "MH_AddTag UNDERLINE" -underline 0 -accelerator "Ctrl-U"
   .mbar.format.menu add command -label Serif -command \
      "MH_AddTag FIXED" -underline 0 -accelerator "Ctrl-S"
   .mbar.format.menu add separator
   .mbar.format.menu add command -label Title -command \
      "MH_AddTag TITLE" -underline 0 -accelerator "Ctrl-T"
   .mbar.format.menu add command -label Hyperlink... -command \
      {MH_BindLink "" "" ""} -underline 0 -accelerator "Ctrl-L"
menu .mbar.justify.menu -tearoff 0
   .mbar.justify.menu add command -label Centre -command \
      "MH_AddTag CENTER" -underline 0
   .mbar.justify.menu add command -label Left -command \
      "MH_AddTag LEFT" -underline 0
   .mbar.justify.menu add command -label Right -command \
      "MH_AddTag RIGHT" -underline 0
menu .mbar.page.menu -tearoff 0
   .mbar.page.menu add command -label New -command {
      if {$gb_current_page_changed == 1 && $gb_page_autocommit == 1} {
         MH_SaveHelpPage
      }
      if {$gb_current_page_changed == 0} {
         MH_NewHelpPage 
      } else {
         if ![Question_Dialog . "Help text on this page has changed!" \
            "Discard Changes" "Cancel"] {
            MH_NewHelpPage 
         }
      }
   } -underline 0
   .mbar.page.menu add command -label "Rename ..." -command \
      {MH_RenamePage} -underline 0
   .mbar.page.menu add command -label Delete -command {
      set name [.name_fr.name get]
      if {$name != ""} {
         MH_DeleteHelpPage $name
      }
   } -underline 0
   .mbar.page.menu add separator
   .mbar.page.menu add checkbutton -label "Display Formatted" -command {
      if {$gb_show_formatted == 1} {
         MH_ParseHelp 0
      } else {
         MH_CreateHelp
      }
   } -variable gb_show_formatted
menu .mbar.navigate.menu -tearoff 0
   .mbar.navigate.menu add command -label "Previous Page" -command {
      if {$gb_current_page_changed == 1 && $gb_page_autocommit == 1} {
         MH_SaveHelpPage
      }
      if {$gb_current_page_changed == 0} {
         MH_EditNavigateBack
      } else {
         if ![Question_Dialog . "Help text on this page has changed!" \
            "Discard Changes" "Cancel"] {
            MH_EditNavigateBack
         }
      }
   } -underline 0 -accelerator "Ctrl-P"
   .mbar.navigate.menu add command -label "History ..." -command {
      if {$gb_current_page_changed == 1 && $gb_page_autocommit == 1} {
         MH_SaveHelpPage
      }
      if {$gb_current_page_changed == 0} {
         MH_SelectHelpPage HISTORY
      } else {
         if ![Question_Dialog . "Help text on this page has changed!" \
            "Discard Changes" "Cancel"] {
            MH_SelectHelpPage HISTORY
         }
      }
   } -underline 0 -accelerator "Ctrl-H"
   .mbar.navigate.menu add command -label "Select Page ..." -command {
      if {$gb_current_page_changed == 1 && $gb_page_autocommit == 1} {
         MH_SaveHelpPage
      }
      if {$gb_current_page_changed == 0} {
         MH_SelectHelpPage FULL
      } else {
         if ![Question_Dialog . "Help text on this page has changed!" \
            "Discard Changes" "Cancel"] {
            MH_SelectHelpPage FULL
         }
      }
   } -underline 0 -accelerator "Ctrl-G"

menu .mbar.image.menu -tearoff 0
   .mbar.image.menu add command -label "Insert..." -command \
      {set gb_current_page_changed 1; .name_fr.save configure \
      -state normal; MH_InsertImage insert} -underline 0
   .mbar.image.menu add command -label "Change Base Dir..." -command \
      {MH_ImageChangeBase} -underline 0
   .mbar.image.menu add separator
   .mbar.image.menu add radiobutton -label "Absolute Pathnames" \
      -variable g_image_path -value "Absolute"
   .mbar.image.menu add radiobutton -label "Relative Pathnames" \
      -variable g_image_path -value "Relative"

menu .mbar.options.menu -tearoff 0
   .mbar.options.menu add checkbutton -label "Autocommit Pages" \
      -variable gb_page_autocommit

###############################################################
# Create the page label
set frame [frame .name_fr -relief groove -borderwidth $RIDGE_BORDER]
label $frame.label -text "Page Label"
entry $frame.name -width 25 -state disabled
button $frame.save -text Save -command {MH_SaveHelpPage}
$frame.save configure -state disabled -disabledforeground ""
button $frame.open -text "..." -command {
   if {$gb_current_page_changed == 1 && $gb_page_autocommit == 1} {
      MH_SaveHelpPage
   }
   if {$gb_current_page_changed == 0} {
      MH_SelectHelpPage FULL
   } else {
      if ![Question_Dialog . "Help text on this page has changed!" \
         "Discard Changes" "Cancel"] {
         MH_SelectHelpPage FULL
      }
   }
}
pack $frame.label $frame.name $frame.open $frame.save -side left \
   -fill x -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING
pack $frame -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING -fill x

###############################################################
# Create the text widget
frame .fr -borderwidth $RIDGE_BORDER -relief groove
pack .fr -padx $DEFAULT_PADDING -pady $DEFAULT_PADDING -side top \
   -expand true -fill both
set gt_mini_help [ScrolledText .fr.help 70 30 0]
$gt_mini_help configure -font $FONT_NORMAL -bg $TEXT_COLOUR 
# $gt_mini_help configure -state disabled
$gt_mini_help tag bind sel <3> {+MH_PopupMenu $gt_mini_help .mbar.format.menu}
$gt_mini_help tag bind sel <Control-3> {+MH_PopupMenu $gt_mini_help .mbar.edit.menu}
$gt_mini_help tag bind sel <Shift-3> {+MH_PopupMenu $gt_mini_help .mbar.justify.menu}
bind $gt_mini_help <KeyPress> {
   set gb_current_page_changed 1
   set gb_project_changed      1
   $frame.save configure -state normal
}
pack .fr.help -side top -anchor w

# Center on the screen
Centre_Dialog . 

# Pop-up the open dialogue
MH_OpenProject
