 /*
	This is pppcosts, version 0.64.

	Calculates PPP connection phone costs.

        Copyright (c) 1996 Tillmann Steinbrecher.
        May be distributed according to the terms of the GNU
        General Public License version 2. No warranty.

	Please send your comments to tst@bigfoot.com     
*/
#define TITLE "PPPCOSTS"
#define VERSION "0.64"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#ifdef FREEBSD
    #include <sys/sysctl.h>
#endif
#include <syslog.h>
#include <signal.h>
#include "pppcosts.h"
#include "telecom/costs.h"

#ifdef FREEBSD
   #define PPP_PID_FILE "/var/run/tun0.pid"
   #define PPP_STAT_FILE "/status"
#else /* all others ? */
   #define PPP_PID_FILE "/var/run/ppp0.pid"
   #define PPP_STAT_FILE "/stat"
#endif
/* #define PPP_PID_FILE "/var/run/cron.pid" */  /* For testing purposes */

#define TOTAL_FILE   ".pppcosts"
#define LOG_FILE     ".pppcosts.log"

#define verbose 0
#define DEFAULT_DELAY_SECS 1
#define DEFAULT_SYSLOGTIME 300

/* declare functions */
void tst_greet();
void read_file(char[256]);
void write_file(char[256]);
void write_logfile(char[256],float, long, long);
void display_total();
void display_current(long, long, long, float);
void wait_for_ppp(char[256],char[256]);
long int get_starttime(char[256]);
long int get_uptime();
int beep_warn(int,int,int);
int connected(char[256]);
void ppp_online(char[256],char[256]);
int isdn_online();
void sig_term(); /* Signal handler for SIGTERM */
void sig_int();  /* Signal handler for SIGINT */
void currency_str( float amount, char amountstr[32]);
	
/* GLOBAL VARS */
long tottime=0;
long totunits=0;
float totcost;
char newline_chr[4];
char offline_str[40];
long starttime=0;
int  usesyslog=0;      /* Are we using syslog() ? */
int  usefilelog=0;     /* Are we loging stats to file ? */
int  showtitle=1;      /* Print Title ? */
int  delay_secs=DEFAULT_DELAY_SECS;     /* Delay between each log */
int  terminate=0;      /* Was SIGTERM or SIGINT received ? */
char beep=1;
char silent=0;

/*===================================================================*/

int main(int argc, char *argv[])
{ 
  char totfilename[256];
  char logfilename[256];
  char *err,*arg,*myname;
  int  syslogtime=DEFAULT_SYSLOGTIME;

  totunits=0;
  tottime=0;

  sprintf(totfilename,"%s/" TOTAL_FILE, getenv("HOME"));
  sprintf(logfilename,"%s/" LOG_FILE,   getenv("HOME"));

  myname=argv[0];
  while(argc>1 && argv[1][0]=='-') {
      argc--;
      arg=*++argv;
      switch (arg[1]) {
      case 'h':
      case '?':
          fprintf(stderr, "Usage: %s [switches]\n",myname);
          fprintf(stderr, "   -h          shows this help\n");
          fprintf(stderr, "   -f file     use 'file' instead of '~/.pppcosts'\n");
          fprintf(stderr, "   -t time     log every 'time' seconds\n");
          fprintf(stderr, "   -l          use syslog instead of stdout for logging and go to background\n");
	  fprintf(stderr, "   -L file     log connections to 'file'\n");
          fprintf(stderr, "   -n          use new line for each output to stdout\n");
          fprintf(stderr, "   -s          install SIGTERM and SIGINT signal handlers\n");
          fprintf(stderr, "   -T          don't display the title\n");
          fprintf(stderr, "   -B          don't beep\n");
          fprintf(stderr, "   -q          don't write to stdout\n");
				  
          exit(0);
          break;
      case 'f':
          ++argv;
          if (!*argv) {
		  fprintf(stderr, "Warning: no filename specified with '-f' option, using default (~/.pppcosts).\n");
	  } else  strcpy(totfilename,*argv);
          argc--;
          fprintf(stderr, "Using cost/time file: %s\n",totfilename);
          break;
      case 'l':
          usesyslog=1;
          silent=1;
          break;
      case 'L':
	  ++argv;
	  if (!*argv) {
		  fprintf(stderr, "Warning: no filename specified with '-L' option, using default (~/"LOG_FILE").\n");
	  } else strcpy(logfilename,*argv);
	  argc--;
	  fprintf(stderr, "Using log file: %s\n",logfilename);
	  usefilelog=1;
	  break;
      case 'T':
	  showtitle=0;
      case 'n':
	  strcpy(newline_chr,"\n");
   	  strcpy(offline_str,"Offline.\n");
	  break;
      case 't':
          ++argv;
          if (!*argv) {
		  fprintf(stderr, "Warning: no log time specified with '-t' option, using default (%u s).\n", delay_secs);
	  } else {	
              syslogtime=strtol(*argv,&err,10);
              if (*argv && err && !(*err)) {
                  /*
                   * Just make sure user can't specify log time of <= 0. In
                   * that case just use default.
                   */
                  if (syslogtime <= 0) {
		    	syslogtime=DEFAULT_SYSLOGTIME;
			fprintf(stderr, "Warning: log time must be > 0, using default (%u s).\n", delay_secs);
			}
              } else {
                  /*
                   * If there was an error in strtol() syslogtime might be
                   * corrupted so set it back to default. This might be
                   * important because if '-l' option was used we set
                   * delay_secs to syslogtime some 17 lines down !!!!
                   */
                  syslogtime=DEFAULT_SYSLOGTIME;
                  fprintf(stderr, "Second argument to '-t' option should be a number\n");
              }
          }
          argc--;
          break;
      case 's':
          signal(SIGTERM,sig_term);  /* install SIGTERM handler */
          signal(SIGINT,sig_int);    /* install SIGINT handler */
          break;
      case 'B':
          beep=0;
          break;
      case 'q':
	  silent=1;
	  showtitle=0;
	  break;
      default:
          fprintf(stderr, "Unknown option %s\n",arg);
          break;
      }
  }  
  
  if (showtitle)  {
      printf("\n%s %s (c) T. Steinbrecher\n", TITLE, VERSION);
      strcpy(newline_chr,"\r");
  }
  strcpy(offline_str,"Offline.                           \r");

  if ((!silent) && (delay_secs!=DEFAULT_DELAY_SECS)) {
      printf("Logging every %d second%s to %s\n",delay_secs,
             delay_secs==1?"":"s",usesyslog?"syslog":"stdout");
  }

  if(usesyslog) delay_secs=syslogtime;

  tst_greet();                       /* run important checking routine */
  read_file(totfilename);            /* read in the info from the file */
  if(!silent) display_total();       /* display total time/cost so far */
  
  if (usesyslog) {
      openlog("pppcosts",0,LOG_INFO);
      switch (fork()) {
      case -1:
          fprintf(stderr, "Yikes - could not fork() - try again\n");
          exit(0);
          break;
      case 0:
          if(!usesyslog) fprintf(stderr, "Running %s in background\n",myname);
          break;
      default:
          exit(0);
          break;
      }
  }
  for(;!terminate;) wait_for_ppp(totfilename,logfilename); /* loop until cancelled */
  if (!silent) printf("\n\n");

  return 0;
}

/*==============================================================*/

void tst_greet()
{
  /* Notice: These lines here are very important so please don't remove them ;-) */
    struct tm* loctim;
    long bdtime;
     
    bdtime=time(0);
    loctim=localtime(&bdtime);
    if(loctim->tm_mon==3 && loctim->tm_mday==18 && !usesyslog) {
        printf("April 18: Send 'Happy Birthday' to tst@bigfoot.com !\a\n");
        fflush(stdout);
  }
}
                

/*==============================================================*/

void read_file(char totfilename[256])
{ 
  FILE *totfile;
  char tottime_chr[32], totunits_chr[32];

  if(!(totfile=fopen(totfilename, "rt"))) {
    if(!usesyslog) {
      fprintf(stderr,"Warning: cost/time file %s not present\n",totfilename);
    }
  } else {
    fgets(tottime_chr,32,totfile);
    fgets(totunits_chr,32,totfile);
    fclose(totfile);
    tottime=strtol(tottime_chr,NULL,10);
    totunits=strtol(totunits_chr,NULL,10);


    if(PRICE_PER_MIN) {
      /* the cost (float) is stored as a long integer */    
      totcost=totunits / 100.0;
    } else {
      totcost=totunits * COSTS_PER_UNIT;
    }
  }
}  
/*=============================================================*/

void write_file(char totfilename[256])
{
  FILE *totfile;

  if((totfile=fopen(totfilename,"wt"))==0) {
    if (!usesyslog) {
       fprintf(stderr, "Warning: can't save costs/time to %s\n", totfilename );
    }
  } else {
    fprintf(totfile,"%ld\n", tottime);

    if(PRICE_PER_MIN) {
       /* the cost (float) is stored as a long integer */    
       totunits=totcost * 100;
    }

    fprintf(totfile,"%ld\n", totunits);
    fclose(totfile);
  }	
}

/*=============================================================*/

void write_logfile(char logfilename[256], float current_cost,
                   long online_secs, long units)
{
  FILE *logfile;
  time_t start_time, curr_time;
  char   startbuf[64], endbuf[64];
  struct tm *startloct, *endloct;
  
  if((logfile=fopen(logfilename,"at"))==0) {
    if (!usesyslog) {
	    fprintf(stderr, "Warning: can't log costs/time to %s\n", logfilename);
    }
  } else {
    char   totcoststr[32], currcoststr[32];

    curr_time=time(NULL);
    endloct=localtime(&curr_time);
    strftime(endbuf, 64, "%H:%M:%S ", endloct);
    
    start_time = curr_time - online_secs;
    startloct=localtime(&start_time);
    strftime(startbuf, 64, "%a, %d %b %Y  %H:%M:%S - ", startloct);

    currency_str(current_cost,currcoststr);
    currency_str(totcost,totcoststr);
   
#ifndef ISOLOG  
    fprintf(logfile, "%s%s %.2ld:%.2ld:%.2ld  %s  (%.2ld:%.2ld:%.2ld  %s)\n",
            startbuf, endbuf, online_secs / 3600 , (online_secs % 3600) / 60,
	    online_secs % 60, currcoststr, tottime / 3600 ,
	    (tottime % 3600) / 60, tottime % 60, totcoststr);
#else
    strftime(endbuf, 64, "%Y-%m-%d %H:%M:%S", endloct);
    fprintf(logfile,"%s %.2ld:%.2ld:%.2ld  %s  (%.2ld:%.2ld:%.2ld  %s)\n",
	    endbuf, online_secs / 3600 , (online_secs % 3600) / 60, 
            online_secs % 60, currcoststr, tottime / 3600 , 
            (tottime % 3600) / 60, tottime % 60, totcoststr);
#endif

  }
  fclose(logfile);
}

/*==============================================================*/

/* because the total units is actually the total cost we do not want them
   displayed here */

void display_total()
{
  if (!usesyslog) {
     char totcostbuf[32];

     currency_str(totcost, totcostbuf);
     if(PRICE_PER_MIN) {
        printf("Total online: %.2ld:%.2ld:%.2ld (%s)\n",
		tottime / 3600 , (tottime % 3600) / 60,
		tottime % 60, totcostbuf);
     } else {
        printf("Total online: %.2ld:%.2ld:%.2ld (%ld units, %s)\n",
                tottime / 3600 , (tottime % 3600) / 60,
		tottime % 60, totunits, totcostbuf);
     }
  }
}



/*==============================================================*/

void display_current(long online_secs, long next_unit, 
		     long units, float current_cost)
{
   
   if (!silent) {
     char       currcostbuf[32];
     
     currency_str(current_cost, currcostbuf);
     printf("ONLINE: %.2ld:%.2ld:%.2ld (%lu unit%s, %s, next in %.2ld:%.2ld) %s",
       online_secs / 3600 , (online_secs % 3600) / 60, online_secs % 60, units,
       units==1?"":"s", currcostbuf, next_unit / 60,
       next_unit % 60, newline_chr);
     fflush(stdout);
   } else {
     time_t     curr_time;
     char 	currcostbuf[32];
     char       timebuf[64];
     struct tm *loctim;

     currency_str(current_cost, currcostbuf);
     curr_time=time(NULL)+next_unit;
     loctim=localtime(&curr_time);
     strftime(timebuf, 64, "%H:%M:%S", loctim);
     syslog(LOG_INFO, "ONLINE %.2ld:%.2ld:%.2ld (%lu unit%s, %s, next at %s)",
       online_secs / 3600 , (online_secs % 3600) / 60, online_secs % 60,
       units, units==1?"":"s", currcostbuf, timebuf);
   }
}


/*==============================================================*/

void ppp_online(char statfile_name[256], char logfilename[256])
{
  /* THIS PROCEDURE ALTERS GLOBAL VARS */

  long lasttime=0;       /* the time that the current unit ends  */
  long online_secs=0;    /* the number of secs ppp has been online  */
  long next_unit;        /* the number of secs until the next unit  */
  long units;		 /* implement units=MIN_UNITS_PER_CALL ? */
  long time0;
  float current_cost=0;
  char beeped=1;
  long old_units=0;        /* the number of units before the calculation
			    * loop. If costs_per_unit is variable then the
			    * original routine breaks when the call
			    * overlaps pricing zones thus we must keep
			    * track of the costif costs_per_unit is
			    * variable then the original routine breaks
			    * when the call overlaps pricing zones
			    * thus we must keep track of the cost
			    */

#ifdef MIN_UNITS_PER_CALL
  units=MIN_UNITS_PER_CALL;
#else
  units=0;
#endif

  while( !terminate && connected(statfile_name)==1 )
    {
      online_secs=(get_uptime() - starttime);

      /* Be careful when trying to modify this calculation routine,
       * as it is is hard to understand and easy to misunderstand */

      time0=time(0);		/* Minimize time() calls */
      for( ; lasttime<=online_secs;
	  lasttime+=getunitlength(time0-online_secs+lasttime )) units++;
      next_unit=lasttime-online_secs;

      if(PRICE_PER_MIN) {
          /* when the unit number increases - so does the cost
           * this falls over when pppcosts is started when ppp is already
           * online and the current call has crossed a pricing zone */
          if (units > old_units) {
	     current_cost+= COSTS_PER_UNIT * (units - old_units);
	     old_units = units;
	  } 
      } else {
         current_cost=INITIAL_COST + units * COSTS_PER_UNIT;
      }
    
      if (current_cost < MIN_COST)
	display_current(online_secs,next_unit,units,MIN_COST);
      else
	display_current(online_secs,next_unit,units,current_cost);

      if(beep) {
        beeped=beep_warn(units,next_unit,beeped);   /* annoy-a-tronic */
      }
      sleep(delay_secs);
    } /* end while */

  if(verbose && !usesyslog) printf("not-connected");

  if(!silent) printf("\n");
  
#ifdef TELL_CONNECTION_TERMINATED
  if (!silent) printf("Connection terminated.\n");
#endif

  fflush(stdout);

  if(current_cost < MIN_COST) current_cost=MIN_COST;
  totcost+=current_cost;   
  tottime+=online_secs;
  totunits+=units;
  if(!silent) display_total();               /* display the new total */
  if(usefilelog) {
     write_logfile(logfilename,current_cost,online_secs,units);
  }
}

/*===============================================================*/
 
int connected(char statfile_name[256])
{
  /* returns 1 if online 0 if not */

#ifndef ISDN
 struct stat statbuf;
 if((stat(statfile_name, &statbuf) == 0))
 return(1);
#endif
  
#ifdef ISDN
  
char buf[8192],*ptr;
int fd;
int len;

/* This code was stolen from 'isdnbutton'. I was too lazy
   to write this function on my own. */

 if ((fd = open("/dev/isdninfo",O_RDONLY|O_NDELAY)) >= 0) {
    if ((len = read(fd,buf,sizeof(buf)-1)) >= 0) {
      buf[len] = '\000';
      for (ptr = buf; ptr && *ptr; (ptr = strchr(ptr,'\n')) ? ptr++ : 0) {
	if (!strncmp(ptr,"usage:\t",7))
	  for (ptr += 7; *ptr && *ptr != '\n'; ptr++)
	    if (*ptr != '0' && *ptr != ' ') {
	      close(fd);
	      return 1;} } }
	close(fd);
  	 }
    
/* End of stolen code */
 
#endif
		       
  return (0);
  
}

/*===============================================================*/

#ifdef FREEBSD

long int get_uptime()
{
  /* return the number of seconds the system has been running */
  		
  /* Need complete rewrite for FreeBSD */
  		
  int mib[2]; /* Men in black array :) */
  size_t len;
  struct timeval uptime;
  struct timeval daytime;
  long int runtime;
  
  mib[0] = CTL_KERN;
  mib[1] = KERN_BOOTTIME;
  len = sizeof (uptime);
  
  if (sysctl(mib, 2, &uptime, &len, NULL, 0) == 0) {
    if (gettimeofday(&daytime, NULL) == 0) {            
  	runtime = daytime.tv_sec - uptime.tv_sec;
  	if(verbose && !usesyslog) printf("System running for %ld secs \n",runtime);
  	return(runtime);
    }
    else {
  	if (!usesyslog) fprintf(stderr, "\nError: can't get day time");
	exit(1);
    }
  }
  else {
    if (!usesyslog) fprintf(stderr, "\nError: can't get boot time");
    exit(1);
  }
}

#else /* other than BSD */

long int get_uptime()
{
  /* return the number of seconds the system has been running */

  FILE *uptimefile;
  char uptime_chr[64];
  long uptime;

  uptimefile=fopen("/proc/uptime","rt");  /* open uptimefile*/
  if(!uptimefile) { 
	if (!usesyslog) fprintf(stderr, "\nError: can't read uptime file - no /proc fs ?");
	exit(1);
    }
  fgets(uptime_chr,12,uptimefile);        /* get first 12 chars */
  fclose(uptimefile);                     /* close file */
  uptime_chr[index(uptime_chr,'.')-uptime_chr]=0;  
  uptime=strtol(uptime_chr,NULL,10);         /* convert uptime_chr to long int */
  if(verbose && !usesyslog) printf("System running for %ld secs, ",uptime);
  return(uptime);
}

#endif

/*==============================================================*/


int beep_warn(int units,int next, int beeped)     
{
   if(beep_before && (next<=beep_before) ) 
    {
      if(!beeped) 	
	{ 
	     if (!usesyslog) fprintf(stderr,"\a");
	  beeped=1;
	}
    } else beeped=0;	
	  
  /* Maximal cost per session used up - beep once every second */
  
  if(maxsession && (units>maxsession) && !usesyslog) fprintf(stderr, "\a");
  
  /* if(maxsession && (units>maxsession)) fprintf(stderr, "eep"); */ 
  
/* Maximal total cost used up - beep twice every second */
/* not implemented for pric_per_minute systems as the costs
   not the units are stored */


  if(!PRICE_PER_MIN && maxtotal && (units+totunits>maxtotal)
                       && !usesyslog)
	     fprintf(stderr,"\a");
   
  return(beeped); 
  
}

/*---------------------------------------------------------*/

void currency_str( float amount, char amountbuff[32])
{
  char format[7];

  /* return a nicely formatted amount of a currency */
  if(CURRENCY_AFTER_COST) {
    sprintf( format, "%%.%cf%%s", DECIMALS);
    sprintf( amountbuff, format, amount, CURRENCY);
  } else {
    sprintf( format, "%%s%%.%cf", DECIMALS);
    sprintf( amountbuff, format, CURRENCY, amount);
  }
}


/************************************************************

	     Modem-specific functions start here
	     
 ************************************************************/	     


#ifndef ISDN

void wait_for_ppp(char totfilename[256],char logfilename[256])
{  
  /* wait for PPP connection via modem */

  FILE *pidfile;
  char pid_chr[12], statfile_name[64];

  while( !terminate && ! (pidfile = fopen(PPP_PID_FILE,"rt")) )
    { 
      /* while offline */
      if (!silent) {
          printf(offline_str);
          fflush(stdout);
      }
      sleep(1);
    }  
  /* when ONLINE */

  /* get the pid and status file */
  if (pidfile) {                          /* if pidfile==NULL ppp link is
                                             inactive and pppcosts received
                                             SIGTERM or SIGINT - skip the rest */
  fgets(pid_chr,12,pidfile);          /* get pid_chr from pidfile */
  fclose(pidfile);                    /* close pidfile */ 
  pid_chr[strlen(pid_chr)-1]='\0';    /* put a NUL at the end of pid_chr */
  strcpy(statfile_name,"/proc/");     /* statfile = /proc */
  strcat(statfile_name,pid_chr);      /* statfile = /proc/~pid~ */	    
  strcat(statfile_name,PPP_STAT_FILE);      /* statfile = /proc/~pid~/stat */
  
  if(verbose && !usesyslog) printf("Now online - PPP daemon PID: %s\n",pid_chr);
  starttime=get_starttime(statfile_name);   
                                      /* get the time ppp came up */
  ppp_online(statfile_name,logfilename);          /*  */
 }
  write_file(totfilename);            /* write new totals to file */
}

/*==============================================================*/

#ifdef FREEBSD
long int get_starttime(char statfile_name[64])
{  
  /* returns ppp link start time (in secs after system startup) */

  /* Many changes for FreeBSD */
		
  long int starttime;  
  FILE *statfile;
  char stat_chrb[512],*stat_chr;
  
  int count;

  int mib[2];
  size_t len;
  struct timeval uptime; /* TIMEVAL struct */

  if(verbose && !usesyslog) printf("Opening stat file: %s\n", statfile_name);
  statfile=fopen(statfile_name,"rt");     /* open /proc/"pid"/stat read (t?) */

  if(!statfile) {
    if (!usesyslog) fprintf(stderr, "\nError: obsolete pid file, did daemon die?\n");
    exit(0);
  }

  fgets(stat_chrb,511,statfile);           /* copy first 512 from stat into stat_chr */
  fclose(statfile);                       /* close statfile */
  stat_chr=stat_chrb;
  if(verbose && !usesyslog) printf(stat_chr);
  
  /* count MUST be < 7 */
  for(count=0;count<7;count++)
  {  
    stat_chr=index(stat_chr,32)+1;      /* Skip first 7 entries */
    if (verbose) printf ("count : %d - stat_chr : %s\n", count, stat_chr);
  }
    		
  /* we use "," for index */
  stat_chr[index(stat_chr,44)-stat_chr]=0;  
  if(verbose && !usesyslog) printf("After truncating: stat_chr: %s\n", stat_chr);
  starttime=strtol(stat_chr,NULL,10);    /* convert stat_chr into long int starttime */

  mib[0] = CTL_KERN;
  mib[1] = KERN_BOOTTIME;
  len = sizeof (uptime);
  
  if (sysctl(mib, 2, &uptime, &len, NULL, 0) == 0) {
    starttime = starttime - uptime.tv_sec;    
  }
  else {
    if (!usesyslog) fprintf(stderr, "\nError: can't get boot time");
    exit(1);
  }
  
  if(verbose && !usesyslog) printf("ppp daemon running since uptime=%ld secs\n\n",starttime);
  
  #ifdef DIAL_TIME        /* Add the time the modem took to dial */
    starttime+=DIAL_TIME;
  #endif
  
  return(starttime);
}

#else /* all other OS */

long int get_starttime(char statfile_name[64])
{  
  /* returns ppp link start time (in secs after system startup) */

  long int starttime;  
  FILE *statfile;
  char stat_chrb[512],*stat_chr;
  int count;

  if(verbose && !usesyslog) printf("Opening stat file: %s\n", statfile_name);
  statfile=fopen(statfile_name,"rt");     /* open /proc/"pid"/stat read (t?) */

  if(!statfile) {
    if (!usesyslog) fprintf(stderr, "\nError: obsolete pid file, did daemon die?\n");
    exit(0); }

/*  stat_chr=malloc(512); */                  /* allocate 512 for stat_chr */
  fgets(stat_chrb,511,statfile);           /* copy first 512 from stat into stat_chr */
  fclose(statfile);                       /* close statfile */
  stat_chr=stat_chrb;
  if(verbose && !usesyslog) printf(stat_chr);
  for(count=0;count<21;count++)
    {  
      stat_chr=index(stat_chr,32)+1;      /* Skip first 21 entries */ 
    }
  stat_chr[index(stat_chr,32)-stat_chr]=0;  
  if(verbose && !usesyslog) printf("After truncating: ");
  starttime=strtol(stat_chr,NULL,10);    /* convert stat_chr into long int starttime */
  starttime=starttime / 100;               

  if(verbose && !usesyslog) printf("ppp daemon running since uptime=%ld secs\n\n",starttime);
  
  #ifdef DIAL_TIME        /* Add the time the modem took to dial */
   starttime+=DIAL_TIME;
  #endif
  
  return(starttime);
}
#endif /* !FREEBSD */

#endif /* !ISDN */



/*****************************************************************
               
               ISDN - specific functions start here 
               
 *****************************************************************/


#ifdef ISDN


/*---------------------------------------------------------*/

void wait_for_ppp(char totfilename[256],char logfilename[256])
{  
  /* waits for PPP via ISDN */

  long int online_secs, units;

  while( !terminate && ! (connected("")) )
    { 
      /* while offline */
      if (!silent) {
          printf(offline_str);
          fflush(stdout);
      }
      sleep(1);
    }  
  starttime=get_uptime();	      /* remember the time ppp came up */
  ppp_online("",logfilename);                     /*   */                  
  write_file(totfilename);            /* write new totals to file */   
}


#endif

/*---------------------------------------------------------*/

void sig_term()
{
  if (!usesyslog) {
    printf("Received SIGTERM - terminating\n");
    fflush(stdout);
  }
  else
    syslog(LOG_INFO, "Received SIGTERM - terminating");
  terminate=1;
  signal(SIGTERM,sig_term);
}

void sig_int()
{
  if (!usesyslog) {
    printf("Received SIGINT (Ctrl-C) - terminating\n");
    fflush(stdout);
  }
  else
    syslog(LOG_INFO, "Received SIGINT (Ctrl-C) - terminating");
  terminate=1;
  signal(SIGINT,sig_int);
}

