#!/usr/bin/perl
#
# Advanced Firewall System v1.0
#
# by, Preston A. Elder <prez@antisocial.com>
#
use strict;

my ($config, $firewall, $ifconfig, $interface, $stream, $option, $test, $command);
$config = "/etc/ppp/afs.cfg";
$firewall = "/sbin/ipchains";
$ifconfig = "/sbin/ifconfig";
$stream = "input";
$test = 0;

# Sanity checks ...
die "Config file ($config) not found.\n" unless (-f $config);
die "Config file ($config) not readable.\n" unless (-r $config);
die "Firewall package ($firewall) not found.\n" unless (-f $firewall);
die "Firewall package ($firewall) not readable.\n" unless (-r $firewall);
die "Firewall package ($firewall) not readable.\n" unless (-x $firewall);
die "Firewall package ($firewall) not found.\n" unless (-f $ifconfig);
die "Firewall package ($firewall) not readable.\n" unless (-r $ifconfig);
die "Firewall package ($firewall) not readable.\n" unless (-x $ifconfig);
die "You must be root to run this package.\n" unless ($ENV{UID} == 0);

if (@ARGV == 0) {
	print STDERR "Syntax: $0 <+|-><interface>\n";
	print STDERR "\n<+|-> is either + or -, anything else is invalid.\n";
	die "<interface> is the network interface to apply $config to.\n";
} else {
	# Check for interface, and wether we're ADDING or REMOVING
	# the filewalls in the config.
	$interface = $ARGV[0];
	if ($interface =~ /^\+/) {
		$option = "-I";
	} elsif ($interface =~ /^\-/) {
		$option = "-D";
	} else {
		die "NO + or - specified with the interface.  Type $0 for help";
	}
	$interface =~ s/^[\+\-]//;
	my $junk = `$ifconfig $interface`;
	die "Specified interface ($interface) does not exist.\n" unless ($junk);
}

open(CONF,$config);
MAIN: while (<CONF>) {
	# Comments and blank lines
	next MAIN if (/^$/ || /^#/);

	my ($access, $ip, $mask, $established, $log, @protos, @ports);
	my ($c_access, $c_ip, $c_mask, $c_proto, $c_port) = split(/\s+/);

	# Get rule (access) condition - and wether we're logging.
	$log = "-l" if ($c_access =~ /^\+/);
	$c_access =~ s/^\+//;
	if ($c_access =~ /^ACC/i || $c_access =~ /^ALL/i) {
		$access = "ACCEPT";
	} elsif ($c_access =~ /^DEN/i) {
		$access = "DENY";
	} elsif ($c_access =~ /^REJ/i) {
		$access = "REJECT";
	} else {
		print STDERR "Unknown ACCESS TYPE on line $. -- skipped.\n";
		next MAIN;
	}

	# Get IP address and sanity check the bitch.
	my ($ip1, $ip2, $ip3, $ip4) = split(/\./, $c_ip);
	if ($ip1 < 0 || $ip1 > 255 || $ip2 < 0 || $ip2 > 255
		|| $ip3 < 0 || $ip3 > 255 || $ip4 < 0 || $ip4 > 255) {
		print STDERR "Invalid IP specified on line $. -- skipped.\n";
		next MAIN;
	} else {
		$ip = $c_ip;
	}

	# Now get the netmask (or bit) and do the same.
	if ($c_mask =~ /\./) {
		my ($ip1, $ip2, $ip3, $ip4) = split(/\./, $c_mask);
		if ($ip1 < 0 || $ip1 > 255 || $ip2 < 0 || $ip2 > 255
			|| $ip3 < 0 || $ip3 > 255 || $ip4 < 0 || $ip4 > 255) {
			print STDERR "Invalid MASK specified on line $. -- skipped.\n";
			next MAIN;
		} else {
			$mask = $c_mask;
		}
	} else {
		if ($c_mask < 0 || $c_mask > 32) {
			print STDERR "Invalid MASK BIT specified on line $. -- skipped.\n";
			next MAIN;
		} else {
			$mask = $c_mask;
		}
	}

	# Protocol processing -- which is optional.
	if ($c_proto && $c_proto !~ /^#/) {
		my (@t_proto) = split(/:/, $c_proto);
		my $tmp;
		foreach $tmp (@t_proto) {
			# +TCP is a special case (ie. for the --syn flag)
			# The rest are standard.
			if ($tmp =~ /^\+TCP$/i) {
				push(@protos, "tcp");
				$established = "--syn";
			} elsif ($tmp =~ /^TCP$/i) {
				push(@protos, "tcp");
			} elsif ($tmp =~ /^UDP$/i) {
				push(@protos, "udp");
			} elsif ($tmp =~ /^ICMP$/i) {
				push(@protos, "icmp");
			} else {
				print STDERR "Invalid PROTOCOL specified on line $. -- skipped.\n";
				next MAIN;
			}
		}
		# Error out if we use '+' (SYN flag) with another protocol
		if (@protos == 0 || ($established && @protos > 1)) {
			print STDERR "Tag + used with NON-TCP protocol on line $. -- skipped.\n";
			next MAIN;
		}
	}

	# Grab ports and ensure they're valid.
	if ($c_port && $c_port !~ /^#/) {
		my (@t_port) = split(/:/, $c_port);

		# We cant have more than 2 ports
		if (@t_port > 2) {
			print STDERR "Too many ports specified on line $. -- skipped.\n";
		} elsif (@t_port > 0) {
			my $tmp;
			foreach $tmp (@t_port) {
				if ($tmp >= 0 && $tmp <= 65535) {
					push(@ports, $tmp);
				} else {
					print STDERR "Invalid PORT specified on line $. -- skipped.\n";
					next MAIN;
				}
			}
		}
	}

	# More sanity checks ... ever get the feeling we're
	# anally retentive?  Well, maybe just slightly.
	if (grep(/ICMP/i, @protos) && @ports > 1) {
		print STDERR "Multipal ICMP types specified on line $. -- skipped.\n";
		next MAIN;
	}
	if (grep(/ICMP/i, @protos) && @protos > 1 && @ports > 0) {
		print STDERR "Multipal protocols specified (including ICMP) with ports defined on line $. -- skipped.\n";
		next MAIN;
	}

	# Actually execute it -- if $test is 1, then
	# it just prints, not executes.
	if (@protos) {
		my $proto;
		foreach $proto (@protos) {
			if (@ports == 2) {
				$command = "$firewall $option $stream -p $proto $established $log -j $access -i $interface -d $ip/$mask $ports[0]:$ports[1]";
			} elsif (@ports == 1) {
				if ($proto =~ /ICMP/i) {
					$command = "$firewall $option $stream -p $proto $established $log -j $access -i $interface -s 0/0 $ports[0] -d $ip/$mask";
				} else {
					$command = "$firewall $option $stream -p $proto $established $log -j $access -i $interface -d $ip/$mask $ports[0]";
				}
			} else {
				$command = "$firewall $option $stream -p $proto $established $log -j $access -i $interface -d $ip/$mask";
			}
			if ($test) {
				print "$command\n";
			} else {
				system($command);
			}
		}
	} else {
		$command = "$firewall $option $stream -j $access $log -i $interface -d $ip/$mask";
		if ($test) {
			print "$command\n";
		} else {
			system($command);
		}
	}
}
