/*
 * This file is part of ispcost.
 *
 * ispcost is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * ispcost is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ispcost; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Copyright 1994 Dominic Giampaolo
 *   Copyright 1996 Torsten Martinsen
 *
 * $Id: stats.c,v 1.9 1996/11/22 09:11:02 torsten Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <time.h>

#include "ispcost.h"

static int get2d(char *s);

static char *mname[] =
{
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

static char ***ilist, **tlist;
static char *totals;


/* 07:45:19 031196-07:46:45 031196   0:01:26    0.62 */
#define STATLINELEN 52
/* Mar 96 99:59:59 123.45 */
#define TOTALLINELEN 22

typedef struct {
    int dy, mo, yr;
    int hr, mi, sc;
} Date;

typedef struct {
    Date start, stop;
    int relmonth;
    time_t duration;	/* 0 if call did not succeed */
    float cost;
} Call;

static Call * calllist;

char ***
getStatistics(int *np, char ***totalsp)
{
    FILE *fp;
    Call * call;
    int n, i, c, h, m, sec, mon, day, year, curmon, lastmonth;
    char tmp[MAXPATHLEN], *buf, *p, *s, *start, *dest, **items;
    size_t len;
    time_t t_now, *timetotal;
    struct tm *now;
    float f, *costtotal;
    Call ** firstitem;

    /* Read entire logfile into memory */
    if ((fp = fopen(global.logfile, "r")) == NULL) {
	sprintf(tmp, "can't open %s", global.logfile);
	perror(tmp);
	return NULL;
    }
    fseek(fp, 0, SEEK_END);
    len = ftell(fp);
    fseek(fp, 0, SEEK_SET);
    buf = xmalloc(len + 1);
    if (fread(buf, 1, len, fp) != len) {
	sprintf(tmp, "can't read %s", global.logfile);
	perror(tmp);
	fclose(fp);
	return NULL;
    }
    fclose(fp);

    /* Count lines */
    buf[len] = 0;
    for (n = 0, p = buf; *p; ++p)
	if (*p == '\n')
	    ++n;
    calllist = xmalloc(n * sizeof(Call));		/* Individual calls */
    firstitem = xmalloc((NMONTHS+1) * sizeof(Call *));	/* Pointers to first call of each month */
    ++firstitem;					/* Allow index of -1 */
    ilist = xmalloc(NMONTHS * sizeof(char **));		/* Pointers to call details as text */
    tlist = xmalloc((NMONTHS + 2) * sizeof(char **));	/* Pointers to monthly totals */
    totals = xmalloc((NMONTHS + 2) * (TOTALLINELEN + 1)); /* Totals as text */
    for (i = 0; i < NMONTHS; ++i) {
	ilist[i] = NULL;
	firstitem[i] = NULL;
    }
    timetotal = xmalloc((NMONTHS + 1) * sizeof(time_t));
    costtotal = xmalloc((NMONTHS + 1) * sizeof(float));
    for (i = 0; i < (NMONTHS + 1); ++i) {
	timetotal[i] = 0;
	costtotal[i] = 0.0;
    }

    t_now = time(NULL);
    now = localtime(&t_now);
    curmon = (now->tm_year % 100) * 12 + now->tm_mon;

    /*
     * Read lines of the form
     * 07:45:19 031196-07:46:45 031196 0:01:26  0.62
     * and stuff them into the Call struct
     */
    call = calllist;
    for (start = p = buf; *p; ++p)
	if (*p == '\n') {
	    s = start;
	    start = p + 1;
	    h = get2d(s);
	    m = get2d(s + 3);
	    sec = get2d(s + 6);
	    mon = get2d(s + 9) - 1;
	    day = get2d(s + 11);
	    year = get2d(s + 13);
	    call->start.hr = h;
	    call->start.mi = m;
	    call->start.sc = sec;
	    call->start.mo = mon;
	    call->start.dy = day;
	    call->start.yr = year;
	    call->relmonth = curmon - (year*12 + mon);
	    c = s[15];
	    s += 16;
	    if (c == '-') {	/* Completed call */
		h = get2d(s);
		m = get2d(s + 3);
		sec = get2d(s + 6);
		mon = get2d(s + 9) - 1;
		day = get2d(s + 11);
		year = get2d(s + 13);
		call->stop.hr = h;
		call->stop.mi = m;
		call->stop.sc = sec;
		call->stop.mo = mon;
		call->stop.dy = day;
		call->stop.yr = year;
		s += 15;
		sscanf(s, "%d:%d:%d", &h, &m, &sec);
		call->duration = h * 3600 + m * 60 + sec;
		while (*s && (*s != ':'))
		    ++s;
		s += 6;
	    } else
		call->duration = 0;
	    sscanf(s, "%f", &f);
	    call->cost = f;
	    ++call;
	}
#if 0
    call = calllist;
    for (i = 0; i < n; ++i) {
	printf("%02d:%02d:%02d %02d%02d%02d",
	       call->start.hr, call->start.mi, call->start.sc,
	       call->start.mo+1, call->start.dy, call->start.yr);
	if (call->duration != 0)
	    printf("-%02d:%02d:%02d %02d%02d%02d"
		   " %ld %f",
		   call->stop.hr, call->stop.mi, call->stop.sc,
		   call->stop.mo+1, call->stop.dy, call->stop.yr,
		   call->duration, call->cost);
	else printf(" The call did not succeed %f",
		   call->cost);
	printf(" %d\n", call->relmonth);
	++call;
    }
#endif

    /* Compute total for each month and grand total */
    call = calllist;
    for (i = 0; i < n; ++i) {
	if (call->relmonth < NMONTHS) {
	    timetotal[call->relmonth] += call->duration;
	    costtotal[call->relmonth] += call->cost;
	    timetotal[NMONTHS] += call->duration;	/* Grand total */
	    costtotal[NMONTHS] += call->cost;
	}
	++call;
    }

    /* Format totals */
    for (i = 0, dest = totals; i < NMONTHS; ++i) {
	tlist[i] = dest;
	mon = curmon - i;
	year = mon / 12;
	mon -= year * 12;
	sec = timetotal[i];
	m = sec / 60;
	sec -= m * 60;
	h = m / 60;
	m -= h * 60;
	f = costtotal[i];
	sprintf(tmp, "%.0f", f);
	sprintf(dest, "%s%02d %2d:%02d:%02d %s%.2f",
	      mname[mon], year % 100, h, m, sec, "   " + strlen(tmp), f);
	dest += TOTALLINELEN;
	*dest++ = 0;
    }
    tlist[NMONTHS] = dest;
    sec = timetotal[NMONTHS];
    m = sec / 60;
    sec -= m * 60;
    h = m / 60;
    m -= h * 60;
    f = costtotal[NMONTHS];
    sprintf(tmp, "%.0f", f);
    sprintf(dest, "Total %2d:%02d:%02d %s%.2f",
	    h, m, sec, "   " + strlen(tmp), f);
    dest += TOTALLINELEN;
    *dest++ = 0;
    tlist[NMONTHS+1] = NULL;
    *totalsp = tlist;
    *np = n;

    /* For each month, store a pointer to the first call for that month */
    lastmonth = -1;
    call = calllist;
    for (i = 0; i < n; ++i) {
	if (call->relmonth < NMONTHS) {
	    if (call->relmonth != lastmonth) {
		firstitem[call->relmonth] = call;
		if (lastmonth >= 0)
		    for (m = call->relmonth+1; firstitem[m] == NULL; ++m)
			firstitem[m] = call;
		lastmonth = call->relmonth;
	    }
	}
	++call;
    }
    firstitem[-1] = call;

    /* Format items for each month */
    for (i = 0; i < NMONTHS; ++i) {
	if ((firstitem[i] == NULL) || (firstitem[i-1] == firstitem[i]))
	    ilist[i] = NULL;
	else {
	    n = firstitem[i-1] - firstitem[i];
	    items = xmalloc((n+1)* sizeof(char *));
	    dest = xmalloc(n * (STATLINELEN + 1));
	    ilist[i] = items;
	    for (call = firstitem[i]; call < firstitem[i-1]; ++call) {
		*items++ = dest;
		sprintf(dest, "%02d:%02d:%02d %02d%02d%02d%c",
			call->start.hr, call->start.mi, call->start.sc,
			call->start.mo+1, call->start.dy, call->start.yr, 0);
		dest += strlen(dest);
		if (call->duration != 0) {
		    sec = call->duration;
		    h = sec/3600;
		    sec -= h*3600;
		    m = sec/60;
		    sec -= m*60;
		    sprintf(dest, "-%02d:%02d:%02d %02d%02d%02d "
			    "%d:%02d:%02d  %c",
			    call->stop.hr, call->stop.mi, call->stop.sc,
			    call->stop.mo+1, call->stop.dy, call->stop.yr,
			    h, m, sec, 0);
		} else sprintf(dest, " The call did not succeed %c", 0);
		dest += strlen(dest);
		sprintf(tmp, "%.0f", call->cost);
		sprintf(dest, "%s%.2f", "   " + strlen(tmp), call->cost);
		dest += strlen(dest)+1;
	    }
	    *items = NULL;
	}
    }

    free(buf);
    free(costtotal);
    free(timetotal);
    return ilist;
}

static int
get2d(char *s)
{
    int d = 0;

    if (*s != ' ')
	d = *s - '0';
    return *++s - '0' + 10 * d;
}

void
stat_free(void)
{
    int i;
    
    for (i = 0; i < NMONTHS; ++i)
	if (ilist[i] != NULL) {
	    free(ilist[i][0]);
	    free(ilist[i]);
	}
    free(ilist);
    free(tlist);
    free(totals);
}
