/*
 * This file is part of ispcost.
 *
 * ispcost is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * ispcost is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ispcost; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright 1996 Torsten Martinsen
 *
 * $Id: readrc.c,v 1.10 1996/11/22 10:31:08 torsten Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>

#include "ispcost.h"
#include "readrc.h"

GlobalOptions global = {
    60,  /* redialinterval */  
    60,  /* connecttimeout */  
     0,  /* redial   */   
   0.0,  /* dialcost */
   NULL, /* logfile  */
   NULL, /* number   */
   "",   /* country */
   "",   /* currency */
   NULL  /* alarmsound */
};

static void parseint(char *s, void *arg);
static void parseflt(char *s, void *arg);
static void parsestr(char * s, void * arg);
static void parsestrte(char * s, void * arg);
static char *expand_tilde(char *path);
static PrefixInfo * currentPrefixInfo(void);
static void parseday(char *s, void * junk);
static void parsenum(char *s, void * junk);
static void parsepre(char *s, void * junk);
static void parsedrt(char * s, void * junk);
static void parsenrt(char * s, void * junk);
static void parsetime(char *s, void * junk);
static void parselog(char *s);
static void parsecty(char * s, void * arg);
static void parseccy(char * s, void * arg);
static void parsedc(char * s, void * junk);
static void rcerror(char * reason);

static int lineno;
static PrefixInfo * prefixinfocur, * prefixinfohead = NULL;
static char ** numberlist = NULL;
static char * rcfile;

/* These must be in the same order that struct tm uses */
static char * weekdays[] = {
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};


static Keywords keywords[] = {
    /* Global parameters */
    { "redialinterval",	parseint,	&global.redialinterval },
    { "connecttimeout",	parseint,	&global.connecttimeout },
    { "logfile",	parsestrte,	&global.logfile },
    { "country",	parsecty,	&global.country },
    { "currency",	parseccy,	&global.currency },
    { "alarmsound",	parsestr,	&global.alarmsound },
    /* Per-number parameters */
    { "dialcost",	parsedc,	NULL  },
    { "number",		parsenum,	NULL  },
    { "prefix",		parsepre,	NULL  },
    { "nightrate",	parsenrt,	NULL  },
    { "dayrate",	parsedrt,	NULL  },
    { "nighttime",	parsetime,	NULL  },
    { "cheapday",	parseday,	NULL  },
    { NULL, NULL, NULL }
};

/*
 * Try to parse specified RC file, returning 1 if file is found.
 */
int
parserc(char * file)
{
    char * fname, buf[200], * p, * cur, * w;
    FILE * fp;
    Keywords * kp;

    rcfile = file;
    fname = expand_tilde(rcfile);
    if ((fp = fopen(fname, "r")) == NULL)
	return 0;
  
    /* Read each line, skip empty or comment lines, and parse the line. */
    lineno = 0;
    while (fgets(buf, sizeof(buf)-1, fp)) {
	++lineno;
	p = buf;
	while (*p && isspace(*p))
	    ++p;
	if ((*p == '#') || (*p == '\n') || (*p == '\0'))
	    continue;
	for (kp = keywords; kp->name != NULL; ++kp) {
	    cur = kp->name;
	    w = p;
	    while (*cur && *w && (*cur == *w)) {
		++cur;
		++w;
	    }
	    if ((*cur == 0) && isspace(*w)) {
		while (*w && isspace(*w))
		    ++w;
		kp->func(w, kp->data);
		break;
	    }
	}
	if (kp->name == NULL)
	    rcerror("unknown keyword");
    }
    fclose(fp);

    /* Hack to get leading tilde expanded */
    if (global.logfile == NULL)
	parselog(DEFAULTLOGFILE);
    return 1;
}

/*
 * Generic parse routines
 */
static void
parseint(char * s, void * arg)
{
    if (sscanf(s, "%d", (int *) arg) != 1)
	rcerror("invalid integer");
}

static void
parseflt(char * s, void * arg)
{
    if (sscanf(s, "%f", (float *) arg) != 1)
	rcerror("invalid floating point number");
}

/*
 * Parse a string.
 * The string is allocated by strdup(), and must therefore be free()d after use.
 */
static void
parsestr(char * s, void * arg)
{
    char temp[200];
    
    if (sscanf(s, "%199s", temp) == 1)
	*((char **) arg) = strdup(temp);
    else
	*((char **) arg) = strdup("");
}

/*
 * Parse a string, expanding any leading tilde.
 * The string is allocated by strdup(), and must therefore be free()d after use.
 */
static void
parsestrte(char * s, void * arg)
{
    char temp[200];
    
    if (sscanf(s, "%199s", temp) == 1)
	*((char **) arg) = strdup(expand_tilde(temp));
    else
	*((char **) arg) = strdup("");
}

/*
 * Keyword-specific parse routines.
 */

static void
parselog(char * s)
{
    char * temp;
    
    parsestr(s, &temp);
    global.logfile = strdup(expand_tilde(temp));
    free(temp);
}

static void
parsedrt(char * s, void * junk)
{
    parseflt(s, &(currentPrefixInfo()->dayrate));
}

static void
parsenrt(char * s, void * junk)
{
    parseflt(s, &(currentPrefixInfo()->nightrate));
}

static void
parsecty(char * s, void * arg)
{
    parsestr(s, (char *) arg);
}

static void
parseccy(char * s, void * arg)
{
    parsestr(s, (char *) arg);
}

static void
parsedc(char * s, void * junk)
{
    PrefixInfo * nip;

    if (prefixinfohead == NULL)
	parseflt(s, &global.dialcost);
    else {
	nip = currentPrefixInfo();
	parseflt(s, &nip->dialcost);
    }
}

/*
 * Read a time specification of the form hh:mm-hh:mm.
 */
static void
parsetime(char * s, void * junk)
{
    int n, sh, sm, eh, em;
    Whennight * new, * cur, * prev;
    PrefixInfo * nip;
  
    n = sscanf(s, "%d:%d-%d:%d", &sh, &sm, &eh, &em);
    if ((n != 4) ||
	(sh < 0) || (sh > 23) || (sm < 0) || (sm > 59) ||
	(eh < 0) || (eh > 23) || (em < 0) || (em > 59) ||
	(sh*60+sm > eh*60+em)) {
	rcerror("invalid time specification");
	return;
    }
    nip = currentPrefixInfo();
    new = xmalloc(sizeof(Whennight));
    new->next = NULL;
    new->s_hour = sh;
    new->s_min = sm;
    new->e_hour = eh;
    new->e_min = em;
    if (nip->whennight == NULL)
	nip->whennight = new;
    else {
	prev = cur = nip->whennight;
	while (cur != NULL) {
	    prev = cur;
	    cur = cur->next;
	}
	prev->next = new;
    }
}

static void
parseday(char * s, void * junk)
{
    int i;
    PrefixInfo * nip;
  
    nip = currentPrefixInfo();
    if (!strncasecmp(s, "hol", 3)) {
	nip->cheapholidays = 1;
	return;
    }
    for (i = 0; i < 7; ++i)
	if (!strncasecmp(s, weekdays[i], 3)) {
	    nip->cheapdays[i] = 1;
	    return;
	}
    rcerror("unrecognized day");
}


/*
 * Parse a phone number prefix and initialize a PrefixInfo structure
 * in which to store the following info.
 * Insert the PrefixInfo struct in the linked list so that the longest
 * prefixes come first.
 */
static void
parsepre(char * s, void * junk)
{
    char * prefix;
    PrefixInfo * new, * nip, * prev;
    int i;
  
    parsestr(s, (void *) &prefix);
    new = xmalloc(sizeof(PrefixInfo));
    new->prefix = prefix;
    new->dayrate = 0.0;
    new->nightrate = 0.0;
    new->whennight = NULL;
    for (i = 0; i < 7; ++i)
	new->cheapdays[i] = 0;
    new->cheapholidays = 0;
    new->dialcost = global.dialcost;
    if (prefixinfohead == NULL) {
	new->next = NULL;
	prefixinfohead = new;
    } else {
	i = strlen(prefix);
	nip = prev = prefixinfohead;
	if (strlen(nip->prefix) <= i) {
	    new->next = prefixinfohead;	/* Insert at list head */
	    prefixinfohead = new;
	} else {			/* Insert in middle of list */
	    while (nip && (strlen(nip->prefix) > i)) {
		prev = nip;
		nip = nip->next;
	    }
	    if (nip == NULL)
		nip = prev;
	    new->next = nip->next;
	    nip->next = new;
	}
    }
    prefixinfocur = new;
}

/*
 * Parse a phone number and store it in the list.
 */
static void
parsenum(char * s, void * junk)
{
    char * num;
    char ** p;
    int n;
    
    parsestr(s, (void *) &num);
    
    if (numberlist == NULL)
	n = 0;
    else
	for (n = 0, p = numberlist; *p != NULL; ++p)
	    ++n;
    numberlist = realloc(numberlist, (n+2)*sizeof(char **));
    numberlist[n++] = num;
    numberlist[n] = NULL;
}

static PrefixInfo *
currentPrefixInfo(void)
{
    int i;
  
    if (prefixinfohead == NULL) {
	prefixinfohead = xmalloc(sizeof(PrefixInfo));
	prefixinfohead->prefix = "";
	prefixinfohead->dayrate = 0.0;
	prefixinfohead->nightrate = 0.0;
	prefixinfohead->whennight = NULL;
	for (i = 0; i < 7; ++i)
	    prefixinfohead->cheapdays[i] = 0;
	prefixinfohead->cheapholidays = 0;
	prefixinfohead->dialcost = global.dialcost;
	prefixinfohead->next = NULL;
	prefixinfocur = prefixinfohead;
    }
    return prefixinfocur;
}  

/*
 * Expand leading tilde in path name.
 */
static char *
expand_tilde(char *path)
{
    static char out[MAXPATHLEN];
    char name[80];
    char *pp, *cp;
    struct passwd *pw;

    if (*path != '~')
	return path;
    ++path;
    pp = path;
    cp = name;
    while (*pp && (*pp != '/'))
	*cp++ = *pp++;
    *cp = 0;
    if (name[0] == '\0')
	pw = getpwuid(getuid());
    else
	pw = getpwnam(name);
    if (pw == NULL)
	return path;
    strcpy(out, pw->pw_dir);
    strcat(out, pp);
    
    return out;
}

static void
rcerror(char * reason)
{
    fprintf(stderr, "error parsing '%s': %s in line %d\n",
	    rcfile, reason, lineno);
    exit(1);
}

static PrefixInfo fallback = {
  "", 0.0, 0.0, 0.0, NULL, { 0, 0, 0, 0, 0, 0, 0 }, 0, NULL
};

PrefixInfo *
getPrefixInfo(char * number)
{
    PrefixInfo * nip = prefixinfohead;
    char * p, * s;
    
    while (nip != NULL) {
	p = nip->prefix;
	s = number;
	while (*s && *p && (*s == *p)) {
	    ++s;
	    ++p;
	}
	if (*p == 0)
	    return nip;
	nip = nip->next;
    }
    return &fallback;
}

/*
 * Get the NULL-terminated list of phone numbers.
 */
char **
getNumberList(int * np)
{
  int n;
  char ** p;
  
  for (n = 0, p = numberlist; *p != NULL; ++p)
      ++n;
  if (np)
      *np = n;
  return numberlist;
}
