/*
 * This file is part of ispcost.
 *
 * ispcost is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * ispcost is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ispcost; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright 1996 Torsten Martinsen
 *
 * The algorithms herein are derived from
 * _Calendrical_Calculations_, by Nachum Dershowitz and Edward M. Reingold,
 * Software -- Practice and Experience, 20(9), 899-928, September 1990.
 *
 * See also the source for the GNU Emacs 'calendar' package.
 *
 * $Id: holidays.c,v 1.3 1996/08/30 11:05:09 torsten Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "ispcost.h"

static int greg2abs(int m, int d, int y);
static int last_day_of_month(int m, int y);
static int kday_on_or_before(int d, int k);
static int easter(int y);

typedef struct
{
  int day;
  int month;
} hday;

/*
 * Convert Gregorian date m/d/y to absolute day.
 */
static int
greg2abs(int m, int d, int y)
{
    int a, i;

    a = d;
    for (i = 1; i < m; ++i)
	a += last_day_of_month(i, y);
    --y;
    a += 365 * y + y / 4 - y / 100 + y / 400;
    return a;
}

/*
 * Return last day of month 'm' in Gregorian year 'y'.
 */
static int
last_day_of_month(int m, int y)
{
    static short days[] =
    {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
    int yy = y % 400;

    if ((m == 2) && (y % 4 == 0) && (yy != 100) && (yy != 200) && (yy != 300))
	return 29;
    return days[m - 1];
}

static int
kday_on_or_before(int d, int k)
{
    return d - ((d - k) % 7);
}

/*
 * Return absolute date of Easter in Gregorian year y.
 */
static int
easter(int y)
{
    int p;
    int c = 1 + y / 100;
    int e = 14 + 11 * (y % 19) - (3 * c) / 4 + (5 + (8 * c)) / 25 + 30 * c;

    e = e % 30;
    if ((e == 0) || ((e == 1) && (10 < (y % 19))))
	++e;
    p = greg2abs(4, 19, y) - e;

    return kday_on_or_before(p + 7, 0);
}


static hday holidays[] =
{
    { 1,  1},			/* New Year's Day */
    {24, 12},			/* Christmas Eve */
    {25, 12},			/* Christmas Day */
    {26, 12},
    {-1, -1}
};

/*
 * Table of Easter-related holidays which do not fall on a Sunday.
 */
static short easterhols[] =
{
    -48,	/* Shrove Monday */
    -47,	/* Shrove Tuesday */
    -46,	/* Ash Wednesday */
     -3,	/* Maundy Thursday */
     -2,	/* Good Friday */
      1,	/* Easter Monday */
     39,	/* Ascension Day */
     50,	/* Whitmundy */
      0
};

static short da_easterhols[] = {
     26,	/* Denmark: Store Bededag */
     0
};

typedef struct {
    char * country;
    short * hols;
} NEhols;

static NEhols national_easterhols[] = {
    { "da", da_easterhols },
    { NULL, NULL }
};


/*
 * Return 1 if the specified day is a holiday, i.e. either
 * - Sunday
 * - Fixed holiday, like Christmas
 * - related to Easter in some way
 * Only does the calculations once for each day.
 */
int
isholiday(struct tm *lt)
{
    hday *hd = holidays;
    int d, m, y;
    short ed, * p;
    static int lastmday = -1, lastresult;
    NEhols * nh;
    
    if (lastmday == lt->tm_mday)
	return lastresult;
    lastmday = lt->tm_mday;

    d = lt->tm_mday;
    m = lt->tm_mon + 1;
    y = lt->tm_year + 1900;

    /* Sundays */
    if (lt->tm_wday == 0) {
	lastresult = 1;
	return 1;
    }

    /* Fixed holidays */
    while (hd->month >= 0) {
	if ((hd->month == m) && (hd->day == d)) {
	    lastresult = 1;
	    return 1;
	}
	++hd;
    }

    /* Easter */
    ed = greg2abs(m, d, y) - easter(y);
    for (p = easterhols; *p; ++p)
	if (*p == ed) {
	    lastresult = 1;
	    return 1;
	}
    nh = national_easterhols;
    while (nh->country) {
	if (!strcmp(nh->country, global.country)) {
	    for (p = nh->hols; *p; ++p)
		if (*p == ed) {
		    lastresult = 1;
		    return 1;
		}
	    break;
	}
	++nh;
    }
    lastresult = 0;
    return 0;
}
