/*
 * This file is part of ispcost.
 *
 * ispcost is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * ispcost is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ispcost; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Copyright 1996 Torsten Martinsen
 *
 * $Id: cost.c,v 1.4 1996/11/20 07:34:34 torsten Exp $
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/param.h>
#include <pwd.h>
#include <sys/types.h>

#include "ispcost.h"
#include "readrc.h"

PrefixInfo * curpreinfo;

static time_t starttime, stoptime;
static int running;
static float cost;
static int inited = 0;
static int callstarted;

static char clockstring[10], coststring[10], ratestring[10];

static float getrate(time_t now);

/*
 * Called every second.
 * If we are not connected, just update rate.
 * Otherwise, also update time and cost.
 */
void
clocktick(XtPointer junk, XtIntervalId * id)
{
  time_t now = time(NULL);
  time_t diff = now-starttime, temp;
  int h, m, s, ratechanged = 0, costchanged = 0, timechanged;
  static float rate = 0.0;
  static time_t lastupdate;
  float t;
  
  h = diff/3600;		/* Number of hours */
  temp = diff - h*3600;		/* Remaining seconds */
  m = temp/60;			/* Number of minutes */
  s = temp - m*60;		/* Number of seconds */

  timechanged = running;
  if (!inited)
    {
      strcpy(clockstring, "0:00:00");
      rate = getrate(now);
      ratechanged = 1;
      costchanged = 1;
      lastupdate = now;
      inited = 1;
      timechanged = 1;

    }
  if ((t = now-lastupdate) >= COSTINTERVAL)
    {
      float oldrate = rate;
      
      rate = getrate(now);
      ratechanged = (rate != oldrate);
      if (running)
	{
	  cost += rate*t/60.0;
	  costchanged = 1;
	}
      lastupdate = now;
    }

  if (running)
    sprintf(clockstring, "%2d:%02d:%02d", h, m, s);
  if (costchanged)
    sprintf(coststring, "%.2f", cost);
  if (ratechanged)
    sprintf(ratestring, "%.2f", rate);
  set_display(clockstring, coststring, ratestring);
  update_display(timechanged, costchanged, ratechanged);
  
  XtAppAddTimeOut(mycontext, TICKINTERVAL, clocktick, NULL);
}

void
startclock(void)
{
  starttime = time(NULL);
  running = 1;
  inited = 0;
}

void
stopclock(void)
{
  stoptime = time(NULL);
  running = 0;
}

void
add_dial_cost(void)
{
  if (curpreinfo == NULL)
    curpreinfo = getPrefixInfo(*getNumberList(NULL));
  
  cost = curpreinfo->dialcost;
  callstarted = 1;
  starttime = time(NULL);
  stoptime = 0;
}

static float
getrate(time_t now)
{
  struct tm * local;
  float hi, lo, rate;
  Whennight * wp;
  int s, e, n;
  
  local = localtime(&now);

  if (curpreinfo == NULL)
    curpreinfo = getPrefixInfo(*getNumberList(NULL));
  
  lo = curpreinfo->nightrate;
  hi = curpreinfo->dayrate;

  rate = hi;

  if (curpreinfo->cheapdays[local->tm_wday])
    rate = lo;
  else if (isholiday(local))
    rate = lo;
  else
    {
      n = local->tm_hour*60+local->tm_min;
      for (wp = curpreinfo->whennight; wp != NULL; wp = wp->next)
	{
	  s = wp->s_hour*60+wp->s_min;
	  e = wp->e_hour*60+wp->e_min;
	  if ((n >= s) && (n < e))
	    return lo;
	}
    }

  return rate;
}
  
/*
 * Append a line describing the just-ended session to the log file.
 */
void
log_costs(void)
{
  FILE * fp;
  time_t diff;
  int h, m, s;
  struct tm * local;
  


  if (!callstarted)
    return;
  
  if ((fp = fopen(global.logfile, "a")) == NULL)
    {
      char buf[MAXPATHLEN];

      sprintf(buf, "can't open %s", global.logfile);
      perror(buf);
      return;
    }

#ifdef TEST
  if (testing)
    fprintf(fp, "TEST ");
#endif

  local = localtime(&starttime);
  fprintf(fp, "%02d:%02d:%02d %02d%02d%02d",
	  local->tm_hour, local->tm_min, local->tm_sec,
	  local->tm_mon+1, local->tm_mday, local->tm_year % 100);
  
  if (stoptime)
    {
      /* Elapsed time in h/m/s */
      diff = stoptime-starttime;
      h = diff/3600;
      diff -= h*3600;
      m = diff/60;
      s = diff - m*60;

      local = localtime(&stoptime);
	  
      fprintf(fp, "-%02d:%02d:%02d %02d%02d%02d %d:%02d:%02d %.3f\n",
	      local->tm_hour, local->tm_min, local->tm_sec,
	      local->tm_mon+1, local->tm_mday, local->tm_year % 100,
	      h, m, s, cost);
    }
  else if (global.dialcost != 0.0)
    /* Call did not get through, log cost of dialling */
    fprintf(fp, "                            %.3f\n", cost);
	  
  fclose(fp);
  callstarted = 0;
}
