/*
 * This file is part of ispcost.
 *
 * ispcost is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * ispcost is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ispcost; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Copyright 1996 Torsten Martinsen
 *
 * $Id: connect.c,v 1.11 1996/11/21 17:27:33 torsten Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>

#include "ispcost.h"
#include "readrc.h"

static pid_t child_pid;
static int aborting = 0;

static void checklink(XtPointer timearg, XtIntervalId * id);
static void do_wait(XtPointer func, XtIntervalId * id);
static void redial(XtPointer junk, XtIntervalId * id);
static void do_redial(XtPointer junk, XtIntervalId * id);
static void checkdown(XtPointer first, XtIntervalId * id);

/*
 * Start the PPP daemon and set up a routine to wait until the link is up.
 */
void
doconnect()
{
    pid_t pid;
    char command[80];

    cstate = CSTATE_TRYING;
    sa_printf("Dialing %s", global.number);

#ifdef TEST
    if (!testing) {
#endif
	pid = fork();
	if (!pid) {
	    sprintf(command, "%s %s", CONNECTSCRIPT, global.number);
	    execl(PPPD, PPPD, "connect", command, NULL);
	    printf("execl() failed\n");
	} else if (pid == -1)
	    printf("fork() failed\n");
	else
	    child_pid = pid;

	XtVaSetValues(connectb, XtNbitmap, abortbitmap, XtNsensitive, 1, NULL);
	aborting = 0;

	/*
	 * Wait a realistic amount of time before calling checklink()
	 * the first time.
	 */
	XtAppAddTimeOut(mycontext, 5000, checklink, (void *) time(NULL));
#ifdef TEST
    } else {
	cstate = CSTATE_CONNECTED;
	add_dial_cost();
	startclock();
	sa_printf("Connected (33.6)");
	XtVaSetValues(connectb, XtNbitmap, disconnectbitmap,
		      XtNsensitive, 1, NULL);
    }
#endif
}

#ifdef SHOW_CONNECT_STATUS
/*
 * The hairy version. Periodically check the syslog file to gather
 * messages from pppd.
 * This is similar to doing a 'tail -f /var/adm/messages'.
 */
static void
checklink(XtPointer timearg, XtIntervalId * id)
{
    static time_t start, now;
    static size_t fpos;
    FILE *fp;
    char s[80], *p, *q;
    static int dialed = 0;
    int done = 0;
    static int bps;

    if (aborting)
	return;

    fp = fopen(SYSLOGFILE, "r");

    if (timearg) {
	start = (time_t) timearg;
	/* Get current size of log file */
	fseek(fp, 0, SEEK_END);
	fpos = ftell(fp);
	bps = 0;
    }
    /* Seek to the old end and see if any stuff has been added */
    fseek(fp, fpos, SEEK_SET);
    while (fgets(s, sizeof(s) - 1, fp) != NULL) {
	/* We got a line; see if it is from chat */
	if ((p = strstr(s, "chat[")) != NULL) {
	    /* Check for some possible chat messages */
	    if (strstr(p, "ATD")) {	/* Dial command */
		dialed = 1;
	    } else if (strstr(p, "Failed")) {	/* Some error occurred */
		if (strstr(p, "BUSY")) {
		    sa_printf("Line was busy!");
		    if (global.redialinterval > 0) {
			cstate = CSTATE_BUSY;
			XtAppAddTimeOut(mycontext, 3000, redial, (XtPointer) 1);
		    } else
			cstate = CSTATE_NC;
		} else if (strstr(p, "NO DIALTONE")) {
		    sa_printf("No dial tone!");
		    cstate = CSTATE_NC;
		} else {
		    sa_printf("Connect failed!");
		    cstate = CSTATE_NC;
		}
		if (dialed) {
		    add_dial_cost();
		    dialed = 0;
		}
		hangup(hangup_done);
		done = 1;
	    } else if (strstr(p, "NO DIALTONE"))
		dialed = 0;
	    else if ((q = strstr(p, "CARRIER")) != NULL)
		sscanf(q, "CARRIER %d", &bps);
	}
	/* See if the message is from pppd */
	else if ((p = strstr(s, "pppd[")) != NULL) {
	    /* Check for some possible pppd messages */
	    if (strstr(p, "connection established")) {
		sa_printf("Getting IP...");
	    } else if (strstr(p, "IP address")) {
		cstate = CSTATE_CONNECTED;
		add_dial_cost();
		startclock();
		if (bps)
		    sa_printf("Connected (%d.%d)",
			      bps / 1000, (bps % 1000) / 100);
		else
		    sa_printf("Connected");
		XtVaSetValues(connectb, XtNbitmap, disconnectbitmap, NULL);
		done = 1;
		XtAppAddTimeOut(mycontext, 1000, checkdown, (void *) 1);
	    }
	}
	fpos = ftell(fp);
    }
    fclose(fp);

    /* Check for timeout */
    now = time(NULL);
    if (now - start > global.connecttimeout) {
	sa_printf("Timed out!");
	cstate = CSTATE_NC;
	hangup(hangup_done);
    } else if (!done)
	/* Do it again, please */
	XtAppAddTimeOut(mycontext, 500, checklink, NULL);
}

/*
 * Once the link is up, this routine is called every 5 seconds
 * to check if the link has gone down in the meantime.
 * We do this by checking if pppd says 'Hangup (SIGHUP)'.
 */
static void
checkdown(XtPointer first, XtIntervalId * id)
{
    static size_t fpos;
    FILE *fp;
    char s[80], *p;
    int done = 0;

    if (cstate != CSTATE_CONNECTED)
	return;

    fp = fopen(SYSLOGFILE, "r");

    if (first) {
	/* Get current size of log file */
	fseek(fp, 0, SEEK_END);
	fpos = ftell(fp);
    }
    /* Seek to the old end and see if any stuff has been added */
    fseek(fp, fpos, SEEK_SET);
    if (fgets(s, sizeof(s) - 1, fp) != NULL) {
	if ((p = strstr(s, "pppd[")) != NULL) {
	    if (strstr(p, "Hangup")) {	/* Some error occurred */
		cstate = CSTATE_NC;
		sa_printf("Connection lost!");
		stopclock();
		hangup(hangup_done);
		done = 1;
	    }
	}
	fpos = ftell(fp);
    }
    fclose(fp);

    if (!done)
	XtAppAddTimeOut(mycontext, 5000, checkdown, NULL);
}
#else
/*
 * The no-frills version. Periodically check for existence
 * of the 'linkinfo' file.
 */
static void
checklink(void *timearg)
{
    static time_t start, now;
    struct stat buf;
    static size_t fpos;
    FILE *fp;
    char s[80];


    if (aborting)
	return;

    if (timearg)
	start = (time_t) timearg;

    now = time(NULL);

    /* Check if the ip-up script says the link is up */
    if (stat(PPPDFILE, &buf) == 0) {
	cstate = CSTATE_CONNECTED;
	startclock();
	sa_printf("Connected");
	XtVaSetValues(connectb, XtNbitmap, disconnectbitmap, NULL);
    } else if (now - start > global.connecttimeout) {
	sa_printf("Timed out");
	cstate = CSTATE_NC;
	hangup(hangup_done);
    } else
	XtAppAddTimeOut(mycontext, 1000, checklink, NULL);
}
#endif

/*
 * Hang up the modem by sending a SIGHUP to the PPP daemon.
 */
void
hangup(Voidfunc func)
{
    FILE *fp;
    char buf[15];
    pid_t pid;

    log_costs();
    if ((cstate == CSTATE_TRYING) || (cstate == CSTATE_BUSY))
	aborting = 1;

    /* Get the pid from the /var/run/ppp0.pid file */
    if ((fp = fopen(PPPDPIDFILE, "r")) == NULL) {
	child_pid = 0;
    } else {
	fgets(buf, sizeof(buf) - 1, fp);
	fclose(fp);
	pid = atoi(buf);
	if (!pid)
	    printf("error getting pid\n");
	kill(pid, SIGHUP);
    }
    XtAppAddTimeOut(mycontext, 1000, do_wait, func);
}

static void
do_wait(XtPointer func, XtIntervalId * id)
{
    if (child_pid)
	wait(NULL);

    ((Voidfunc) func) ();
}

/*
 * Let the status area show that redialing will happen.
 * Then set up a timeout to actually do it.
 */
static void
redial(XtPointer firstcall, XtIntervalId * id)
{
    static int secondsleft;

    if (firstcall)
	secondsleft = global.redialinterval;
    else
	--secondsleft;

    if (secondsleft) {
	sa_printf("Will retry in %d s", secondsleft);
	XtAppAddTimeOut(mycontext, 1000, redial, 0);
    } else
	do_redial(NULL, NULL);
}

static void
do_redial(XtPointer junk, XtIntervalId * id)
{
    sa_printf("Retrying..");
    doconnect();
}
