#include "dialtty.h"
#include "config.h"


static char            buffer[384];  /* storage buffer for various purposes */
static char*           daemon;       /* daemon name */
static struct termios  tp;           /* terminal settings */
static jmp_buf         stack;
static unsigned int    connected_speed;


/* Storage for things detected while the login name was read. */
static struct chardata  init_chardata =
  {  DEF_ERASE,	  /* default erase character */
     DEF_KILL,	  /* default kill character */
     13,	  /* default eol char */
     0,		  /* space parity */
     0,		  /* no capslock */
  };


static struct options  options = 
  {  PATH_LOGIN,         /* login */
     NULL,               /* tty line */
     INIT_STR,           /* modem init string */
     ISSUE,              /* default issue file */
     RING_COUNT,         /* connect after xx rings */
     DEBUG_MODE,         /* debug mode */
     0                   /* software connect = false */
  };


int main(int argc, char *argv[])
{
  char            *logname = NULL;  /* login name, given to /bin/login */
  struct chardata  chardata;	    /* set by get_logname() */

  /* Let's find out our name... */
  {  register char*  dp = argv[0] + strlen(argv[0]);

     while(dp != argv[0] && dp[-1] != '/')
       dp--;
     if (dp == argv[0] || !(daemon = strdup(dp)))
       daemon = "dialtty";
  }

  /* Ignore ^C */
  signal(SIGINT, SIG_IGN);

  /* Parse command-line arguments */
  parse_args(argc, argv);

  /* TERM, QUIT or HUP signal will force dialtty to hangup and deinitialize */
  signal(SIGTERM, term_signal);
  signal(SIGQUIT, term_signal);
  signal(SIGHUP, term_signal);

  /* USR1 signal will detach dialtty from modem line */
  signal(SIGUSR1, usr1_signal);

  /* let them know we're running */
  debug(0, D_WELCOME, "%s v"VERSION" started...", daemon);

  /* force vt100 terminal mode */
  setenv("TERM", "vt100", 1);

  /* become a process group leader */
  setsid();
	
  /* Update the utmp file. */
  update_utmp();

  /* Open the tty as standard { input, output, error }. */
  debug(0, D_GENERAL, "opening /dev/%s", options.tty);
  open_tty();
  {  int  iv = getpid();
     ioctl(0, TIOCSPGRP, &iv);
  }

  /* Initialize the termio settings (raw mode, eight-bit, blocking i/o). */
  termio_init();

  /* send escape sequence and write the modem init string */
  write(1, "+++", 3);
  if (!send(options.initstring) || !expect("OK\r", 4, "+++"))
    debug(1, D_MODEMINIT, "modem initialization failed");

  /* Initialize number of rings */
  {  char  rings[15];

     sprintf(rings, "ATS7=35S0=%d", options.softconnect ? 0 : options.rings);
     if (!send(rings) || !expect("OK\r", 4, ""))
       debug(1, D_MODEMINIT, "modem-ring initialization failed");
  }

  /* Write modem signature */
  (void)(send("ATI0") && expect("OK\r", 3, "ATI0"));

  /* go to blocking write mode */
  fcntl(1, F_SETFL, fcntl(1, F_GETFL, 0) & ~O_NONBLOCK);

  /* wait for caller to connect */
  ringer();

  /* we're connected; now DON'T accept USR1 signals!! */
  signal(SIGUSR1, SIG_IGN);

  /* Set timer... better login before it goes off! */
  signal(SIGALRM, login_alarm); alarm(TIMEOUT);

  /* Read the login name. */
  debug(0, D_GENERAL, "get login name...");
  chardata = init_chardata;
  if ((logname = get_logname(&chardata)) == 0)
    debug(2, D_CONNECT, "unable to match line speed, aborting");

  /* Disable timer. */
  signal(SIGALRM, SIG_DFL); alarm(0);

  /* Finalize the termio settings. */
  termio_final(&chardata);

  /* Now the newline character should be properly written. */
  write(1, "\n", 1);

  /* Let the login program take care of password validation. */
  errno = 0;
  execl(options.login, options.login, "--", logname, (char*)0);
  debug(3, D_LOGIN, "%s: can't exec %s: %s", options.tty, 
                    options.login, sys_errlist[errno]);

  return 0;  /* satisfy compiler - this will never be executed */
}


/* parse-args - parse command-line arguments */
static void parse_args(int argc, char *argv[])
{
  extern char  *optarg;  /* getopt */
  extern int    optind;	 /* getopt */
  int           c;

  while(isascii(c = getopt(argc, argv, "I:r:d:c"))) 
    {  switch (c) 
         {  case 'I':
	      {  char  *p, *q;

		 /* copy optarg into op->initstring decoding \ddd
		    octal codes into chars */
		 q = options.initstring = malloc(strlen(optarg) + 1);
		 p = optarg;
		 while (*p) 
                   {  if (*p == '\\') 
                        {  char  ch;
			   /* know \\ means \ */
			   p++;
			   if (*p == '\\') 
                             {  ch = '\\';
			        p++;
			     } 
                           else 
                             {  int  i;
			        /* handle \000 - \177 */
			        ch = 0;
			        for(i = 1; i <= 3; i++) 
                                  {  if (*p >= '0' && *p <= '7') 
                                       {  ch <<= 3;
				          ch += *p - '0';
				          p++;
				       } 
                                     else 
				       break;
				  }
			     }
			   *q++ = ch;
			}
		      else 
			*q++ = *p++;
		   }
		 *q = '\0';
	      }
	      break;

	    case 'r':  /* Connect after x rings */
	      options.rings = atoi(optarg);
	      if (options.rings <= 0 || options.rings > 15)
		debug(1, D_CMDLINE, "bad number of rings: %d", options.rings);
	      break;

	    case 'd':  /* Debug mode on */
	      sscanf(optarg, "%o", &options.debug);
	      break;

	    case 'c':  /* Enable software connect */
	      options.softconnect = 1;
	      break;

	    default:
	      debug(5, D_CMDLINE, "Usage: dialtty [-d mode] [-c] [-I init] [-r rings] tty");
	 }
    }

  /* print command line options */
  debug(0, D_CMDLINE, "debug options = %o", options.debug);
  debug(0, D_CMDLINE, "init string = %s", options.initstring);
  debug(0, D_CMDLINE, "connect after %d ring%c", options.rings,
      	              (options.rings == 1) ? ' ' : 's');
  debug(0, D_CMDLINE, "software connect %sbled", 
                      options.softconnect ? "ena" : "disa");

  /* check parameter count */
  if (argc != optind + 1)
    debug(1, D_CMDLINE, "expecting serial line (e.g. ttyS0)");

  /* tty */
  options.tty = argv[optind];
  debug(0, D_CMDLINE, "device = /dev/%s", options.tty);
}


/* update_utmp - update our utmp entry */
static void update_utmp(void)
{
    struct utmp   ut;
    int           ut_fd;
    int           mypid = getpid();
    struct utmp  *utp;

    /*
     * The utmp file holds miscellaneous information about things started by
     * /sbin/init and other system-related events. Our purpose is to update
     * the utmp entry for the current process, in particular the process type
     * and the tty line we are listening to. Return successfully only if the
     * utmp file can be opened for update, and if we are able to find our
     * entry in the utmp file.
     */

    utmpname(_PATH_UTMP);
    setutent();
    while((utp = getutent()) && !(utp->ut_type == INIT_PROCESS
		&& utp->ut_pid == mypid)) /* nothing */;

    if (utp) 
      memcpy(&ut, utp, sizeof(ut));
    else 
      {	 /* some inits don't initialize utmp... */
	 memset(&ut, 0, sizeof(ut));
	 strncpy(ut.ut_id, options.tty + 3, sizeof(ut.ut_id));
      }
    endutent();
	
    strncpy(ut.ut_user, "LOGIN", sizeof(ut.ut_user));
    strncpy(ut.ut_line, options.tty, sizeof(ut.ut_line));
    time(&ut.ut_time);
    ut.ut_type = LOGIN_PROCESS;
    ut.ut_pid = mypid;

    pututline(&ut);
    endutent();

    if ((ut_fd = open(_PATH_WTMP, O_APPEND|O_WRONLY)) >= 0) 
      {	 flock(ut_fd, LOCK_EX);
	 write(ut_fd, &ut, sizeof(ut));
	 flock(ut_fd, LOCK_UN);
	 close(ut_fd);
      }
}


/* open_tty - set up tty as standard { input, output, error } */
static void open_tty(void)
{
  struct stat  st;
  bool         busy;

  /* close standard file descriptors */
  close(0);  /* stdin */
  close(1);  /* stdout */
  close(2);  /* stderr */
  errno = 0;

  /* sanity checks... */
  if (chdir("/dev"))
    debug(1, D_LINEOPEN, "/dev: chdir() failed: %s", sys_errlist[errno]);
  if (stat(options.tty, &st) < 0)
    debug(1, D_LINEOPEN, "/dev/%s: %s", options.tty, sys_errlist[errno]);
  if ((st.st_mode & S_IFMT) != S_IFCHR)
    debug(1, D_LINEOPEN, "/dev/%s: not a character device", options.tty);

  /* open tty as standard input. */
  for(busy = 0;;)
    {  errno = 0;
       if (!(open(options.tty, O_RDWR | O_NONBLOCK, 0) == -1 &&
             errno == EBUSY))
         {  if (errno)
	      debug(1, D_LINEOPEN, "/dev/%s: cannot open as standard "
		       "input: %s", options.tty, sys_errlist[errno]);
	    else
	      break;
	 }
       if (!busy)
         {  busy = 1;
	    debug(0, D_LINEOPEN, "/dev/%s: %s",
                                 options.tty, sys_errlist[EBUSY]);
         }
       sleep(20);
    }
  if (busy)
    debug(0, D_LINEOPEN, "/dev/%s: Device free; continue", options.tty);
  
  /* set up stdout and stderr file descriptors. */
  if (dup(0) != 1 || dup(0) != 2)     
    debug(1, D_LINEOPEN, "%s: dup problem: %s", options.tty, 
	                 sys_errlist[errno]);

  /*
   * The following ioctl will fail if stdin is not a tty, but also when
   * there is noise on the modem control lines. In the latter case, the
   * common course of action is (1) fix your cables (2) give the modem more
   * time to properly reset after hanging up. SunOS users can achieve (2)
   * by patching the SunOS kernel variable "zsadtrlow" to a larger value;
   * 5 seconds seems to be a good value.
   */

  if (ioctl(0, TCGETA, &tp) < 0)
    debug(1, D_LINEOPEN, "%s: ioctl: %s", options.tty, sys_errlist[errno]);

  /* all devices are unbuffered */
  setbuf(stdin, (char*)NULL);
  setbuf(stdout, (char*)NULL);
  setbuf(stderr, (char*)NULL);

  /* set permissions */
  chown(options.tty, 0, 0);  /* root, root */
  chmod(options.tty, 0622);  /* crw--w--w- */
  errno = 0;
}


/* termio_init - initialize termio settings */
static void termio_init(void)
{
  /*
   * Initial termio settings: 8-bit characters, raw-mode, blocking i/o.
   * Special characters are set after we have read the login name; all
   * reads will be done in raw mode anyway. Errors will be dealt with
   * lateron.
   */
  /* flush input and output queues, important for modems! */
  ioctl(0, TCFLSH, TCIOFLUSH);

  tp.c_cflag = CS8 | HUPCL | CRTSCTS | CREAD | SPEED;
  tp.c_iflag = tp.c_lflag = tp.c_oflag = tp.c_line = 0;
  tp.c_cc[VMIN] = 1;
  tp.c_cc[VTIME] = 0;
  ioctl(0, TCSETA, &tp);

  /* go to blocking input mode */
  fcntl(0, F_SETFL, fcntl(0, F_GETFL, 0) & ~O_NONBLOCK);
}


/* do_prompt - show login prompt, optionally preceded by /etc/issue contents */
static void do_prompt(void)
{
  FILE           *fd;
  int             oflag;
  char            c;
  struct utsname  uts;

  uname(&uts);

  if ((fd = fopen(options.issue, "r"))) 
    {  oflag = tp.c_oflag;		/* save current setting */
       tp.c_oflag |= (ONLCR | OPOST);	/* map NL in output to CR-NL */
       ioctl(0, TCSETAW, &tp);

       while((c = getc(fd)) != EOF)
	 {  if (c == '\\')
	      {	 switch(c = getc(fd))
		   {  case 's':
		        printf("%s", uts.sysname);
		        break;
		    
		      case 'n':
		        printf("%s", uts.nodename);
		        break;
		    
		      case 'r':
		        printf("%s", uts.release);
		        break;
		    
		      case 'v':
		        printf("%s", uts.version);
		        break;
		    
		      case 'm':
		        printf("%s", uts.machine);
		        break;

		      case 'o':
		        printf("%s", uts.domainname);
		        break;

		      case 'd':
		      case 't':
		        {  static char *weekday[] = {  "Sun", "Mon", "Tue", "Wed", 
					        "Thu", "Fri", "Sat" 
			                     };
		           static char *month[] = {  "Jan", "Feb", "Mar", "Apr", 
					      "May", "Jun", "Jul", "Aug", 
					      "Sep", "Oct", "Nov", "Dec" 
			                   };
		           time_t      now;
		           struct tm  *tm;

		           time(&now);
		           tm = localtime(&now);

		           if (c == 'd')
			     printf ("%s %s %d, %d",
				weekday[tm->tm_wday], month[tm->tm_mon],
				tm->tm_mday, 
				tm->tm_year < 70 ? tm->tm_year + 2000 :
				tm->tm_year + 1900);
		           else
			     printf ("%02d:%02d:%02d",
				tm->tm_hour, tm->tm_min, tm->tm_sec);
		      
		           break;
			}

		      case 'l':
		        printf("%s", options.tty);
		        break;

		      case 'b':
		        printf("%u", connected_speed);
			break;

		      case 'u':
		      case 'U':
		        {  int           users = 0;
		           struct utmp  *ut;

			   setutent();
			   while((ut = getutent()))
		             if (ut->ut_type == USER_PROCESS)
			       users++;
		           endutent();
		           printf("%d", users);
		           if (c == 'U')
		             printf(" user%s", (users == 1) ? "" : "s");
		           break;
			}

		      default:
		        putchar(c);
			break;
		   }
	      }
	    else
	      putchar(c);
	 }
       fflush(stdout);

       tp.c_oflag = oflag;     /* restore settings */
       ioctl(0, TCSETAW, &tp);  /* wait till output is gone */
       fclose(fd);
    }

  {  char  hn[MAXHOSTNAMELEN + 1];

     gethostname(hn, MAXHOSTNAMELEN);
     write(1, hn, strlen(hn));
  }
  write(1, LOGIN, sizeof(LOGIN) - 1);  /* always show login prompt */
}


/* get_logname - get user name, establish parity, speed, erase, kill, eol */
static char *get_logname(struct chardata *cp)
{
  static char  logname[256];
  char        *bp;
  char         c;		 /* input character, full eight bits */
  char         ascval;		 /* low 7 bits of input character */
  int          bits;		 /* # of "1" bits per character */
  int          mask;		 /* mask with 1 bit up */
  static char *erase[] = /* backspace-space-backspace */
    {  "\010\040\010",	/* space parity */
       "\010\040\010",	/* odd parity */
       "\210\240\210",	/* even parity */
       "\210\240\210",	/* no parity */
    };

  /* Initialize kill, erase, parity etc. (also after switching speeds). */
  *cp = init_chardata;

  /* Flush pending input (esp. after parsing or switching the baud rate). */
  sleep(1);
  ioctl(0, TCFLSH, TCIFLUSH);

  /* Prompt for and read a login name. */
  for(*logname = 0; *logname == 0;) 
    {  /* Write issue file and prompt, with "parity" bit == 0. */
       do_prompt();

       /* Read name, watch for break, parity, erase, kill, end-of-line. */
       for(bp = logname, cp->eol = 0; cp->eol == 0;) 
         {  /* Do not report trivial EINTR/EIO errors. */
	    if (read(0, &c, 1) < 1) 
              {  if (errno == EINTR || errno == EIO)
		   exit(0);
		 debug(2, D_LOGIN, "%s: read: %s", options.tty, 
		                   sys_errlist[errno]);
	      }

	    /* Do parity bit handling. */
	    if (c != (ascval = (c & 0177)))  /* "parity" bit on ? */
              {	 for(bits = 1, mask = 1; mask & 0177; mask <<= 1)
		   if (mask & ascval)
		     bits++;  /* count "1" bits */
		 cp->parity |= ((bits & 1) ? 1 : 2);
	      }

	    /* Do erase, kill and end-of-line processing. */
	    switch(ascval) 
              {  case CR:
	         case NL:
		   *bp = 0;			/* terminate logname */
		   cp->eol = ascval;		/* set end-of-line char */
		   break;

	         case BS:
	         case DEL:
	         case '#':
		   cp->erase = ascval;		/* set erase character */
		   if (bp > logname) 
                     {  write(1, erase[cp->parity], 3);
		        bp--;
		     }
		   break;

	         case CTL('U'):
	         case '@':
		   cp->kill = ascval;		/* set kill character */
		   while (bp > logname) 
                     {  write(1, erase[cp->parity], 3);
		        bp--;
		     }
		   break;

	         case CTL('D'):
		   if (!raise(SIGHUP))  /* added by me */
		     sleep(6);          /* added by me */
		   exit(0);

	         default:
		   if (!isascii(ascval) || !isprint(ascval)) 
                     {  /* ignore garbage characters */ ;
		     } 
                   else 
                     if ((unsigned)(bp - logname) >= sizeof(logname) - 1)  
                       debug(2, D_LOGIN, "%s: input overrun", options.tty);
                     else 
                       {  write(1, &c, 1);	/* echo the character */
		          *bp++ = ascval;	/* and store it */
		       }
		   break;
	      }
	 }
    }

  /* Handle names with upper case and no lower case. */
  if ((cp->capslock = caps_lock(logname))) 
    {  for(bp = logname; *bp; bp++)
	 if (isupper(*bp))
	   *bp = tolower(*bp);		/* map name to lower case */
    }
  return logname;
}


/* termio_final - set the final tty mode bits */
static void termio_final(struct chardata *cp)
{
  /* General terminal-independent stuff. */
  tp.c_iflag |= IGNPAR | IXON | IXOFF;
  tp.c_lflag |= ICANON | ISIG | ECHO | ECHOE | ECHOK| ECHOKE;
  tp.c_oflag |= OPOST;
  tp.c_cc[VINTR] = DEF_INTR;	 /* default interrupt */
  tp.c_cc[VQUIT] = DEF_QUIT;	 /* default quit */
  tp.c_cc[VEOF] = DEF_EOF;	 /* default EOF character */
  tp.c_cc[VEOL] = DEF_EOL;
  tp.c_cc[VSWTC] = DEF_SWITCH;   /* default switch character */

  /* Account for special characters seen in input. */
  if (cp->eol == CR) 
    {  tp.c_iflag |= ICRNL;	 /* map CR in input to NL */
       tp.c_oflag |= ONLCR;	 /* map NL in output to CR-NL */
    }
  tp.c_cc[VERASE] = cp->erase;	 /* set erase character */
  tp.c_cc[VKILL] = cp->kill;	 /* set kill character */

  /* Account for the presence or absence of parity bits in input. */
  switch(cp->parity) 
    {  case 0:					/* space (always 0) parity */
	 break;
       case 1:					/* odd parity */
	 tp.c_cflag |= PARODD;
	 /* FALLTHROUGH */
       case 2:					/* even parity */
	 tp.c_cflag |= PARENB;
	 tp.c_iflag |= INPCK | ISTRIP;
	 /* FALLTHROUGH */
       case (1 | 2):				/* no parity bit */
	 tp.c_cflag &= ~CSIZE;
	 tp.c_cflag |= CS7;
	 break;
    }

  /* Account for upper case without lower case. */
  if (cp->capslock) 
    {  tp.c_iflag |= IUCLC;
       tp.c_lflag |= XCASE;
       tp.c_oflag |= OLCUC;
    }

  /* Finally, make the new settings effective */
  if (ioctl(0, TCSETA, &tp) < 0)
    debug(2, D_LINEOPEN, "%s: ioctl: TCSETA: %s", options.tty, 
	                 sys_errlist[errno]);
}


/* caps_lock - string contains upper case without lower case */
static int caps_lock(char *s)
{
  int  capslock;

  for(capslock = 0; *s; s++) 
    {  if (*s >= 'a' && *s <= 'z')
	 return 0;
       if (!capslock && *s >= 'A' && *s <= 'Z')
	 capslock = 1;
    }
  return capslock;
}


/* update syslog */
static void debug(unsigned char xit, unsigned int mode, char *fmt, ...)
{
  /* log this message? */
  if (mode & options.debug)
    {  va_list  ap;
       char     message[256];

       /* fill in message parameters */
       va_start(ap, fmt);
       vsprintf(message, fmt, ap);
       va_end(ap);

       /* open syslog and write message */
       openlog(daemon, LOG_PID, LOG_USER);
       syslog(LOG_ERR, message);
       closelog();
    }

  /* terminate program if necessary */
  if (xit)
    {  close(0);
       close(1);
       close(2);
       exit(xit);
    }
}


/* Send a command to the modem */
static bool send(register char *s)
{
  bool   retval;
  char*  bp;

  /* send string and build syslog message */
  for(bp = strcpy(buffer, "send [") + 5; (*(++bp) = *s); s++)
    if (write(1, s, 1) != 1)
      break;

  /* send CR */
  retval = (!*s && write(1, "\r", 1) == 1) ? 1 : 0;

  /* success or failure? */
  strcpy(bp, (retval) ? "]" : "] -- FAIL");
  debug(0, D_MODEMRSLT, buffer);

  return retval;
}


/* Try to match modem output with string */
static void expect_alarm()
{
  signal(SIGALRM, SIG_DFL);
  longjmp(stack, 1);
}


static bool expect(char *s, unsigned int timeout, char *strip)
{
  char            ch;
  volatile short  match_length = -strlen(s);
  static char*    strip_ptr;
  static char*    buffer_ptr;
  static char*    got;
  volatile bool   sequence = 0;

  /* timeout entry point */
  if (setjmp(stack))
    {  while(buffer_ptr[-1] == ' ')
         buffer_ptr--;
       strcpy(buffer_ptr, " -- FAIL");
       debug(0, D_MODEMRSLT, buffer);
       debug(0, D_MODEMRSLT, "[timed out after %d seconds]", timeout);
       return 0;
    }

  /* enable alarm signal */
  if (timeout)
    {  signal(SIGALRM, expect_alarm);
       alarm(timeout);
    }

  /* match modem output */
  buffer_ptr = got = buffer + sprintf(buffer, "expect [%s] ", s);
  strip_ptr = strip;  /* workaround for "clobbered" message */
  while(read(0, &ch, 1) == 1)
    {  /* remove superfluous cr/nl characters */
       if (ch == '\n' || (ch == '\r' && buffer_ptr[-1] == '\r'))
	 continue;

       if (!sequence)
	 {  /* strip sequence */
	    if (strip_ptr && *strip_ptr)
              {  if (ch == *(strip_ptr++))
	           continue;
	         strip_ptr = NULL;
	      }

	    /* strip heading ^M and spaces */
	    if (ch == '\r' || ch == ' ')
              continue;
	 }

       /* ok we've got serious data here... */
       if (!sequence)
         {  sequence = 1;
            got = buffer_ptr;
	 }

       /* build expect string */
       *(buffer_ptr++) = ch;

       /* match modem output? */
       if (++match_length >= 0)
	 {  *buffer_ptr = 0;
            if (!strcmp(got + match_length, s))
	      {  sequence = 2;
	         break;
	      }
	 }
    }

  /* Disable alarm signal */
  if (timeout)
    {  alarm(0);
       signal(SIGALRM, SIG_DFL);
    }

  /* Success or failure? */
  if (!(sequence = (sequence == 2) ? 1 : 0))
    {  while(buffer_ptr[-1] == ' ')
         buffer_ptr--;
       strcpy(buffer_ptr, " -- FAIL");
    }

  /* Update syslog */
  debug(0, D_MODEMRSLT, buffer);
  
  return sequence;
}


static void ringer(void)
{
  bool  retry;
  
  again:
  do
    {  int  remaining = options.rings;

       /* wait until somebody calls (infinite) */
       debug(0, D_GENERAL, "wait for connection...");
       expect("RING", 0, "");

       /* count the remaining rings */
       for(remaining--, retry = 0; !retry && remaining; remaining--)
         if (!expect("RING", INTERRING, ""))
           retry = 1;
    } while(retry);

  /* ok, at this point the modem will try to connect */
  debug(0, D_GENERAL, "%d ring%s, try to connect...", options.rings, 
	   (options.rings == 1) ? "" : "s");

  /* software connection? */
  if (options.softconnect)
    if (!send(CONNECT_STR))
      {  debug(0, D_CONNECT, "failed software connection");
	 goto again;
      }

  /* CONNECT */
  if (!expect("CONNECT", 40, ""))
    {  debug(0, D_CONNECT, "unable to connect");
       goto again;
    }
  if (!(connected_speed = extract_linespeed()))
    {  debug(0, D_CONNECT, "unable to determine linespeed");
       goto again;
    }
  debug(0, D_CONNECT, "connected, linespeed = %u baud", connected_speed);
}


/* wait for linespeed digits */
static unsigned int extract_linespeed(void)
{
  volatile unsigned int  speed = 0;

  /* timeout entry point */
  if (setjmp(stack))
    return 0;

  /* set timeout = 6 seconds */
  signal(SIGALRM, expect_alarm);
  alarm(6);

  /* wait for digits */
  for(;;)
    {  char  ch;

       if (read(0, &ch, 1) != 1)
         debug(2, D_CONNECT, "aborting");
       if (!speed && ch == ' ')
	 continue;
       if (ch < '0' || ch > '9')
         break;
       speed = speed * 10 + (unsigned int)(ch - '0');
    }

  /* remove alarm signal */
  alarm(0);
  signal(SIGALRM, SIG_DFL);

  return speed;
}


/* Hangup line */
static void hangup(void)
{
  debug(0, D_SIGNAL, "hangup...");

  /* sending the reset sequence will result in another SIGHUP, so ignore it */
  signal(SIGHUP, SIG_IGN);

  /* Send escape sequence and wait for modem to enter command mode */
  if (write(1, "+++", 3) == 3)
    {  sleep(1);

       /* Wait 2 seconds (at most) for modem to hangup */
       (void)(send(RESET_STR) && expect(RESET_STR"\r", 2, "+++"));
    }

  /* Close tty connection */
  close(2);
  close(1);
  close(0);
}


/* No login name within time, disconnecting */
static void login_alarm()
{
  printf("\r\nlogin timed out after %d seconds!\r\n"
	          "closing connection...\r\n", TIMEOUT);
  debug(2, D_LOGIN, "[login timed out after %d seconds]", TIMEOUT);
}


static void term_signal()
{
  hangup();
  debug(5, D_SIGNAL, "exiting on TERM/QUIT/HUP signal");
}


static short  usr1_busy = 0;

static void usr1_signal()
{
  debug(0, D_SIGNAL, "caught USR1 signal %s", usr1_busy ? "again" : "");
  signal(SIGUSR1, usr1_signal);

  if (!usr1_busy++)
    {  debug(0, D_GENERAL, "detach dialtty from /dev/%s", options.tty);
       hangup();
       debug(0, D_GENERAL, "delayed exit; waiting for 15 seconds");
    }

  sleep(15);

  if (--usr1_busy)
    sleep(15);
  else
    exit(5);
}
