/* Copyright 2001  Mark Pulford <mark@kyne.com.au>
 * This file is subject to the terms and conditions of the GNU General Public
 * License. Read the file COPYING found in this archive for details, or
 * visit http://www.gnu.org/copyleft/gpl.html
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <gdbm.h>
#include <errno.h>

#include "common.h"
#include "error.h"
#include "md5.h"

static int verbose = 0;
static char *exec_program = NULL;

static void decode_args(int argc, char **argv);
static void proto_main(int client_fd, const char *addr);
static const char *generate_seed();
static const char *get_address(int sock);
static int sat_log(const char *token, const char *addr, const char *msg);
static void sat_exec(const char *token, const char *addr, const char *msg);
void log(int prio, int type, const char *fmt, ...);
int lockfd(int fd, int type);

int main(int argc, char **argv)
{
	const char *addr;

	srandom(time(NULL) + getpid());

	decode_args(argc, argv);

	addr = get_address(0);
	if(!addr) {
		if(ENOTSOCK == errno) {
			printf("satellited v%s\n", VERSION);
			die("must be run from inetd");
		}
		log(1, LOG_WARNING, "Cannot get client address");
		return 0;
	}

	proto_main(0, addr);

	return -1;	/* Shouldn't happen */
}

/* Handle the connection to the client
 * Doesn't return. */
static void proto_main(int client_fd, const char *addr)
{
	char *token;
	const char *line;
	const char *secret;
	const char *seed;
	char *hash;
	const char *errmsg;

	/* Get token */
	line = get_line(client_fd, &errmsg);
	if(!line) {
		if(!errmsg)
			errmsg = "Connection closed";
		log(1, LOG_WARNING, "Client %s: network error: %s", addr, errmsg);
		exit(0);
	}
	token = strdup(line);

	/* Find corresponding secret */
	secret = get_secret_by_token(token, &errmsg);
	if(!secret) {
		if(errmsg)
			log(0, LOG_CRIT, "Client %s: Token '%s' lookup failed: %s", addr, token, errmsg);
		else
			log(1, LOG_WARNING, "Client %s: unknown token '%s'", addr, token);
		exit(0);
	}

	/* Poke client with seed */
	seed = generate_seed();
	if(!put_line(client_fd, seed)) {
		log(1, LOG_WARNING, "Client %s: token '%s' not verifed before closed connection", addr, token);
		exit(0);
	}

	/* Get client's hash */
	line = get_line(client_fd, &errmsg);
	if(!line) {
		if(!errmsg)
			errmsg = "Connection closed";
		log(1, LOG_WARNING, "Client %s: token '%s' not verified: %s", addr, token, errmsg);
		exit(0);
	}

	/* Skip past the magic string and the seed */
	hash = strrchr(crypt_md5(secret, seed), '$')+1;

	if(strcmp(hash, line)) {
		log(1, LOG_WARNING, "Client %s: Bad response for token '%s'", addr, token);
		put_line(client_fd, "ERROR");
		exit(0);
	}

	/* From this point it is ok for the client to close the connection */
	put_line(client_fd, "OK");

	/* Get message */
	line = get_line(client_fd, &errmsg);
	if(!line && errmsg)	/* not logging closed connections */
		log(1, LOG_WARNING, "Client %s: [%s] Error receiving message: %s", addr, token, errmsg);
	sat_log(token, addr, line);

	if(exec_program)
		sat_exec(token, addr, line);
	
	exit(0);
}

/* Returns: 0 failure
 *          1 success */
/* NULL message is ok */
static int sat_log(const char *token, const char *addr, const char *msg)
{
	char buf[TOKEN_SIZE*4];
	int fd;
	int len;
	time_t tm;

	fd = open(LOG_FILE, O_CREAT|O_WRONLY|O_APPEND, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
	if(-1 == fd) {
		log(0, LOG_ERR, "Cannot open log: %s", strerror(errno));
		return 0;
	}

	if(!lockfd(fd, F_WRLCK)) {
		log(0, LOG_ERR, "Cannot get lock: %s", strerror(errno));
		close(fd);
		return 0;
	}

	time(&tm);
	strncpy(buf, ctime(&tm), sizeof(buf));
	len = strlen(buf) - 1;	/* Strip \n */
	snprintf(buf+len, sizeof(buf)-len, "  %s  %s  %s\n",
			token, addr, msg?msg:"");

	len = strlen(buf);
	if(len != safe_write(fd, buf, len)) {
		log(0, LOG_ERR, "Write failed: %s", strerror(errno));
		close(fd);
		return 0;
	}

	close(fd);
	return 1;
}

/* Attempt to run the program. Return immediately. */
static void sat_exec(const char *token, const char *addr, const char *msg)
{
	int ret, i;

	ret = fork();
	if(-1 == ret) {
		log(0, LOG_ERR, "cannot fork(): %s", strerror(errno));
		return;
	}
	if(ret)
		return;

	ret = open("/dev/null", O_RDWR);
	if(-1 == ret) {
		log(0, LOG_ERR, "cannot open /dev/null for rw");
		_exit(0);
	}
	
	for(i=0; i<3; i++) {
		if(-1 == dup2(ret, i)) {
			log(0, LOG_ERR, "unable to run program. dup failed: %s", strerror(errno));
			_exit(0);
		}
	}

	if(msg)
		execl(exec_program, exec_program, token, addr, msg, NULL);
	else
		execl(exec_program, exec_program, token, addr, NULL);

	log(0, LOG_ERR, "exec(%s) failed: %s", exec_program, strerror(errno));
	_exit(0);
}


/* Returns: 0	failure (errno set)
 *          1	success */
int lockfd(int fd, int type)
{
	struct flock l;

	l.l_type = type;
	l.l_whence = SEEK_SET;
	l.l_start = 0;
	l.l_len = 0;
	if(-1 == fcntl(fd, F_SETLKW, &l))
		return 0;

	return 1;
}

/* returns NULL on error (errno set) */
static const char *get_address(int sock)
{
	struct sockaddr_in sin;
	int size;
	static char addr[16];

	size = sizeof(sin);
	if(-1 == getpeername(sock, (struct sockaddr *)&sin, &size))
		return NULL;

	strncpy(addr, (char *)inet_ntoa(sin.sin_addr), sizeof(addr));

	return addr;
}

/* Use all random bits, unlike % */
#define RND(x) (int) (((double)(x))*random()/(RAND_MAX+1.0))

/* randomly generate 8 char long random seed */
static const char *generate_seed()
{
	char *ch = "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
	static char seed[9];
	int i;

	for(i=0; i<sizeof(seed)-1; i++)
		seed[i] = ch[RND(64)];
	seed[sizeof(seed)-1] = 0;

	return seed;
}

/* fmt passed to log() should be a string constant */
void log(int prio, int type, const char *fmt, ...)
{
	va_list args;
	static int ready = 0;

	if(prio <= verbose) {
		char buf[TOKEN_SIZE*4];
		if(!ready) {
			openlog("satellited", LOG_PID, LOG_DAEMON);
			ready = 1;
		}

		va_start(args, fmt);
		vsnprintf(buf, sizeof(buf), fmt, args);
		va_end(args);
		if(strtok(buf, "%"))
			syslog(LOG_WARNING, "The following log message was truncated, possible format string attack");
		syslog(type, buf);
	}
}

/* Exit on failure */
static void decode_args(int argc, char **argv)
{
	int ch;

	char *optstr = "vf:e:";

	opterr = 0;
	ch = getopt(argc, argv, optstr);
	while(ch != -1) {
		switch(ch) {
		case ':':
			log(0, LOG_CRIT, "Missing parameter");
			exit(-1);
		case '?':
			log(0, LOG_CRIT, "Unknown option '%c'", optopt);
			exit(-1);
		case 'v':
			verbose++;
			break;
		case 'f':
			db_file = optarg;
			break;
		case 'e':
			exec_program = optarg;
			break;
		default:
			abort();
		}
		ch = getopt(argc, argv, optstr);
	}
}
