/* Copyright 2001  Mark Pulford <mark@kyne.com.au>
 * This file is subject to the terms and conditions of the GNU General Public
 * License. Read the file COPYING found in this archive for details, or
 * visit http://www.gnu.org/copyleft/gpl.html
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>

#include "common.h"
#include "error.h"
#include "md5.h"

static int verbose = 0;

static const char *dest_host;
static int dest_port = DEFAULT_PORT;
static const char *site_token = NULL;
static const char *site_secret = NULL;
static const char *site_message = NULL;

static void print_usage();
static void decode_args(int argc, char **argv);
static int make_connect(const char *host, int port, const char **err);
static int resolv_addr(struct in_addr *in, const char *addr);
static int service2port(const char *name, const char *proto);

int main(int argc, char **argv)
{
	int sock;
	const char *line;
	const char *errmsg;

	dest_port = service2port("satellite", "tcp");
	if(!dest_port)
		dest_port = DEFAULT_PORT;

	decode_args(argc, argv);

	sock = make_connect(dest_host, dest_port, &errmsg);
	if(-1 == sock)
		die("connect failed: %s", errmsg);

	if(!put_line(sock, site_token))
		die("remote closed connection");

	line = get_line(sock, &errmsg);
	if(!line) {
		if(errmsg)
			die("read error: %s", errmsg);
		else
			die("server refused token");
	}

	/* Calculate & send hash */
	line = strrchr(crypt_md5(site_secret, line), '$') + 1;
	if(!put_line(sock, line))
		die("remote closed connection");

	line = get_line(sock, &errmsg);
	if(!line) {
		if(errmsg)
			die("read error: %s", errmsg);
		else
			die("connection closed");
	}
	if(strcmp(line, "OK"))
		die("server refused secret");

	/* Send message if requested */
	if(site_message && !put_line(sock, site_message))
		die("remote closed connection");

	if(verbose >= 1)
		puts("Server successfully contacted");

	return 0;
}

static int service2port(const char *name, const char *proto)
{
	struct servent *s;

	s = getservbyname(name,proto);
	if(!s)
		return 0;

	return ntohs(s->s_port);
}

/* Returns	0  error (h_errno set)
 *		1  success */
static int resolv_addr(struct in_addr *in, const char *addr)
{
	struct hostent *h;

	if(inet_aton(addr, in))
		return 1;

	h = gethostbyname(addr);
	if(!h)
		return 0;
	memcpy(in, h->h_addr_list[0], sizeof(*in));

	return 1;
}

/* Returns:  -1		error. **err set to description.
 *	     otherwise file descriptor */
static int make_connect(const char *host, int port, const char **err)
{
	struct sockaddr_in addr;
	int fd;

	if(!resolv_addr(&addr.sin_addr, host)) {
		*err = hstrerror(h_errno);
		return -1;
	}
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);

	fd = socket(PF_INET, SOCK_STREAM, 0);
	if(-1 == fd) {
		*err = strerror(errno);
		return -1;
	}

	if(-1 == connect(fd, (struct sockaddr *)&addr, sizeof(addr))) {
		*err = strerror(errno);
		close(fd);
		return -1;
	}

	return fd;
}

static void print_usage()
{
	puts("usage: satellite [-v] [-m message] [-p port] host token [secret]");
}

/* Process cli args. Ensure required vars are set. */
static void decode_args(int argc, char **argv)
{
	const char *optstr = "vhm:p:f:";
	const char *errmsg;
	char *end;
	int ch;
	int argleft;

	if(1 == argc) {
		printf("satellite v%s\n", VERSION);
		exit(0);
	}

	ch = getopt(argc, argv, optstr);
	while(ch != -1) {
		switch(ch) {
		case ':':
			print_usage();
			die("missing parameter");
		case '?':
			print_usage();
			die("unknown option");
		case 'v':
			verbose++;
			break;
		case 'h':
			print_usage();
			exit(0);
		case 'p':
			dest_port = strtol(optarg, &end, 10);
			if(0 == *optarg || 0 != *end)
				die("bad port number");
			break;
		case 'm':
			site_message = optarg;
			if(strlen(optarg) > TOKEN_SIZE)
				die("message too long");
			break;
		case 'f':
			db_file = optarg;
			break;
		default:
			abort();
		}
		ch = getopt(argc, argv, optstr);
	}

	argleft = argc - optind;
	if(argleft!=2 && argleft!=3)
		die("incorrect number of arguments");

	dest_host = argv[optind++];
	site_token = argv[optind++];
	if(optind < argc) {
		site_secret = argv[optind++];
	} else {
		site_secret = get_secret_by_token(site_token, &errmsg);
		if(!site_secret) {
			if(!errmsg)
				errmsg = "not found";
			die("token lookup: %s", errmsg);
		}
	}

	if(strlen(site_token) > TOKEN_SIZE)
		die("token too long");
	if(strchr(site_token, '\n'))
		die("token cannot contain newlines");
	if(strlen(site_secret) > TOKEN_SIZE)
		die("secret too long");
	if(strchr(site_secret, '\n'))
		die("secret cannot contain newlines");
}
