/* Copyright 2001  Mark Pulford <mark@kyne.com.au>
 * This file is subject to the terms and conditions of the GNU General Public
 * License. Read the file COPYING found in this archive for details, or
 * visit http://www.gnu.org/copyleft/gpl.html
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gdbm.h>

#include "error.h"

static char *db_file = TOKEN_FILE;
static GDBM_FILE db_handle = NULL;

static void db_close();
static void db_open();
static void db_add(char *token, char *secret);
static void db_delete(char *token);
static void db_list(char *token);
static int db_print_datum(datum token);
static void db_listall();
static void print_usage();
static char *datum2str(datum d);

int main(int argc, char **argv)
{
	int ch;

	if(1 == argc) {
		printf("satcfg v%s\n", VERSION);
		return 0;
	}

	if(-1 == atexit(db_close))
		die("atexit() failed!");

	while(-1 != (ch=getopt(argc, argv, "hf:a:d:l:L"))) {
		switch(ch) {
		case 'h':
			print_usage();
			return 0;
		case 'f':
			db_file = optarg;
			db_close();
			printf("Config set to %s\n", db_file);
			break;
		case 'a':
			if(optind >= argc)
				die("missing secret");
			db_add(optarg, argv[optind++]);
			break;
		case 'd':
			db_delete(optarg);
			break;
		case 'l':
			db_list(optarg);
			break;
		case 'L':
			db_listall();
			break;
		case '?':
			return -1;	/* getopt has already complained */
		case ':':
			print_usage();
			die("missing parameter");
		default:
			db_close();	/* db close must be run */
			abort();
		}
	}

	return 0;
}

static void print_usage()
{
	puts("usage: satcfg [ -f FILE | -a TOKEN SECRET | -d TOKEN | -l TOKEN | -L ]...");
	puts("       -f FILE           Select configuration file");
	puts("       -a TOKEN SECRET   Add token/secret to file");
	puts("       -d TOKEN          Delete token from file");
	puts("       -l TOKEN          List token/secret pair");
	puts("       -L                List all token/secret pairs");
}

static void db_close()
{
	if(db_handle) {
		gdbm_close(db_handle);
		db_handle = NULL;
	}
}

static void db_open()
{
	db_handle = gdbm_open(db_file, 512, GDBM_WRCREAT,
			S_IRUSR|S_IWUSR, NULL);
	if(!db_handle)
		die("%s: %s", db_file, gdbm_error_str());
}

static void db_add(char *token, char *secret)
{
	datum t, s;
	int err;

	if(!db_handle)
		db_open();

	t.dptr = token;
	t.dsize = strlen(token);
	s.dptr = secret;
	s.dsize = strlen(secret);
	err = gdbm_store(db_handle, t, s, GDBM_REPLACE);
	if(err)
		die("'%s' add failed: %s", token, gdbm_error_str());
	else
		printf("'%s' updated.\n", token);
}

static void db_delete(char *token)
{
	datum t;
	int err;

	if(!db_handle)
		db_open();

	t.dptr = token;
	t.dsize = strlen(token);
	err = gdbm_delete(db_handle, t);
	if(err)
		die("'%s' delete failed: %s", token, gdbm_error_str());
	else
		printf("'%s' deleted.\n", token);
}

static void db_list(char *token)
{
	datum t;

	if(!db_handle)
		db_open();

	t.dptr = token;
	t.dsize = strlen(token);

	if(!db_print_datum(t))
		die("'%s' lookup failed: %s", datum2str(t), gdbm_error_str());
}

/* Returns malloc'd null terminated string */
static char *datum2str(datum d)
{
	char *s;

	s = malloc(d.dsize + 1);
	memcpy(s, d.dptr, d.dsize);
	s[d.dsize] = 0;

	return s;
}

/* Returns:	0 error (gdbm_errno set)
 * 		1 success */
static int db_print_datum(datum token)
{
	datum secret;
	char *t, *s;

	secret = gdbm_fetch(db_handle, token);
	if(!secret.dptr)
		return 0;

	t = datum2str(token);
	s = datum2str(secret);

	printf("%-20s\t%-20s\n", t, s);

	free(secret.dptr);
	free(t);
	free(s);

	return 1;
}

static void db_listall()
{
	datum t;

	if(!db_handle)
		db_open();

	t = gdbm_firstkey(db_handle);
	while(t.dptr) {
		if(!db_print_datum(t))
			die("'%s' lookup failed: %s", datum2str(t), gdbm_error_str());
		free(t.dptr);
		t = gdbm_nextkey(db_handle, t);
	}
}
