/* Copyright 2001  Mark Pulford <mark@kyne.com.au>
 * This file is subject to the terms and conditions of the GNU General Public
 * License. Read the file COPYING found in this archive for details, or
 * visit http://www.gnu.org/copyleft/gpl.html
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <gdbm.h>
#include <errno.h>

#include "common.h"
#include "error.h"

char *db_file = TOKEN_FILE;

static int timed_out;

static RETSIGTYPE alarm_handler(int sig);

/* Returns NULL if the look up fails */
const char *get_secret_by_token(const char *token, const char **dberr)
{
	GDBM_FILE db;
	static char secret[TOKEN_SIZE+1];
	datum t, s;
	int size;

	*dberr = NULL;

	db = gdbm_open(db_file, 512, GDBM_READER, 0, NULL);
	if(!db) {
		*dberr = gdbm_error_str();
		return NULL;
	}

	t.dptr = (char *)token;
	t.dsize = strlen(token);
	s = gdbm_fetch(db, t);
	if(!s.dptr) {
		if(GDBM_NO_ERROR != gdbm_errno)
			*dberr = gdbm_error_str();
		gdbm_close(db);
		return NULL;
	}
	gdbm_close(db);

	/* leave space for null terminator */
	size = s.dsize<(sizeof(secret)-1) ? s.dsize : sizeof(secret)-1;
	memcpy(secret, s.dptr, size);
	secret[size] = 0;

	return secret;
}

static RETSIGTYPE alarm_handler(int sig)
{
	timed_out = 1;
}

/* Returns: 0	error (socket closed)
 * 	    1	success */
int put_line(int fd, const char *data)
{
	int ret;
	int len;
	char *buf;

	buf = strdup(data);
	if(!buf)
		abort();

	len = strlen(buf);
	buf[len++] = '\n';	/* Overwrite NULL */

	signal(SIGPIPE, SIG_IGN);

	ret = safe_write(fd, buf, len);
	free(buf);
	if(ret != len) {
		close(fd);
		return 0;
	}
	return 1;
}

/* Wrapper for write. Handles partial & interrupted writes.
 * Returns: -1 error (errno set)
 *          otherwise bytes written */
int safe_write(int fd, const char *buf, int size)
{
	int i=0;
	int ret;

	while (i < size) {
		ret = write(fd, buf+i, size-i);
		if(-1 == ret) {
			if(EINTR == errno)
				continue;
			else
				return -1;
		}
		if(0 == ret)
			return i;
		i += ret;
	}

	return i;
}

/* Gets a line at a time. Discards anything after a '\n'.
 * FILE* don't have this problem, but their signal handling gets in the way */
/* Returns NULL on error. *err set:
 * 	NULL remote closed
 * 	otherwise description */
const char *get_line(int fd, const char **err)
{
	static char buf[TOKEN_SIZE+1];	/* includes \n */
	int rsize;
	char *end;
	int len = 0;

	timed_out = 0;
	signal(SIGALRM, alarm_handler);
	alarm(180);

	while(len < sizeof(buf)) {
		rsize = read(fd, buf+len, sizeof(buf)-len);
		if(-1 == rsize) {
			if(timed_out) {
				*err = "Connection timeout";
				close(fd);
				return NULL;
			} else if(EINTR == errno) {
				continue;
			} else {
				*err = strerror(errno);
				close(fd);
				return NULL;
			}
		}
		if(0 == rsize) {
			*err = NULL;
			return NULL;
		}
		end = memchr(buf+len, '\n', rsize);
		len += rsize;
		if(end)
			break;
	}
	if(!end) {
		*err = "Line too long. Connection closed.";
		close(fd);
		return NULL;
	}

	*end = 0;
	len = end - buf;

	/* Strip CR for telnet debugging. Note the string can't reach
	 * TOKEN_SIZE when '\r' is sent */
	if(len > 0 && '\r' == buf[len-1])
		buf[--len] = 0;

	if(!len) {
		*err = "Blank line received. Connection closed.";
		close(fd);
		return NULL;
	}

	return buf;
}
