/* ------------------------------------------------------------------------ */
/* NEtalk.c : Network Element talk program.				    */
/*									    */
/* 12/01/99 Forget older versions. This is completely revised.		    */
/*									    */
/* Created by Dimitris Evmorfopoulos (devmorfo@algo.com.gr)		    */
/*									    */
/* This program was originaly uploaded to ftp://ftpeng.cisco.com/incoming   */
/* and can be retrieved from there.					    */
/*									    */
/* You can also find this program under all linux structures of sunsite     */
/* ftp servers arround the globe under the incoming directory or wherever   */
/* their admins decided to place it.					    */
/* ------------------------------------------------------------------------ */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <termios.h>
#ifdef HAS_GETOPT
#include <getopt.h>
#else
#include <stdlib.h>
#endif

#include <arpa/telnet.h>

/* ------------------------------------------------------------------------ */
/* A few defs								    */
/* ------------------------------------------------------------------------ */

#ifndef CONFIG_FILE
#define CONFIG_FILE		"/etc/netalk.conf"
#endif

#define	 RETRIES 		3
#define  DELAY_BETWEEN_RETRIES 	2
#define	 CONNECTION_TIMEOUT	120
#define	 WAIT_TIMEOUT		(200*1000)

#define LINE_BUFFER_SIZE	(16*1024)

/* ------------------------------------------------------------------------ */
/* Some needed structures.						    */
/* ------------------------------------------------------------------------ */

struct telcmd {
  unsigned char cmd, opt;
  };

/* ------------------------------------------------------------------------ */
/* The only needed global var. I'm so ashamed of this.			    */
/* ------------------------------------------------------------------------ */

struct termios orig;

/* ------------------------------------------------------------------------ */
/* Lets exit in a proper way.						    */
/* ------------------------------------------------------------------------ */

void __exit(int value) {
  tcsetattr(0, TCSANOW, &orig);
  exit(value);
  }

/* ------------------------------------------------------------------------ */
/* Function to read a single byte from the socket. Takes into consideration */
/* the fact that there might be a telnet option.			    */
/* ------------------------------------------------------------------------ */

int read_sock(int 	    sockf, 
              unsigned char *c, 
	      int           capture, 
	      struct telcmd *cmd) {

  /* We are comming from a select on sockf. If we cannot read then we have
     a broken connection, thus we should exit. */
  if (read(sockf, c, 1) == 0) __exit(0);
  if (*c == IAC) {
    if (read(sockf, c, 1) == 0) __exit(0);
    cmd->cmd = *c;
    if ((*c == DO) || (*c == DONT) || (*c == WILL) || (*c == WONT)) {
      if (read(sockf, c, 1) == 0) __exit(0);
      cmd->opt = *c;
      }
    return 0;
    }
  else {
    if (capture) {
      fputc((int)*c, stdout);
      fflush(stdout);
      }
    }
  return 1;
  }

/* ------------------------------------------------------------------------ */
/* This is our support band. How can WE have a concert without one of those */
/* This is read from the socket with timeout storing results into a sliding */
/* buffer ... We just hope that our buffer is long enough. Still we wont    */
/* segfault if the buffer is small, we will simply abort.		    */
/* ------------------------------------------------------------------------ */

int read_t(int  sockf, 
	   char *expect,
	   int  timeout, 
	   int  waittime, 
	   int  capture,
	   int  inputfail) {

  int  		 retval, i = 0;
  int		 max_len, wrap;
  unsigned char  c;
  unsigned char	 line[LINE_BUFFER_SIZE]; 
  fd_set 	 rfds;
  struct timeval tv;
  struct telcmd  cmd;

  max_len = strlen(expect);
  wrap = max_len * 4;
  if (wrap >= LINE_BUFFER_SIZE) __exit(-2);

  line[0] = (char)0;

  while (1) {

    FD_ZERO(&rfds); 
    FD_SET(sockf, &rfds); 
    tv.tv_sec = timeout;
    tv.tv_usec = 0;

    if (select(sockf+1, &rfds, NULL, NULL, &tv) > 0) {
      if (read_sock(sockf, &c, capture, &cmd)) {
        line[i++] = c; line[i] = (char)0;
        if (i >= wrap) {
          memmove(&line[0], &line[wrap - max_len], max_len);
          line[max_len] = (char)0;
          i = max_len;
          }

        /* Check if we got the expected value */
        if ( ((expect) ? strstr((char *)line, expect) : NULL) ) break;
        }
      }
    else 
      return (!inputfail);
    }

  /* If we found what is expected make sure there is nothing else waiting */
  while (1) {
    FD_ZERO(&rfds); 
    FD_SET(sockf, &rfds); 
    tv.tv_sec = 0;
    tv.tv_usec = waittime;
    if (select(sockf+1, &rfds, NULL, NULL, &tv) > 0)
      read_sock(sockf, &c, capture, &cmd);
    else 
      return 1;
    }
  }

/* ------------------------------------------------------------------------ */
/* The network setup part of talk; nough said.				    */
/* ------------------------------------------------------------------------ */

int net_setup(char *host,
	      int  verbose, 
	      int  retries, 
	      int  delay) {

  struct hostent 	*hostptr;
  struct sockaddr_in	 sock; 
  int	 		sockf, retry_num = 0;

  /* Get the hostent structure for the given host */
  if ((hostptr = gethostbyname(host)) == NULL) {
    if (verbose) fprintf(stderr, "Host %s cannot be found\n", host);
    return -1;
    }

  /* Open a socket to use for our connection */
  if ((sockf = socket(hostptr->h_addrtype, SOCK_STREAM, 0)) < 0) {
    if (verbose) perror("Cannot create socket");
    return -1;
    }

  /* Fix the socket record to show host and service port */
  sock.sin_family = hostptr->h_addrtype;
  sock.sin_port = htons(23);
  memcpy(&sock.sin_addr, hostptr->h_addr, hostptr->h_length);

  /* Fancy connection message if allowed */
  if (verbose) {
    fprintf(stderr, "Connecting to %s ", host);
    fflush(stderr);
    }

  /* Loop a few times until we get a connection, or until we reach retries */
  while (connect(sockf, (struct sockaddr*)&sock, sizeof(sock)) < 0) {

    /* Fancy connection message if allowed (cont) */
    if (verbose) {
      fprintf(stderr, ".");
      fflush(stderr);
      }

    /* We failed miserably ... sleep over it for a while */
    sleep(delay);

    /* Redo socket record. Its not safe after the connect for some OS's */
    sock.sin_family = hostptr->h_addrtype;
    sock.sin_port = htons(23);
    memcpy(&sock.sin_addr, hostptr->h_addr, hostptr->h_length);

    /* Count the retries we've made so far */
    if (++retry_num == retries) {
      /* Since we hit rock bottom here, end this fancy message. */
      if (verbose) fprintf(stderr, ". failed \nAborting \n\n");
      return -1;
      }
    }

  /* Finish that fancy connection message if allowed (end) */
  if (verbose) fprintf(stderr, ". connected\n");

  return sockf;
  }

/* ------------------------------------------------------------------------ */
/* Negotiate the telnet protocol. Promote our ignorance about it ;)	    */
/* For comments on how this works, look for the appropriate RFC's. 	    */
/* ------------------------------------------------------------------------ */

void do_telnet(int sockf, int verbose, int capture) {
  unsigned char  to_send[1024];
  unsigned char  c;
  int		 len = 0, res, sentout = 0;
  struct telcmd  cmd;

  if (verbose) fprintf(stderr, "Attempting to negotiate telnet options\n");

  /* submit to server our state */
  to_send[len] = IAC; len++;
  to_send[len] = DONT; len++;
  to_send[len] = TELOPT_LINEMODE; len++;

  to_send[len] = IAC; len++;
  to_send[len] = DO; len++;
  to_send[len] = TELOPT_SGA; len++;

  to_send[len] = IAC; len++;
  to_send[len] = DONT; len++;
  to_send[len] = TELOPT_ECHO; len++;

  write(sockf, &to_send, len);

  len = 0;

  /* Loop until we get the servers state */
  while (1) {
    fd_set 	   rfds;
    struct timeval tv;

    FD_ZERO(&rfds); 
    FD_SET(sockf, &rfds); 
    tv.tv_sec = 1;
    tv.tv_usec = 0;

    if ((res = select(sockf + 1, &rfds, NULL, NULL, &tv)) > 0) {
      if (!read_sock(sockf, &c, capture, &cmd)) {
	if ((cmd.cmd >= WILL) && (cmd.cmd <= DONT)) {
	  switch (cmd.opt) {
	    case TELOPT_NAWS : 
	    case TELOPT_ECHO :
	    case TELOPT_SGA  : switch (cmd.cmd) {
			         case DO   : cmd.cmd = WILL; break;
			         case DONT : cmd.cmd = WILL; break;
			         case WILL : cmd.cmd = DO; break;
			         case WONT : cmd.cmd = DO;
			         }
			       break;
	    default	     : switch (cmd.cmd) {
			         case DO   : cmd.cmd = WONT; break;
			         case DONT : cmd.cmd = WONT; break;
			         case WILL : cmd.cmd = DONT; break;
			         case WONT : cmd.cmd = DONT; break;
			         default   : __exit(-4);
			         }
	    }
          to_send[len] = IAC; len++;
          to_send[len] = cmd.cmd; len++;
          to_send[len] = cmd.opt; len++;
          }
        }
      else {
        sentout = 1;
        write(sockf, &to_send, len);
	if ((c != '\n') && (c != '\r') && (c != ' ')) break;
        }
      } 
    else {
      if (!sentout) {
        write(sockf, &to_send, len);
	sentout = 0;
        }
      }
    }
  }

/* ------------------------------------------------------------------------ */
/* Chat script processing. Simple but effective.			    */
/* ------------------------------------------------------------------------ */

int chat(int   sockf, 
	 char *script, 
	 int   timeout, 
	 int   waittime, 
	 int   verbose,
	 int   capture,
	 int   inputfail) {

  int	i = 0,j;
  int   expect = 0;
  char  string[1024] = "'";
  char  quote;

  /* I had trouble remembering how to quote a single quote in single quotes */
  quote = string[0];

  while (i < strlen(script)) {
    if (script[i] == quote) {
      expect = (!expect);
      for (j = i+1; (script[j] != quote) && (j < strlen(script)); j++) 
        string[j - (i + 1)] = script[j];
      string[j - (i + 1)] = (char)0;
      if (expect) {
	if (strlen(string) != 0) {
          if (verbose) fprintf(stderr, "Chat expect (%s)\n", string);
          if (!read_t(sockf, string, timeout, waittime, capture, inputfail)) {
            if (verbose) fprintf(stderr, "Error during read \nAborting\n\n");
            return 0;
            }
	  }
        }
      else {
        if (verbose) fprintf(stderr, "Chat send (%s)\n", string);
	write(sockf, string, strlen(string));
        write(sockf, "\r\n", 2);
        }
      i = j + 1;
      }
    else
      i++;
    }
  if (expect) {
    if (verbose) fprintf(stderr, "Chat ended with expect\nAborting\n\n");
    return 0;
    }
  write(1, "\n", 1);
  return 1;
  }

/* ------------------------------------------------------------------------ */
/* This is the heart of the whole thing. Connects,talks and gets the output */
/* of the router, prints it and exits. Oh .. did I mentioned that it works  */  
/* on ther hosts as well ... not all of course, but most. Those pesky IBM's */
/* refuse to have a proper telnet server with them ... nough said.	    */
/* ------------------------------------------------------------------------ */

int talk(char *host, 
	 int   retries, 
	 int   delay, 
	 int   timeout, 
	 int   waittime, 
	 int   verbose,
	 int   capture,
	 int   inputfail,
	 char *autologin,
	 char *command, 
	 char *script) {

  int	 	 sockf;
  int 		 wait_a_byte = 0, break_on_clean = 0;
  unsigned char  c; 
  fd_set 	 rfds;
  struct timeval tv;
  struct telcmd  cmd;
  struct termios term;

  if ((sockf = net_setup(host, verbose, retries, delay)) < 0) return 0;
  do_telnet(sockf, verbose, capture);

  /* If we have a login script, run it and hope it works */
  if (strlen(autologin) > 0) {
    if (verbose) fprintf(stderr, "Starting autologin\n");
    if (!chat(sockf, autologin, 
	      timeout, waittime, 
              verbose, capture, inputfail))
      return 0;
    if (verbose) fprintf(stderr, "Autologin completed\n");
    }

  /* Write out our commands and get results back. */
  if (strlen(command) > 0) {
    if (verbose) fprintf(stderr, "Running script from command line\n");
    return chat(sockf, command, 
		timeout, waittime, 
		verbose, capture, inputfail);
    }

  /* If we have a script to run execute it now */
  if (strlen(script) > 0) {
    if (verbose) fprintf(stderr, "Running script from config file\n");
    return chat(sockf, script, 
		timeout, waittime, 
		verbose, capture, inputfail);

    }

  /* Now all we have is an interactive shell over our stdin. */ 

  /* Make sure our stdin does not echo and does process input a character 
     at a time */
  memcpy(&term, &orig, sizeof(term));
#ifdef NEEDS_CRNL
  term.c_lflag &= ~(ECHO | ICANON);
#else
  term.c_lflag &= ~(ECHO | ICANON);
#endif
  tcsetattr(0, TCSANOW, &term);

  if (verbose) fprintf(stderr, "Running interactively\n");
  while (1) {
    FD_ZERO(&rfds); 
    if ((!break_on_clean) && (!wait_a_byte)) FD_SET(0, &rfds); 
    FD_SET(sockf, &rfds); 
    if (!break_on_clean) {
      tv.tv_sec = timeout;
      tv.tv_usec = 0;
      }
    else {
      tv.tv_sec = 0;
      tv.tv_usec = waittime;
      }
      
    if (select(sockf + 1, &rfds, NULL, NULL, &tv) > 0) {
      if (FD_ISSET(sockf, &rfds)) {
        while (1) {
	  wait_a_byte = 0;
          read_sock(sockf, &c, capture, &cmd);

          FD_ZERO(&rfds); 
          FD_SET(sockf, &rfds); 
          tv.tv_sec = 0;
          tv.tv_usec = waittime;

          if (select(sockf+1, &rfds, NULL, NULL, &tv) <= 0) {
            if (break_on_clean) {
              write(1, "\n", 1);
              return 1;
              }
     	    break;
            }
          }
        }
      else {
        if (FD_ISSET(0, &rfds)) {
          if (!read(0, &c, 1)) break_on_clean = 1;
          else {
            if (c == '\n') {
              write(sockf, "\r\n", 2);
	      wait_a_byte = 1;
              }
            else 
              write(sockf, &c, 1);
            }
          }
        }
      }
    else {
      if (verbose) fprintf(stderr, "Timeout.\nAborting\n\n");
      break;
      }
    }

  write(1, "\n", 1);
  return 1;
  }

/* ------------------------------------------------------------------------ */
/* Config file parsing. These read in the necessary host specific variables */
/* along with the autologin script for the host (if any) and the requested  */
/* script (if any). 							    */
/* ------------------------------------------------------------------------ */

char *key_words[] = {"verbose",   "capture", "prompt-fail",
		     "retries",   "delay",   "timeout", 
		     "wait-time", "host",    "script", 
		     "end",       "login",   "chat",    NULL};

/* ------------------------------------------------------------------------ */

int get_line(FILE *fconf, int *key, char *value) {
  char line[1024] = "'", 
       *point, *rest, quote;
  int  i, even;
  
  quote = line[0];
  *key = -1;
  while (*key == -1) {
    if (fgets(line, sizeof(line), fconf) == NULL) return 0;
    i = strlen(line) - 1;

    /* clean up spaces/newlines at the end of the line */
    while ( ((line[i] == ' ') || (line[i] == '\t') || 
	     (line[i] == '\n') || (line[i] == '\r')) && (i >= 0) ) 
      line[i--] = (char)0;
    if (strlen(line) == 0) continue;

    /* clean up leading spaces */
    i = 0;
    point = &line[i];
    while ( ((*point == ' ') || (*point == '\t')) && (*point != (char)0) ) 
      point++;

    /* Is this an empty comment only line ?? */
    if (*point == '#') continue;

    /* point has the keyword. Lets isolate this from the rest */
    rest = point;
    while ((*rest != ' ') && (*rest != '\t') && (*rest != (char)0)) rest++;
    *rest = (char)0;

    /* Now lets place the value into rest */
    rest = rest++;

    /* remove leading spaces from rest */
    while ( ((*rest == ' ') || (*rest == '\t')) && (*rest != (char)0) ) rest++;
    
    /* remove any comments from the rest */
    i = 0;
    even = 1;
    while (rest[i] != (char)0) {
      if (rest[i] == quote) even = !even;
      if (even) {
        if (rest[i] == '#') {
          rest[i] = (char)0;
          break;
          }
        }
      i++;
      }
    sprintf(value, "%s", rest);

    /* find out the keyword key value */
    for (i = 0; key_words[i] != NULL; i++)
      if (strcasecmp(key_words[i], point) == 0) *key = i;
    }
  return 1;
  }

/* ------------------------------------------------------------------------ */

int correct_host(char *value, char *hostname) {
  char host[1024], *point;

  sprintf(host, "%s", value);
  while ((point = rindex(host, (int)'.')) != NULL) {
    if (strcasecmp(hostname, host) == 0) return 1;
    *point = (char)0;
    }
  if (strcasecmp(hostname, host) == 0) return 1;
  return 0;
  }

/* ------------------------------------------------------------------------ */

int correct_name(char *value, char *script) {
  if (strcasecmp(value, script) == 0) return 1;
  return 0;
  }

/* ------------------------------------------------------------------------ */

int boolean(char *value) {
  if (strcasecmp(value, "on") == 0) return 1;
  return 0;
  }

/* ------------------------------------------------------------------------ */

void get_config(char *hostname, 
		char *script, int script_len, 
		char *autologin, int autologin_len, 
		int  *verbose, 
		int  *capture, 
		int  *inputfail, 
		int  *retries, 
		int  *delay, 
		int  *timeout, 
		int  *waittime,
		char *config_file) {

  FILE		*fconf;
  int		key;
  char		value[1024];

  if ((fconf = fopen(config_file, "r")) != NULL) {
    while (get_line(fconf, &key, value)) {
      switch (key) {
	case 0 : *verbose = boolean(value);
		 break;
	case 1 : *capture = boolean(value);
		 break;
	case 2 : *inputfail = boolean(value);
		 break;
	case 3 : *retries = atoi(value);
		 break;
	case 4 : *delay = atoi(value);
		 break;
	case 5 : *timeout = atoi(value);
		 break;
	case 6 : *waittime = atoi(value) * 1000;
		 break;
	case 7 : if (correct_host(value, hostname)) {
                   int end_loop = 0;

		   while (!end_loop) {
		     if (get_line(fconf, &key, value)) {
		       switch (key) {
			 case 0 : *verbose = boolean(value);
		 		  break;
		 	 case 1 : *capture = boolean(value);
		 		  break;
		 	 case 2 : *inputfail = boolean(value);
		 		  break;
		  	 case 3 : *retries = atoi(value);
		 		  break;
		 	 case 4 : *delay = atoi(value);
		 		  break;
		 	 case 5 : *timeout = atoi(value);
		 		  break;
		 	 case 6 : *waittime = atoi(value);
		 		  break;
			 case 9 : end_loop = 1;
				  break;
	                 case 10: strncpy(autologin, value, autologin_len);
			 default: break;
			 }
                       }
		     else {
                       fclose(fconf);
		       return;
                       }
                     }
		   }
		 else {
		   while (1) {
		     if (get_line(fconf, &key, value)) {
		       if (key == 9) break;
                       }
		     else {
                       fclose(fconf);
		       return;
                       }
                     }
		   }
		 break;
	case 8 : if (correct_name(value, script)) {
		   int end_loop = 0;

		   while (!end_loop) {
		     if (get_line(fconf, &key, value)) {
		       switch (key) {
			 case 9 : end_loop = 1;
				  break;
		 	 case 11: strncpy(script, value, script_len);
			 default: break;
			 }
                       }
		     else {
                       fclose(fconf);
		       return;
		       }
		     }
		   }
		 else {
		   while (1) {
		     if (get_line(fconf, &key, value)) {
		       if (key == 9) break;
                       }
		     else {
                       fclose(fconf);
		       return;
                       }
                     }
		   }
		 break;
        }
      }
    fclose(fconf);
    }
  else {
    if (*verbose) fprintf(stderr, "Cannot open file (%s)\n", config_file);
    }
  }

/* ------------------------------------------------------------------------ */
/* Usage. Need I say more ? We are just being a little fancy here.	    */
/* ------------------------------------------------------------------------ */

char *usage_message[] = {
  "netalk [-vnf] [-s name] [-c chat] [-rdtw num] [-C file] host",
  "   -v      : Be verbose. Prints debug messages",
  "   -n      : No output capture. Nothing will be returned.",
  "   -f      : Fail when the expect values are not matched.",
  "   -s name : Run chat script from configuration file.",
  "   -c chat : Execute this command line chat script.",
  "   -r num  : Number of retries to connect.",
  "   -d num  : Delay in seconds between retries.",
  "   -t num  : Timeout in seconds for all reads. Zero disables.",
  "   -w num  : Wait time in miliseconds for net read completion.",
  "   -C file : Read configuration from specified file.\n",
  "The chat scripts defined for netalk are sets of expect/send pairs.",
  "Each pair HAS to end with a send value if you want further scripts",
  "to work for the same connection. Strings must be in single quotes.\n",
  "For help, comments or your depest desires contact <devmorfo@algo.com.gr>.",
  NULL};

void usage(int ret_value) {
  int i;
  for (i = 0; usage_message[i] != NULL; i++) 
    fprintf(stderr, "%s\n", usage_message[i]);
  __exit(ret_value);
  }

/* ------------------------------------------------------------------------ */
/* Glorious main ... cant do shit without you.				    */
/* ------------------------------------------------------------------------ */

int main(int argc, char *argv[]) {
  int   i;
  char  hostname[1024] = "", 
  	command[1024] = "",
	script[1024] = "",
	autologin[1024] = "",
	config_file[1024] = CONFIG_FILE;
  int   verbose = 0, v_ = -1, 
        capture = 1, c_ = -1, 
	inputfail = 0, i_ = -1, 
        retries = RETRIES, r_ = -1, 
        delay = DELAY_BETWEEN_RETRIES, d_ = -1, 
        timeout = CONNECTION_TIMEOUT, t_ = -1, 
	waittime = WAIT_TIMEOUT, w_ = -1;

  tcgetattr(0, &orig); 

  while ((i = getopt(argc, argv, "?vnfs:c:r:d:t:w:C:")) != EOF) {
    switch (i) {
      case 'v' : v_ = 1;
      		 break;
      case 'n' : c_ = 0;
      		 break;
      case 'f' : i_ = 1;
      		 break;
      case 's' : strncpy(script, optarg, sizeof(script));
      		 break;
      case 'c' : strncpy(command, optarg, sizeof(command));
      		 break;
      case 'r' : r_ = atoi(optarg);
      		 break;
      case 'd' : d_ = atoi(optarg);
      		 break;
      case 't' : t_ = atoi(optarg);
      		 break;
      case 'w' : w_ = atoi(optarg) * 1000;
      		 break;
      case 'C' : strncpy(config_file, optarg, sizeof(config_file));
      		 break;
      case '?' :
      default  : usage(1);
      }
    }

  /* The option that does not have a switch is the host. */
  /* Only the first non switched option is taken into consideration */
  if (optind < argc) strncpy(hostname, argv[optind], sizeof(hostname));

  /* Did we get a hostname to connect to ?? */
  if (strlen(hostname) == 0) usage(1);

  /* Get from configuration file scripts and values for variables */
  get_config(hostname, script, sizeof(script), 
  	     autologin, sizeof(autologin), 
  	     &verbose, &capture, &inputfail, &retries, 
  	     &delay, &timeout, &waittime, config_file);
 
  /* Values over the command line override all other values */
  if (v_ != -1) verbose = v_;
  if (c_ != -1) capture = c_;
  if (i_ != -1) inputfail = i_;
  if (r_ != -1) retries = r_;
  if (d_ != -1) delay = d_;
  if (t_ != -1) timeout = t_;
  if (w_ != -1) waittime = w_;

  /* We are ready, so lets have a chat with the host */
  if (!talk(hostname, retries, delay, 
  	    timeout, waittime, verbose, 
  	    capture, inputfail, autologin, 
  	    command, script)) {
    if (!verbose) fprintf(stderr, "An error has occured. Please try again.\n");
    __exit(-1);
    }
  __exit(0);
  }

/* ------------------------------------------------------------------------ */
/*  Thats all folks.							    */
/* ------------------------------------------------------------------------ */
