// This is the wipld daemon process.
//
// Possible improvements: 
//
// 1) Make the process fork later making it possible to write error messages 
//    to the console instead of only to the log.

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>
#include <errno.h>
#include <sys/stat.h>
#include <stdio.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include "common.h"
#include "parseprg.h"
#include "shmem.h"
#include "config.h"
#include "logfile.h"
#include "pcapintf.h"
#include "sigctl.h"
#include "prgcommon.h"
#include "proxyremap.h"
#include "proxydict.h"

// This class creates the shared memory area. When given packets to
// the consume function it then updates the area accordingly.
class CardCollect:ScriptExec {
  static ip_port zero_port;
  
  ShMemServer serv;
  LogFile* lf;      // File we log to
  Config* cnf;
  int memsize;     // Used by getShMemSize()
  card_id_type cit;
  
  // Called on compilation:
  virtual ParseProgram::types getSymbol(char* name, void*& location);
  
  // Called on running:
  char printbuffer[128]; // Buffer for printing
  virtual void print(const char* string);  

  // Update IP/MAC information of the given card:
  void updateIpMac(ServerMem* sa, mac_addr& m, ip_addr& ip);

  public:
  
  // This inits the daemon.
  // Returns 0 or an error message
  char* init(Config*, LogFile*);    
  
  // After successfull call to init this can be called:
  int getShMemSize() { return memsize; }
  
  PcapInterface::Packet pack;

  // Consumes the packet pack. Returns 0 on success or an error message on
  // a fatal error.
  char* Consume();
};

ip_port CardCollect::zero_port=0;

ParseProgram::types CardCollect::getSymbol(char* name, void*& location) {
  if(!strcmp(name,"srcmac"))  { location=&pack.maca_src; return tMAC; }
  if(!strcmp(name,"dstmac"))  { location=&pack.maca_dst; return tMAC; }
  if(!strcmp(name,"srcip"))   { location=&pack.ip_src; return tIP; }
  if(!strcmp(name,"dstip"))   { location=&pack.ip_dst; return tIP; }
  if(!strcmp(name,"srcport")) { if(pack.proto==IPPROTO_TCP || pack.proto==IPPROTO_UDP) location=&pack.port_src; else location=&zero_port; return tINT32; }
  if(!strcmp(name,"dstport")) { if(pack.proto==IPPROTO_TCP || pack.proto==IPPROTO_UDP) location=&pack.port_dst; else location=&zero_port; return tINT32; }
  if(!strcmp(name,"size"))    { location=&pack.ip_len; return tINT32; }
 
  // Else do default operation:
  return ScriptExec::getSymbol(name,location);
}  

char* CardCollect::init(Config* cnf, LogFile* l) {
  lf=l;
  int counterc=cnf->counterc();
  int maxcardc=cnf->maxcardc();
  cit=cnf->cit();
  this->cnf=cnf;
      
  // Check parameters:
  if(maxcardc<=0) return "At least one card should be allowed";
  if(counterc<=0 || counterc>=maxcounterc) {
    snprintf(errmsg,sizeof(errmsg),"Between 1 and %i counters must be defined",maxcounterc); return errmsg;
  }    
  
  // Compile program and return error message on error:
  assert_cit(cit);
  char* c=setProgram(cnf->prgline(),cnf->program());
  if(c) return c;

  // Create the shared memory:
  void* memory;
  memsize=sizeof(ServerMem)+maxcardc*sizeof(CardInfo);
  if(cnf->proxymaxcon()) memsize+=proxyGetMemSize(cnf->proxymaxcon());
  c=serv.init(cnf->daemonfile(),memsize,memory);
  if(c) return c;

  // An initialize it:
  ServerMem* ca=(ServerMem*)memory;      
  ca->Counterc=counterc;
  for(int a=0; a<counterc; a++) {
    strncpy(ca->CounterNames[a],cnf->countername(a),MaxCounterNameLen);
    ca->CounterNames[a][MaxCounterNameLen]='\x0';
  }
  for(int a=counterc; a<maxcounterc; a++) {
    strcpy(ca->CounterNames[a],"(unused)");
  }
  
  ca->maxcardc=maxcardc;
  ca->curcardc=0;
  ca->cit=cit;
  
  ca->proxymaxcon=cnf->proxymaxcon();
  if(ca->proxymaxcon) {
    proxyInitMem(&ca->cards[ca->maxcardc],cnf->proxymaxcon());
  }

  gettimeofday(&ca->start.tv,0); 
  serv.UnLock(ca);
  
  return 0;
}

char* CardCollect::Consume() {
  // Lock the shared memory:
  ServerMem* ca=(ServerMem*)serv.Lock();
  if(ca==0) return LockErrString;
  
  // Change address if we are redirecting from a proxy server
  if(ca->proxymaxcon) {
    void* proxydata=&ca->cards[ca->maxcardc]; // Here the proxy data are

    // Remap source address:
    ProxyRemapBlock* prm=proxyLookup(proxydata,pack.ip_src.asUnsigned(),htons(pack.port_src),pack.proto);
    if(prm) {
      // And we have a match in the proxy server - redirect:
      pack.ip_src=prm->caddr;
      pack.maca_src=prm->macaddr;
    }
      
    // This is symetric on above just on destination address:
    prm=proxyLookup(proxydata,pack.ip_dst.asUnsigned(),htons(pack.port_dst),pack.proto);
    if(prm) {
      pack.ip_dst=prm->caddr;
      pack.maca_dst=prm->macaddr;
    }
  }

  // CONFIGURABLE: Do not make statisics on a per-machine basis.
  //
  // You might be in a situation where you do not want to make
  // statistics on a per-machine basis. For example you might want to
  // consider two differnt machines as one. Or you migh want 
  // to consider different packets comming from the same machine as 
  // comming from different machines.
  //
  // This is the place where you will want to put code to change
  // the contents of a packet before giving it to the user program.
  // The packet is placed in the variable pack. It is of type
  // PcapInterface::Packet which you can find in pcapintf.h.
    
  // Execute the program:
  printbuffer[0]='\x0';
  char* error=ScriptExec::Consume(ca,pack.time.tv.tv_sec);
  if(printbuffer[0]) { print("\n"); }
  
  // Look for ip/mac address change. Note: We do this after we execute the
  // program. So if the user for some reason changes the mac or ip
  // address of the packet the modfied information is updated
  updateIpMac(ca,pack.maca_src,pack.ip_src);
  updateIpMac(ca,pack.maca_dst,pack.ip_dst);

  // And unlock the memory:
  serv.UnLock(ca);
  
  if(error) return error;  
  return 0;
}  

void CardCollect::print(const char* string) {
  for(;;) {
    char* n=strchr(string,'\n');
    if(!n) {
      int bl=strlen(printbuffer);
      int bufrest=sizeof(printbuffer)-1-bl;
      int strrest=strlen(string);
      int tocopy=strrest<bufrest ? strrest : bufrest;
      strncpy(printbuffer+bl,string,tocopy); 
      printbuffer[bl+tocopy]='\x0';
      return;
    }              
    int l=n-string;
    lf->log(LogFile::print,"User program printed: %s%.*s",printbuffer, l, string);
    printbuffer[0]='\x0';
    string=n+1;
  }
}

void CardCollect::updateIpMac(ServerMem* sa, mac_addr& mac, ip_addr& ip) {
  card_id cid;
  memset(&cid,0,sizeof(cid));
  switch(cit) {
    case idIP:
      cid.ipaddr=ip;
      break;
    case idMAC:
      cid.macaddr=mac;
      break;
  }

  char* foo=0;
  int idx=getIdIdx(cid,0,foo);
  if(idx<0) return;
  CardInfo* a=&sa->cards[idx];
  
  if(cit==idMAC && a->ipaddr!=ip) {
    // If IP address changed:
    char buf[40];
    snprintf(buf,40,"%s",ip.tostr());
    lf->log(LogFile::addrchange,
      "IP address for card %s changed from %s to %s",
      mac.tostr(),a->ipaddr.tostr(),buf);
    a->ipaddr=ip;
  }
  
  if(cit==idIP && a->macaddr!=mac) {
    // MAC adress changed:
    char buf[40];
    snprintf(buf,40,"%s",mac.tostr());
    lf->log(LogFile::addrchange,
      "MAC address for %s changed from %s to %s",      
      ip.tostr(),a->macaddr.tostr(),buf);
    a->macaddr=mac;
  }    
}
  
//-----------------------------------------------------------------------------

// Class used to create file with the process id of the daemon.
class PidFile {
  char* file;

  static void programAbort(void *p) {
    // Delete file:
    unlink(((PidFile*)p)->file);
  };
  
  public:
  PidFile()  : file(0) { registerSignalHandler(this,programAbort,signalTerm); }
  ~PidFile() { unregisterSignalHandler(this,programAbort); programAbort(this); delete[] file; }
  
  char* init(const char* filename) {
    blockSignals();
    file=new char[strlen(filename)+1];
    strcpy(file,filename);
    unblockSignals();
    int h=open(filename,O_CREAT|O_WRONLY|O_APPEND,
       S_IRUSR|S_IWUSR|
       S_IRGRP|S_IWGRP|
       S_IROTH);
    if(h==-1) {    
      snprintf(errmsg,sizeof(errmsg),"Could not create pid file %s: %s",filename,strerror(errno));
      return errmsg;
    }
    char buf[40];
    snprintf(buf,sizeof(buf),"%i\n",(int)getpid());
    write(h,buf,strlen(buf));
    close(h);
    return 0;
  }
};

main(int argc, char*argv[]) {  
  // Parse parameters:
  int debug=0;
  char* conffile=DEFAULT_CNFFILE;

  for(;;) {
    static option long_options[]=
      {
       {"debug",0,0,'b'},
       {"config", 1,0,'c'},
      };
     
     int c=getopt_long(argc,argv,"bc:",long_options,0);
     if(c==EOF) break;
     
     switch(c) {
       case 'b':
         debug=1;
	 break;
       case 'c':
         conffile=optarg;
	 break;
	 
       // Error situations:
       case ':':
         fprintf(stderr,"Missing parameter for option\n");
	 return 1;
       case '?':
         return 1; // The getopt library prints an error message
	 break;	 
       default:
         // Should not happen, I guess
	 fprintf(stderr,"Error parsing commandline arguments\n");
	 return 1;       
     }
  }
  
  if(optind!=argc) {
    fprintf(stderr,"Unknown argument(s) given\n");
    return 1;
  }
    
  // Read configuration file:
  Config* cnf=new Config();
  char* c=cnf->readConfFile(conffile);
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 2;
  }
  
  // Open log file:
  LogFile lf;
  c=lf.init(cnf,debug);
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 2;
  }
  
  if(!debug) {  
    // We have now opened the logfile. Because of this we are able to write
    // errormssages to the log file instead of to the terminal. We can there-
    // fork and let the parent process exit. On the other hand we can't fork
    // later because of problems with destructor calls and other things.
    int i=fork();
    if(i==-1) { // On error:
      fprintf(stderr,"Could not fork: %s\n",strerror(errno));
      return 3;
    }
    if(i!=0) { // On sucess and parent process
      return 0;
    }
  
    // We are now the child process.
  
    // Disassociate from process group so we don't receive signals for the group:
    setpgrp();
  
    // Disassociate from controling terminal:
    int h;
    if(h=open("/dev/tty",O_RDWR)>=0) {
      ioctl(h,TIOCNOTTY,(char*)0);
      close(h);
    }

    // Close all filehandels except for the logfile:
    lf.closeAllOtherFiles();
  }
  
  // Initialize daemon and compile program:  
  CardCollect cc;
  c=cc.init(cnf,&lf);

  // Create pid file.
  PidFile pf;
  c || (c=pf.init(cnf->pidfile()));

  // Change to root directory to allow unmount of cwd:
  if(!debug) {
    chdir("/");
  }
  
  // Start reading from device:
  PcapInterface pi;  
  c || (c=pi.init(cnf->filter(),cnf->netdev(),&lf));
  
  lf.log(LogFile::status,"---");

  if(c) {
    lf.log(LogFile::status,"Daemon could not start: %s",c);
    return 1;
  }
  
  // Print status information:
  lf.log(LogFile::status,"Daemon initialized with pid %i and shared memory size %i",getpid(),cc.getShMemSize());
  lf.log(LogFile::status,"Device: %s, Logfile: %s, Pidfile: %s, Daemonfile: %s",pi.getDev(),cnf->logfile(),cnf->pidfile(),cnf->daemonfile());
  
  delete cnf; // Save a bit memory...
    
  for(;;) { 
    c=pi.nextPacket(cc.pack);
    c || (c=cc.Consume());
    if(c) {
      lf.log(LogFile::status,"Daemon terminated: %s",c);
      return 1;
    }
  }
}  
