#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <getopt.h>

#include "common.h"
#include "shmem.h"

//------------------------------------------------------------------------------
// Overview of this module:
//
//   The class ServerMemClient has the responsibility of all communication
//   with the server. On reqeust it will read all information from the server
//   and return an UserServerMem object indicating the sever data.
//
//   The class OutFile is used to write outdata on the lowest level. It is
//   inherited by HtmlFile which will write an HtmlFile instead of plain ascii.
//
//   The class Print can given an OutFile object write a UserServerMem object
//   
//   The class History is used to compute speeds. It can be feeded by
//   UserServerMem objects from the server and will return other UserServerMem
//   objects indicating the speeds. More precisly the returned objects will
//   indicate the changes to the counters in the last period and this can be
//   printed by the Print class as speeds.
//
//------------------------------------------------------------------------------

// Has information like ServerMem but has additional functionallity:
class UserServerMem {
  friend class ServerMemClient;

  int cardc;    
  int counterc; 
  Time start;
  Time stop;
  ServerMemClient* parent;
  
  CardInfo* cards; 
  // Is always sorted by MAC address. This is just a copy from the shared
  // memory area.
  
  public:
  UserServerMem()  { cardc=0; cards=0; counterc=0; parent=0; }
  ~UserServerMem() { delete[] (char*)cards; }
  
  UserServerMem(const UserServerMem&);
  UserServerMem& operator=(const UserServerMem&);

  // This will subtract all counters in the parameter from the counters in
  // this object. Cards for which all counters become zero are discarded.    
  // Start and stop times are always set apropriate.
  UserServerMem& operator-=(const UserServerMem&);  

  // Computes the totals for all cards:
  CardInfo Total() const; 
    
  // These can be used to retriview the different cards:
  int CardCount() const { return cardc; }
  const CardInfo& operator[](int i) const { assert(i>=0 && i<cardc); return cards[i]; }  
     
  Time GetStart() const { return start; }
  Time GetStop() const { return stop;  }
  
  // Returns a sorted array with all the cards. 
  // If order is -5 the arrays is sorted by MAC address
  // If -4<=order<=-1 the array is sorted by IP address
  // If order>=0 the array is sorted to the counter with number order.
  // If rev is true the order is reversed.
  // NOTE: The caller muse delete this object!
  CardInfo* GetSorted(int order, int reverse) const;
  
  // Returns the object this object was created by:
  ServerMemClient* getParent() const {
    return parent;
  }
};

CardInfo* UserServerMem::GetSorted(int a, int reverse) const {
  // Copy data to a new loaction:
  CardInfo* ret=new CardInfo[cardc];
  for(int b=0; b<cardc; b++) ret[b]=cards[b];
  
  // And sort them:
  CardInfo::reverse=reverse;  
  if(a==-5)
    qsort(ret,cardc,sizeof(CardInfo),CardInfo::compare_mac);    
  else if(a<0)
    qsort(ret,cardc,sizeof(CardInfo),CardInfo::compare_ip);
  else {
    CardInfo::counter=a;
    qsort(ret,cardc,sizeof(CardInfo),CardInfo::compare_counter);    
  } 
  CardInfo::reverse=0;  
  
  return ret;
}

CardInfo UserServerMem::Total() const {
  CardInfo ret;
  ret.setZero();
  ret.ipaddr.setZero();
  ret.macaddr.setZero();
  for(int a=0; a<cardc; a++) {
    ret.add(cards[a],counterc);
  }
  return ret;
}

UserServerMem::UserServerMem(const UserServerMem& c) {
  cards=0; (*this)=c;
}

UserServerMem& UserServerMem::operator = (const UserServerMem& c) {
  cardc=c.cardc;
  counterc=c.counterc;
  start=c.start;
  stop=c.stop;    
  parent=c.parent;
  delete[] (char*)cards;
  cards=(CardInfo*)new char[sizeof(CardInfo)*c.cardc];
  memcpy(cards,c.cards,sizeof(CardInfo)*c.cardc);
  return *this;
}
  
UserServerMem& UserServerMem::operator-=(const UserServerMem& ca) {
  assert(ca.start==start);
  assert(ca.stop<=stop);
  assert(ca.counterc==counterc);
  assert(ca.parent==parent);
  
  int a=0;
  int b=0;
  while(a!=cardc && b!=ca.cardc) {      
    int c=mac_addr::compare(cards[a].macaddr,ca.cards[b].macaddr);
    if(c>0)  // If we try to subtract from a non existing card.
      b++;
    else if(c<0) // We have a card we don't subtract anything from
      a++;
    else { // We are in the same element in both arrays.
      cards[a].sub(ca.cards[b],counterc);
      a++; b++;
    }  
  }
  
  // Now throw away cards having all counters set to 0:
  a=0;
  b=0;
  for(;a<cardc;a++) {
    if(cards[a].isZero()) continue;
    cards[b++]=cards[a];
  }
  cardc=b;
     
  start=ca.stop;
          
  return *this;
}

//------------------------------------------------------------------------------

// Has all communication with the daemon process:
class ServerMemClient {
  // From constructor parameters:
  ShMemClient clnt;
  int cardc,exclude;
  const mac_addr* interested;

  // Read from server:
  int counterc;
  char counternames[maxcounterc][MaxCounterNameLen+1];
  
  public:
  // Inittializes with the given daemon.
  //   cardc:      Number of elements in the array interested
  //   interested: If this is !=0 it only information for the these
  //               cards are handled. THIS MUST BE SORTED BY MAC ADDRESS!
  //   inv:        If this is true and interested!=0 information for all cards
  //               except for the ones in interested are handled.
  char* Init(char* ServerID, int cardc=0, const mac_addr* interested=0, int inv=0);
  
  // Reads from the server and sets its parameter. Returns 0 or an error message:
  char* ReadArray(UserServerMem&);
  
  // Returns counter information from the server:
  int getCounterC() { return counterc; }
  char* getCounterName(int i) { return counternames[i]; }
};

char* ServerMemClient::Init(char* ServerID, int c, const mac_addr* i, int in) {
  char* er=clnt.init(ServerID,1);
  if(er) return er;

  cardc=c;
  interested=i;
  exclude=in;
  
  ServerMem* src=(ServerMem*)clnt.Lock();
  if(!src) return LockErrString;
  
  counterc=src->Counterc;
  memcpy(counternames,src->CounterNames,sizeof(counternames));
  
  clnt.UnLock(src);
  
  return er;
}

char* ServerMemClient::ReadArray(UserServerMem& dst) {
  ServerMem* src=(ServerMem*)clnt.Lock();
  if(!src) return LockErrString;
  delete[] (char*)dst.cards;
  
  dst.start=src->start;
  dst.counterc=src->Counterc;
  dst.parent=this;
  gettimeofday(&dst.stop.tv,0);
  card_id_type cit=src->cit;
    
  if(!cardc) { // Just copy everything
    dst.cardc=src->curcardc;
    dst.cards=(CardInfo*)new char[sizeof(CardInfo)*dst.cardc];
    memcpy(dst.cards,src->cards,sizeof(CardInfo)*dst.cardc);
  } else {
    // We shall not display information for all cards:
    dst.cardc=0;
    dst.cards=(CardInfo*)new char[sizeof(CardInfo)*src->curcardc];
    int i=0;    
    for(int s=0; s<src->curcardc;) {
      int c;
      c=(i==cardc) ? 1 : mac_addr::compare(interested[i],src->cards[s].macaddr);
      if((exclude && c>0) || (!exclude && c==0)) dst.cards[dst.cardc++]=src->cards[s];
      if(c==0) { // The two addresses are equal
        i++; s++;
      } else if(c>0) 
        s++;
      else
        i++;
    }
  }
  clnt.UnLock(src);
  
  // Hack - this program excepts everything to be sorted by MAC address:
  if(cit!=idMAC)
    qsort(dst.cards,dst.cardc,sizeof(CardInfo),CardInfo::compare_mac);      
    
  return 0;
}

//-----------------------------------------------------------------------------

// This is used to compute speeds:
class History {
  UserServerMem* buffer; // A ring buffer with the data given to consume
  int bufalloc;          // Elements allocated in buffer
  int place;             // Index to be written to the next time
  int buflen;            // Number of elements in buffer
  
  public:

  // The class will save len UserServerMem objects. This will, for example, make
  // it possible to show speeds during the last miniute and possible to update
  // this list every 10th second:
  History(int len) { buffer=new UserServerMem[len]; place=0; bufalloc=len; buflen=0; }
  
  ~History() { delete[] buffer; }
  
  // Consums NewPacket and puts information to speed about what has been
  // transmitted for the last period:  
  void Consume(const UserServerMem& NewPacket, UserServerMem& speed);
};

void History::Consume(const UserServerMem& read, UserServerMem& write) {
  write=read;
  if(buflen==bufalloc) {  
    write-=buffer[place];
  } else {
    write-=buflen==0 ? write : buffer[0]; 
    buflen++;
  }
  
  buffer[place++]=read;
  place%=bufalloc;
}

//-----------------------------------------------------------------------------

// This class is used to read a new-line seperated file with MAC addresses.
// These can be given to the ServerMemClient class.

const int MaxMacers=100; // Maximal number of cards that can be read
class ReadMacFile {
  int macc;
  mac_addr* macas;

  public:
  
  ReadMacFile()  : macc(0) { macas=new mac_addr[MaxMacers]; }
  ~ReadMacFile() { delete[] macas; }
  
  // Tries to read the list and returns an error message or 0.
  char* Read(char* filename);
  
  // When the list is read from the file it can be read from:
  int GetMacac() { return macc; }
  const mac_addr* GetMacas() { return macas; }
};

char* ReadMacFile::Read(char* name) {
  FILE* f=fopen(name,"r");
  if(!f) return "Could not read file with MAC addresses";
  macc=0;
  while(!feof(f) && !ferror(f)) {
    char buf[80]; buf[0]='\x0';
    fgets(buf,79,f);
    if(buf[0]) {
      if(macc==MaxMacers) return "Too many MAC adresses in file";
      buf[17]='\x0'; // Fjern eventuelle kommentarer
      if(macas[macc].fromstr(buf))
        return "Invalid MAC address read from file";
      macc++;
    }
  }
  // Sort the addresses:
  qsort(macas,macc,sizeof(mac_addr),&mac_addr::compare);
  
  return 0;	
}

//-----------------------------------------------------------------------------

// This writes to a text file:
class OutFile {
  int lnxclear;
  char* filename;
  protected:  
  FILE *f;
  public:
  OutFile(char* file, int clearscr) : f(0), lnxclear(clearscr), filename(file) { }
  virtual ~OutFile() { assert(!f); }

  virtual void StartPage();
  virtual void put(const char*);
  virtual void NewLine() { put("\n"); }
  virtual void EndPage();
};

void OutFile::StartPage() {
  assert(!f);
  if(!filename) f=stdout;
  else f=fopen(filename,"w");
  if(lnxclear) put("\E[H\E[J");
}

void OutFile::EndPage() {
  assert(f);
  if(filename) fclose(f);
  else fflush(f);
  f=0;
}

void OutFile::put(const char* p) {
  if(f) {
    fwrite(p,1,strlen(p),f);
  }
}

// This writes to a HTML file:
class HtmlFile:public OutFile {
  char* Title;
  int Refresh;
  public:
  // title:   Title to put in HTML page
  // refresh: A refresh rate in seconds to put in file making the browser
  //          update the page automatically
  HtmlFile(char* file, int clearscr, char* title, int refresh=0) 
    : OutFile(file,clearscr), Title(title), Refresh(refresh) {}
  
  void StartPage();
  void EndPage();
};

void HtmlFile::StartPage() {
  OutFile::StartPage();
  put("<HTML><HEAD><TITLE>");
  put(Title ? Title : "");
  put("</TITLE></HEAD><BODY bgcolor=\"white\" fgcolor=\"black\"><PRE>");
  if(Refresh) {
    NewLine();
    char buf[128];
    sprintf(buf,"<META http-equiv=\"refresh\" content=\"%i\">",Refresh);
    put(buf);
    NewLine();
  }
}

void HtmlFile::EndPage() {
  put("</PRE></BODY></HTML>");
  OutFile::EndPage();
}

//-----------------------------------------------------------------------------

class Print {
  int Colc;  // How many colums?
  int* Cols; // And which colums?
  
  int SpeedPrint, PrintTotal, SortOrder, LongFormat, SortReverse, Mega;
  
  // Returns the speed in bytes / second.
  double ComputeSpeed(const UserServerMem& ca, CardInfo& ci, int col);
  
  int PrintHeader(const UserServerMem& ca, OutFile& of);
  void LongPrint(const UserServerMem& ca, OutFile& f);
  void ShortPrint(const UserServerMem& ca, OutFile& f);
  
  public:
  // colc: Number of colums
  // cols: Which values should be put in colums (indexs in the server counters)
  Print(int colc, int* cols, 
       int speedprint, int totalprint, int sortorder, int sortrev,
       int longformat, int mega)
    : Cols(cols), Colc(colc),
      SpeedPrint(speedprint), PrintTotal(totalprint), SortReverse(sortrev),
      SortOrder(sortorder), LongFormat(longformat), Mega(mega) {}
      
  void Prnt(const UserServerMem& ca, OutFile& f) {
    f.StartPage();
    if(LongFormat) LongPrint(ca,f); else ShortPrint(ca,f);
    f.EndPage();
  }
};

double Print::ComputeSpeed(const UserServerMem& ca, CardInfo& ci, int col) {
  double period=(ca.GetStop()-ca.GetStart()).asDouble();
  if(period==0) period=1; // Use one secons as a default
  return double(ci.Counters[Cols[col]])/period;

//  const a=32;
//  double b=pow(2,a);
//  return b;
}

int Print::PrintHeader(const UserServerMem& ca, OutFile& f) {
  char buf[256];
  int a;

  sprintf(buf,"Start time:  %s",ca.GetStart().tostr()); f.put(buf); f.NewLine();
  sprintf(buf,"End time  :  %s",ca.GetStop().tostr());  f.put(buf); f.NewLine();
  f.NewLine();

  if(!ca.CardCount()) {
    f.put("Nothing transmitted"); f.NewLine();
    return -1;
  }
  
  
  return 0;  
}

void Print::LongPrint(const UserServerMem& ca, OutFile& f) {
  char buf[256];

  if(PrintHeader(ca,f)) return;

  CardInfo* ci=ca.GetSorted(SortOrder,SortReverse);
  CardInfo Sum=ca.Total();

  int a;
  for(a=0; a<ca.CardCount()+(PrintTotal ? 1 : 0); a++) {  
    CardInfo* c=(a==ca.CardCount()) ? &Sum : &ci[a];
  
    if(a==ca.CardCount()) {
      f.put("------------------------------"); f.NewLine();
      f.put("For all shown cards             "); f.NewLine();
      f.put("------------------------------"); f.NewLine();
    } else {
      f.put("------------------------------"); f.NewLine();  	
    }
	    
    for(int b=0; b<Colc; b++) {
      if(Cols[b]==-5) { // Print MAC adress:
        if(a!=ca.CardCount()) {
          sprintf(buf,"MAC addr.  : %17s",c->macaddr.tostr());
	  f.put(buf); f.NewLine();
        }
      } else if(Cols[b]>=-4 && Cols[b]<=-1) { // Print IP adress:
        if(a!=ca.CardCount()) {
	  sprintf(buf,"IP addr.   : %17s",c->ipaddr.tostr(-Cols[b]));
	  f.put(buf); f.NewLine();	     
	}
      } else if(SpeedPrint) {
        sprintf(buf,"%-10s : %14.1f%s",
	  ca.getParent()->getCounterName(Cols[b]),
	  ComputeSpeed(ca,*c,b)/(Mega ? 1024.0*1024.0 : 1024.0),
	  Mega ? "MBS" : "KBS");
	f.put(buf); f.NewLine();
      } else {
        sprintf(buf,"%-10s : %16i%s",
	  ca.getParent()->getCounterName(Cols[b]),
	  (int)(c->Counters[Cols[b]]>>(Mega ? 20 : 10)),Mega ? "M" : "K");
        f.put(buf); f.NewLine();	     
      }
    }	
    f.NewLine();
  }	
  delete[] ci;	
}

void Print::ShortPrint(const UserServerMem& ca, OutFile& f) {  
  if(PrintHeader(ca,f)) return;
    
  CardInfo* ci=ca.GetSorted(SortOrder,SortReverse);
  CardInfo Sum=ca.Total();
  
  // Find the width of what we display and print out the headline
  int width=0;
  int a;

  for(a=0; a<Colc; a++) {    
    char buf[256+MaxCounterNameLen];
    if(Cols[a]==-5) {
      width+=17;
      f.put("MAC address      ");
    } else if(Cols[a]>=-4 && Cols[a]<=-1) {
      width+=ip_addr::getmaxlen_tostr(-Cols[a]);
      sprintf(buf,"%-*s",ip_addr::getmaxlen_tostr(-Cols[a]),"IP");
      f.put(buf);
    } else {
      width+=10;
      sprintf(buf,"%-10s",ca.getParent()->getCounterName(Cols[a]));
      f.put(buf);
    }
    
    if(a!=Colc-1) {
      width+=3;
      f.put(" | ");
    }
  }
  f.NewLine();
  
  char* buf=new char[width+50];

  for(a=0; a<ca.CardCount()+PrintTotal ? 1 : 0; a++) {  
    CardInfo* c=(a==ca.CardCount()) ? &Sum : &ci[a];
    
    if(a==ca.CardCount() || a==0) { // Print vertical line
      int b;
      for(b=0; b<width; b++) buf[b]='-';
      buf[b]='\x0';
      f.put(buf);
      f.NewLine();
    }

    for(int b=0; b<Colc; b++) {
      if(Cols[b]<0) {      
        if(Cols[b]==-5) {
          if(a<ca.CardCount()) {    
            f.put(c->macaddr.tostr());
	  } else if(b==0) {
	    f.put("Total:           ");
	  }
        } else {
          int l=ip_addr::getmaxlen_tostr(-Cols[b]);
	  if(a<ca.CardCount()) {
	    if(Cols[b]==-1) { // Right justify
	      sprintf(buf,"%*s",l,c->ipaddr.tostr(-Cols[b]));
	    } else { // Left justify
	      sprintf(buf,"%-*s",l,c->ipaddr.tostr(-Cols[b]));
	    }
	  } else  {
	    if((b==0) && (l>=7)) {
	      strcpy(buf,"Total:");
	    } else buf[0]='\x0';
  	    for(int a=strlen(buf); a<l; a++) strcat(buf," ");
	  }
	  f.put(buf);	    
	}
      } else {
        if(SpeedPrint) {	  
          sprintf(buf,"%7.1f%s",
  	    ComputeSpeed(ca,*c,b)/(Mega ? 1024.0*1024.0 : 1024.0),
	    Mega ? "MBS" : "KBS"	
	  );
	  f.put(buf);
        } else {
          sprintf(buf,"%9i%s",
  	    (int)(c->Counters[Cols[b]]>>(Mega ? 20 : 10)),Mega ? "M" : "K");
          f.put(buf); 
        }
      }
      if(b!=Colc-1) f.put(" | ");
    }
    f.NewLine();
  }  
  delete[] ci;  
  delete[] buf;
}

//-----------------------------------------------------------------------------

main(int argc, char* argv[]) {
  // Read arguments in to these. First the arguments common with
  // the advanced wiplc:
  char* ipcid=DEFAULT_DAEMONFILE;
                     // -d --daemon    Daemon file to use  
  char* outfile=0;   // -o --outfile   The file to print to or 0 on stdout 
  int clearscr=0;    // -c --clearscr  Should a clear-page command be sent to 
                     //                the linux console before each table?
  int sortorder=-5;  // -s --sort      The client colum to sort by.
  int sortrev=0;     // -r --sortrev   Reverse sort order    
  int period_len=0;  // -p --periodlen The length of each period in seconds.
  int end_time=-1;   // -q --quittime  The time_t time where the client should 
                     //                exit.
  char* htmltitle=0; // -t --title
  
  // And the private arguments:
  int html=0;        // -h --html
  int longf=0;       // -l --long
  char* macfile=0;   // -e --macfile
  int mac_exclude=0; // -x --macexclude
  int mega=0;        // -m --mega
  int print_speed=0; // -e --speed
  int history_len=-1;// -i --history
  


  // Parse the option arguments:
  for(;;) {
    static option long_options[]=
      {
       {"daemon",    1,0,'d'},
       {"outfile",   1,0,'o'},
       {"clearscr",  0,0,'c'},
       {"sort",      1,0,'s'},
       {"sortrev",   0,0,'r'},
       {"periodlen", 1,0,'p'},
       {"quittime",  1,0,'q'},
       {"title",     1,0,'t'},
       {"html",      0,0,'h'},
       {"long",      0,0,'l'},
       {"macfile",   1,0,'f'},
       {"macexclude",0,0,'x'},
       {"mega",      0,0,'m'},
       {"speed",     0,0,'e'},
       {"history",   1,0,'i'}
      };
     int c=getopt_long(argc,argv,"d:o:cs:rp:q:t:hlf:xmei:",long_options,0);
     if((c==EOF) || c=='-') break;
     
     switch(c) {
       // Simple arguments:
       case 'c':
         clearscr=1;
	 break;
       case 'r':
         sortrev=1;
	 break;
       case 'h':
         html=1;
	 break;        
       case 'l':
         longf=1;
	 break;       
       case 'x':
         mac_exclude=1;
	 break;       
       case 'm':
         mega=1;
	 break;       
       case 'e':
         print_speed=1;
	 break;
	                        
       // Options taking an integer argument:
       case 's':
       case 'p':
       case 'q':
       case 'i':
          { 
	    int n;
  	    if(sscanf(optarg,"%i",&n)!=1 ||
	      (c=='s' && (n<-5 || n>=(int)maxcounterc)) ||
	      (c=='p' && n<0) ||
	      (c=='q' && n<0) ||
	      (c=='i' && n<0)) {
	      fprintf(stderr,"Number has wrong format or wrong range\n");
	      return 1;
	    }
            switch(c) {
              case 's': sortorder=n; break;
              case 'p': period_len=n; break;
  	      case 'q': end_time=n; break;
	      case 'i': history_len=n; break;
	    }
	  }
	  break;
	  
       // Options taking other arguments:
       case 'o':
         outfile=optarg;
	 break;	  
       case 'd':
         ipcid=optarg;
	 break;	  
       case 't':
         htmltitle=optarg;
	 break;
       case 'f':
         macfile=optarg;
	 break;
	 
       // Error situations:
       case ':':
         fprintf(stderr,"Missing parameter for option\n");
	 return 1;
       case '?':
         return 1; // The getopt library prints an error message
	 break;	 
       default:
         // Should not happen, I guess
	 fprintf(stderr,"Error parsing commandline arguments\n");
	 return 1;
       
     }
  }
  
  // Find the colums to show:
  int print[maxcounterc+5];
  int printc=0; // Elements in print
  
  // Find out which counters to show:
  for(int a=optind;a<argc;a++) {
    int n;
    if(sscanf(argv[a],"%i",&n)==1 && n>=-5 && n<(int)maxcounterc) {
      if(printc==maxcounterc+5) {
        fprintf(stderr,"Too many counters\n");
        return 1;
      }
      print[printc++]=n;
    } else {
      fprintf(stderr,"Invalid counter specification: %s\n",argv[a]);
      return 1;
    }
  }
  if(printc==0) {
    fprintf(stderr,"At least one counter must be displayed\n");
    return 1;
  }

  // The parameters are read - check for consistence:  
  if(htmltitle && !html) {
    fprintf(stderr,"Can only specify a html title when writing a html file\n");
    return 1; 
  }

  if(end_time==-1) { // Der er ingen perioder
    if(period_len) {
      fprintf(stderr,"It is not possible to give a period length when the program is not\n"
                     "going to go through any periods\n");
      return 1;
    }
  } else
    if(!period_len) {
      fprintf(stderr,"Period length must be specified\n");
      return 1;
    }

  if(end_time==1 || end_time==-1)  // Make exactly one statictic
    if(history_len!=-1) {
      fprintf(stderr,"A time to quit can not be specified when there are no or less than two periods\n");
      return 1;
    }
    
  if(history_len!=-1 && period_len==-1) {
    fprintf(stderr,"Cannot specify a history lenght without setting a period length\n");
    return 1;
  }    
  
  if(!macfile && mac_exclude) {
    fprintf(stderr,"Can only exclude MAC adresses when there is some to exclude\n");
    return 1;
  }  
  
  // Read the file with MAC addresses:
  ReadMacFile ref;
  char* c=macfile ? ref.Read(macfile) : 0;
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 1;
  }
  
  // Connect to the server:
  ServerMemClient clnt;
  c=clnt.Init(ipcid,ref.GetMacac(),ref.GetMacas(),mac_exclude);
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 0;
  }
  
  // Make class to display output:
  OutFile* of=html ? 
    new HtmlFile(outfile,clearscr,htmltitle,period_len) : 
    new OutFile(outfile,clearscr);
        
  //--------------------------
  // And start!
  
  UserServerMem dst;
    
  if((end_time==1) || (end_time==-1)) {
    // Only print one statictic:
    
    if(end_time==1) {
      // But go through one period first:
      UserServerMem tmp;
      c=clnt.ReadArray(tmp); if(c) { fprintf(stderr,"%s\n",c); return 10; }
      sleep(period_len);
      c=clnt.ReadArray(dst); if(c) { fprintf(stderr,"%s\n",c); return 10; }
      dst-=tmp;
    } else {
      // Dont go through any periods - just print information since the server
      // start:
      c=clnt.ReadArray(dst); if(c) { fprintf(stderr,"%s\n",c); return 10; }
    }
    
    Print p(printc,print,print_speed,1,sortorder,sortrev,longf,mega);
    p.Prnt(dst,*of);
  } else {
    // We are going to make severel statictics. 
    // First read the current status:
    UserServerMem src;
    c=clnt.ReadArray(src); if(c) { fprintf(stderr,"%s\n",c); return 10; }
    
    if(history_len>0) { // Print with help of the history class
      History hist(history_len);
      for(;;) {
        hist.Consume(src,dst);
          
        Print p(printc,print,print_speed,1,sortorder,sortrev,longf,mega);
        p.Prnt(dst,*of);

        if(end_time!=0 && src.GetStop()>=end_time) break;
        sleep(period_len);
	
	c=clnt.ReadArray(src); if(c) { fprintf(stderr,"%s\n",c); return 10; }
      }
    } else { // Don't use the history class:
      dst=src;
      for(;;) {
        if(history_len==0) dst-=src;
	
        Print p(printc,print,print_speed,1,sortorder,sortrev,longf,mega); 
	p.Prnt(dst,*of);	

        if(end_time!=0 && dst.GetStop()>=end_time) break;
        sleep(period_len);
	
	c=clnt.ReadArray(dst); if(c) { fprintf(stderr,"%s\n",c); return 10; }
      } 
    }	
  }
  
  delete of;
  
  return 0;
}
